<?php
/**
 * Ruler Widget class
 *
 * @project     CWC2
 * @revision    $Id: Ruler.widget.php,v 1.4 2004/05/26 20:44:47 pspencer Exp $
 * @purpose     Ruler Widget class
 * @author      DM Solutions Group (assefa@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


include_once(dirname(__FILE__)."/../NavTool.php");


/**
 * Ruler
 *
 * @desc A widget to measure distances
 */
class Ruler extends NavTool
{
    var $mnNumberOfPoints = 100;
    var $mnSpaceBetweenPoints = 15;

    /**
     * construct a new ruler widget
     */
    function Ruler()
    {
        // invoke constructor of parent
        parent::NavTool();

        $this->NavTool();
        $this->SetNavCommand("RULER");
        $this->maAttributes["UNITS"] = new StringAttribute( "UNITS", true,
            array( "INCHES", "FEET", "MILES", "METERS", "KILOMETERS", "DEGREES", "PIXELS" ) );
        $this->maAttributes["NUMBEROFPOINTS"] = new IntegerAttribute( "NUMBEROFPOINTS", false, 0 );        $this->maAttributes["SPACEBETWEENPOINTS"] = new IntegerAttribute( "SPACEBETWEENPOINTS", false, 0 );

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The Ruler widget is a tool that allows the user to interactively drawn line segments on the map
and display the cumulative distance of the segments in ground units.
EOT;

    }

    /**
     * validate a unit of measurement
     */
    function IsUnitValid($szUnit)
    {
        $aValidUnits = array(trim($this->moMLT->get("6", "Inches")), 
                             trim($this->moMLT->get("7", "Feet")), 
                             trim($this->moMLT->get("8", "Miles")), 
                             trim($this->moMLT->get("9", "Meters")),
                             trim($this->moMLT->get("10", "Kilometers")), 
                             trim($this->moMLT->get("11", "Degrees")), 
                             trim($this->moMLT->get("12", "Pixels")));

        $nCount = count($aValidUnits);
        $bFound = false;

        for ($i=0; $i<$nCount; $i++)
        {
            if(strcasecmp($aValidUnits[$i], trim($szUnit)) == 0)
            {
                $bFound = true;
                break;
            }
        }
        return $bFound;
    }

    /**
     * return an abbreviation for a unit of measurement
     */
    function GetUnitAbbbreviation($szUnit)
    {
        //assuming the unit is valid
        //if ($this->IsUnitValid($szInUnit))
        $szUnit = trim(strtoupper($szUnit));

        $szResult = "";
        switch ($szUnit)
        {
            case "INCHES":
              $szResult = trim($this->moMLT->get("13", "in."));
              break;

            case "FEET":
              $szResult = trim($this->moMLT->get("14", "ft"));
              break;

            case "MILES":
              $szResult = trim($this->moMLT->get("15", "mi"));
              break;

            case "METERS":
              $szResult = trim($this->moMLT->get("16", "m"));
              break;

            case "KILOMETERS":
              $szResult = trim($this->moMLT->get("17", "km"));
              break;

            case "DEGREES":
              $szResult = trim($this->moMLT->get("18", "dd"));
              break;

            case "PIXELS":
              $szResult = trim($this->moMLT->get("19", "px"));
              break;

            default:
              break;
        }

        return $szResult;
    }

    /**
     * convert a measurement from a unit to meters
     */
    function ConvertToMeter($szInUnit, $dfValue)
    {
        //assuming the unit is valid
        //if ($this->IsUnitValid($szInUnit))
        $dfResult = -1.0;

        $aMeterPerUnit = array(); //how many meters are there in the unit
        $aMeterPerUnit[0] =  0.0254; //inches
        $aMeterPerUnit[1] =  0.3048; //feet
        $aMeterPerUnit[2] =  1609.3445; //Miles
        $aMeterPerUnit[3] =  1.0; //Meters
        $aMeterPerUnit[4] =  1000.0; //Kilometers
        $aMeterPerUnit[5] =  110570; //using value a 0 latitude 1degree = 110.57km


        $szInUnit = trim(strtoupper($szInUnit));
        switch ($szInUnit)
        {
            case "INCHES":
              $dfResult = $dfValue * $aMeterPerUnit[0];
              break;

            case "FEET":
               $dfResult = $dfValue * $aMeterPerUnit[1];
              break;

            case "MILES":
              $dfResult = $dfValue * $aMeterPerUnit[2];
              break;

            case "METERS":
              $dfResult = $dfValue * $aMeterPerUnit[3];
              break;

            case "KILOMETERS":
              $dfResult = $dfValue * $aMeterPerUnit[4];
              break;

            case "DEGREES":
              $dfResult = $dfValue * $aMeterPerUnit[5];
              break;

            default:
              break;
        }

        return $dfResult;

    }

    /**
     * convert a measurement from meters to a unit
     */
    function ConvertFromMeter($szInUnit, $dfValue)
    {
        //assuming the unit is valid
        //if ($this->IsUnitValid($szInUnit))
        $dfResult = -1.0;

        $aUnitPerMeter = array(); //how many units in one meters
        $aUnitPerMeter[0] =  39.37; //inches
        $aUnitPerMeter[1] =  3.2808; //feet
        $aUnitPerMeter[2] =  0.00062137; //Miles
        $aUnitPerMeter[3] =  1.0; //Meters
        $aUnitPerMeter[4] =  0.001; //Kilometers
        $aUnitPerMeter[5] =  0.000009044; //using value a 0 latitude 1degree = 110.57km

        $szInUnit = trim(strtoupper($szInUnit));
        switch ($szInUnit)
        {
            case "INCHES":
              $dfResult = $dfValue * $aUnitPerMeter[0];
              break;

            case "FEET":
               $dfResult = $dfValue * $aUnitPerMeter[1];
              break;

            case "MILES":
              $dfResult = $dfValue * $aUnitPerMeter[2];
              break;

            case "METERS":
              $dfResult = $dfValue * $aUnitPerMeter[3];
              break;

            case "KILOMETERS":
              $dfResult = $dfValue * $aUnitPerMeter[4];
              break;

            case "DEGREES":
              $dfResult = $dfValue * $aUnitPerMeter[5];
              break;

            default:
              break;
        }

        return $dfResult;

    }

    /**
     * arbitrarily convert a measurement from one unit to another
     */
    function ConvertUnit($szInUnit, $szOutUnit, $dfValue)
    {
        $bInFound = $this->IsUnitValid($szInUnit);
        $bOutFound = $this->IsUnitValid($szOutUnit);


        if (!$bInFound || !$bOutFound)
        {
            //echo "units invalid <br>";
            return -1.0;
        }

        $dfValue = $this->ConvertToMeter($szInUnit, $dfValue);
        //echo "after convert to meter = $dfValue <br>";
        $dfValue = $this->ConvertFromMeter($szOutUnit, $dfValue);
        //echo "after convert from meter = $dfValue <br>";

        return $dfValue;
    }

    /**
     * the javascript functions to be called as the page loads
     */
    function GetJavascriptInitFunctions()
    {
        $aReturn = array();

        $aReturn = parent::GetJavascriptInitFunctions();

        $szJsFunctionName = "RulerInit";
        $szFunction = "$szJsFunctionName();\n";
        $aReturn[$szJsFunctionName] = $szFunction;


        return $aReturn;
    }

    /**
     * javascript include files
     */
    function GetJavascriptIncludeFunctions()
    {
        $aReturn = array();
        
        $aReturn = parent::GetJavascriptIncludeFunctions();

        $szJsIncludeName = "cwc_dhtml.js";
        $szInclude = "<script language=\"JavaScript\" src=\"".$_SESSION["gszCoreWebPath"]."/widgets/js/cwc_dhtml.js\"></script>";
        $aReturn[$szJsIncludeName] = $szInclude;

        $szJsIncludeName = "dhtmlapi.js";
        $szInclude = "<script language=\"JavaScript\" src=\"".$_SESSION["gszCoreWebPath"]."/widgets/js/dhtmlapi.js\"></script>";
        $aReturn[$szJsIncludeName] = $szInclude;

        $szJsIncludeName = "toolsobj.js";
        $szInclude = "<script language=\"JavaScript\" src=\"".$_SESSION["gszCoreWebPath"]."/widgets/js/toolsobj.js\"></script>";
        $aReturn[$szJsIncludeName] = $szInclude;

        $szJsIncludeName = "eventslib.js";
        $szInclude = "<script language=\"JavaScript\" src=\"".$_SESSION["gszCoreWebPath"]."/widgets/js/eventslib.js\"></script>";
        $aReturn[$szJsIncludeName] = $szInclude;
        return $aReturn;
    }

    /**
     * @desc javascript mousemove functions
     */
    function GetJavascriptOnMouseMoveFunctions()
    {
        $aReturn = array();

        $aReturn = parent::GetJavascriptOnMouseMoveFunctions();

        $szJsFunctionName = "RulerWMouseMove";
        $szFunction = "$szJsFunctionName(e);\n";
        $aReturn[$szJsFunctionName] = $szFunction;

        return $aReturn;
    }

    /**
     * @desc general javascript functions
     */
    function GetJavascriptFunctions()
    {
        $oApp = GetChameleonApplication();
        $aReturn = array();

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        $aReturn = parent::GetJavascriptFunctions();
        
        $szRulerPix = $oApp->findFile( "images/ruler_pix.gif" );
        $szRulerPix = $oApp->fileSystemToURL( $szRulerPix );
        $szRulerNode = $oApp->findFile( "images/ruler_node.gif" );
        $szRulerNode = $oApp->fileSystemToURL( $szRulerNode );

        $szJsFunctionName = "RulerInit";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called to initialize the pan widget.
 */
function {$szJsFunctionName}()
{
    // draw all distance points
    //drawDistPt();
    var sId = "";
    var content = "<IMG  SRC='{$szRulerPix}'><BR>";
    var content2 = "<IMG  SRC='{$szRulerNode}'><BR>";
    for (var i=1; i<= gRulerNbPoints; i++)
    {
        sId = "pix"+i;
        CWCDHTML_CreateLayer(sId, 10, 10, 0, 0, false, content);
        CWCDHTML_SetLayerZOrder( sId, 20 );
        
        sId = "pInf"+i;
        CWCDHTML_CreateLayer(sId, 10, 10, 0, 0, false, content2);
        CWCDHTML_SetLayerZOrder( sId, 20 );
        
    }
    // instantiate the distancetool object
    //MapLayerDiv is created in MapDHTMLWidget.php.
    //gMapWhspc and gMapWvspc are created in MapDHTMLWidget and are
    //the TL position of the map.
    DistTool = new distancetool(getObject("MapLayerDiv"), gRulerNbPoints,
                                gMapWhspc, gMapWvspc);

    // initialize mouse event for the layer named 'MapLayerDiv'
    //setMapDraggable("MapLayerDiv");
}
EOT;

    $aReturn[$szJsFunctionName] = $szFunction;

    $szJsFunctionName = "RulerWMouseMove";
    $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the mouse moves for the Ruler widget.
 */
function {$szJsFunctionName}(e)
{

    //make sure that the forms varaibales are initialized.
    //It seems like that in IE, the mouse move may be called before.
    if ({$this->mszHTMLForm} == null ||
      {$this->mszHTMLForm}.NAV_CMD == null)
      return true;

    if({$this->mszHTMLForm}.NAV_CMD.value != "RULER")
    {
        //if ruler result is present rest the text field.
        if ({$this->mszHTMLForm}.RULER_RESULT != null)
        {
            {$this->mszHTMLForm}.RULER_RESULT.value = "";
        }

        //reset the first click.
        gRulerfirstClickDist = true;

        if (gRulerHideLayers)
        {
            for(var i=1; i<=gRulerNbPoints; i++)
            {
                hide("pix" + i);
            }
            for(var i=1; i<=gRulerNbPoints; i++)
            {
                hide("pInf" + i);
            }
        }
        document.onDblClick = null;
        document.onKeyPress= null;

        //if (isNav4)
        //{
        //   getObject("MapLayerDiv").onmouseout= null;
        //}

        if (isIE4 || isNav6)
        {
            //obj = getParObject("MapLayerDiv");
            obj = document.getElementById("MapLayerDiv");
            if (obj != null)
              obj.onmouseout= null;
        }
        return true;
    }
    if ({$this->mszHTMLForm}.NAV_CMD.value == "RULER" && !MapWinsideMap() && isNav4)
    {
        document.onmousedown = null;
        return true;
    }
    var obj = "MapLayerDiv";
    if (isNav4 || isNav6)
    {
        document.captureEvents(Event.MOUSEDOWN);
        document.captureEvents(Event.MOUSEMOVE);
        document.captureEvents(Event.DBLCLICK);
        document.captureEvents(Event.RESIZE);
    }
    if(isNav4)
    {
        //getObject(obj).onmouseover= mouseOnMap;
        getObject(obj).onmouseout= mouseOutMap;
        document.onmousedown=startDist;
        document.onDblClick=stopDist ;
        //document.addEventListener("dblclick",stopDist,true);
        document.onKeyPress=stopDist;
        //getObject(obj).onmousemove=distIt
    }
    else if (isIE4 || isNav6)
    {
        //var layer = getParObject(obj);
        var layer  = document.getElementById("MapLayerDiv");
        document.onmousedown=startDist;
        document.ondblclick=stopDist;
        document.onkeypress=stopDist;
        layer.onmouseout=outIE;
        //layer.onmousemove=distIt
        if (isNav6)
        {
          document.addEventListener("dblclick",stopDist,true);
          document.addEventListener("keypress",stopDist,true);
        }

    }
    //alert(gRulerFirstUse);
    //if (gRulerFirstUse)//first time the ruler tool is used
    //{
    //    DistTool = new distancetool(getObject("MapLayerDiv"), gRulerNbPoints,
    //                               gMapWhspc, gMapWvspc);
    //     gRulerFirstUse = false;
    //}

    distIt(e);

}
EOT;
    $aReturn[$szJsFunctionName] = $szFunction;

    if ($bCWCJSAPI)
    {

        $szJsFunctionName = "RulerWRegisterForEvent";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called to register and even when the map extents chnages (JSAPI)
 */
function {$szJsFunctionName}()
{
    goCWCJSAPI.RegisterEvent(MAP_EXTENT_CHANGED, "RulerWMapExtentsChanged");
}
EOT;
    
       $aReturn[$szJsFunctionName] = $szFunction;

       $szJsFunctionName = "RulerWMapExtentsChanged";
       $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the mapextents are changed to update the ruler global variables(JSAPI)
 */
function {$szJsFunctionName}()
{
    gRulercMapWidth = goCWCJSAPI.oMap.width;
    gRulercMapHeight = goCWCJSAPI.oMap.height;
    gRulermapMinX = goCWCJSAPI.oMap.minx;
    gRulermapMaxY = goCWCJSAPI.oMap.maxx;

    dfPixelSize = (goCWCJSAPI.oMap.maxx - goCWCJSAPI.oMap.minx)/goCWCJSAPI.oMap.width;

    szMapUnit = goCWCJSAPI.oMap.GetUnitString(goCWCJSAPI.oMap.units);

    gRulergblPixelSize = goCWCJSAPI.oMap.ConvertUnit(szMapUnit, gRulerUserUnit, dfPixelSize);

}
EOT;
    
    $aReturn[$szJsFunctionName] = $szFunction;
    }//if jsapi 
 
    return $aReturn;
    }


    /**
     * javascript variables
     */
    function GetJavascriptVariables()
    {
        if (isset($this->moMapObject))
        {
            $poMap = $this->moMapObject->oMap;

            $dfPixelSize = ($poMap->extent->maxx - $poMap->extent->minx)/
                          $poMap->width;
            $nUnits = $this->moMapObject->oMap->units;
        }
        else
        {
            if (!defined( "MS_INCHES" ))
            {
                define( "MS_INCHES", 0 );
                define( "MS_FEET", 1 );
                define( "MS_MILES", 2 );
                define( "MS_METERS", 3 );
                define( "MS_KILOMETERS", 4 );
                define( "MS_DEGREES", 5 );
                define( "MS_PIXELS", 6 );
                define( "MS_DD", 7 );
            }
            $nUnits = 3;
            $dfPixelSize = 1;
        }
        //echo "orig pix value = $dfPixelSize <br>";
        $szUserUnit = "";
        if (isset($this->maParams["UNITS"]))
            $szUserUnit = strtoupper($this->maParams["UNITS"]);

        $szMapUnits = "";
        switch ($nUnits)
        {
            case MS_INCHES:
              $szMapUnits = trim($this->moMLT->get("6", "Inches"));
              break;

            case MS_FEET:
               $szMapUnits = trim($this->moMLT->get("7", "Feet"));
              break;

            case MS_MILES:
              $szMapUnits = trim($this->moMLT->get("8", "Miles"));
              break;

            case MS_METERS:
              $szMapUnits = trim($this->moMLT->get("9", "Meters"));
              break;

            case MS_KILOMETERS:
              $szMapUnits = trim($this->moMLT->get("10", "Kilometers"));
              break;

            case MS_DD:
              $szMapUnits = trim($this->moMLT->get("11", "Degrees"));
              break;

            case MS_PIXELS:
              $szMapUnits = trim($this->moMLT->get("12", "Pixels"));
              break;
        }

/* -------------------------------------------------------------------- */
/*      if the user given units are diffrent fromn the map units,       */
/*      convert the pixel size.                                         */
/* -------------------------------------------------------------------- */
        if ($szMapUnits != "" && $szUserUnit != "" &&
            $szUserUnit != $szMapUnits)
        {
            //echo "convert from $szMapUnits to $szUserUnit <br>";
            $dfPixelSize =
              $this->ConvertUnit($szMapUnits, $szUserUnit,  $dfPixelSize);

            //echo "converted pix value = $dfPixelSize <br>";

            if ($dfPixelSize > 0)
              $szUnits = $szUserUnit;
            else
            {
                $_SESSION['gErrorManager']->setError(ERR_WARNING,
                  trim($this->moMLT->get("22", "ERROR: Unit value could not be used in Ruler.widget.php.")));
                $szUnits = trim($this->moMLT->get("12", "Pixels"));
                $dfPixelSize = 1.0;
            }
        }
        else
        {
            $szUnits = $szMapUnits;
        }

        $szUserUnit = $szUnits;
        $szUnits = $this->GetUnitAbbbreviation($szUnits);

        $aReturn = array();
        $aReturn = parent::GetJavascriptVariables();

        $szVariable = "gRulergblPixelSize";
        $szValue = " var $szVariable = $dfPixelSize;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulermapMinX";
        if (is_object($this->moMapObject))
        {
            $nMinX = $poMap->extent->minx;
        }
        else
            $nMinX = 0;
        $szValue = " var $szVariable = ".$nMinX.";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulermapMaxY";
        if (is_object($this->moMapObject))
        {
            $nMaxX = $poMap->extent->maxx;
        }
        else
            $nMaxX = 0;
        $szValue = " var $szVariable = ".$nMaxX.";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerMapImgOX";
        $szValue = " var $szVariable = gMapWhspc;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerMapImgOY";
        $szValue = " var $szVariable = gMapWvspc;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerCurrentTool";
        $szValue = " var $szVariable = \"distance\";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerfirstClickDist";
        $szValue = " var $szVariable = \"true\";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulercMapWidth";
        if (is_object($this->moMapObject))
        {
            $nWidth = $poMap->width;
        }
        else
            $nWidth = 1;
        $szValue = " var $szVariable = ".$nWidth.";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulercMapHeight";
        if (is_object($this->moMapObject))
        {
            $nHeight = $poMap->height;
        }
        else
            $nHeight = 1;
        $szValue = " var $szVariable = ".$nHeight.";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerg_digitTool";
        $szValue = " var $szVariable = \"true\";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerg_end_digit";
        $szValue = " var $szVariable = \"false\";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerareaMode";
        $szValue = " var $szVariable = \"false\";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerareaString";
        $szValue = " var $szVariable = \"\";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulersUnits";
        $szValue = " var $szVariable = \"$szUnits\";\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerHideLayers";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerFirstUse";
        $szValue = " var $szVariable = true;\n";
        $aReturn[$szVariable] = $szValue;

        $nPoints = $this->mnNumberOfPoints;
        if (isset($this->maParams["NUMBEROFPOINTS"]))
          $nPoints = intval($this->maParams["NUMBEROFPOINTS"]);

        $szVariable = "gRulerNbPoints";
        $szValue = " var $szVariable = $nPoints;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gRulerUserUnit";
        $szValue = " var $szVariable = \"$szUserUnit\";\n";
        $aReturn[$szVariable] = $szValue;
        
        $nSpaceBetween = $this->mnSpaceBetweenPoints;
        if (isset($this->maParams["SPACEBETWEENPOINTS"]))
          $nSpaceBetween = intval($this->maParams["SPACEBETWEENPOINTS"]);
        if ($nSpaceBetween <=0)
          $nSpaceBetween = $this->mnSpaceBetweenPoints;

        $szVariable = "gRulerSpaceBetweenPoints";
        $szValue = " var $szVariable = $nSpaceBetween;\n";
        $aReturn[$szVariable] = $szValue;

        return $aReturn;
    }
    
    /**
     * return an array of javascript functions needed by the Ruler widget
     * and called when the page is loaded. 
     * @return array of name = function values
     */
    function GetJavascriptOnLoadFunctions()
    {
        $aReturn = array();
        
        $aReturn = parent::GetJavascriptOnLoadFunctions();

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
        {
             $szJsFunctionName = "RulerWRegisterForEvent";
             $szFunction = "$szJsFunctionName();\n";
             $aReturn[$szJsFunctionName] = $szFunction;
        }

        return $aReturn;
    }
     
}
?>
