<?php
/**
 * ROIPolygonTool Widget class
 *
 * @project     CWC2
 * @revision    $Id: ROIPolygonTool.widget.php,v 1.6 2004/05/26 20:44:32 pspencer Exp $
 * @purpose     ROI Polygon Tool Widget class.  Allows the user to select
 * a circular region of interest on the map.  
 * @author      DM Solutions Group (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002-2003, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../Button.php");

/**
 * ROIPolygonTool
 *
 * @desc A widget to draw a polagonal Region of Interest
 */
class ROIPolygonTool extends CWCWidget
{
    /* the button that is the interface for this widget */
    var $moButton;
    var $mnNumPoints = 12;  // default number of points
    var $mnPointSpacing = 10; // default the point spacing;
    
    /**
     * construct a new ROI Polygon Tool widget
     */
    function ROIPolygonTool()
    {
        // set the language resource
        $this->mszLanguageResource = str_replace("\\","/",dirname(__FILE__))."/ROIPolygonTool.dbf";        
        
        // invoke constructor of parent
        parent::CWCWidget();
        
        // set the attributes
        $this->maAttributes["NUMBEROFPOINTS"] = new IntegerAttribute( "NUMBEROFPOINTS", false, 0 );
        $this->maAttributes["POINTSPACING"] = new IntegerAttribute( "POINTSPACING", false, 0 );
        
        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The ROIPolygonTool widget allows the user to draw a circular region
of interest on the map interface.
EOT;
        $this->moButton = new CWCButton( $this );
    }
    
    /**
     * 
     */
    function InitDefaults()
    {
        parent::InitDefaults();
        
        $this->moButton->InitDefaults();
        $this->moButton->SetOnClick('ROIPolygonTool'.$this->mnId);
    }
    
    /**
     * pass URL to button
     */
    function  ParseURL()
    {
        return $this->moButton->ParseURL();
    }

    function GetHTMLHiddenVariables()
    {
        $aReturn = $this->moButton->GetHTMLHiddenVariables();
                
        $szCmd = '';
        if ($this->isVarSet('NAV_CMD'))
            $szCmd = $this->getVar('NAV_CMD');
        $szVariable = "NAV_CMD";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        
        $szVariable = "ROI_TOOL_LOCKED";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"0\">\n";
        $aReturn[$szVariable] = $szValue;
        
        return $aReturn;
    }
    
    /**
     * GetJavascriptInitFunctions
     *
     * Functions to be called at the end of the load.
     */
    function GetJavascriptInitFunctions()
    {
        $aReturn = $this->moButton->GetJavascriptInitFunctions();

        $szJsFunctionName = "ROIPolygonToolInit";
        $szFunction = "$szJsFunctionName();\n";
        $aReturn[$szJsFunctionName] = $szFunction;

        $aReturn['InitCWCROIManagerForm'] = 'goCWCROIManager.SetFormObject('.$this->mszHTMLForm.');';
        
        return $aReturn;
    }

    /**
     * javascript include files
     */
    function GetJavascriptIncludeFunctions()
    {
        $aReturn = parent::GetJavascriptIncludeFunctions();
        
        $szVar = "cwc_roi.js";
        $aReturn[$szVar] = '<script src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_roi.js"></script>';
                            
        $szVar = "cwc_line.js";
        $aReturn[$szVar] = '<script src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_line.js"></script>';                            
        
        $szVar = "cwc_dhtml.js";
        $aReturn[$szVar] = '<script src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_dhtml.js"></script>';
        return $aReturn;
    }

    /**
     * @desc javascript mousemove functions
     */
    function GetJavascriptOnMouseMoveFunctions()
    {
        $aReturn = parent::GetJavascriptOnMouseMoveFunctions();
        
        $aReturn['ROIPolygonTool'] = 'ROIPolygonToolMouseMove(e);';

        return $aReturn;
    }

    /**
     * @desc general javascript functions
     */
    function GetJavascriptFunctions()
    {
        $aReturn = parent::GetJavascriptFunctions();

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;
          
        // get the chameleon images url
        $szImagesURL = trim(str_replace("\\", '/', $_SESSION['gszCoreWebPath']));
        $szImagesURL = substr( $szImagesURL, -1 ) == '/' ? $szImagesURL.'images/' : 
                                                            $szImagesURL.'/images/';          
          
        // determine the number of points to use
        if ( isset($this->maParams["NUMBEROFPOINTS"] ) )
            $nPoints = intval( $this->maParams["NUMBEROFPOINTS"] );
        else
            $nPoints = $this->mnNumPoints;
            
        // add JSAPI mode functions  
        if ($bCWCJSAPI)
        {
            $szJsFunctionName = "ROIPolygonToolWRegisterForEvent";
            $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called to register events when the layers change (JSAPI)
 */
function {$szJsFunctionName}()
{
    goCWCJSAPI.RegisterEvent(MAP_EXTENT_CHANGED, "ROIPolygonToolWExtentChanged");
    goCWCJSAPI.RegisterEvent(MAP_PROJECTION_CHANGED, "ROIPolygonToolWExtentChanged");
}
EOT;

           $aReturn[$szJsFunctionName] = $szFunction;
           $szJsFunctionName = "ROIPolygonToolWExtentChanged";
           $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the extent is changed (JSAPI)
 */
function {$szJsFunctionName}()
{
    // re-draw the ROI
    goCWCROIManager.SetGeoExtents( goCWCJSAPI.oMap.minx, goCWCJSAPI.oMap.miny, 
                                    goCWCJSAPI.oMap.maxx, goCWCJSAPI.oMap.maxy );
    goCWCROIManager.Redraw();
}
EOT;
            $aReturn[$szJsFunctionName] = $szFunction;

        }
        
        $nId = $this->mnId;
        $szName = 'ROIPolygonTool'.$nId;
        $szFunction = <<<EOT
        function {$szName}()
        {
            goCWCROIManager.SetROITool('{$nId}');
        }
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szName = 'ROIPolygonToolInit';
        $szFunction = <<<EOT
function {$szName}()
{
    // create new layers to use with line drawing
    var sId = "";
    gaszPolygonDivPointNames = new Array();
    gaszPolygonDivNodeNames = new Array();
    var content = "<img  src='{$szImagesURL}ruler_pix.gif' width='6' height='6'><br>";
    var content2 = "<img  src='{$szImagesURL}ruler_node.gif' width='7' height='7'><br>";
    for (var i=1; i < {$nPoints}; i++)
    {
        // create the point layers
        sId = 'polygonPointLayer'+i;
        CWCDHTML_CreateLayer(sId, 10, 10, 0, 0, false, content);
        CWCDHTML_SetLayerZOrder( sId, 20 );
        gaszPolygonDivPointNames.push( sId );

        // create the node layers
        sId = 'polygonNodeLayer'+i;
        CWCDHTML_CreateLayer(sId, 10, 10, 0, 0, false, content2);
        CWCDHTML_SetLayerZOrder( sId, 20 );
        gaszPolygonDivNodeNames.push( sId );
    }
    
    // init polygon object
    goROIPolygonObj = new CWCPolygonROI({$nPoints});
    goROIPolygonObj.pointImageWidth = 6;
    goROIPolygonObj.nodeImageWidth = 7;
    goROIPolygonObj.left = gMapWhspc;
    goROIPolygonObj.top = gMapWvspc;
    goROIPolygonObj.right = gMapWhspc + gMapWiWidth;
    goROIPolygonObj.bottom = gMapWvspc + gMapWiHeight;
    goROIPolygonObj.edgeColor = goCWCROIManager.szEdgeColor;
    goROIPolygonObj.edgeWidth = goCWCROIManager.nEdgeWidth;
    goROIPolygonObj.fillLayer = goCWCROIManager.bUseFill;
    goROIPolygonObj.fillColor = goCWCROIManager.szFillColor;
    goROIPolygonObj.fillOpacity = goCWCROIManager.nFillOpacity;  
    
    // show the polygon
    goROIPolygonObj.Show();    
    
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szObjectJS = "    var jsObj;\n";
        
        /**
         * TODO: this does not reconstitute the points associated with the
         * ROI tool because this is complicated ;)  In reality, the points may
         * not be at valid locations.  The only reason to reconstitute the points
         * would be to enable editing, which can't happen
         */
        if (isset($_SESSION['ROIRENDERER']))
        {
            $nROI = count($_SESSION['ROIRENDERER']);
            for ($i=0; $i<$nROI; $i++)
            {
                $aROI = $_SESSION['ROIRENDERER'][$i];
                if ($aROI['type'] == 'polygon')
                {
                    $szObjectJS .= "    goCWCROIManager.SetMode(".
                                   $aROI['mode'] . ");\n";
                    $szObjectJS .= "    jsObj = new CWCPolygonROI();\n";
                    $szObjectJS .= "    jsObj.edgeColor = \"".$aROI['edgecolor']."\";\n";
                    $szObjectJS .= "    jsObj.edgeWidth = ".$aROI['edgewidth'].";\n";
                    $szObjectJS .= "    jsObj.fillLayer = goCWCROIManager.bUseFill;\n";
                    $szObjectJS .= "    jsObj.fillColor = \"".$aROI['fillcolor']."\";\n";
                    $szObjectJS .= "    jsObj.fillOpacity = ".$aROI['fillopacity'].";\n";
                    $szObjectJS .= "    goCWCROIManager.Insert( jsObj, $i );\n";
                }
            }
            if (strlen( $szObjectJS ) > 0)
            {
                $szObjectJS .= "    goCWCROIManager.UpdateLayerVisibility();\n";
            }
        }
        
        
        $szName = 'ROIPolygonToolInitObjects';
        $szFunction = <<<EOT
function {$szName}()
{
{$szObjectJS}
}
EOT;
        $aReturn[$szName] = $szFunction;

        $szName = 'ROIPolygonToolDrawFloater';
        $szFunction = <<<EOT
function {$szName}()
{
        // set or default the start points
        var nTmpStartX = 0;
        var nTmpStartY = 0;
        nTmpNodeSpacing = 10;
        if ( goROIPolygonObj.aNodeCoords[0] )
        {
            nTmpStartX = goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][0];
            nTmpStartY = goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][1];
            nTmpNodeSpacing = gnROIPolygonPointSpacing;
        }
        
        // delete old floater
        goFloaterLine.deleteLine();

        // create and setup the new floater line
        goFloaterLine = new Line( nTmpStartX, nTmpStartY, 
            gROIToolmouseX, gROIToolmouseY, false );
        goFloaterLine.pointSpacing = nTmpNodeSpacing;
        goFloaterLine.layerNodeList = gaszPolygonDivNodeNames;
        goFloaterLine.layerPointList = gaszPolygonDivPointNames;
        
        // limit points to 3
        goFloaterLine.numPoints = 3
        
        // draw the line
        goFloaterLine.drawLine();
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        // get translated text for this section
        $szTmpNotice = $this->moMLT->get( '2', 'Press `esc` to cancel or any other key to close the polygon.' );
        
        $szName = 'ROIPolygonToolMouseDown';
        $szFunction = <<<EOT
function {$szName}(e)
{
    // skip if not on map or locked
    if ( !MapWinsideMap() || {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value == 1 )
    {
        return true;
    }
          
    ROIPolygonToolGetMousePosition(e);

    if (gROIToolmouseX >=0 && gROIToolmouseY >=0)
    {
        gROIPolygonToolx1 = gROIToolmouseX;
        gROIPolygonTooly1 = gROIToolmouseY;
        
        // flag that the polygon is open
        gROIPolygonOpen = true;
        
        // draw next node
        ROIPolygonDrawNode();
    } 
    
    // set status message
    //window.status='{$szTmpNotice}';
    
    return false;
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szName = 'ROIPolygonToolMouseUp';
        $szFunction = <<<EOT
function {$szName}(e)
{
    return null;
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szName = 'ROIPolygonToolGetMousePosition';
        $szFunction = <<<EOT
function {$szName}(e)
{
    if (CWCIsNav4 || CWCIsNav6) // Netscape
    {                
        gROIToolmouseX = e.pageX;
        gROIToolmouseY = e.pageY;
    } 
    else if (CWCIsIE) // IE
    {            
        gROIToolmouseX = event.clientX + document.body.scrollLeft;
        gROIToolmouseY = event.clientY + document.body.scrollTop;
    } 
    else 
    {                              // Don't know
        gROIToolmouseX = gROIToolmouseY = 0;
    }
    return true;
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szName = 'ROIPolygonToolMouseMove';
        $szFunction = <<<EOT
function {$szName}(e)
{
    ROIPolygonToolGetMousePosition(e);
    
    if (gROIToolmouseX >=0 && gROIToolmouseY >=0 &&
        {$this->mszHTMLForm}.NAV_CMD != null &&
        {$this->mszHTMLForm}.NAV_CMD.value == "ROI_TOOL" &&
        {$this->mnId} == goCWCROIManager.GetROITool() )
    {
        // capture events
        if ( CWCIsNav4 || CWCIsNav6 )
        {
            document.captureEvents(Event.MOUSEDOWN);
            document.captureEvents(Event.MOUSEMOVE);
            document.captureEvents(Event.RESIZE);
        }
        if( CWCIsNav4 )
        {
            document.onmousedown = ROIPolygonToolMouseDown;
            document.onmouseup = ROIPolygonToolMouseUp;
            document.onKeyPress = ROIPolygonToolKeyPress;
        }
        else if ( CWCIsIE || CWCIsNav6 )
        {
            var layer  = document.getElementById("MapLayerDiv");
            document.onmousedown = ROIPolygonToolMouseDown;
            document.onmouseup = ROIPolygonToolMouseUp;            
            document.onkeypress = ROIPolygonToolKeyPress;
            
            if ( CWCIsNav6 )
              document.addEventListener("keypress", ROIPolygonToolKeyPress, true);
        }

        if ( gROIPolygonOpen )
        {
            // update end point of current line
            gROIPolygonToolx2 = gROIToolmouseX;
            gROIPolygonTooly2 = gROIToolmouseY;
            
            // continuously update the floater line
            ROIPolygonToolDrawFloater();
        }    
    }

    return false;
}
EOT;
        $aReturn[$szName] = $szFunction;

        // get translated text for this function
        $szTmpNotice1 = $this->moMLT->get( '0', 'A polygon may not overlap itself.  The side you are attempting to draw intersects with an existing side of the polygon.' );
        $szTmpNotice2 = $this->moMLT->get( '1', 'Unable to close polygon.  Closing the polygon would cause the final side to intersect with an existing side of the polygon.  Click `OK` and either add another line segment so that the polygon will not overlap itself or use the escape key to cancel.' );

        $szName = 'ROIPolygonDrawNode';
        $szFunction = <<<EOT
function {$szName}()
{  
    // calculate coordinates
    var nPreviousX = gROIPolygonToolx1;
    var nPreviousY = gROIPolygonTooly1;
    var bDefaulted = 1;
    var nNumNodes = 0;
    if ( goROIPolygonObj.aNodeCoords )
        nNumNodes = goROIPolygonObj.aNodeCoords.length;
        
    if ( nNumNodes > 0 )
    {
        nPreviousX = goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][0];
        nPreviousY = goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][1];
        bDefaulted = 0;
    }
    
    // skip if coordinates match (i.e. double event triggered)
    if ( nPreviousX == gROIPolygonToolx1 && nPreviousY == gROIPolygonTooly1 && bDefaulted == 0 )
        return null;
        
    // loop through array of lines an check for overlap
    if ( {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value != 1 && gaROIPolygonObj )
    {
        var oTmpLine = new Line( nPreviousX, nPreviousY, gROIPolygonToolx1, gROIPolygonTooly1, false );
        for ( var i=0; i<gaROIPolygonObj.length; i++ )
        {
            // check for intersection
            if ( oTmpLine.intersectsWith( gaROIPolygonObj[i] ) )
            {
                // give message
                alert('{$szTmpNotice1}');
                
                // lock the tool on a timer
                {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value = 1;
                setTimeout( "ROIPolygonToolUnlockTool()", 500 );            
                
                // exit function
                return null;
            }
        }
    }
    
    // delete floater
    goFloaterLine.deleteLine();    
    
    // draw line
    ROIPolygonToolDrawLine( nPreviousX, nPreviousY, gROIPolygonToolx1, gROIPolygonTooly1 );   
    
    // return 
    return null;
}
EOT;
        $aReturn[$szName] = $szFunction;        
        
        $szName = 'ROIPolygonToolUnlockTool';
        $szFunction = <<<EOT
function {$szName}()
{
    // reset the lock
    {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value = 0;
    return null;
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szName = 'ROIPolygonToolKeyPress';
        $szFunction = <<<EOT
function {$szName}(e)
{
    // only process if polygon tool is selected
    if ( {$this->mszHTMLForm}.NAV_CMD != null &&
        {$this->mszHTMLForm}.NAV_CMD.value == "ROI_TOOL" &&
        {$this->mnId} == goCWCROIManager.GetROITool() && 
        {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value != 1 )
    {
        // delete floater
        goFloaterLine.deleteLine();
    
        // check which key was pressed
        var nKey;
        
        // capture based on browser type
        if ( CWCIsIE )
            nKey = event.keyCode;
        else
            nKey = e.keyCode;
        
        switch ( nKey )
        {
            case 27:
                // esc was pressed so abort        
                ROIPolygonToolClearPolygon();
                break;
                
            default:
                // close polygon
                ROIPolygonToolClosePolygon();
        }
    }

    // return   
    return null;
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szName = 'ROIPolygonToolClosePolygon';
        $szFunction = <<<EOT
function {$szName}()
{
    // loop through array of lines an check for overlap
    if ( {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value != 1 )
    {
        var oTmpLine = new Line(  goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][0], 
                                goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][1], 
                                goROIPolygonObj.aNodeCoords[0][0], 
                                goROIPolygonObj.aNodeCoords[0][1], false );
        for ( var i=0; i<gaROIPolygonObj.length; i++ )
        {
            // check for intersection
            if ( oTmpLine.intersectsWith( gaROIPolygonObj[i] ) )
            {
                // give message
                alert('{$szTmpNotice2}');
                
                // lock the tool on a timer
                {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value = 1;
                setTimeout( "ROIPolygonToolUnlockTool()", 500 );            
                
                // exit function
                return null;
            }
        }
    }

    // close polygon
    gROIPolygonOpen = false;
    
    // clear statusbar
    //window.status='';

    // draw line
    ROIPolygonToolDrawLine( goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][0], 
                            goROIPolygonObj.aNodeCoords[goROIPolygonObj.aNodeCoords.length-1][1], 
                            goROIPolygonObj.aNodeCoords[0][0], 
                            goROIPolygonObj.aNodeCoords[0][1], false );
    // add aoi
    goCWCROIManager.Add( goROIPolygonObj );

    // wait to hide the points (need to see the polygon close)
    setTimeout( 'ROIPolygonToolClearPolygon()', 800 );
    
    // lock the tool on a timer
    {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value = 1;
    setTimeout( "ROIPolygonToolUnlockTool()", 500 );
    
    // return
    return true;    
}
EOT;
        $aReturn[$szName] = $szFunction;        

        $szName = 'ROIPolygonToolDrawLine';
        $szFunction = <<<EOT
function {$szName}( x1, y1, x2, y2, bDrawStartNode )
{ 
    // draw a new line if necessary
    var nLineCount = 0;
    if ( goROIPolygonObj.aNodeCoords )
        nLineCount = goROIPolygonObj.aNodeCoords.length;
        
    if ( nLineCount == 0 )
    {
        var oLine = new Line( x1, y1, x2, y2, bDrawStartNode ); 
        oLine.layerNodeList = gaszPolygonDivNodeNames;
        oLine.layerPointList = gaszPolygonDivPointNames;
        
        // ensure that no nodes are drawn
        oLine.numPoints = 0;
                              
        // draw the line
        oLine.drawLine();
        
        // add to array of line objects
        gaROIPolygonObj.push( oLine );        
    }
    else if ( nLineCount > 0 )
    {
        var oLine = new Line( x1, y1, x2, y2, bDrawStartNode );
        oLine.pointSpacing = gnROIPolygonPointSpacing;
        oLine.layerNodeList = gaszPolygonDivNodeNames;
        oLine.layerPointList = gaszPolygonDivPointNames;
        
        // draw the line
        oLine.drawLine();
        
        // add to array of line objects
        gaROIPolygonObj.push( oLine );
    }
    
    // check if no more nodes then redraw all lines using double spacing
    if ( gaszPolygonDivPointNames.length == 0 )
    {
        // update global point spacing
        gnROIPolygonPointSpacing = gnROIPolygonPointSpacing * 2;
        
        // loop through the list lines, delete, and recreate
        nLineCount = gaROIPolygonObj.length;       
        for ( var i=0; i<nLineCount; i++ )
        {
            gaROIPolygonObj[i].deleteLine();
            gaROIPolygonObj[i].pointSpacing = gnROIPolygonPointSpacing;
            gaROIPolygonObj[i].drawLine();
        }
    }
    
    // record node click if polygon is open
    if ( gROIPolygonOpen )
    {
        var aTmpArray = new Array( gROIPolygonToolx1, gROIPolygonTooly1 );
        goROIPolygonObj.aNodeCoords.push( aTmpArray );
    }
}
EOT;
        $aReturn[$szName] = $szFunction;        
           
        $szName = 'ROIPolygonToolClearPolygon';
        $szFunction = <<<EOT
function {$szName}()
{
    // delete all lines
    var nLineCount = gaROIPolygonObj.length;       
    for ( var i=0; i<nLineCount; i++ )
        gaROIPolygonObj[i].deleteLine();
    
    // reset
    goROIPolygonObj.aNodeCoords = new Array();
    gaROIPolygonObj = new Array();
    gROIPolygonOpen = false;
    //window.status='';
    gnROIPolygonPointSpacing = gnInitialROIPolygonPointSpacing;
    return true;    
}
EOT;
        $aReturn[$szName] = $szFunction;        
        
        $szName = 'ROIPolygonToolUnlockTool';
        $szFunction = <<<EOT
function {$szName}()
{
    // reset the lock
    {$this->mszHTMLForm}.ROI_TOOL_LOCKED.value = 0;
    return null;
}
EOT;
        $aReturn[$szName] = $szFunction;        
        
        return $aReturn;
    }

    /**
     * GetJavascriptVariables
     *
     * Return JS global variables and global code.
     */
    function GetJavascriptVariables()
    {
        $aReturn = array();
        
        $szVariable = "gROIToolmouseX";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gROIToolmouseY";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gROIPolygonToolx1";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gROIPolygonTooly1";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gROIPolygonToolx2";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gROIPolygonTooly2";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gROIPolygonOpen";
        $szValue = " var $szVariable = false;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "goROIPolygonObj";
        $szValue = " var $szVariable = null;\n";
        $aReturn[$szVariable] = $szValue;
        
        $szVariable = "gaROIPolygonObj";
        $szValue = " var $szVariable = new Array();\n";
        $aReturn[$szVariable] = $szValue; 
        
        $szVariable = "gaszPolygonDivPointNames";
        $szValue = " var $szVariable;\n";
        $aReturn[$szVariable] = $szValue; 
        
        $szVariable = "gaszPolygonDivNodeNames";
        $szValue = " var $szVariable;\n";
        $aReturn[$szVariable] = $szValue; 
        
        $szVariable = "goFloaterLine";
        $szValue = " var $szVariable = new Line( 0, 0, 0, 0, false ); ;\n";
        $aReturn[$szVariable] = $szValue;         
        
        // determine point spacing
        if ( isset($this->maParams["POINTSPACING"] ) )
            $nPointSpacing = intval( $this->maParams["POINTSPACING"] );
        else
            $nPointSpacing = $this->mnPointSpacing;
        $szVariable = "gnROIPolygonPointSpacing";
        $szValue = " var $szVariable = $nPointSpacing;\n";
        $aReturn[$szVariable] = $szValue;            
            
        $szVariable = "gnInitialROIPolygonPointSpacing";
        $szValue = " var $szVariable = $nPointSpacing;\n";
        $aReturn[$szVariable] = $szValue;
        
        return $aReturn;
    }
    
    /**
     * return an array of javascript functions needed by the Ruler widget
     * and called when the page is loaded. 
     * @return array of name = function values
     */
    function GetJavascriptOnLoadFunctions()
    {
        $aReturn = parent::GetJavascriptOnLoadFunctions();

        $aReturn["ROIPolygonToolInitObjects"] = "    ROIPolygonToolInitObjects();\n";
        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
        {
            $szJsFunctionName = "ROIPolygonToolWRegisterForEvent";
            $szFunction = "$szJsFunctionName();\n";
            $aReturn[$szJsFunctionName] = $szFunction;
        }          

        return $aReturn;
    }

    /**
     * draw this widget on the page
     */
    function DrawPublish($szPostEvent = "")
    {
        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        if (!$this->mbVisible)
            return "<!-- ROIPolygonTool hidden -->";

        $szReturn = $this->moButton->DrawPublish();

        return $szReturn;
    }
}
?>
