<?php
/**
 * ROIFeatureTool Widget class
 *
 * @project     CWC2
 * @revision    $Id: ROIFeatureTool.widget.php,v 1.4 2004/05/19 21:19:14 pspencer Exp $
 * @purpose     ROI Feature Tool Widget class.  Allows the user to select
 * a region of interest from a feature in an existing layer.  
 * @author      DM Solutions Group (spencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../Button.php");

/**
 * ROIFeatureTool
 *
 * @desc A widget to draw a Region of Interest from an existing feature.  User
 * clicks on the map and the selected layer is queried by point to select a feature
 * then the feature is used to create an ROI.
 *
 * Only works on polygon layers
 */
class ROIFeatureTool extends CWCWidget
{
    /* the button that is the interface for this widget */
    var $moButton;
    
    /**
     * construct a new ROI Feature Tool widget
     */
    function ROIFeatureTool()
    {
        // invoke constructor of parent
        parent::CWCWidget();
        
        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The v widget allows the user to draw a region
of interest on the map interface by selecting an existing
feature from a polygon layer.
EOT;
        $this->moButton = new CWCButton( $this );
    }
    
    /**
     * 
     */
    function InitDefaults()
    {
        parent::InitDefaults();
        
        $this->moButton->InitDefaults();
        $this->moButton->SetOnClick('ROIFeatureTool'.$this->mnId);
    }

    /**
     * pass URL to button
     */
    function  ParseURL()
    {
        return $this->moButton->ParseURL();
    }

    function GetHTMLHiddenVariables()
    {
        $aReturn = $this->moButton->GetHTMLHiddenVariables();
                
        $szCmd = '';
        if ($this->isVarSet('NAV_CMD'))
            $szCmd = $this->getVar('NAV_CMD');
        $szVariable = "NAV_CMD";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        
        return $aReturn;
    }
    
    /**
     * GetJavascriptInitFunctions
     *
     * Functions to be called at the end of the load.
     */
    function GetJavascriptInitFunctions()
    {
        $aReturn = $this->moButton->GetJavascriptInitFunctions();

        /*
        $szJsFunctionName = "ROIFeatureToolInit";
        $szFunction = "$szJsFunctionName();\n";
        $aReturn[$szJsFunctionName] = $szFunction;
        */
        $aReturn['InitCWCROIManagerForm'] = 'goCWCROIManager.SetFormObject('.$this->mszHTMLForm.');';
        
        return $aReturn;
    }

    /**
     * javascript include files
     */
    function GetJavascriptIncludeFunctions()
    {
        $aReturn = parent::GetJavascriptIncludeFunctions();
        
        $szVar = "cwc_roi.js";
        $aReturn[$szVar] = '<script src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_roi.js"></script>';
        
        $szVar = "cwc_dhtml.js";
        $aReturn[$szVar] = '<script src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_dhtml.js"></script>';
        return $aReturn;
    }

    /**
     * @desc javascript mousemove functions
     */
    function GetJavascriptOnMouseMoveFunctions()
    {
        $aReturn = parent::GetJavascriptOnMouseMoveFunctions();
        
        $aReturn['ROIFeatureTool'] = 'ROIFeatureToolMouseMove(e);';

        return $aReturn;
    }

    /**
     * @desc general javascript functions
     */
    function GetJavascriptFunctions()
    {
        $aReturn = parent::GetJavascriptFunctions();

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;
          
        // add JSAPI mode functions  
        if ($bCWCJSAPI)
        {
            $szJsFunctionName = "ROIFeatureToolWRegisterForEvent";
            $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called to register events when the layers change (JSAPI)
 */
function {$szJsFunctionName}()
{
    goCWCJSAPI.RegisterEvent(MAP_EXTENT_CHANGED, "ROIFeatureToolWExtentChanged");
    goCWCJSAPI.RegisterEvent(MAP_PROJECTION_CHANGED, "ROIFeatureToolWExtentChanged");
}
EOT;

           $aReturn[$szJsFunctionName] = $szFunction;
           $szJsFunctionName = "ROIFeatureToolWExtentChanged";
           $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the extent is changed (JSAPI)
 */
function {$szJsFunctionName}()
{
    // re-draw the ROI
    goCWCROIManager.SetGeoExtents( goCWCJSAPI.oMap.minx, goCWCJSAPI.oMap.miny, 
                                    goCWCJSAPI.oMap.maxx, goCWCJSAPI.oMap.maxy );
    goCWCROIManager.Redraw();
}
EOT;
            $aReturn[$szJsFunctionName] = $szFunction;

        }
        
        $nId = $this->mnId;
        $szName = 'ROIFeatureTool'.$nId;
        $szFunction = <<<EOT
function {$szName}()
{
    goCWCROIManager.SetROITool('{$nId}');
}
EOT;
        $aReturn[$szName] = $szFunction;

        $szName = 'ROIRectToolInit';
        $szFunction = <<<EOT
function {$szName}()
{

    //ROIRectToolCreateDHTMLLayers();
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        
        /**
         * TODO: this does not reconstitute the points associated with the
         * ROI tool because this is complicated ;)  In reality, the points may
         * not be at valid locations.  The only reason to reconstitute the points
         * would be to enable editing, which can't happen
         */
        $szObjectJS = "";
         if (isset($_SESSION['ROIRENDERER']))
        {
            $nROI = count($_SESSION['ROIRENDERER']);
            for ($i=0; $i<$nROI; $i++)
            {
                $aROI = $_SESSION['ROIRENDERER'][$i];
                if ($aROI['type'] == 'feature')
                {
                    $szObjectJS = "    var jsObj;\n";
                    $szObjectJS .= "    goCWCROIManager.SetMode(".
                                   $aROI['mode'] . ");\n";
                    $szObjectJS .= "    jsObj = new CWCFeatureROI();\n";
                    $szObjectJS .= "    jsObj.edgeColor = \"".$aROI['edgecolor']."\";\n";
                    $szObjectJS .= "    jsObj.edgeWidth = ".$aROI['edgewidth'].";\n";
                    $szObjectJS .= "    jsObj.fillLayer = goCWCROIManager.bUseFill;\n";
                    $szObjectJS .= "    jsObj.fillColor = \"".$aROI['fillcolor']."\";\n";
                    $szObjectJS .= "    jsObj.fillOpacity = ".$aROI['fillopacity'].";\n";
                    $szObjectJS .= "    jsObj.szLayer = \"".$aROI['selectedLayer']."\";\n";
                    $szObjectJS .= "    jsObj.geoX = ".$aROI['aOrigCoords'][0].";\n";
                    $szObjectJS .= "    jsObj.geoY = ".$aROI['aOrigCoords'][1].";\n";
                    $szObjectJS .= "    goCWCROIManager.Insert( jsObj, $i );\n";
                }
            }
            if (strlen( $szObjectJS ) > 0)
            {
                $szObjectJS .= "    goCWCROIManager.UpdateLayerVisibility();\n";
            }
        }
        
        
        $szName = 'ROIFeatureToolInitObjects';
        $szFunction = <<<EOT
function {$szName}()
{
{$szObjectJS}
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        
        $szName = 'ROIFeatureToolMouseUp';
        $szFunction = <<<EOT
function {$szName}(e)
{
    ROIFeatureToolGetMousePosition(e);

    if (gROIToolmouseX >=0 && gROIToolmouseY >=0)
    {
        if ({$this->mszHTMLForm}.SELECTED_LAYERS != null &&
            {$this->mszHTMLForm}.SELECTED_LAYERS.value != "")
        {
            var szLayer = {$this->mszHTMLForm}.SELECTED_LAYERS.value;
            var oROIFeatureObj = new CWCFeatureROI();
            oROIFeatureObj.edgeColor = goCWCROIManager.szEdgeColor;
            oROIFeatureObj.edgeWidth = goCWCROIManager.nEdgeWidth;
            oROIFeatureObj.fillLayer = goCWCROIManager.bUseFill;
            oROIFeatureObj.fillColor = goCWCROIManager.szFillColor;
            oROIFeatureObj.fillOpacity = goCWCROIManager.nFillOpacity;
            oROIFeatureObj.Set( gROIToolmouseX, gROIToolmouseY, szLayer );
            goCWCROIManager.Add( oROIFeatureObj );
            oROIFeatureObj = null;
                    
        }
        else
        {
            alert( "you must select a layer before using the ROI Feature Tool." );
        }
        
    } 
    return false;
}
EOT;
        $aReturn[$szName] = $szFunction;
        
        $szName = 'ROIFeatureToolGetMousePosition';
        $szFunction = <<<EOT
function {$szName}(e)
{
    if (CWCIsNav4 || CWCIsNav6) // Netscape
    {                
        gROIToolmouseX = e.pageX;
        gROIToolmouseY = e.pageY;
    } 
    else if (CWCIsIE) // IE
    {            
        gROIToolmouseX = event.clientX + document.body.scrollLeft;
        gROIToolmouseY = event.clientY + document.body.scrollTop;
    } 
    else 
    {                              // Don't know
        gROIToolmouseX = gROIToolmouseY = 0;
    }
    var oImage = CWCDHTML_GetImage('mapimage');
    
    if (oImage == null)
    {
        gROIToolmouseX = -1;
        gROIToolmouseY = -1
        return false;
    }
    var left = CWCDHTML_FindObjectPosX( oImage );
    var top = CWCDHTML_FindObjectPosY( oImage );
    
    var width = oImage.width;
    var height = oImage.height;
    
    // subtract offsets from page left and top
    gROIToolmouseX = gROIToolmouseX - left;
    gROIToolmouseY = gROIToolmouseY - top;
    
    if (gROIToolmouseX < 0 || gROIToolmouseX > width ||
        gROIToolmouseY < 0 || gROIToolmouseY > height)
    {
        gROIToolmouseX = -1;
        gROIToolmouseY = -1
    } 
    return true;
}
EOT;
        $aReturn[$szName] = $szFunction;
        $szName = 'ROIFeatureToolMouseMove';
        $szFunction = <<<EOT
function {$szName}(e)
{
    ROIFeatureToolGetMousePosition(e);
    
    if (gROIToolmouseX >=0 && gROIToolmouseY >=0 &&
        {$this->mszHTMLForm}.NAV_CMD != null &&
        {$this->mszHTMLForm}.NAV_CMD.value == "ROI_TOOL" &&
        {$this->mnId} == goCWCROIManager.GetROITool())
    {
        document.onmousedown = null;
        document.onmouseup = ROIFeatureToolMouseUp;   
    }
    else
    {
        if (document.onmouseup == ROIFeatureToolMouseUp)
            document.onmouseup = null;
    }
    return false;
}
EOT;
        $aReturn[$szName] = $szFunction;

        return $aReturn;
    }

    /**
     * GetJavascriptVariables
     *
     * Return JS global variables and global code.
     */
    function GetJavascriptVariables()
    {
        $aReturn = array();

        $szVariable = "gROIToolmouseX";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gROIToolmouseY";
        $szValue = " var $szVariable = 0;\n";
        $aReturn[$szVariable] = $szValue;
        
        $szVariable = "goROIFeatureObj";
        $szValue = " var $szVariable = null;\n";
        $aReturn[$szVariable] = $szValue;
        

        return $aReturn;
    }
    
    /**
     * return an array of javascript functions needed by the Ruler widget
     * and called when the page is loaded. 
     * @return array of name = function values
     */
    function GetJavascriptOnLoadFunctions()
    {
        $aReturn = parent::GetJavascriptOnLoadFunctions();

        $aReturn["ROIFeatureToolInitObjects"] = "    ROIFeatureToolInitObjects();\n";
        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
        {
            $szJsFunctionName = "ROIFeatureToolWRegisterForEvent";
            $szFunction = "$szJsFunctionName();\n";
            $aReturn[$szJsFunctionName] = $szFunction;
        }        

        return $aReturn;
    }
     
    /**
     * draw this widget on the page
     */
    function DrawPublish($szPostEvent = "")
    {
        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        if (!$this->mbVisible)
            return "<!-- ROIFeatureTool hidden -->";

        $szReturn = $this->moButton->DrawPublish();

        return $szReturn;
    }
}
?>
