<?php
/**
 * QuickZoom Widget Class
 *
 * @project     CWC2
 * @revision    $Id: QuickZoom.widget.php,v 1.4 2004/07/08 15:58:58 pspencer Exp $
 * @purpose     Display a quick zoom selector
 * @author      DM Solutions Group (spencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../Label.php");

/**
 * a widget that displays a list of predefined extents that can be zoomed to
 */
class QuickZoom extends CWCWidget
{
    var $moLabel;
    var $mszTitle = "Select a Location";
    var $mszWidgetClass = "";

    /**
     * build a new quick zoom widget
     */
    function QuickZoom()
    {
        $this->mszLanguageResource = str_replace("\\","/",dirname(__FILE__))."/QuickZoom.dbf";

        // invoke constructor of parent
        parent::CWCWidget();

        $this->mszTitle = trim($this->moMLT->get("0", "Select a location"));

        $this->mnPriority = PRIORITY_HIGH;

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The QuickZoom widget defines a list of known extents with human-readable
names that the user can select to zoom to quickly.  The format for the
sub-tags is: &lt;view name="Human Name" minx="-2200000" miny="-712631" maxx="3072800" maxy="3840000" SRS="epsg:42304"/&gt;
where the SRS is the projection that the coordinates are defined in.  There may be any number of views defined.
EOT;

        $this->moLabel = new CWCLabel( $this );

        $this->maAttributes["TITLE"] = new StringAttribute( "TITLE", false );
        $this->maAttributes["WIDGETCLASS"] = new StringAttribute( "WIDGETCLASS", false );
    }
    
    
    function InitDefaults()
    {
        parent::InitDefaults();
        if (isset($this->maParams["WIDGETCLASS"]))
            $this->mszWidgetClass = strtoupper($this->maParams["WIDGETCLASS"]);
    }

    /**
     * get the javascript functions for this widget
     */
    function GetJavascriptFunctions()
    {
        parent::GetJavascriptFunctions();

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        $aReturn = array();
        $szJsFunctionName = "changeQuickZoom";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the quickzoom value is changed.
 */
function {$szJsFunctionName}(obj)
{
    extents=obj.options[obj.selectedIndex].value;
    if (extents != "")
    {
        {$this->mszHTMLForm}.NAV_QUICKZOOM.value = extents;
        {$this->mszHTMLForm}.SEL_QUICKZOOM.value = obj.options[obj.selectedIndex].text;
        if ({$bCWCJSAPI})
        {
            goCWCJSAPI.NAV_QUICKZOOM = extents;
            goCWCJSAPI.UpdateNavQuickZoom();
        }
        else
        {
            {$this->mszHTMLForm}.submit();
        }
    }
    return;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

        return $aReturn;
    }

    /**
     * get the HTML variables for this widget
     */
    function GetHTMLHiddenVariables()
    {
        parent::GetHTMLHiddenVariables();

        $szVariable = "NAV_QUICKZOOM";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "SEL_QUICKZOOM";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;

        return $aReturn;
    }

    /**
     * process the url looking for quickzoom requests
     */
    function  ParseURL()
    {
        parent::ParseURL();

        if ($this->isVarSet( "NAV_QUICKZOOM" ) &&
            $this->getVar( "NAV_QUICKZOOM" ) != "")
        {
            if ($this->isVarSet( "NAV_INPUT_COORDINATES" ))
                $this->setVar( "NAV_INPUT_COORDINATES", "");

            $szQuickZoom = $this->getVar( "NAV_QUICKZOOM" );
            $aExtentSRS = explode("|", $szQuickZoom);

            if (count($aExtentSRS) == 1)
            {
                // this is just a static title
                // do nothing
            }
            else
            if (count($aExtentSRS) == 2)
            {
                // validate SRS
                $aSRS = explode(":", $aExtentSRS[1]);
                if (count($aSRS) == 2 && strtoupper($aSRS[0]) == "EPSG" &&
                    is_numeric($aSRS[1]))
                {
                    // reproject to map unit
                    $szTmpMapProj = $this->moMapObject->oMap->getProjection ();
                    $projOutObj = ms_newprojectionobj ($szTmpMapProj);
                    $projInObj = ms_newprojectionobj ("init=epsg:".$aSRS[1]);
                    $oRect = ms_newrectobj ();

                    $aPixPos = explode(";", $aExtentSRS[0]);
                    $aPixMin = explode(",", $aPixPos[0]);
                    $aPixMax = explode(",", $aPixPos[1]);

                    $oRect->setextent($aPixMin[0],$aPixMin[1],
                              $aPixMax[0],$aPixMax[1]);

                    $oRect->project ($projInObj, $projOutObj);

                    $this->moMapObject->oMap->setextent($oRect->minx, $oRect->miny,
                                    $oRect->maxx, $oRect->maxy);
/* -------------------------------------------------------------------- */
/*      Call the reporjectauto function in case the map projection      */
/*      is set to AUTO:XXX. If it is not, the function will do          */
/*      nothing.                                                        */
/* -------------------------------------------------------------------- */
                    $this->ReprojectAuto();                       
                }
                else
                {
                    $_SESSION['gErrorManager']->setError(ERR_WARNING,
                       trim($this->moMLT->get("1", "ERROR: Invalid SRS specified in QuickZoom.php.")));
                    return;
                }
            }
            else
                {
                    $_SESSION['gErrorManager']->setError(ERR_WARNING,
                       trim($this->moMLT->get("2", "ERROR: Wrong number of parameters in QuickZoom.php.")));
                    return;
                }
        }
        return true;
    }

    /**
     * draw this widget
     */
    function DrawPublish()
    {
        if (!$this->mbVisible)
            return "<!-- QuickView is hidden -->";

        if ($this->isVarSet( "SEL_QUICKZOOM" ))
            $szSelQuickZoom = $this->getVar( "SEL_QUICKZOOM" );
        else
            $szSelQuickZoom = "";

        if ($this->mszWidgetClass != "")
            $szWidgetClass = " CLASS=\"".$this->mszWidgetClass."\"";
        else
            $szWidgetClass = "";

        $szResult = "<SELECT NAME=\"QUICK_ZOOM\" onchange=\"changeQuickZoom(this)\"$szWidgetClass>\n";

        if (isset( $this->maszContents["VIEW"] ))
        {
            foreach( $this->maszContents["VIEW"] as $aView )
            {
                if (isset($aView["MINX"]) && isset($aView["MINY"]) &&
                    isset($aView["MAXX"]) && isset($aView["MAXY"]) &&
                    isset($aView["NAME"]) && isset($aView["SRS"]))
                {
                    $oRect = ms_newRectObj();
                    $oRect->setextent( $aView["MINX"], $aView["MINY"],
                                       $aView["MAXX"], $aView["MAXY"] );
                    $oRect = $this->CheckExtents( $oRect );
                    $szExtents = $oRect->minx.",".$oRect->miny.";".
                        $oRect->maxx.",".$oRect->maxy;
                    $szResult .= "<OPTION VALUE=\"".$szExtents.
                        "|".$aView["SRS"]."\" ".
                        (($szSelQuickZoom==$aView["NAME"]) ? "SELECTED":"").
                        ">".$aView["NAME"]."</OPTION>\n";
                }
                else
                if (isset($aView["NAME"]))
                {
                    // This is just a static title
                    $szResult .= "<OPTION VALUE=\"\"".
                        (($szSelQuickZoom==$aView["NAME"]) ? "SELECTED":"").
                        ">".$aView["NAME"]."</OPTION>\n";
                }
                else
                {
                    $_SESSION['gErrorManager']->setError(ERR_WARNING,
                      trim($this->moMLT->get("3", "ERROR: Invalid VIEW tag definition in QuickZoom.php.")));
                }
            }
        }

        $szResult .= "</SELECT>\n";
        $szResult = $this->moLabel->DrawPublish( $szResult );
        return $szResult;
    }

    /**
     * check the projection of the extents
     */
    function CheckExtents( $oRect )
    {
        $szMapSRS = substr($this->moMapObject->oMap->getprojection(), 5);
        $szSRS = "";
        if (isset( $this->maParams["SRS"]))
            $szSRS = $this->maParams["SRS"];
        if ( $szSRS == "" )
            return $oRect;
        if (strcasecmp($szMapSRS, $szSRS) != 0)
        {
            $oProjOut = ms_newProjectionObj( $this->moMapObject->oMap->getprojection() );
            $oProjIn = ms_newProjectionObj( "init=".strtolower($szSRS ));
            $oRect = $oRect->project( $oProjIn, $oProjOut );
        }
        return $oRect;
    }
}
?>
