<?php
/**
 * MapImageSharedResource class
 *
 * @project     CWC2
 * @revision    $Id: MapImageSharedResource.widget.php,v 1.1 2004/04/23 16:48:54 pspencer Exp $
 * @purpose     MapImageSharedResource class
 * @author      DM Solutions Group (lacroix@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2003, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
include_once("MapImageWidget.widget.php");
include_once("geomath.php");
include_once("expression.php");

/**
 * MapImageSharedResource
 *
 * @desc Map image widget. Provides a single click on the map and
 * updates the NAV_INPUT_COORD parameter.
 * Possibility to add point on the map from a SharedResource.
 */
class MapImageSharedResource extends MapImageWidget
{
    var $mnWidth;
    var $mnHeight;
    var $mbAutoExtent = true;
    var $mnAutoExtentBuffer = "";
    var $maaszMapComponents = array();
    var $maaszPointLists = array();
    var $maaszLineLists = array();
    var $mbQueryable = false;
    var $mszPointQueryField = "";
    var $mszPointQueryFormVar = "";
    var $mszPointQueryPoints = "";
    var $mszPointQueryPointTolerance = 2;
    var $mbSymbolOverlap = true;
    var $mnMinAutoExtentBuffer = 0.001;

    function MapImageSharedResource()
    {
        // invoke constructor of parent
        parent::MapImageWidget();
        
        // set the description for this widget
        $this->szWidgetDescription= "This widget display a simple non navigable map with point taken from a shared resource on it.";

        // Set priority to medium since the shared resource should be loaded before it.
        $this->mnPriority = PRIORITY_MEDIUM;

        $this->maAttributes["AUTOEXTENT"] = new BooleanAttribute( "AUTOEXTENT", false);
        $this->maAttributes["AUTOEXTENTBUFFER"] = new FloatAttribute( "AUTOEXTENTBUFFER", false);
        $this->maAttributes["MINAUTOEXTENTBUFFER"] = new FloatAttribute( "MINAUTOEXTENTBUFFER", false);
        $this->maAttributes["QUERYABLE"] = new BooleanAttribute( "QUERYABLE", false);
        $this->maAttributes["POINTQUERYFIELD"] = new StringAttribute( "POINTQUERYFIELD", false );
        $this->maAttributes["POINTQUERYFORMVAR"] = new StringAttribute( "POINTQUERYFORMVAR", false );
        $this->maAttributes["POINTQUERYPOINTS"] = new StringAttribute( "POINTQUERYPOINTS", false );
        $this->maAttributes["POINTQUERYTOLERANCE"] = new IntegerAttribute( "POINTQUERYTOLERANCE", false );
        $this->maAttributes["SYMBOLOVERLAP"] = new BooleanAttribute( "SYMBOLOVERLAP", false);

    }

    /**
     * validate the attributes
     */
    function ValidateAttributes()
    {
        $nMinx = "";
        $nMiny = "";
        $nMaxx = "";
        $nMaxy = "";
        if(isset( $this->maParams["MINX"]) && 
           substr($this->maParams["MINX"], 0, 2) == "[$" &&
           substr($this->maParams["MINX"], -1) == "]")
        {
            $nMinx = $this->maParams["MINX"];
            $this->maParams["MINX"] = 0;
        }
        if(isset( $this->maParams["MINY"]) && 
           substr($this->maParams["MINY"], 0, 2) == "[$" &&
           substr($this->maParams["MINY"], -1) == "]")
        {
            $nMiny = $this->maParams["MINY"];
            $this->maParams["MINY"] = 0;
        }
        if(isset( $this->maParams["MAXX"]) && 
           substr($this->maParams["MAXX"], 0, 2) == "[$" &&
           substr($this->maParams["MAXX"], -1) == "]")
        {
            $nMaxx = $this->maParams["MAXX"];
            $this->maParams["MAXX"] = 0;
        }
        if(isset( $this->maParams["MAXY"]) && 
           substr($this->maParams["MAXY"], 0, 2) == "[$" &&
           substr($this->maParams["MAXY"], -1) == "]")
        {
            $nMaxy = $this->maParams["MAXY"];
            $this->maParams["MAXY"] = 0;
        }

        $bRes = parent::ValidateAttributes();

        if($nMinx != "")
            $this->maParams["MINX"] = $nMinx;
        if($nMinx != "")
            $this->maParams["MINY"] = $nMiny;
        if($nMinx != "")
            $this->maParams["MAXX"] = $nMaxx;
        if($nMinx != "")
            $this->maParams["MAXY"] = $nMaxy;

        return $bRes;
    }

    /**
     * initialize respectable defaults for the map
     */
    function InitDefaults()
    {
        if (isset($this->maszContents))
        {
            if(isset($this->maszContents["POINTS"]))
                $this->maaszPointLists = $this->maszContents["POINTS"];

            if(isset($this->maszContents["LINES"]))
                $this->maaszLineLists = $this->maszContents["LINES"];

            if(isset($this->maszContents["MAP"]))
                $this->maaszMapComponents = $this->maszContents["MAP"];
        }

        if (isset($this->maParams["AUTOEXTENT"]))
        {
            $this->mbAutoExtent = ((strcasecmp($this->maParams["AUTOEXTENT"],"true")==0)?true:false);
        }
        if (isset($this->maParams["AUTOEXTENTBUFFER"]))
            $this->mnAutoExtentBuffer = $this->maParams["AUTOEXTENTBUFFER"];

        if (isset($this->maParams["MINAUTOEXTENTBUFFER"]))
            $this->mnMinAutoExtentBuffer = $this->maParams["MINAUTOEXTENTBUFFER"];


        if (isset($this->maParams["QUERYABLE"]))
            $this->mbQueryable = ((strcasecmp($this->maParams["QUERYABLE"],"true")==0)?true:false);

        if (isset($this->maParams["POINTQUERYFIELD"]))
            $this->mszPointQueryField = $this->maParams["POINTQUERYFIELD"];

        if (isset($this->maParams["POINTQUERYFORMVAR"]))
            $this->mszPointQueryFormVar = $this->maParams["POINTQUERYFORMVAR"];

        if (isset($this->maParams["POINTQUERYPOINTS"]))
            $this->mszPointQueryPoints = $this->maParams["POINTQUERYPOINTS"];

        if (isset($this->maParams["POINTQUERYTOLERANCE"]))
            $this->mszPointQueryPointTolerance = $this->maParams["POINTQUERYTOLERANCE"];
        if (isset($this->maParams["SYMBOLOVERLAP"]))
            $this->mbSymbolOverlap = ((strcasecmp($this->maParams["SYMBOLOVERLAP"],"false")==0)?false:true);

        parent::InitDefaults();
    }

    /**
     * ParseURL
     */
    function ParseURL()
    {
        $bRes = parent::ParseURL();

        if(!$this->mbQueryable || $this->mszPointQueryPoints == "")
            return $bRes;

        // Check if shared resource is clicked.
        if(($this->isVarSet( $this->msMapImageName."_X" ) && 
            $this->getVar( $this->msMapImageName."_X" ) != "") &&
           ($this->isVarSet( $this->msMapImageName."_Y" ) && 
            $this->getVar( $this->msMapImageName."_Y" ) != ""))
        {
            $nX = $this->getVar( $this->msMapImageName."_X" );
            $nY = $this->getVar( $this->msMapImageName."_Y" );

            $aszSessKey = array_keys($_SESSION);
            $numSessKey = count($aszSessKey);
            $aszPointLists = array();
            $aszSessName = array();
            $bPointClicked = false;
            $j = 0;
            for($i=0; $i<$numSessKey; $i++)
            {
                if(strpos($aszSessKey[$i], $this->mszPointQueryPoints) !== false &&
                   ($szPointPos = strpos($aszSessKey[$i], "_POINTS_".$this->GetValue("id", 
                                         $this->GetValue("name")))))
                {
                    $aszPointLists[$j] =& $_SESSION[$aszSessKey[$i]];
                    $aszSessName[$j] = substr($aszSessKey[$i], 
                                              strlen($this->mszPointQueryPoints)+1,
                                              $szPointPos - strlen($this->mszPointQueryPoints)-1);
                    $j++;
                }
            }

            $nPointListName = count($aszPointLists);
            if ($nPointListName == 0)
                break;

            if ( isset($_SESSION["CLICKED_POINTS"][$this->GetValue("id", $this->GetValue("name"))][$nX][$nY]) )
            {
                $aIndex = $_SESSION["CLICKED_POINTS"][$this->GetValue("id", $this->GetValue("name"))][$nX][$nY];

                $bPointClicked = true;

                $bPointRes = $this->PointClicked($aszSessName[$aIndex[0]], $aszPointLists[$aIndex[0]], $aIndex[1]);

                if (!$bPointRes)
                    return false;
            }
            else
            {
                // Get Extent
                $nMinX = $this->moMapObject->oMap->extent->minx;
                $nMaxX = $this->moMapObject->oMap->extent->maxx;
                $nMinY = $this->moMapObject->oMap->extent->miny;
                $nMaxY = $this->moMapObject->oMap->extent->maxy;
                $dfGeoWidth = ($nMaxX-$nMinX);
                $dfGeoHeight = ($nMaxY-$nMinY);
                $dfGeoMiddleX = $nMinX + ($dfGeoWidth / 2 );
                $dfGeoMiddleY = $nMinY + ($dfGeoHeight / 2 );

                $aLatLon = Pix2Georef ( 
                    $nX, 
                    $nY,
                    $dfGeoMiddleX, $dfGeoMiddleY,
                    $dfGeoWidth,   $dfGeoHeight,
                    $this->moMapObject->oMap->width,  
                    $this->moMapObject->oMap->height  );

                // Check if this latlon is in the dealers list.
                $dfGeoToleranceW = $dfGeoWidth / $this->moMapObject->oMap->width *
                    $this->mszPointQueryPointTolerance ;
                $dfGeoToleranceH = $dfGeoHeight / $this->moMapObject->oMap->height*
                    $this->mszPointQueryPointTolerance ;

                $adClickExtent[0] = $aLatLon[0] - $dfGeoToleranceW;
                $adClickExtent[1] = $aLatLon[0] + $dfGeoToleranceW;
                $adClickExtent[2] = $aLatLon[1] - $dfGeoToleranceH;
                $adClickExtent[3] = $aLatLon[1] + $dfGeoToleranceH;

                for($i=0; $i<$nPointListName; $i++)
                {
                    $nKey = $this->whichClickedUnique($adClickExtent, 
                                                      $aszPointLists[$i]);

                    if($nKey !== false)
                    {
                        $bPointRes = $this->PointClicked($aszSessName[$i], $aszPointLists[$i], $nKey);

                        $bPointClicked = true;
                        $_SESSION["CLICKED_POINTS"][
                           $this->GetValue("id", $this->GetValue("name"))][$nX][$nY] = array($i, $nKey);

                        if (!$bPointRes) return false;
                    }
                }
            }

            if(!$bPointClicked)
            {
                $szErrorLevel = $this->ProcessEvent("NoPointClicked");
                if($szErrorLevel == ONEVENT_FATAL)
                    return false;
            }
        }

        return $bRes;
    }

    function PointClicked($szSessName, &$aszPoint, $nKey)
    {
        if($this->mszPointQueryFormVar != "" && $this->mszPointQueryField != "")
        {
             $aszQueryField=explode(",", $this->mszPointQueryField);
             $aszFormVar =explode(",", $this->mszPointQueryFormVar);
             $numFields = count($aszQueryField);
             for($field=0; $field<$numFields; $field++)
             {
                  if(isset($aszPoint[$nKey][$aszQueryField[$field]]))
                  {
                       $this->setVar( $aszFormVar[$field], 
                                      $aszPoint[$nKey][$aszQueryField[$field]]);
                  }
             }
        }

        $aszPoint = "";

        if ($this->isVarSet( 'NAV_CMD' ))
             $this->setVar( 'NAV_CMD', "");

        // Process event
        $szErrorLevel = $this->ProcessEvent("PointClicked");

        if($szErrorLevel == ONEVENT_FATAL)
             return false;
        $szErrorLevel = $this->ProcessEvent("PointClicked_". $szSessName);

        if($szErrorLevel == ONEVENT_FATAL)
            return false;

        return true;
}

    /**
     * DrawPublish
     *
     * Draw the map and return the HTML.
     */
    function DrawPublish()
    {
        // Get the BBox and add the list of point in the session
        $nMinLon = $nMaxLon = $nMinLat = $nMaxLat = "";
        $numPointList = count($this->maaszPointLists);
        $bArrayDiff = false;
        for($j=0; $j<$numPointList; $j++)
        {
            $szPointListName = "";
            $szSRName = "";
            $szSRLatCol = "";
            $szSRLonCol = "";
            $szType = "MULTI";
            $szLabel = "__i__";
            $szCondition = "";
            $aTmpPointList = array();
            $aPointList =& $aTmpPointList;
            if(isset($this->maaszPointLists[$j]["SRNAME"]))
                $szSRName = $this->maaszPointLists[$j]["SRNAME"];
            else
                $szSRName = $this->maaszPointLists[$j]["SRNAME"] = "SQLQuery";
            if(isset($this->maaszPointLists[$j]["NAME"]))
                $szPointListName = $this->maaszPointLists[$j]["NAME"];
            if(isset($this->maaszPointLists[$j]["SRLATCOL"]))
                $szSRLatCol = $this->maaszPointLists[$j]["SRLATCOL"];
            else
                $szSRLatCol = $this->maaszPointLists[$j]["SRLATCOL"] = 
                    "LATITUDE";
            if(isset($this->maaszPointLists[$j]["SRLONCOL"]))
                $szSRLonCol = $this->maaszPointLists[$j]["SRLONCOL"];
            else
                $szSRLonCol = $this->maaszPointLists[$j]["SRLONCOL"] = 
                    "LONGITUDE";
            if(isset($this->maaszPointLists[$j]["LABEL"]))
            {
                $szLabel = $this->parseSRLabel($this->maaszPointLists[$j]["LABEL"]);
                if(!$this->mbSymbolOverlap && $szLabel == "")
                    $szLabel = " ";
            }
            if(isset($this->maaszPointLists[$j]["TYPE"]))
                $szType = strtoupper($this->maaszPointLists[$j]["TYPE"]);
            if(isset($this->maaszPointLists[$j]["CONDITION"]) && 
               $this->maaszPointLists[$j]["CONDITION"] != "")
            {
                $szCondition = $this->maaszPointLists[$j]["CONDITION"];
            }

            $aszSRName = explode(".", $szSRName);
            if(isset($this->maSharedResourceWidgets[$aszSRName[0]]))
            {
                $aSR =& $this->maSharedResourceWidgets[$aszSRName[0]];
                array_shift($aszSRName);
            }
            else
                $aSR =& $this->maSharedResourceWidgets;

            $bInSRWidget = false;
            // Check if the Shared resource is a widget
            if (isset($aSR->mszWidgetName) &&
                $aSR->mszWidgetName == "SharedResource")
            {
                $bInSRWidget = true;
                $aSR =& $aSR->maszContents;
            }

            foreach($aszSRName as $szSRName2)
            {
                if($bInSRWidget)
                    $szSRName2 = strtoupper($szSRName2);
                if (isset($aSR[$szSRName2]))
                    $aSR =& $aSR[$szSRName2];
                else
                    if ($szSRName2 == "_LAST_")
                        $aSR =& $aSR[count($aSR)-1];
            }

            // Get the shared resource from DrivingDirections widget
            if($szType == "UNIQUE" && 
               isset($aSR[$szSRLatCol]) &&
               isset($aSR[$szSRLonCol]))
            {
                $aPointList[1] =& $aSR;
            }
            else if(is_array($aSR))
            {
                $aPointList =& $aSR;
            }

            $nCnt = count($aPointList);
            if($nCnt)
            {
                $aTmpPoint = array();
                $aTmpPoint["LongitudeField"] = $szSRLonCol;
                $aTmpPoint["LatitudeField"] = $szSRLatCol;

                $i = 0;
                $aszKeys = array_keys($aPointList);
                for($i=0;$i<$nCnt;$i++)
                {
                    $aPoint = $aPointList[$aszKeys[$i]];
                    $szKey = $aszKeys[$i];
                    if(is_array($aPoint) && isset($aPoint[$szSRLonCol]) && 
                       isset($aPoint[$szSRLatCol]) && 
                       ($szCondition == "" || 
                        $this->processCondition($szCondition, $aPoint)))
                    {
                        $aTmpPoint[$i][$szSRLonCol] = $aPoint[$szSRLonCol];
                        $aTmpPoint[$i][$szSRLatCol] = $aPoint[$szSRLatCol];

                        if($szLabel == "__i__")
                              $aTmpPoint[$i]["TXT"] = $szKey;
                        else
                              $aTmpPoint[$i]["TXT"] = $szLabel;

                        if($this->mszPointQueryField != "" && 
                           $szSRName == $this->mszPointQueryPoints)
                        {
                            $aszQueryField = explode(",", 
                                                    $this->mszPointQueryField);
                            $numFields = count($aszQueryField);
                            for($field=0; $field<$numFields; $field++)
                            {
                                if(isset($aPoint[$aszQueryField[$field]]))
                                    $aTmpPoint[$i][$aszQueryField[$field]] = 
                                        $aPoint[$aszQueryField[$field]];
                                else
                                    $aTmpPoint[$i][$aszQueryField[$field]]="";
                            }
                        }
 
                        if($nMinLon > $aPoint[$szSRLonCol] || $nMinLon=="")
                            $nMinLon = $aPoint[$szSRLonCol];
                        if($nMaxLon < $aPoint[$szSRLonCol] || $nMaxLon=="")
                            $nMaxLon = $aPoint[$szSRLonCol];
                        if($nMinLat > $aPoint[$szSRLatCol] || $nMinLat=="")
                            $nMinLat = $aPoint[$szSRLatCol];
                        if($nMaxLat < $aPoint[$szSRLatCol] || $nMaxLat=="")
                            $nMaxLat = $aPoint[$szSRLatCol];
                    }
                }
                $bArrayDiff = (($bArrayDiff) ? $bArrayDiff :$this->DiffArray($aTmpPoint, 
                                       $_SESSION[$szSRName."_".$szPointListName."_POINTS_".$this->GetValue("id", $this->GetValue("name"))]));

                $_SESSION[$szSRName."_".$szPointListName."_POINTS_".$this->GetValue("id", $this->GetValue("name"))] = array();
                $_SESSION[$szSRName."_".$szPointListName."_POINTS_".$this->GetValue("id", $this->GetValue("name"))] = $aTmpPoint;
            }
        }
        $this->drawPoints();







        // Get the BBox and add the list of line in the session
        $numLineList = count($this->maaszLineLists);
        for($j=0; $j<$numLineList; $j++)
        {
            $szLineListName = "";
            $szSRName = "";
            $szSRLatCol = "";
            $szSRLonCol = "";
            $szType = "MULTI";
            $szLabel = "__i__";
            $aLineList = array();
            if(isset($this->maaszLineLists[$j]["SRNAME"]))
                $szSRName = $this->maaszLineLists[$j]["SRNAME"];
            else
                $szSRName = $this->maaszLineLists[$j]["SRNAME"] = "SQLQuery";
            if(isset($this->maaszLineLists[$j]["NAME"]))
                $szLineListName = $this->maaszLineLists[$j]["NAME"];
            if(isset($this->maaszLineLists[$j]["SRLATCOL"]))
                $szSRLatCol = $this->maaszLineLists[$j]["SRLATCOL"];
            else
                $szSRLatCol = $this->maaszLineLists[$j]["SRLATCOL"] = 
                    "LATITUDE";
            if(isset($this->maaszLineLists[$j]["SRLONCOL"]))
                $szSRLonCol = $this->maaszLineLists[$j]["SRLONCOL"];
            else
                $szSRLonCol = $this->maaszLineLists[$j]["SRLONCOL"] = 
                    "LONGITUDE";
            if(isset($this->maaszLineLists[$j]["LABEL"]))
                $szLabel = $this->parseSRLabel($this->maaszLineLists[$j]["LABEL"]);
            if(isset($this->maaszLineLists[$j]["TYPE"]))
                $szType = strtoupper($this->maaszLineLists[$j]["TYPE"]);

            $aszSRName = explode(".", $szSRName);
            $aSR =& $this->maSharedResourceWidgets;
            foreach($aszSRName as $szSRName2)
                if (isset($aSR[$szSRName2]))
                    $aSR =& $aSR[$szSRName2];
                else
                if ($szSRName2 == "_LAST_")
                    $aSR =& $aSR[count($aSR)-1];

            // Get the shared resource
            if(is_array($aSR))
                $aPointList =& $aSR;

            $nCnt = count($aPointList);
            if($nCnt)
            {
                $aTmpPoint = array();
                $aTmpPoint["LongitudeField"] = $szSRLonCol;
                $aTmpPoint["LatitudeField"] = $szSRLatCol;

                $i = 0;
                $aszKeys = array_keys($aPointList);
                for($i=0;$i<$nCnt;$i++)
                {
                    $aPoint = $aPointList[$aszKeys[$i]];
                    $szKey = $aszKeys[$i];
                    if(is_array($aPoint) && isset($aPoint[$szSRLonCol]) && 
                       isset($aPoint[$szSRLatCol]))
                    {
                        $aTmpPoint[$i][$szSRLonCol] = $aPoint[$szSRLonCol];
                        $aTmpPoint[$i][$szSRLatCol] = $aPoint[$szSRLatCol];

                        if($szLabel == "__i__")
                              $aTmpPoint[$i]["TXT"] = $szKey;
                        else
                              $aTmpPoint[$i]["TXT"] = $szLabel;

                        if($nMinLon > $aPoint[$szSRLonCol] || $nMinLon=="")
                            $nMinLon = $aPoint[$szSRLonCol];
                        if($nMaxLon < $aPoint[$szSRLonCol] || $nMaxLon=="")
                            $nMaxLon = $aPoint[$szSRLonCol];
                        if($nMinLat > $aPoint[$szSRLatCol] || $nMinLat=="")
                            $nMinLat = $aPoint[$szSRLatCol];
                        if($nMaxLat < $aPoint[$szSRLatCol] || $nMaxLat=="")
                            $nMaxLat = $aPoint[$szSRLatCol];
                    }
                }

                $bArrayDiff = (($bArrayDiff) ? $bArrayDiff :$this->DiffArray($aTmpPoint, 
                                       $_SESSION[$szSRName."_".$szLineListName."_LINES_".$this->GetValue("id", $this->GetValue("name"))]));

                $_SESSION[$szSRName."_".$szLineListName."_LINES_".$this->GetValue("id", $this->GetValue("name"))] = array();
                $_SESSION[$szSRName."_".$szLineListName."_LINES_".$this->GetValue("id", $this->GetValue("name"))] = $aTmpPoint;
            }
        }
        $this->drawLines();








//print_r($this->moMapObject->aoAnnotation);


        /* Add buffer space and make sure we don't end up with a 0-size map
         * (0.001 degre is approx 100 meters)
         */
        $bUseMinXY = false;
        if(!$this->mbNavigable)
        {
            $nMinx = $this->GetValue("minx", false);
            $nMiny = $this->GetValue("miny", false);
            $nMaxx = $this->GetValue("maxx", false);
            $nMaxy = $this->GetValue("maxy", false);

            if($nMinx === false || $nMaxx === false)
                $nMinx = $nMaxx;
            if($nMiny === false || $nMaxy === false)
                $nMiny = $nMaxy;

            if($nMinx !== false && $nMaxx !== false && 
               $nMiny !== false && $nMaxy !== false)
            {
                $nMinx = $this->parseSRLabel($nMinx);
                $nMiny = $this->parseSRLabel($nMiny);
                $nMaxx = $this->parseSRLabel($nMaxx);
                $nMaxy = $this->parseSRLabel($nMaxy);

                $nDiffx = $nMaxx-$nMinx;
                $nDiffy = $nMaxy-$nMiny;

                if ($nDiffx < $this->mnMinAutoExtentBuffer)
                {
                    $nDiffx = $this->mnMinAutoExtentBuffer;
                    $this->SetValue("minx", ($nMinx - $nDiffx/2));
                    $this->SetValue("maxx", ($nMaxx + $nDiffx/2));
                }
                else
                {
                    $this->SetValue("minx", $nMin);
                    $this->SetValue("maxx", $nMaxx);
                }

                if ($nDiffy < $this->mnMinAutoExtentBuffer)
                {
                    $nDiffy = $this->mnMinAutoExtentBuffer;
                    $this->SetValue("miny", ($nMiny - $nDiffy/2));
                    $this->SetValue("maxy", ($nMaxy + $nDiffy/2));
                }
                else
                {
                    $this->SetValue("miny", $nMiny);
                    $this->SetValue("maxy", $nMaxy);
                }

                $bUseMinXY = true;
            }
        }

        if (($this->isVarSet( "NAV_CMD" ) && $this->getVar( "NAV_CMD" ) == "") || 
            $bArrayDiff || !$this->isVarSet( "NAV_CMD" ) && !$bUseMinXY)
        {
            if ($this->mbAutoExtent)
            {
                $nDiffLon = $nMaxLon-$nMinLon;
                $nDiffLat = $nMaxLat-$nMinLat;

                if ($nDiffLon < $this->mnMinAutoExtentBuffer) $nDiffLon = $this->mnMinAutoExtentBuffer;
                if ($nDiffLat < $this->mnMinAutoExtentBuffer) $nDiffLat = $this->mnMinAutoExtentBuffer;

                $nMaxLon += ($this->mnAutoExtentBuffer != "") ? $this->mnAutoExtentBuffer : $nDiffLon/10;
                $nMinLon -= ($this->mnAutoExtentBuffer != "") ? $this->mnAutoExtentBuffer : $nDiffLon/10;
                $nMaxLat += ($this->mnAutoExtentBuffer != "") ? $this->mnAutoExtentBuffer : $nDiffLat/10;
                $nMinLat -= ($this->mnAutoExtentBuffer != "") ? $this->mnAutoExtentBuffer : $nDiffLat/10;

                $poRect = ms_newRectObj();

                $poRect->setextent($nMinLon, $nMinLat, $nMaxLon, $nMaxLat);
    
                $szTmpMapProj = $this->moMapObject->oMap->getProjection();
                if ($szTmpMapProj != "")
                {
                    $projOutObj = ms_newprojectionobj( $szTmpMapProj );
                    $projInObj = ms_newprojectionobj("init=epsg:4326");

                    $poRect->project($projInObj, $projOutObj);
                }

                if ($nMaxLon != "" && $nMinLon != "" && $nMaxLat != "" && $nMinLat != "")
                {
                    if($this->mbNavigable)
                        $this->moMapObject->oMap->setExtent($poRect->minx, 
                                                            $poRect->miny, 
                                                            $poRect->maxx, 
                                                            $poRect->maxy);
                    else
                    {
                        $this->SetValue("minx", $poRect->minx);
                        $this->SetValue("maxx", $poRect->maxx);
                        $this->SetValue("miny", $poRect->miny);
                        $this->SetValue("maxy", $poRect->maxy);
                    }
                }
            }
        }

        return parent::DrawPublish();
    }

    /**
     * DiffArray
     *
     * This function check if arrays are different.
     */
    function DiffArray(&$aString1, &$aString2)
    {
        $aReturn = array();

        if (is_array($aString1) && is_array($aString2))
        {
            $nCnt = count($aString1);
            for ($i=0;$i<$nCnt;$i++)
            {
                if (isset($aString2[$i]))
                {
                    $aTmp = array_diff($aString1[$i], $aString2[$i]);
                    if (count($aTmp) > 0)
                        return true;
                }
            }
        } 
        else
        if ($aString2 != $aString1) return true;

        return false;
    }
 
    /**
     * drawPoints()
     *
     * Draw a list of points on the map from a list (Shared Resource)
     */
    function drawPoints()
    {
        // Get the BBox and add the list of point in the session
        $numPointList = count($this->maaszPointLists);
        $aoAnnotationArray = array();
        $aszActiveAnnotationLayers = array();
        $bAlreadySet = false;
        for($j=0; $j<$numPointList; $j++)
        {
            if(isset($this->maaszPointLists[$j]["NAME"]))
                $szPointListName = $this->maaszPointLists[$j]["NAME"];
            else
                $szPointListName = "";
            $szSRName = "";
            if(isset($this->maaszPointLists[$j]["SRNAME"]))
                $szSRName = $this->maaszPointLists[$j]["SRNAME"].
                    "_".$szPointListName."_POINTS_".$this->GetValue("id", 
                                                     $this->GetValue("name"));

            $bAlreadySet = false;
            $nCnt = count($this->moMapObject->aoAnnotation);
            for ($i=0;$i<$nCnt;$i++)
            {
                if($this->moMapObject->aoAnnotation[$i]->name == $szSRName)
                {
                    $bAlreadySet = true;
                    break;
                }
            }

            array_push( $aszActiveAnnotationLayers, $szSRName );

            // Get the shared resource
            if(isset($_SESSION[$szSRName]) && count($_SESSION[$szSRName]) &&
               !$bAlreadySet)
            {
                // Create a new default layer for this point list
                $poAnnotationLayer =& $this->moMapObject->addAnnotationLayer();
                if(isset($this->maaszPointLists[$j]["NAME"]))
                    $aoAnnotationArray[$this->maaszPointLists[$j]["NAME"]] =& 
                        $poAnnotationLayer;

                $poAnnotationLayer->set("projection", "init=epsg:4326");
                $poAnnotationLayer->set("name", $szSRName);
                $poAnnotationLayer->set("status", MS_ON);

                if(!$this->mbSymbolOverlap)
                {
                    $poAnnotationLayer->set("type", MS_LAYER_ANNOTATION);
                    $poAnnotationLayer->set("labelcache", MS_ON);
                }
                else
                {
                    $poAnnotationLayer->set("type", MS_LAYER_POINT);
                    $poAnnotationLayer->set("labelcache", MS_OFF);
                }

                $poAnnotationClass =& $poAnnotationLayer->addClass();
                $poAnnotationClass->set("name","dynamic point class");

                $poAnnotationStyle =& $poAnnotationClass->addStyle();
                if(isset($this->maaszPointLists[$j]["TYPE"]) &&
                   strtoupper($this->maaszPointLists[$j]["TYPE"]) == "UNIQUE")
                {
                    $poAnnotationStyle->set("symbolname", "star");
                    $poAnnotationStyle->set("size", 15);
                    $poAnnotationStyle->color->setRGB(156, 5, 5);
                }
                else
                {
                    $poAnnotationStyle->set("symbolname", "circle");
                    $poAnnotationStyle->set("size", 15);
                    $poAnnotationStyle->color->setRGB(156, 5, 5);
                    $poAnnotationStyle->outlinecolor->setRGB(156, 5, 5);
                    $poAnnotationStyle->backgroundcolor->setRGB(156, 5, 5);
                }

                $poAnnotationClass->label->set("font", "arial-bold");
                $poAnnotationClass->label->set("type", "TRUETYPE");
                $poAnnotationClass->label->color->setRGB(255, 255, 255);
                $poAnnotationClass->label->set("size", 8);
                $poAnnotationClass->label->set("position", 8);
                if(!$this->mbSymbolOverlap)
                {
                    $poAnnotationClass->label->set("buffer", 2);
                }
            }
        }

        // Set the Map Component from the widget
        $numComponent = count($this->maaszMapComponents);
        for($j=0; $j<$numComponent; $j++)
        {
            if(isset($this->maaszMapComponents[$j]["POINT"]) &&
               isset($this->maaszMapComponents[$j]["TYPE"]) &&
               isset($this->maaszMapComponents[$j]["PARAM"]) &&
               isset($this->maaszMapComponents[$j]["VALUE"]) &&
             isset($aoAnnotationArray[$this->maaszMapComponents[$j]["POINT"]]))
            {
                $poAnnotationObj[$j] = "";
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="LAYER")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[
                        $this->maaszMapComponents[$j]["POINT"]];
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="CLASS")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[
                        $this->maaszMapComponents[$j]["POINT"]]->aoClass[0];
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="STYLE")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[$this->
                      maaszMapComponents[$j]["POINT"]]->aoClass[0]->aoStyle[0];
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="LABEL")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[$this->
                           maaszMapComponents[$j]["POINT"]]->aoClass[0]->label;

                if($poAnnotationObj[$j] != "")
                {
                    $szParam = strtolower(
                        $this->maaszMapComponents[$j]["PARAM"]);
                    $szValue = $this->maaszMapComponents[$j]["VALUE"];
                    if($this->maaszMapComponents[$j]["PARAM"] != "color" &&
                       $this->maaszMapComponents[$j]["PARAM"]!="outlinecolor"&&
                     $this->maaszMapComponents[$j]["PARAM"]!="backgroundcolor")
                    {
                        $poAnnotationObj[$j]->set($szParam, $szValue);
                    }
                    else
                    {
                        $Red = (hexdec($szValue) & 0xFF0000) / 0x010000;
                        $Green = (hexdec($szValue) & 0x00FF00) / 0x000100;
                        $Blue = (hexdec($szValue) & 0x0000FF) / 0x000001;

                        if($this->maaszMapComponents[$j]["PARAM"] == 
                           "outlinecolor")
                            $poAnnotationObj[$j]->outlinecolor->setRGB(
                                $Red,$Green,$Blue);
                        else 
                        if($this->maaszMapComponents[$j]["PARAM"] == 
                           "backgroundcolor")
                            $poAnnotationObj[$j]->backgroundcolor->setRGB(
                                $Red,$Green,$Blue);
                        else
                            $poAnnotationObj[$j]->color->setRGB(
                                $Red,$Green,$Blue);

                    }
                }
            }
        }
    }



    /**
     * drawLines()
     *
     * Draw a list of line on the map from a list (Shared Resource)
     */
    function drawLines()
    {
        // Get the BBox and add the list of point in the session
        $numLineList = count($this->maaszLineLists);
        $aoAnnotationArray = array();
        $aszActiveAnnotationLayers = array();
    $bAlreadySet = false;
        for($j=0; $j<$numLineList; $j++)
        {
            if(isset($this->maaszLineLists[$j]["NAME"]))
                $szLineListName = $this->maaszLineLists[$j]["NAME"];
            else
                $szLineListName = "";
            $szSRName = "";
            if(isset($this->maaszLineLists[$j]["SRNAME"]))
                $szSRName = $this->maaszLineLists[$j]["SRNAME"].
                    "_".$szLineListName."_LINES_".$this->GetValue("id", 
                                                      $this->GetValue("name"));

            $bAlreadySet = false;
            $nCnt = count($this->moMapObject->aoAnnotation);
            for($i=0;$i<$nCnt;$i++)
            {
                if($this->moMapObject->aoAnnotation[$i]->name == $szSRName)
                {
                    $bAlreadySet = true;
                    break;
                }
            }

            array_push( $aszActiveAnnotationLayers, $szSRName );

            // Get the shared resource
            if(isset($_SESSION[$szSRName]) && count($_SESSION[$szSRName]) &&
               !$bAlreadySet)
            {
                // Create a new default layer for this point list
                $poAnnotationLayer =& $this->moMapObject->addAnnotationLayer();
                if(isset($this->maaszLineLists[$j]["NAME"]))
                    $aoAnnotationArray[$this->maaszLineLists[$j]["NAME"]] =& 
                        $poAnnotationLayer;


                $poAnnotationLayer->set("projection", "init=epsg:4326");
                $poAnnotationLayer->set("name", $szSRName);
                $poAnnotationLayer->set("status", MS_ON);

                $poAnnotationLayer->set("type", MS_LAYER_LINE);

                $poAnnotationLayer->set("labelcache", MS_OFF);
                $poAnnotationLayer->set("status", MS_ON);

                $poAnnotationClass =& $poAnnotationLayer->addClass();
                $poAnnotationClass->set("name","dynamic line class");

                $poAnnotationStyle =& $poAnnotationClass->addStyle();

                $poAnnotationStyle->set("symbolname", "circle");
                $poAnnotationStyle->set("size", 10);
                $poAnnotationStyle->color->setRGB(156, 5, 5);

                $poAnnotationClass->label->set("font", "arial-bold");
                $poAnnotationClass->label->set("type", "TRUETYPE");
                $poAnnotationClass->label->color->setRGB(255, 255, 255);
                $poAnnotationClass->label->set("size", 8);
                $poAnnotationClass->label->set("position", 8);
                $poAnnotationClass->label->set("force", 1);
            }
        }

        // Set the Map Component from the widget
        $numComponent = count($this->maaszMapComponents);
        for($j=0; $j<$numComponent && !$bAlreadySet; $j++)
        {
            if(isset($this->maaszMapComponents[$j]["LINE"]) &&
               isset($this->maaszMapComponents[$j]["TYPE"]) &&
               isset($this->maaszMapComponents[$j]["PARAM"]) &&
               isset($this->maaszMapComponents[$j]["VALUE"]) &&
             isset($aoAnnotationArray[$this->maaszMapComponents[$j]["LINE"]]))
            {
                $poAnnotationObj[$j] = "";
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="LAYER")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[
                        $this->maaszMapComponents[$j]["LINE"]];
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="CLASS")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[
                        $this->maaszMapComponents[$j]["LINE"]]->aoClass[0];
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="STYLE")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[$this->
                      maaszMapComponents[$j]["LINE"]]->aoClass[0]->aoStyle[0];
                if(strtoupper($this->maaszMapComponents[$j]["TYPE"]) =="LABEL")
                    $poAnnotationObj[$j] =& $aoAnnotationArray[$this->
                           maaszMapComponents[$j]["LINE"]]->aoClass[0]->label;

                if($poAnnotationObj != "")
                {
                    $szParam = strtolower(
                        $this->maaszMapComponents[$j]["PARAM"]);
                    $szValue = $this->maaszMapComponents[$j]["VALUE"];
                    if($this->maaszMapComponents[$j]["PARAM"] != "color" &&
                       $this->maaszMapComponents[$j]["PARAM"]!="outlinecolor"&&
                     $this->maaszMapComponents[$j]["PARAM"]!="backgroundcolor")
                    {
                        $poAnnotationObj[$j]->set($szParam, $szValue);
                    }
                    else
                    {
                        $Red = (hexdec($szValue) & 0xFF0000) / 0x010000;
                        $Green = (hexdec($szValue) & 0x00FF00) / 0x000100;
                        $Blue = (hexdec($szValue) & 0x0000FF) / 0x000001;

                        if($this->maaszMapComponents[$j]["PARAM"] == 
                           "outlinecolor")
                            $poAnnotationObj[$j]->outlinecolor->setRGB(
                                $Red,$Green,$Blue);
                        else 
                        if($this->maaszMapComponents[$j]["PARAM"] == 
                           "backgroundcolor")
                            $poAnnotationObj[$j]->backgroundcolor->setRGB(
                                $Red,$Green,$Blue);
                        else
                            $poAnnotationObj[$j]->color->setRGB(
                                $Red,$Green,$Blue);

                    }
                }
            }
        }
    }

    /**
     * ParseSQLQuery()
     *
     * Replace the dynamic values (SharedResource) in the SQL query
     */
    function ParseSRLabel($szLabel)
    {
        if(stristr($szLabel, "[$") === false)
            return $szLabel;

        $nPos = 0;
        while (strpos($szLabel, "[$", $nPos) !== false)
        {
            $nStart = strpos($szLabel, "[$", $nPos);
            $nEnd   = strpos($szLabel, "]", $nStart);

            $szPath = substr($szLabel, $nStart+2, $nEnd-$nStart-2);

            $aszPath = explode(".", $szPath);
            $nCount = count($aszPath);

            $szEvalPath = "\$this->maSharedResourceWidgets";

            for ($i=0; $i<$nCount; $i++)
            {
                if(is_numeric($aszPath[$i]))
                    $szEvalPath .= "[".$aszPath[$i]."]";
                else if($aszPath[$i] == "_LAST_")
                    $szEvalPath .= "[(count($szEvalPath)-1)]";
                else
                    $szEvalPath .= "['".$aszPath[$i]."']";
            }

            @eval("\$szReturn = $szEvalPath;");
            if (isset($szReturn))
                $szLabel = str_replace("[$".$szPath."]", $szReturn, $szLabel);
            else
                $szLabel = str_replace("[$".$szPath."]", "", $szLabel);
            $nPos = $nStart;
        }

        return $szLabel;
    }

    function whichClickedUnique($adClickExtent, &$aPointList)
    {
        if(!isset($aPointList["LatitudeField"]) || 
           !isset($aPointList["LongitudeField"]))
            return false;

        $szLatField = $aPointList["LatitudeField"];
        $szLonField = $aPointList["LongitudeField"];

        $aszKeys = array_keys($aPointList);
        $nCnt = count($aszKeys);

        $oPoint = ms_newPointObj();
        $szTmpMapProj = $this->moMapObject->oMap->getProjection();

        for($j=0;$j<$nCnt;$j++)
        {
            $aPoint = $aPointList[$aszKeys[$j]];
            if(is_array($aPoint) && isset($aPoint[$szLonField]) && 
               isset($aPoint[$szLatField]))
            {
                $oPoint->setXY($aPoint[$szLonField], $aPoint[$szLatField]);
    
                if ($szTmpMapProj != "")
                {
                    $projOutObj = ms_newprojectionobj( $szTmpMapProj );
                    $projInObj = ms_newprojectionobj("init=epsg:4326");

                    $oPoint->project($projInObj, $projOutObj);
                }

                if($oPoint->x >= $adClickExtent[0] &&
                   $oPoint->x <= $adClickExtent[1] &&
                   $oPoint->y >= $adClickExtent[2] &&
                   $oPoint->y <= $adClickExtent[3])
                {
                    return $aszKeys[$j];
                }
            }
        }

        return false;
    }

    /**
     * Evaluate the condition and return true/false
     */
    function processCondition($szCondition, $aszFields)
    {
        if(preg_match("/^%([^% ]*)%$/", $szCondition, $matches))
        {
            if(isset($aszFields[$matches[1]]) && $aszFields[$matches[1]] != "")
                return true;
            else
                return false;
        }

        $szCondition = str_replace("&quot;", "\"", $szCondition);

        $bResult = false;
        $nCnt = count($aszFields);
        $aszKeys = array_keys($aszFields);
        for($j=0;$j<$nCnt;$j++)
        {
            $szCondition = str_replace("\"%".$aszKeys[$j]."%\"", "\"".$aszFields[$aszKeys[$j]]."\"", 
                                       $szCondition);
            $szCondition = str_replace("%".$aszKeys[$j]."%", "\"".$aszFields[$aszKeys[$j]]."\"", 
                                       $szCondition);
        }

        $bResult = evaluateExpression($szCondition);

        if($bResult == -1)
            $bResult = false;

        return $bResult;
    }
}
?>
