<?php
/**
 * Locate Widget class
 *
 * @project     CWC2
 * @revision    $Id:
 * @purpose     Locate Popup Widget class
 * @author      DM Solutions Group (sfournier@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
include_once(dirname(__FILE__)."/../Widget.php");
include_once (dirname(__FILE__)."/../Button.php");
include_once (dirname(__FILE__)."/../Popup.php");
include_once ( COMMON."wrapper/map_navigator.php" );

/**
 * Locate
 *
 * @desc Locate widget class
 */
class Locate extends CWCWidget
{
    var $moButton;
    var $moPopup;
    var $mnScale = 300000;

    /**
     * Locate
     *
     * Constructor method for the Locate widget.
     */
    function Locate ()
    {
        $this->mszLanguageResource = str_replace("\\","/",dirname(__FILE__))."/Locate.dbf";

        // invoke constructor of parent
        parent::CWCWidget();

        $this->mnPriority = PRIORITY_HIGH;

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The Locate widget provides a dialog box that can be used to search for
postal codes, place names and NTS map sheets and zoom the map to one of
the results of the search.
EOT;

        $this->moButton = new CWCButton($this);
        $this->moPopup = new CWCPopup ($this);
        $this->moPopup->mszLink = $_SESSION['gszCoreWebPath']."/widgets/Locate/Locate.phtml";
        $this->moPopup->mszParam = "&szCallbackFunc=LocateCB";
        $this->maAttributes['ZOOMTOSCALE'] = new IntegerAttribute( "ZOOMTOSCALE", false, 1);
    }

    function InitDefaults()
    {
        parent::InitDefaults();
        //this widget should never belong to a toolset
        $this->maParams["TOOLSET"] = "";
        $this->moButton->InitDefaults();
        $this->moButton->SetOnClick( "onClickLocate" );
        if(isset($this->maParams['ZOOMTOSCALE']))
        {
            $this->mnScale = intval($this->maParams['ZOOMTOSCALE']);
        }
    }

    /**
     * SetMap
     *
     * Set the map session and create a navigation tool.
     */
    function SetMap($oMapSession)
    {
        $this->moMapObject = $oMapSession;
    }

    /**
     * ParseURL
     *
     * Look for the extents in the URL and set the map.
     */
    function ParseURL ()
    {
        $szLatLong = "";
        $bZoomTo = false;
        $bAddPoint = false;
        
        $oMap = $this->moMapObject->oMap;
        
        $szOriginalExtents = trim($oMap->getMetaData( "original_extents" ));
        if ($szOriginalExtents == "")
        {
            $szOriginalExtents = $oMap->extent->minx;
            $szOriginalExtents .= ",".$oMap->extent->miny;
            $szOriginalExtents .= ",".$oMap->extent->maxx;
            $szOriginalExtents .= ",".$oMap->extent->maxy;
            $oMap->setMetaData( "original_extents", $szOriginalExtents );
        }

        
        if ($this->isVarSet( "LOCATE_LATLONG" ))
            $szLatLong = trim( $this->getVar( "LOCATE_LATLONG" ));

        if ($this->isVarSet( "LOCATE_ZOOM_TO" ))
            $bZoomTo = trim( $this->getVar( "LOCATE_ZOOM_TO" ));

        if ($this->isVarSet( "LOCATE_ADD_ELEMENT" ))
            $bAddElement = trim( $this->getVar( "LOCATE_ADD_ELEMENT" ));

        if ($this->isVarSet( "LOCATE_COORD_UNIT" ))
          $szUnit = trim( $this->getVar( "LOCATE_COORD_UNIT" ));
        else
        {
            $szUnit = "LATLONG";
        }

        if ($this->isVarSet( "LOCATE_ELEMENT_TYPE" ))
            $szType = trim( $this->getVar( "LOCATE_ELEMENT_TYPE" ));
        else
          $szType = "POINT";

        if ($szLatLong != "" && ($bZoomTo || $bAddElement))
        {
            if ($this->isVarSet( "NAV_INPUT_COORDINATES" ))
                $this->setVar( "NAV_INPUT_COORDINATES", "");

            $oMapNavigator = new MapNavigator( $this->moMapObject );
            $aTmpGeo = explode ("|", $szLatLong);

            $szPlaceName = $aTmpGeo[0];
/* -------------------------------------------------------------------- */
/*      extract coordinates :                                           */
/*                                                                      */
/*      Point : label|x,y                                               */
/*      Rectangle : label|x1,y1;x2,y2                                   */
/* -------------------------------------------------------------------- */
            if ($szType == "POINT")
              $anCoord = explode (",", $aTmpGeo[1]);
            else if ($szType == "RECTANGLE" || $szType == "FEATURE" )
            {
                $anRectCoord = explode(";", $aTmpGeo[1]);
                $anRectFirstPoint = explode(",", $anRectCoord[0]);
                $anRectSecondPoint = explode(",", $anRectCoord[1]);                
            }

            if ($bAddElement)
            {
                if ($szType == "POINT")
                {
                    // convert pixels to geo if necessary
                    if ( $szUnit == "PIX" || $szUnit == "MAP")
                    {

                        if ($szUnit == "PIX")
                        {
                        // convert pixel to geo
                            $dGeoX = $this->pixel_to_geo( $anCoord[0], 0, $oMap->width,
                                                          $oMap->extent->minx,
                                                          $oMap->extent->maxx );
                            $dGeoY = $this->pixel_to_geo( $anCoord[1], 0, $oMap->height,
                                                          $oMap->extent->miny,
                                                          $oMap->extent->maxy, 1);
                        }
                        else
                        {
                            $dGeoX = $anCoord[0];
                            $dGeoY = $anCoord[1];
                        }
                        // reproject the point from map units to EPSG:4326

                        $szTmpMapProj = $oMap->getProjection();
                        $projInObj = ms_newprojectionobj( $szTmpMapProj );
                        $projOutObj = ms_newprojectionobj("init=epsg:4326");
                        $poPoint = ms_newpointobj();
                        $poPoint->setXY( $dGeoX, $dGeoY );
                        $poPoint->project($projInObj, $projOutObj);

                        // set the variables
                        $dGeoX = $poPoint->x;
                        $dGeoY = $poPoint->y;
                    }
                    else //latlong
                    {
                        $dGeoX = $anCoord[0];
                        $dGeoY = $anCoord[1];
                    }

                    $anCoord[0] = $dGeoX;
                    $anCoord[1] = $dGeoY;

/* -------------------------------------------------------------------- */
/*      Extract rendering parameters : If parameters are not found,     */
/*      use the default values.                                         */
/*                                                                      */
/*      Also extract the layer name on which this point is going to     */
/*      be added : 2 cases                                              */
/*        - when this function is called from the Locate interface,     */
/*      the new points are going to be created on a defined tmp cwc     */
/*      layer (cwc_tmp_point)                                           */
/*        - when it is called from the jsapi, the layer name is         */
/*      passed as a parameter.                                          */
/*                                                                      */
/*                                                                      */
/* -------------------------------------------------------------------- */
                    //symbol id
                    $nSymbolId = 6;
                    if($this->isVarSet( "SYMBOL" ) &&
                       $this->getVar( "SYMBOL" ) >=0 )
                      $nSymbolId =  $this->getVar( "SYMBOL" );

                    //symbol colour
                    $szSymbolColour = "0,0,0";
                    if($this->isVarSet( "SYMBOL_COLOUR" ) &&
                       $this->getVar( "SYMBOL_COLOUR" ) != "" )
                      $szSymbolColour =  $this->getVar( "SYMBOL_COLOUR" );

                    //symbol size
                    $nSymbolSize = 20;
                    if($this->isVarSet( "SYMBOL_SIZE" ) &&
                       $this->getVar( "SYMBOL_SIZE" ) >0 )
                      $nSymbolSize =  $this->getVar( "SYMBOL_SIZE" );

                    //symbol outiline colour
                    $szSymbolOutlineColour = "255,255,255";
                    if($this->isVarSet( "SYMBOL_OUTLINECOLOUR" ) &&
                       $this->getVar( "SYMBOL_OUTLINECOLOUR" ) !="" )
                      $szSymbolOutlineColour =
                        $this->getVar( "SYMBOL_OUTLINECOLOUR" );

                    //font
                    $szFont = "arial";
                    if($this->isVarSet( "FONT" ) &&
                       $this->getVar( "FONT" ) !="" )
                      $szFont =  $this->getVar( "FONT" );

                    //font size
                    $nFontSize = MS_MEDIUM;
                    if($this->isVarSet( "FONT_SIZE" ) &&
                       $this->getVar( "FONT_SIZE" ) >= 0 &&
                       $this->getVar( "FONT_SIZE" ) <= 4)
                      $nFontSize =  $this->getVar( "FONT_SIZE" );

                    $szFontcolour = "0,0,0";
                    if($this->isVarSet( "FONT_COLOUR" ) &&
                       $this->getVar( "FONT_COLOUR" ) != "" )
                      $szFontcolour =  $this->getVar( "FONT_COLOUR" );

                    $szFontoutcolour = "255,255,255";
                    if($this->isVarSet( "FONT_OUTLINECOLOUR" ) &&
                       $this->getVar( "FONT_OUTLINECOLOUR" ) != "" )
                      $szFontoutcolour =  $this->getVar( "FONT_OUTLINECOLOUR" );

                    $nLabelPos = MS_CL;
                    if($this->isVarSet( "LABEL_POSITION" ) &&
                       $this->getVar( "LABEL_POSITION" ) >= 0 &&
                       $this->getVar( "LABEL_POSITION" ) <= 9)
                      $nLabelPos =  $this->getVar( "LABEL_POSITION" );

                    $nXOff = 5;
                    if($this->isVarSet( "LABEL_X_OFF" ) &&
                       $this->getVar( "LABEL_X_OFF" ) >= 0)
                      $nXOff = $this->getVar( "LABEL_X_OFF" );

                    $nYOff = 0;
                    if($this->isVarSet( "LABEL_Y_OFF" ) &&
                       $this->getVar( "LABEL_Y_OFF" ) >= 0)
                      $nYOff = $this->getVar( "LABEL_Y_OFF" );

                    $szLayerName = "cwc_tmp_point";
                    if ($this->isVarSet( "LAYER_NAME" ) &&
                       $this->getVar( "LAYER_NAME" ) != "")
                    {
                        $szLayerName = $this->getVar( "LAYER_NAME" );
                    }

                    $szProjection = "init=epsg:4326";
                    if ($this->isVarSet( "LAYER_PROJECTION" ))
                      $szProjection = $this->getVar( "LAYER_PROJECTION" );

                    $nNewId = $this->addPoint( $anCoord[0], $anCoord[1],
                                               array(-1,
                                                     $anCoord[0], $anCoord[1],
                                                     $nSymbolId, $szSymbolColour,
                                                     $nSymbolSize,
                                                     $szSymbolOutlineColour,
                                                     $szPlaceName, $szFont,
                                                     $nFontSize,
                                                     $szFontcolour,
                                                     $szFontoutcolour, "",
                                                     $nLabelPos, $nXOff, $nYOff),
                                               $szLayerName, $szProjection, 
                                               $szPlaceName );
                    $oLayer = $oMap->getlayerbyname($szLayerName);
                    $this->drawPoint($oLayer, $nNewId);
                }
                else if ($szType = "RECTANGLE")
                {
                    $aTmpGeo = explode ("|", $szLatLong);
                    $szPlaceName = $aTmpGeo[0];
                    
                    if ( $szUnit == "PIX" || $szUnit == "MAP")
                    {
                        if ($szUnit == "PIX")
                        {
                            $anRectFirstPoint[0] =
                              $this->pixel_to_geo($anRectFirstPoint[0],
                                                  0, $oMap->width,
                                                  $oMap->extent->minx,
                                                  $oMap->extent->maxx );

                            $anRectFirstPoint[1]=
                              $this->pixel_to_geo($anRectFirstPoint[1], 0,
                                                  $oMap->height,
                                                  $oMap->extent->miny,
                                                  $oMap->extent->maxy, 1);
                            $anRectSecondPoint[0] =
                              $this->pixel_to_geo($anRectSecondPoint[0],
                                                  0, $oMap->width,
                                                  $oMap->extent->minx,
                                                  $oMap->extent->maxx );

                            $anRectSecondPoint[1]=
                              $this->pixel_to_geo($anRectSecondPoint[1], 0,
                                                  $oMap->height,
                                                  $oMap->extent->miny,
                                                  $oMap->extent->maxy, 1);
                        }

                        $szTmpMapProj = $oMap->getProjection();
                        $projInObj = ms_newprojectionobj( $szTmpMapProj );
                        $projOutObj = ms_newprojectionobj("init=epsg:4326");
                        $poRect = ms_newrectobj();
                        $poRect->setextent($anRectFirstPoint[0],
                                           $anRectFirstPoint[1],
                                           $anRectSecondPoint[0],
                                           $anRectSecondPoint[1]);
                        $poRect->project($projInObj, $projOutObj);

                    }
                    else
                    {
                        $poRect = ms_newrectobj();
                        $poRect->setextent($anRectFirstPoint[0],
                                           $anRectFirstPoint[1],
                                           $anRectSecondPoint[0],
                                           $anRectSecondPoint[1]);
                    }
                    $szCoords = "".$poRect->minx.",".$poRect->miny.";".$poRect->maxx.",".$poRect->maxy;

/* -------------------------------------------------------------------- */
/*      extract parameters for the drawing.                             */
/* -------------------------------------------------------------------- */
                    $nSymbol = 0;
                    if ($this->isVarSet( "SYMBOL" ) &&
                        $this->getVar( "SYMBOL" ) >= 0)
                      $nSymbol = $this->getVar( "SYMBOL" );

                    $nSymbolSize = 0;
                    if ($this->isVarSet( "SYMBOL_SIZE" ) &&
                        $this->getVar( "SYMBOL_SIZE" ) >= 0)
                      $nSymbolSize = $this->getVar( "SYMBOL_SIZE" );

                    $szColour = "0,0,128";
                    if ($this->isVarSet( "COLOUR" ) &&
                        $this->getVar( "COLOUR" ) != "")
                      $szColour = $this->getVar( "COLOUR" );

                    $szLayerName = "cwc_tmp_line";
                    if ($this->isVarSet( "LAYER_NAME" ) &&
                       $this->getVar( "LAYER_NAME" ) != "")
                    {
                        $szLayerName = $this->getVar( "LAYER_NAME" );
                    }

                    $szProjection = "init=epsg:4326";
                    if ($this->isVarSet( "LAYER_PROJECTION" ))
                      $szProjection = $this->getVar( "LAYER_PROJECTION" );

                    $nNewId = $this->addRectangle( $poRect,
                                                   array(-1, $szCoords,
                                                         $nSymbol, $szColour,
                                                         $nSymbolSize),
                                                   $szLayerName, $szProjection, $szPlaceName);
                    $oLayer = $oMap->getlayerbyname($szLayerName);
                    $this->drawRectangle($oLayer, $nNewId);


                }

            }

            if ($bZoomTo)
            {
                // reproject full_extents metadata if different from projection
                $szOriginalProjection = $oMap->getMetaData("original_projection");
                $szOriginalExtents = $oMap->getMetaData("original_extents");
                
                //if there is no metadata, use the map's projection.
                if (strlen($szOriginalProjection) == 0)
                {
                    $szOriginalProjection = $oMap->getProjection();
                }
                
                if (strlen($szOriginalExtents) > 0)
                {
                    $aszExtents = explode(",", $szOriginalExtents);
                    if (count($aszExtents) == 4)
                    {
                        $aszExtents = $oMapNavigator->reprojectExtentFromCenter($aszExtents, $oMap->width, $oMap->height, $szOriginalProjection,$oMap->getProjection());

                        $oMap->setMetadata("full_extents", $aszExtents[0].",".$aszExtents[1].",".$aszExtents[2].",".$aszExtents[3]);
                    }
                    else
                    {
                        $_SESSION['gErrorManager']->setError(ERR_WARNING,
                           trim($this->moMLT->get("15", "ERROR: Invalid extent specified in Locate.widget.php.")));
                    }
                }
                else
                {
                    $_SESSION['gErrorManager']->setError(ERR_WARNING,
                           trim($this->moMLT->get("15", "ERROR: Invalid extent specified in Locate.widget.php.")));
                }
                
                $szExtents="";
                $szExtents = $oMap->getMetaData("full_extents");
                if (strlen($szExtents) > 0)
                {
                    $aExtents = explode(",", $szExtents);
                    if (count($aExtents) == 4)
                    {
                        $oMap->setextent($aExtents[0], $aExtents[1],
                                         $aExtents[2], $aExtents[3]);

                        $aExtents[0] = $oMap->extent->minx;
                        $aExtents[1] = $oMap->extent->miny;
                        $aExtents[2] = $oMap->extent->maxx;
                        $aExtents[3] = $oMap->extent->maxy;
                    }
                }
                else
                {
                    // cant find if no orginal extents.
                    $_SESSION['gErrorManager']->setError(ERR_WARNING,
                          trim($this->moMLT->get("16", "ERROR: No original extents defined in current map in Locate.widget.php.")));
                    return;
                }
                
                $szTmpMapProj = $oMap->getProjection ();
                
                if (isset($anCoord))
                {
                    // reproject the point from EPSG:4326 to map unit
                    $poPoint1 = ms_newpointobj ();
                    $poPoint1->setXY ($anCoord[0], $anCoord[1]);
    
                    if (strcasecmp($szTmpMapProj,"init=epsg:4326") != 0)
                    {
                        $projOutObj = ms_newprojectionobj ($szTmpMapProj);
                        $projInObj = ms_newprojectionobj ("init=epsg:4326");
                        $poPoint1->project ($projInObj, $projOutObj);
                    }
                    
                    if (count($anCoord) == 3)
                    {
                        $nScale = $anCoord[2];
                    }
                    $nx1 = $this->geo_to_pixel ($poPoint1->x, 0, $oMap->width,
                                                $aExtents[0],
                                                $aExtents[2]);
                    $ny1 = $this->geo_to_pixel ($poPoint1->y, 0, $oMap->height,
                                                $aExtents[1],
                                                $aExtents[3]);
                    
                    // zoom
                    $oMapNavigator->zoomScale ($this->mnScale, $nx1,
                                               $oMap->height - $ny1);
                                           
                }
                else if (isset($anRectCoord))
                {
                    $poRect = ms_newrectobj();
                    $poRect->setextent( $anRectFirstPoint[0], $anRectFirstPoint[1],
                                        $anRectSecondPoint[0], $anRectSecondPoint[1] );
                                        
                    if (strcasecmp($szTmpMapProj,"init=epsg:4326") != 0)
                    {
                        $projOutObj = ms_newprojectionobj ($szTmpMapProj);
                        $projInObj = ms_newprojectionobj ("init=epsg:4326");
                        $poRect->project ($projInObj, $projOutObj);
                    }
                    $this->moMapObject->oMap->setextent( $poRect->minx, $poRect->miny,
                                                         $poRect->maxx, $poRect->maxy );
                }
               
            }
        }
        
        // return success
        return true;
    }

    /**
     * GetJavascriptFunctions
     *
     * Build and return the array of functions needed in the
     * widget.
     */
    function GetJavascriptFunctions ()
    {
        $aReturn = array();

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        $szJsFunctionName = "onClickLocate";
        $szFunction = "function $szJsFunctionName() \n".
            "{\n".$this->moPopup->DrawPublish ()."\n"."return;\n"."}\n";
        $aReturn[$szJsFunctionName] = $szFunction;

        $szJsFunctionName = "LocateCB";
        $szFunction = <<<EOF
        function {$szJsFunctionName}(actionID, wh)
        {
            if (actionID == 1)
            {
                // Here we should handle the load mapcontext
                {$this->mszHTMLForm}.LOCATE_LATLONG.value = wh.szLatLong;
                {$this->mszHTMLForm}.LOCATE_ADD_ELEMENT.value = wh.bAddPoint;
                {$this->mszHTMLForm}.LOCATE_ZOOM_TO.value = wh.bZoomTo;
                {$this->mszHTMLForm}.LOCATE_ELEMENT_TYPE.value = wh.szElementType;
                if ({$bCWCJSAPI})
                {
                    aLabelCoords = wh.szLatLong.split("|");
                    szCoords = aLabelCoords[1];
                    aCoords = szCoords.split(",");
                    if ({$this->mszHTMLForm}.LOCATE_ELEMENT_TYPE.value == "POINT")
                    {
                        goCWCJSAPI.oMap.AddPointWidget(aCoords[0], aCoords[1],
                                             aLabelCoords[0], "LATLONG",
                                             wh.bAddPoint, wh.bZoomTo);
                    }
                    else if ({$this->mszHTMLForm}.LOCATE_ELEMENT_TYPE.value = "RECTANGLE" )
                    {
                        goCWCJSAPI.oMap.AddRectangleWidget(aCoords[0], aCoords[1],
                                             aCoords[2], aCoords[3],
                                             aLabelCoords[0], "LATLONG",
                                             wh.bAddPoint, wh.bZoomTo);
                    }
                }
                else
                {
                    {$this->mszHTMLForm}.submit ();
                }
            }
            else if (actionID == 0)
            {
                // Here we should handle the load mapcontext
                {$this->mszHTMLForm}.LOCATE_LATLONG.value = wh.szLatLong;
                {$this->mszHTMLForm}.LOCATE_ADD_ELEMENT.value = wh.bAddPoint;
                {$this->mszHTMLForm}.LOCATE_ZOOM_TO.value = wh.bZoomTo;
                {$this->mszHTMLForm}.LOCATE_ELEMENT_TYPE.value = wh.szElementType;
                wh.close ();

                if ({$bCWCJSAPI})
                {
                    aLabelCoords = wh.szLatLong.split("|");
                    szCoords = aLabelCoords[1];
                    aCoords = szCoords.split(",");

                    if ({$this->mszHTMLForm}.LOCATE_ELEMENT_TYPE.value == "POINT")
                    {
                        goCWCJSAPI.oMap.AddPointWidget(aCoords[0], aCoords[1],
                                             aLabelCoords[0], "LATLONG",
                                             wh.bAddPoint, wh.bZoomTo);
                    }
                    else if ({$this->mszHTMLForm}.LOCATE_ELEMENT_TYPE.value = "RECTANGLE" )
                    {
                        goCWCJSAPI.oMap.AddRectangleWidget(aCoords[0], aCoords[1],
                                             aCoords[2], aCoords[3],
                                             aLabelCoords[0], "LATLONG",
                                             wh.bAddPoint, wh.bZoomTo);
                    }
                }
                else
                {
                    {$this->mszHTMLForm}.submit();
                }
            }
            else if (actionID == 2) // close window
                wh.close ();

            return;
        }

EOF;
        $aReturn[$szJsFunctionName] = $szFunction;

        return $aReturn;
    }

    function GetJavascriptVariables()
    {
        if ($this->mbVisible)
            return $this->moButton->GetJavascriptVariables();
        else return array();
    }

    function GetJavascriptInitFunctions()
    {
        if ($this->mbVisible)
            return $this->moButton->GetJavascriptInitFunctions();
        else return array();
    }

    function GetJavascriptOnLoadFunctions()
    {
        if ($this->mbVisible)
            $aReturn = $this->moButton->GetJavascriptOnLoadFunctions();
        else
            $aReturn = array();
        return $aReturn;
    }

    function GetJavascriptIncludeFunctions()
    {
        if ($this->mbVisible)
            $aReturn = $this->moButton->GetJavascriptIncludeFunctions();
        else
            $aReturn = array();
        return $aReturn;
    }
    /**
     * GetHTMLHiddenVariables
     *
     * Return HTML hidden variables.
     */
    function GetHTMLHiddenVariables ()
    {
        $aReturn = $this->moButton->GetHTMLHiddenVariables();
        
        $szVariable = "LOCATE_ZOOM_TO";
        /*        if ($this->isVarSet( $szVariable ))
           $szVal = $this->getVar( $szVariable );
           else */
        $szVal = "";
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"$szVal\">\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "LOCATE_ADD_ELEMENT";
        /*        if ($this->isVarSet( $szVariable ))
           $szVal = $this->getVar( $szVariable );
           else */
        $szVal = "";
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"$szVal\">\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "LOCATE_LATLONG";
        /*        if ($this->isVarSet( $szVariable ))
           $szVal = $this->getVar( $szVariable );
           else */
        $szVal = "";
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"$szVal\">\n";
        $aReturn[$szVariable] = $szValue;
        
        $szVariable = "LOCATE_ELEMENT_TYPE";
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        

        return $aReturn;
    }

    /**
     * DrawPublish
     *
     * Return the HTML code to display the link to the legend popup
     */
    function DrawPublish ()
    {
        
        if (!$this->mbVisible)
            return "<!-- Locate widget hidden -->";

        if (!isset($_SESSION['WAITIMAGE']))
        {
            $oApp = GetChameleonApplication();
            if (isset($this->maSharedResourceWidgets["WaitImage"]->maszContents["WAITIMAGE"]))
            {
                foreach($this->maSharedResourceWidgets["WaitImage"]->maszContents["WAITIMAGE"] as $aWaitImage)
                {
                    if (isset($aWaitImage["LANGUAGE"]) && $aWaitImage["LANGUAGE"] == $_SESSION["gszCurrentLanguage"])
                    {
                        if (isset($aWaitImage["WAITIMAGE"]))
                        {
                            $wait = $aWaitImage["WAITIMAGE"];
                            if (strcasecmp( "http://", substr($wait, 0, 7)) == 0 ||
                                strcasecmp( "/", substr($wait, 0, 1)) == 0 ||
                                strcasecmp( ":", substr($wait, 1, 1)) == 0 )
                                $_SESSION["WAITIMAGE"] = $aWaitImage["WAITIMAGE"];
                            else
                                $_SESSION["WAITIMAGE"] = $oApp->findFile($aWaitImage["WAITIMAGE"]);
                                $_SESSION["WAITIMAGE"] = $oApp->fileSystemToURL( $_SESSION['WAITIMAGE'] );
                        }
                        if (isset($aWaitImage["WAITIMAGEWIDTH"]))
                            $_SESSION["WAITIMAGEWIDTH"] = $aWaitImage["WAITIMAGEWIDTH"];
                        if (isset($aWaitImage["WAITIMAGEHEIGHT"]))
                            $_SESSION["WAITIMAGEHEIGHT"] = $aWaitImage["WAITIMAGEHEIGHT"];
                    }
                }
            }
        }

        $szReturn = $this->moButton->DrawPublish();

        return $szReturn;
    }

    /**
     * convert a pixel position to geocoded position
     *
     * @param nPixPos double pixel position
     * @param dfPixMin double minimum map pixel value
     * @param dfPixMax double maximum map pixel value
     * @param dfGeoMin double minimum map geocoded value
     * @param dfGeoMax double maximum map geocoded value
     * @param nInversePix integer optional flag to inverse , set to 1 for
     *                            Y pixel coordinates where UL > LR
     * @return double geocoded position
     */
    function pixel_to_geo($nPixPos, $dfPixMin, $dfPixMax, $dfGeoMin, $dfGeoMax,
                          $nInversePix = "")
    {
        // calculate the geocoded & pixel width
        $dfWidthGeo = $dfGeoMax - $dfGeoMin;
        $dfWidthPix = $dfPixMax - $dfPixMin;

        // get ratio
        $dfPixToGeo = $dfWidthGeo / $dfWidthPix;

    // get difference
        if (!$nInversePix)
            $dfDeltaPix = $nPixPos - $dfPixMin;
        else
            $dfDeltaPix = $dfPixMax - $nPixPos;

        // calculate
        $dfDeltaGeo = $dfDeltaPix * $dfPixToGeo;
        $dfPosGeo = $dfGeoMin + $dfDeltaGeo;

        // return value
        return ($dfPosGeo);

        // end pixel_to_geo function
    }


    /**
     * convert a geocoded position to pixel coord
     *
     * @param nGeoPos double Geocoded position
     * @param dfPixMin double minimum map pixel value
     * @param dfPixMax double maximum map pixel value
     * @param dfGeoMin double minimum map geocoded value
     * @param dfGeoMax double maximum map geocoded value
     * @param nInverseGeo integer optional flag to inverse , set to 1 for
     *                            Y pixel coordinates where UL > LR
     * @return double geocoded position
     */
    function geo_to_pixel ($nGeoPos, $dfPixMin, $dfPixMax, $dfGeoMin,
                           $dfGeoMax, $nInverseGeo = "")
    {
        // calculate the geocoded & pixel width
        $dfWidthGeo = abs($dfGeoMax - $dfGeoMin);
        $dfWidthPix = abs($dfPixMax - $dfPixMin);

        // get ratio
        $dfGeoToPix = $dfWidthPix / $dfWidthGeo;

        // get difference
        if (!$nInverseGeo)
            $dfDeltaGeo = $nGeoPos - $dfGeoMin;
        else
            $dfDeltaGeo = $dfGeoMax - $nGeoPos;

        // calculate
        $dfDeltaPix = $dfDeltaGeo * $dfGeoToPix;
        $dfPosPix = $dfPixMin + $dfDeltaPix;

        // return value
        return round ($dfPosPix);

        // end pixel_to_geo function
    }

    /**
     * This function adds a new record for the co-ordinates given.
     *
     * @param $nPixX integer - X pixel position to add point at.
     * @param $nPixY integer - Y pixel position to add point at.
     * @return boolean - True is successful, false if not.
     **/
    function addPoint( $dX, $dY, $axValues="", $szLayerName, $szProjection="", $szPlaceName)
    {
        // get the map object
        $oMap = $this->moMapObject->getMapObj();

        $oLayer = false;
        $aszLayerName = $oMap->getAllLayerNames();
        foreach ($aszLayerName as $szKey => $szTmp)
            if ($szTmp == $szLayerName)
                $oLayer = $oMap->getlayer($szKey);

        if ($oLayer === false)
        {
            $oLayer = ms_newLayerObj( $oMap );
            $oLayer->set( "name", $szLayerName);
            $oLayer->setMetaData("wms_title", "Locations");
            $oLayer->set( "type", MS_LAYER_POINT );
            $oLayer->set( "status", MS_ON );
            $szData = $_SESSION['gszTmpPath'].session_id();// . uniqid("");
            $oLayer->set( "data",  $szData);
            $oLayer->set("labelitem", "label");
            $oLayer->set("labelcache", MS_ON);
            $oLayer->set("classitem", "id" );
            //the projection is set only for cwc_tmp_point layer
            //used by the locate widget
            if ($szProjection != "")
              $oLayer->setprojection($szProjection);

            $_SESSION['gszCurrentState'] = $this->moMapObject->saveState();
        }
        else if ($oLayer->data == "")
        {
            $szData = $_SESSION['gszTmpPath'].session_id();// . uniqid("");
            $oLayer->set( "data",  $szData);
            $oLayer->set("labelitem", "label");
            $oLayer->set("labelcache", MS_ON);
            $oLayer->set("classitem", "id" );
        }

        clearstatcache();
        // check if we have to create the shape file and dbf
        if (!file_exists($oLayer->data.".dbf"))
        {
            // create the temp shapefile files
            $shpFname = $oLayer->data;

            $shpFile = ms_newShapeFileObj( $shpFname, MS_SHP_POINT );

            // create an array for the dbf structure
            $aDbfStruct =   array(
                                  array("id", "N", 7, 0),
                                  array("x","N", 19, 10),
                                  array("y","N", 19, 9),
                                  array("symbol","N", 3, 0),
                                  array("s_colour","C", 11),
                                  array("s_size","N", 3, 0),
                                  array("s_ocolour","C", 11),
                                  array("label","C", 255),
                                  array("font","C", 50),
                                  array("f_size","N", 3, 0),
                                  array("f_colour","C", 11),
                                  array("f_ocolour","C", 11),
                                  array("marquee","C", 3),
                                  array("position","C", 25),
                                  array("x_off","N", 8, 0),
                                  array("y_off","N", 8, 0)
                                  );

            // create the dbase file
            $dbfFile = dbase_create( $shpFname.".dbf", $aDbfStruct);

            // close the dbf
            dbase_close( $dbfFile );

            // release resources
            $shpFile->free();
        }


        // open the dbase file
        $dbfFile = dbase_open($oLayer->data.".dbf", 2);

        // get the record count
        $nCount = dbase_numrecords( $dbfFile );

        // determine the ID
        if ( $nCount > 0 )
        {
            // get the last record
            $aLastRec = dbase_get_record_with_names( $dbfFile, $nCount);
            $nNewId = $aLastRec["id"] + 1;
        }
        else
            $nNewId = 1;

        $axValues[0] = $nNewId;

        // write the attributes record
        dbase_add_record( $dbfFile, array_values( $axValues ) );

        // update count
        $nCount++;

        // create a new one
        //$shpFile = ms_newShapeFileObj($oLayer->data, MS_SHP_POINT );
        //$shpFile->free();

        $aRec = dbase_get_record( $dbfFile, $nCount);
        $shpFile = ms_newShapeFileObj($oLayer->data, -2 );
        $oShp = ms_newShapeObj(MS_SHAPE_POINT);
        $oLine = ms_newLineObj();
        $oLine->addXY( $aRec[1], $aRec[2] );
        $oShp->add( $oLine );
        $shpFile->addShape( $oShp );
        // release resources
        $shpFile->free();

        // loop through the dbase file and re-build the shape file
        /*
        for($i = 1; $i <= $nCount; $i++)
        {
            // get the next record
            $aRec = dbase_get_record( $dbfFile, $i );
            if ($aRec['deleted'] != 1)
            {
                // get the shape objects
                $shpFile = ms_newShapeFileObj($oLayer->data, -2 );
                $oShp = ms_newShapeObj(MS_SHAPE_POINT);
                $oLine = ms_newLineObj();
                $oLine->addXY( $aRec[1], $aRec[2] );
                $oShp->add( $oLine );
                $shpFile->addShape( $oShp );

                // release resources
                $shpFile->free();
            }
        }
        */

        // close the dbf
        dbase_close( $dbfFile );

        // return success
        return $nNewId;

        // end addNewPoint() function
    }


    /**
     * This function will draw the point in the temp dbase file onto the map specify by newId.
     *
     * @return boolean - true if successful, false if not.
     **/
    function drawPoint($o_tmp_layer, $nNewId)
    {
        // get the map object
        $oMap = $this->moMapObject->getMapObj();

        // open the dbase file
        $dbfFile = dbase_open( $o_tmp_layer->data . ".dbf", 0 );

        // get record count
        $nCount = dbase_numrecords( $dbfFile );


       // only continue if there are points
        if ( $nCount < 1 )
        {
            // close and return
            dbase_close( $dbfFile );
            return true;
        }


        // loop through each point and add a class
        for ( $i=1; $i<=$nCount; $i++ )
        {
            // get tne next record
            $aRec = dbase_get_record_with_names( $dbfFile, $i);

            if ($aRec['id'] != $nNewId) continue;

            // create the class object
            $o_tmp_class = ms_newClassObj($o_tmp_layer);
            $o_tmp_style = ms_newStyleObj($o_tmp_class);

            // get the colour values
            $aSColour = explode( ",", $aRec["s_colour"] );
            $aSOColour = explode( ",", $aRec["s_ocolour"] );
            $aFColour = explode( ",", $aRec["f_colour"] );
            $aFOColour = explode( ",", $aRec["f_ocolour"] );

            //set class properties
            $o_tmp_style->color->setRGB($aSColour[0], $aSColour[1],
                                        $aSColour[2]);
            $o_tmp_style->outlinecolor->setRGB($aSOColour[0], $aSOColour[1],
                                               $aSOColour[2]);
            $o_tmp_style->set("minsize" , $aRec["s_size"] );
            $o_tmp_style->set("maxsize" , $aRec["s_size"] );
            $o_tmp_style->set("size"    , $aRec["s_size"] );
            $o_tmp_style->set("symbol"  , $aRec["symbol"] );
            $o_tmp_class->set("template"  , "ttt" );
            $o_tmp_class->setexpression( "([id] = ".$aRec["id"].")" );

            // set the label properties
            $nTmpPos =  $aRec["position"];
            eval ("\$nTmpPos = $nTmpPos;");
            $o_tmp_class->label->set("position", $nTmpPos );
            $o_tmp_class->label->set("mindistance", 100 );
            $o_tmp_class->label->set("offsetx", $aRec["x_off"]);
            $o_tmp_class->label->set("offsety", $aRec["y_off"]);
            $o_tmp_class->label->set("buffer", 2 );

            $o_tmp_class->label->set("size", 10);
            //$o_tmp_class->label->set("size", MS_MEDIUM );
            //$o_tmp_class->label->set("minsize", $aRec["f_size"] );
            //$o_tmp_class->label->set("minsize", MS_SMALL);
            //$o_tmp_class->label->set("maxsize", $aRec["f_size"] );
            //$o_tmp_class->label->set("maxsize", MS_MEDIUM);
            $o_tmp_class->label->set("partials", MS_TRUE );
            $o_tmp_class->label->set("force", MS_TRUE );
            $o_tmp_class->label->color->setRGB($aFColour[0], $aFColour[1],
                                               $aFColour[2]);
            $o_tmp_class->label->outlinecolor->setRGB($aFOColour[0],
                                                      $aFOColour[1],
                                                      $aFOColour[2]);
            $o_tmp_class->label->set("font", "sans" );
            $o_tmp_class->label->set("type", MS_TRUETYPE );
            //$o_tmp_class->label->set("type", MS_BITMAP);

            // add the marquee if necessary
            if ( strtoupper( trim( $aRec["marquee"] ) ) == "ON" )
            {
                $o_tmp_class->label->backgroundcolor->setRGB(255,255,204);
                $o_tmp_class->label->backgroundshadowcolor->setRGB(0, 0, 0);
                $o_tmp_class->label->set("backgroundshadowsizex", 1 );
                $o_tmp_class->label->set("backgroundshadowsizey", 1 );
            }
        }

        // close the dbase file
        dbase_close( $dbfFile );

        // return success
        return true;

        // end drawPoints() function
    }

    /**
     * This function adds a new record for the co-ordinates given.
     * Used to add a rectangular element.
     * @param $nPixX integer - X pixel position to add point at.
     * @param $nPixY integer - Y pixel position to add point at.
     * @return boolean - True is successful, false if not.
     **/
    function addRectangle( $oRect, $axValues="", $szLayerName, $szProjection="", $szPlaceName)
    {
        // get the map object
        $oMap = $this->moMapObject->getMapObj();

        $oLayer = false;
        $aszLayerName = $oMap->getAllLayerNames();
        foreach ($aszLayerName as $szKey => $szTmp)
        if ($szTmp == $szLayerName)
            $oLayer = $oMap->getlayer($szKey);

        if ($oLayer === false)
        {
            $oLayer = ms_newLayerObj( $oMap );
            $oLayer->set( "name", $szLayerName );
            $oLayer->set( "type", MS_LAYER_POLYGON );
            $oLayer->set( "status", MS_ON );
            $szData = $_SESSION['gszTmpPath'].session_id();// . uniqid("");
            $oLayer->set( "data",  $szData);
            $oLayer->set("classitem", "id" );
            $oLayer->set("labelitem", "label" );
            if ($szProjection != "")
              $oLayer->setprojection( "init=epsg:4326" );

            $_SESSION['gszCurrentState'] = $this->moMapObject->saveState();
        }
        else if ($oLayer->data == "")
        {
            $szData = $_SESSION['gszTmpPath'].session_id();// . uniqid("");
            $oLayer->set( "data",  $szData);
            $oLayer->set("classitem", "id" );
        }

        clearstatcache();
        if (!file_exists($oLayer->data.".dbf"))
        {
            // create the temp shapefile files
            $shpFname = $oLayer->data;
            $shpFile = ms_newShapeFileObj( $shpFname, MS_SHP_ARC);

            // create an array for the dbf structure
            $aDbfStruct =   array(
                                  array("id", "N", 7, 0),
                                  array("coord","C", 255),
                                  array("symbol","N", 3, 0),
                                  array("colour","C", 11),
                                  array("size","N", 3, 0),
                                  array("label","C", 255 )
                                  );
            // create the dbase file
            $dbfFile = dbase_create( $shpFname.".dbf", $aDbfStruct);

            // close the dbf
            dbase_close( $dbfFile );

            // release resources
            $shpFile->free();
        }

        // open the dbase file
        $dbfFile = dbase_open($oLayer->data.".dbf", 2);

        // get the record count
        $nCount = dbase_numrecords( $dbfFile );
        
        array_push( $axValues, $szPlaceName );
        
        // determine the ID
        if ( $nCount > 0 )
        {
            // get the last record
            $aLastRec = dbase_get_record_with_names( $dbfFile, $nCount);
            $nNewId = $aLastRec["id"] + 1;
        }
        else
            $nNewId = 1;

        $axValues[0] = $nNewId;

        // write the attributes record
        dbase_add_record( $dbfFile, array_values( $axValues ) );

        // update count
        $nCount++;

        // create a new one
        $shpFile = ms_newShapeFileObj($oLayer->data, MS_SHP_POLYGON );
        $shpFile->free();

        // loop through the dbase file and re-build the shape file
        for($i = 1; $i <= $nCount; $i++)
        {
            // get the next record
            $aRec = dbase_get_record( $dbfFile, $i );
            if ($aRec['deleted'] != 1)
            {
                // get the shape objects
                $shpFile = ms_newShapeFileObj($oLayer->data, -2 );
                $oShp = ms_newShapeObj(MS_SHAPE_POLYGON);
                $oLine = ms_newLineObj();

                //cordinates are x1,y1;x2,y2
                $szCoords = $aRec[1];
                $aCoords = explode(";",  $szCoords);
                $aFirstPoint = explode("," , $aCoords[0]);
                $aSecondPoint = explode("," , $aCoords[1]);
                $nX1 = $aFirstPoint[0];
                $nY1 = $aFirstPoint[1];
                $nX2 = $aSecondPoint[0];
                $nY2 = $aSecondPoint[1];
                /*
                $oLine->addXY($aFirstPoint[0], $aFirstPoint[1]); //xmin,ymin
                $oLine->addXY($aSecondPoint[0], $aFirstPoint[1]); //xmax, ymin
                $oLine->addXY($aSecondPoint[0], $aSecondPoint[1]); //xmax, ymax
                $oLine->addXY($aFirstPoint[0], $aSecondPoint[1]); //xmin, ymax
                $oLine->addXY($aFirstPoint[0], $aFirstPoint[1]); //xmin,ymin
                */
                $nPoints = 20;
                $dx = ($nX2 - $nX1) / ($nPoints - 1);
                for( $j=0; $j<$nPoints; $j++)
                {
                    $oLine->addXY( $nX1 + ($j * $dx), $nY1 );
                }
                $dy = ($nY2 - $nY1) / ($nPoints - 1);
                for( $j=1; $j<$nPoints; $j++)
                {
                    $oLine->addXY( $nX2, $nY1 + ($j * $dy) );
                }
                $dx = ($nX1 - $nX2) / ($nPoints - 1);
                for( $j=1; $j<$nPoints; $j++)
                {
                    $oLine->addXY( $nX2 + $j*$dx, $nY2 );
                }
                $dy = ($nY1 - $nY2) / ($nPoints - 1);
                for( $j=1; $j<$nPoints; $j++)
                {
                    $oLine->addXY( $nX1, $nY2 + ($j*$dy));
                }
                
                
                $oShp->add( $oLine );
                $shpFile->addShape( $oShp );

                // release resources
                $shpFile->free();
            }
        }

        // close the dbf
        dbase_close( $dbfFile );

        // return success
        return $nNewId;

        // end addNewRectangle() function
    }

    /**
     * This function will draw the recatangle in the temp dbase file onto
     * the map specify by newId.
     *
     * @return boolean - true if successful, false if not.
     **/
    function drawRectangle($o_tmp_layer, $nNewId)
    {
        // get the map object
        $oMap = $this->moMapObject->getMapObj();

        // open the dbase file
        $dbfFile = dbase_open($o_tmp_layer->data . ".dbf", 0 );

        // get record count
        $nCount = dbase_numrecords( $dbfFile );

        // only continue if there are points
        if ( $nCount < 1 )
        {
            // close and return
            dbase_close( $dbfFile );
            return true;
        }
        
        $o_tmp_layer->set( 'transparency', 40 );
            
        // loop through each element and add a class
        for ( $i=1; $i<=$nCount; $i++ )
        {
            // get the next record
            $aRec = dbase_get_record_with_names( $dbfFile, $i);

            if ($aRec['id'] != $nNewId) continue;

            // create the class object
            $o_tmp_class = ms_newClassObj($o_tmp_layer);
            $o_tmp_style = ms_newStyleObj($o_tmp_class);
            $o_tmp_style->set( 'size', 3 );
            $aSColour = explode( ",", $aRec["colour"] );
            $o_tmp_style->outlinecolor->setRGB($aSColour[0], $aSColour[1],
                                        $aSColour[2]);
            $o_tmp_style->color->setRGB( 255, 255, 255 );
            //$o_tmp_style->set("size"    , $aRec["size"] );
            $o_tmp_style->set("symbol"  , 0 );
            $o_tmp_class->set("template"  , "ttt" );
            $o_tmp_class->setexpression( "([id] = ".$aRec["id"].")" );
                        
            $o_tmp_class->label->set("position", MS_AUTO );
            $o_tmp_class->label->set("buffer", 2 );

            $o_tmp_class->label->set("size", MS_LARGE);
            $o_tmp_class->label->set("partials", MS_TRUE );
            $o_tmp_class->label->set("force", MS_TRUE );
            $o_tmp_class->label->color->setRGB( 0,0,0 );
            $o_tmp_class->label->outlinecolor->setRGB(255,255,255);
            $o_tmp_class->label->set("size", 10 );
            $o_tmp_class->label->set("font", "sans" );
            $o_tmp_class->label->set("type", MS_TRUETYPE);

            // add the marquee if necessary
            if ( false )
            {
                $o_tmp_class->label->backgroundcolor->setRGB(255,255,204);
                $o_tmp_class->label->backgroundshadowcolor->setRGB(0, 0, 0);
                $o_tmp_class->label->set("backgroundshadowsizex", 1 );
                $o_tmp_class->label->set("backgroundshadowsizey", 1 );
            }
        }

        // close the dbase file
        dbase_close( $dbfFile );

        // return success
        return true;
    }
}
?>
