<?php
/**
 * KeyMap Widget class
 *
 * @project     CWC2
 * @revision    $Id:
 * @purpose     KeyMap Widget class
 * @author      DM Solutions Group (sfournier@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../NavTool.php");

/**
 * display a Key Map
 */
class KeyMap extends NavTool
{
    var   $mszAnchorName = "keyMapAnchor";
    var   $mszKeyMapImageName = "KeyMapImage";

    function KeyMap()
    {
        parent::NavTool();

        $this->SetNavCommand("KEY_MAP");
        $this->moButton->mszAnchorName = $this->mszAnchorName;

        $this->mnPriority = PRIORITY_HIGH;

        $this->maAttributes["WIDTH"] = new IntegerAttribute( "WIDTH", false, 0 );
        $this->maAttributes["HEIGHT"] = new IntegerAttribute( "HEIGHT", false, 0 );
        $this->maAttributes["IMAGE"] = new StringAttribute( "IMAGE", false );
        $this->maAttributes["COLOR"] = new RGBColorAttribute( "COLOR", false );
        $this->maAttributes["OUTLINECOLOR"] = new RGBColorAttribute( "OUTLINECOLOR", false );
        $this->maAttributes["MINX"] = new FloatAttribute( "MINX", false );
        $this->maAttributes["MINY"] = new FloatAttribute( "MINY", false );
        $this->maAttributes["MAXX"] = new FloatAttribute( "MAXX", false );
        $this->maAttributes["MAXY"] = new FloatAttribute( "MAXY", false );
        $this->maAttributes["SRS"] = new StringAttribute( "SRS", false );

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The KeyMap widget displays a key map (or reference map) that displays the users
currently viewed extents within a larger reference frame.
EOT;
    }

    /**
     * process the url changes
     */
    function  ParseURL()
    {
        parent::ParseURL();
        $szCmd = "";
        if ($this->getVar("NAV_CMD"))
          $szCmd = trim($this->getVar("NAV_CMD"));


        if ($szCmd == "KEY_MAP")
        {
            $oMap = $this->moMapObject->oMap;

            $szInputCoords = "";
            $szInputCoords = $this->getVar("NAV_INPUT_COORDINATES");
            $szInputType = $this->getVar("NAV_INPUT_TYPE");

            if(strlen($szInputCoords) <= 0)
            {
                $szInputCoords = ($oMap->reference->width/2).",".($oMap->reference->height/2);
                $szInputType = "POINT";
            }

            if ($szInputType == "POINT")
            {
                $aPixPos = explode(",", $szInputCoords);
                $this->keymapNavigate ($aPixPos[0], $aPixPos[1] );
            }
            else
            {
                $_SESSION['gErrorManager']->setError(ERR_WARNING,
                   trim($this->moMLT->get("1", "ERROR: Invalid input type specified in"))." KeyMap.php.");
            }
        }
        
        // return success
        return true;
    }

    /**
     * render the keymap widget
     */
    function DrawPublish()
    {
        if (!$this->mbVisible)
            return "<!-- KeyMap Widget hidden -->";

        if ( PHP_OS == "WINNT" || PHP_OS == "WIN32" )
            $szGDModule = "php_gd2.dll";
        else
        {
            $szGDModule = "php_gd2.so";
        }

        if (!extension_loaded("gd"))
            dl($szGDModule);

        $oMap = $this->moMapObject->oMap;

        $nWidth = $oMap->reference->width;
        $nHeight = $oMap->reference->height;
        
        if ($nWidth <= 0)
          $szWidthHTMLTag = "";
        else
          $szWidthHTMLTag = "WIDTH=".$nWidth;

         if ($nHeight <= 0)
          $szHeightHTMLTag = "";
        else
          $szHeightHTMLTag = "HEIGHT=".$nHeight;

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $szOnLoad = "onLoad=\"KeyMapLoaded()\"";
        else
          $szOnLoad = "";
          
        $szCoreWebPath = $_SESSION["gszCoreWebPath"];
        $nMapSessionMode = $_SESSION["gnMapSessionMode"];
        $szKeyMapImageName = $this->mszKeyMapImageName;
        $WEBCOMMON = WEBCOMMON;
        $szSID = SID;
        $szReturn = <<<EOT
<table cellpadding="0" cellspacing="0" border="0">
<tr>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1" id="keymapFrameTL" name="keymapFrameTL"></td>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1"></td>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1" id="keymapFrameTR" name="keymapFrameTR"></td>
  </tr>
  <tr>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1"></td>
    <td><a name="mapanchor"  HREF="javascript:KeyMapClicked()"><IMG SRC="{$WEBCOMMON}/wrapper/drawmap.php?map_session_mode={$nMapSessionMode}&REQUEST=KEYMAP&{$szSID}" {$szWidthHTMLTag} {$szHeightHTMLTag} NAME="{$szKeyMapImageName}" {$szOnLoad} border="0" onmouseover="KeyMapCaptureMouse()" onmouseout="KeyMapReleaseMouse()"></a></td>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1"></td>
  </tr>
  <tr>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1" id="keymapFrameBL" name="keymapFrameBL"></td>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1"></td>
    <td bgcolor="000000"><img src="./images/a_pixel.gif" width="1" height="1" id="keymapFrameBR" name="keymapFrameBR"></td>
  </tr>
</table>
EOT;

        return $szReturn;
    }

    /**
     * SetMap($oMapObject)
     *
     * Set the map object.
     *
     * @param oMapObject
     */
    function SetMap(&$oMapObject)
    {
        $this->moMapObject =& $oMapObject;
        
        $oRef =& $this->moMapObject->oMap->reference;
        $oExt =& $oRef->extent;
        if (isset($this->maParams["MINX"]))
        {
            $oExt->set( "minx",  $this->maParams["MINX"] );
        }
        if (isset($this->maParams["MAXX"]))
        {
            $oExt->set( "maxx",  $this->maParams["MAXX"] );
        }
        if (isset($this->maParams["MINY"]))
        {
            $oExt->set( "miny",  $this->maParams["MINY"] );
        }
        if (isset($this->maParams["MAXY"]))
        {
            $oExt->set( "maxy",  $this->maParams["MAXY"] );
        }
        if (isset($this->maParams["WIDTH"]))
        {
            $oRef->set("width",$this->maParams["WIDTH"]);
        }
        if (isset($this->maParams["HEIGHT"]))
        {
            $oRef->set("height",$this->maParams["HEIGHT"]);
        }
        if (isset($this->maParams["IMAGE"]))
        {
            $oApp = GetChameleonApplication();
            $szPath = $oApp->resolvePath2( $this->maParams['IMAGE'], $_SESSION['gszAppPath'] );
            $oRef->set("image",$szPath);
        }
        if (isset($this->maParams["COLOR"]))
        {
            $aColor = split( " ", $this->maParams["COLOR"] );
            if (count($aColor) != 3)
            {
                if(isset( $_SESSION["gErrorManager"] ))
                {
                    $_SESSION["gErrorManager"]->setError( ERR_WARNING, "invalid color parameter in KeyMap widget" );
                }
            }
            else
            {
                $oRef->color->setRGB( $aColor[0], $aColor[1], $aColor[2] );
            }
        }
        if (isset($this->maParams["OUTLINECOLOR"]))
        {
            $aColor = split( " ", $this->maParams["OUTLINECOLOR"] );
            if (count($aColor) != 3)
            {
                $_SESSION["gErrorManager"]->setError( ERR_WARNING, "invalid outline color parameter in KeyMap widget" );
            }
            else
            {
                $oRef->outlinecolor->setRGB( $aColor[0], $aColor[1], $aColor[2] );
            }
        }
        if (isset($this->maParams["SRS"]))
        {
            //TODO: enable this, but also add init= if SRS starts with epsg:
            //$oRef->setprojection($this->maParams["SRS"]));
            $_SESSION["KEYMAP_SRS"] = $this->maParams["SRS"];
        }
     }

    /**
     * return javascript functions required by the keymap
     */
    function GetJavascriptFunctions()
    {
        $oMap = $this->moMapObject->oMap;

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        $nWidth = $oMap->reference->width;
        $nHeight = $oMap->reference->height;
        
        $szJsFunctionName = "KeyMapTrackMouseXY";
        $szFunction = <<<EOT
function {$szJsFunctionName}(e)
{
    if (navigator.appName == "Microsoft Internet Explorer")
    {
        gnMouseX = event.clientX + document.body.scrollLeft;
        gnMouseY = event.clientY + document.body.scrollTop;
    }
    else
    {
        gnMouseX = e.pageX;
        gnMouseY = e.pageY;
    }
    return true;
}
EOT;

        $aReturn[$szJsFunctionName] = $szFunction;
        
        $szJsFunctionName = "KeyMapCaptureMouse";
        $szFunction = <<<EOT
function {$szJsFunctionName}()
{
    document.onmousemove=KeyMapTrackMouseXY;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

        $szJsFunctionName = "KeyMapReleaseMouse";
        $szFunction = <<<EOT
function {$szJsFunctionName}()
{
    document.onmousemove=null;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;


        $szJsFunctionName = "KeyMapClicked";
        $szFunction = <<<EOT
function {$szJsFunctionName}( )
{
    if (gnMouseX == -1 || gnMouseY == -1)
        return;
        
    var nAnchorX = CWCDHTML_FindObjectPosX( CWCDHTML_GetImage('keymapFrameTL') );
    var nAnchorY = CWCDHTML_FindObjectPosY( CWCDHTML_GetImage('keymapFrameTL') );;
    
    nAnchorX +=1;
    nAnchorY +=1;
    
    nMapX = gnMouseX - nAnchorX;
    nMapY = gnMouseY - nAnchorY;
    
    szCoord= nMapX + "," + nMapY;
    
    {$this->mszHTMLForm}.NAV_INPUT_COORDINATES.value=szCoord;
    {$this->mszHTMLForm}.NAV_INPUT_TYPE.value='POINT';
    {$this->mszHTMLForm}.NAV_CMD.value='KEY_MAP';
    
    if ({$bCWCJSAPI})
    {
        goCWCJSAPI.NAV_INPUT_COORDINATES = szCoord;
        goCWCJSAPI.NAV_INPUT_TYPE = 'POINT';
        goCWCJSAPI.NAV_CMD='KEY_MAP';
        goCWCJSAPI.UpdateNavTools();
    }
    else
    {
        {$this->mszHTMLForm}.submit();
    }
}
EOT;

        $aReturn[$szJsFunctionName] = $szFunction;

        if ($bCWCJSAPI)
        {
            
          $url = //$_SESSION["gszCoreWebPath"].
                     WEBCOMMON."/wrapper/drawmap.php?map_session_mode=".
                     $_SESSION["gnMapSessionMode"].
                     "&run_query=0&REQUEST=KEYMAP&".SID;
            
            $szJsFunctionName = "KeyMapWRegisterForEvent";
            $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called to register and even when the map extents chnages (JSAPI)
 */
function {$szJsFunctionName}()
{
    goCWCJSAPI.RegisterEvent(MAP_EXTENT_CHANGED, "KeyMapWMapExtentsChanged");
}
EOT;

       $aReturn[$szJsFunctionName] = $szFunction;

       $szJsFunctionName = "KeyMapWMapExtentsChanged";
       $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the mapextents are changed to update the scalebar images(JSAPI)
 */
function {$szJsFunctionName}()
{
     var sImgName = "{$this->mszKeyMapImageName}";
     var url = "{$url}";
     var d = new Date();
     var unique = d.getTime() + '' + Math.floor(1000 * Math.random());
     url = url + "&UniqId="+unique;
     document.images[sImgName].src = url;
}
EOT;
       $aReturn[$szJsFunctionName] = $szFunction;

       $szJsFunctionName = "KeyMapLoaded";
       $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the image is finished loading
 */
function {$szJsFunctionName}()
{
    // Get all errors from ErrorManager on server side.
    //goCWCJSAPI.oErrorManager.GetServerErrors();
}
EOT;
       $aReturn[$szJsFunctionName] = $szFunction;
      }
       return $aReturn;
   }


    /**
     * return javascript variables required by the keymap
     */
    function GetJavascriptVariables()
    {
        $aReturn = parent::GetJavascriptVariables();

        //?TODO should we create another function for global code
        $szVariable = "KeyMapWidget".$this->mnId;
        $szValue = "if (navigator.appName != \"Microsoft Internet Explorer\") document.captureEvents(Event.MOUSEMOVE);\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gnMouseX";
        $szValue = "var $szVariable = -1;\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "gnMouseY";
        $szValue = "var $szVariable = -1;\n";
        $aReturn[$szVariable] = $szValue;

        return $aReturn;
    }
    
    function GetJavascriptIncludeFunctions()
    {
        $aReturn = parent::GetJavascriptIncludeFunctions();

        $szVar = 'cwc_dhtml.js';
        $aReturn[$szVar] = '<script language="JavaScript" src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_dhtml.js"></script>';
        return $aReturn;
    }


    /**
     * return an array of javascript functions needed by keymap widget
     * and called when the page is loaded.
     * @return array of name = function values
     */
    function GetJavascriptOnLoadFunctions()
    {
        $aReturn = parent::GetJavascriptOnLoadFunctions();

        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
        {
             $szJsFunctionName = "KeyMapWRegisterForEvent";
             $szFunction = "$szJsFunctionName();\n";
             $aReturn[$szJsFunctionName] = $szFunction;
        }

        return $aReturn;
    }

    /**
     * return variables required by KeyMap
     */
    function GetHTMLHiddenVariables()
    {
        $aReturn = parent::GetHTMLHiddenVariables();
        
        $szVariable = "NAV_INPUT_TYPE";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;

        $szVariable = "NAV_INPUT_COORDINATES";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;

        return $aReturn;
    }

    /**
     * This function takes the keymap click co-ordinates and recenters the map
     * accordingly.
     *
     * @param $nX integer - The keymap X click position in pixels.
     * @param $nY integer - The keymap Y click position in pixels.
     **/
    function keymapNavigate ( $nX, $nY )
    {
        // get local map object
        $oMap = $this->moMapObject->getMapObj();
        $oRef = $oMap->reference;
        $oExt = $oRef->extent;

        $nWidth = $oRef->width;
        $nHeight = $oRef->height;
        
                   
/* -------------------------------------------------------------------- */
/*      get the click geographic coordinates and then project the       */
/*      value to the map projection.                                    */
/* -------------------------------------------------------------------- */
        $nGeoClickX = $this->Pix2Geo($nX, 0, $nWidth, $oExt->minx,
                                     $oExt->maxx, 0);
        $nGeoClickY = $this->Pix2Geo($nY, 0, $nHeight, $oExt->miny,
                                     $oExt->maxy, 1);
        
        $oPoint = ms_newpointobj();
        $oPoint->setxy($nGeoClickX, $nGeoClickY);
        
        //TODO: this may not be necessary if MapServer supports a projection
        // object in the ReferenceObj
        if (isset($_SESSION["KEYMAP_SRS"]))
        {
            $szKeyProjection = $_SESSION['KEYMAP_SRS'];
            if (stristr( $szKeyProjection, "epsg" ))
                $szKeyProjection = "init=".strtolower($szKeyProjection);
                
            if (strcasecmp( $szKeyProjection, $oMap->getProjection()) != 0)    
            {
                $oKeyProjection = ms_newProjectionObj( $szKeyProjection );
                $oMapProjection = ms_newProjectionObj( $oMap->getProjection() );
                $oPoint->project( $oKeyProjection, $oMapProjection );
            }
        }
        
        //calculate the new extents of the map
        $dfDeltaX = $oMap->extent->maxx - $oMap->extent->minx;
        $dfDeltaY = $oMap->extent->maxy - $oMap->extent->miny;
        
        $dfNewMinX = $oPoint->x - ($dfDeltaX/2);
        $dfNewMinY = $oPoint->y - ($dfDeltaY/2);
        $dfNewMaxX = $oPoint->x + ($dfDeltaX/2);
        $dfNewMaxY = $oPoint->y + ($dfDeltaY/2);
        // recenter to the click position
        $oMap->setExtent($dfNewMinX, $dfNewMinY, $dfNewMaxX, $dfNewMaxY);
    }

    /**
     * This function calculates the corresponding map click given a keymap click.
     *
     * @param $nMapPix
     * @param $nMapMinGeo
     * @param $nMapMaxGeo
     * @param $nMapMinGeoMax
     * @param $nMapMaxGeoMax
     * @param $nKeyPixClick
     * @param $nKeyPix
     * @param $bX
     * @return integer - The map click position in pixels.
     **/
    function keymapToMap ( $nMapPix, $nMapMinGeo, $nMapMaxGeo, $nMapMinGeoMax,
                           $nMapMaxGeoMax, $nKeyPixClick, $nKeyPix,
                           $bX )
    {
       // calculate the difference in extents
        $nMapGeo = $nMapMaxGeo - $nMapMinGeo;
        $nMapGeoMax = $nMapMaxGeoMax - $nMapMinGeoMax;

        // calculate the pixel value of the maximum map extents (i.e. width or
        // height of map at maximum extents)
        $nMapPixMax = ( $nMapPix * $nMapGeoMax ) / $nMapGeo;

        // calculate the map click distance
        $dClickDistance =  ( $nKeyPixClick * $nMapPixMax ) / $nKeyPix;

        // calculate the value to translate the origin
        if ($bX)
            $dDeltaOrigin = (($nMapMinGeo - $nMapMinGeoMax) * $nMapPix) / $nMapGeo;
        else
            $dDeltaOrigin = (($nMapMaxGeoMax - $nMapMaxGeo) * $nMapPix) / $nMapGeo;

        // calculate the return value
        $dReturnValue = $dClickDistance - $dDeltaOrigin;

        return $dReturnValue;
    }

/************************************************************************/
/*function GMapPix2Geo($nPixPos, $dfPixMin, $dfPixMax, $dfGeoMin, dfGeoMax,*/
/*                           $nInversePix)                              */
/*                                                                      */
/*      Utility function to convert a pixel position to geocoded        */
/*      position.                                                       */
/*                                                                      */
/*       The parameter $nInversePix could be set to 1 for Y pixel       */
/*      coordinates where the UL > LR. Else set to 0.                   */
/************************************************************************/
    function Pix2Geo($nPixPos, $dfPixMin, $dfPixMax, $dfGeoMin, $dfGeoMax,
                     $nInversePix)
      {
        $dfWidthGeo = $dfGeoMax - $dfGeoMin;
        $dfWidthPix = $dfPixMax - $dfPixMin;

        $dfPixToGeo = $dfWidthGeo / $dfWidthPix;

        if (!$nInversePix)
          $dfDeltaPix = $nPixPos - $dfPixMin;
        else
          $dfDeltaPix = $dfPixMax - $nPixPos;

        $dfDeltaGeo = $dfDeltaPix * $dfPixToGeo;


        $dfPosGeo = $dfGeoMin + $dfDeltaGeo;

        return ($dfPosGeo);
      }
}
?>
