<?php
/**
 * ExpressionBuilder Widget Class
 *
 * @project     Chameleon
 * @revision    $Id: ExpressionBuilder.widget.php,v 1.12 2004/07/08 17:49:07 pspencer Exp $
 * @purpose     Display a dialog box to build expressions
 * @author      DM Solutions Group (dev@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2003, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../Button.php");
include_once(dirname(__FILE__)."/../Popup.php");

/**
 * ExpressionBuilder
 *
 * @desc Display a widget that pops up a projection selection widget.
 */
class ExpressionBuilder extends CWCWidget
{
    var $moButton;
    var $moPopup;
    
    /**
     * ExpressionBuilder
     *
     * Constctor method for the ExpressionBuilder
     */
    function ExpressionBuilder()
    {
        $this->mszLanguageResource = str_replace("\\","/",dirname(__FILE__))."/ExpressionBuilder.dbf";

        parent::CWCWidget();

        $this->moButton = new CWCButton( $this );
        $this->moPopup = new CWCPopup( $this );
        $this->moPopup->mszLink = $_SESSION['gszCoreWebPath']."/widgets/ExpressionBuilder/ExpressionBuilder.phtml";

        // add sld attribute support (not yet implemented)
        $this->maAttributes["SLD"] = new StringAttribute( "SLD", false );

        // add like wildcard attributes
        $this->maAttributes["WILDCARD"] = new StringAttribute( "WILDCARD", true );
        $this->maAttributes["SINGLECHAR"] = new StringAttribute( "SINGLECHAR", true );
        $this->maAttributes["ESCAPE"] = new StringAttribute( "ESCAPE", true );

        // set the description for this widget
        $this->szWidgetDescription = "The ExpressionBuilder widget displays a ".
            "popup window that allows the user to interactively build ".
            "expressions.";
    }

    function InitDefaults()
    {
        parent::InitDefaults();
        //this widget should never belong to a toolset
        $this->maParams["TOOLSET"] = "";
        $this->moButton->InitDefaults();
        $this->moButton->SetOnClick('clickExpressionBuilder');
    }

    /**
     * SetMap
     *
     * Set the map session and create a navigation tool.
     */
    function SetMap($oMapSession)
    {
        $this->moMapObject = $oMapSession;
    }


    function GetJavascriptInitFunctions()
    {
        return $this->moButton->GetJavascriptInitFunctions();
    }

    function GetJavascriptVariables()
    {
        return $this->moButton->GetJavascriptVariables();
    }

    function GetJavascriptOnLoadFunctions()
    {
        $aReturn = parent::GetJavascriptOnLoadFunctions();
        
        return $aReturn;
    }
    
        function GetJavascriptIncludeFunctions()
    {
        return $this->moButton->GetJavascriptIncludeFunctions();
    }

    /**
     * GetHTMLHiddenVariables
     */
    function GetHTMLHiddenVariables()
    {
        // initialize return array
        $aReturn = $this->moButton->GetHTMLHiddenVariables();
        
        // setup the variables
        $szVariable = 'WFS_FILTER';
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        $szVariable = 'WFS_FILTER_LAYER_INDEX';
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        $szVariable = 'WFS_FILTER_EXPRESSIONLIST';
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        $szVariable = 'WFS_FILTER_APPLYTOMAP';
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        $szVariable = 'WFS_FILTER_SLD';
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;  
        $szVariable = 'WFS_FILTER_SLD_FILE';
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        $szVariable = 'WFS_FILTER_CONNECTION';
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        $szVariable = 'WFS_FILTER_LAYERTYPE';
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        $szVariable = 'WFS_FILTER_SLD_NAMEDLAYER';
        $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">\n";
        $aReturn[$szVariable] = $szValue;
        // return array of hidden variables
        return $aReturn;
    }

    /**
     * ParseURL
     *
     */
    function  ParseURL()
    {       
        // process WFS_FILTER_EXPRESSIONLIST
        if ( $this->isVarSet("WFS_FILTER_EXPRESSIONLIST") &&
            $this->isVarSet("WFS_FILTER_LAYER_INDEX") &&
             strlen( $this->getVar("WFS_FILTER_LAYER_INDEX") ) > 0 )
        {
            // set metadata
            $oLayer = $this->moMapObject->oMap->getlayer(
                               $this->getVar("WFS_FILTER_LAYER_INDEX") );
            $oLayer->setmetadata("cham_x_build_expressions", 
                                $this->getVar("WFS_FILTER_EXPRESSIONLIST") );
        }
        
        // process the WFS filter if set
        if ( $this->isVarSet("WFS_FILTER") &&
             strlen( $this->getVar("WFS_FILTER") ) > 0 &&
             $this->isVarSet("WFS_FILTER_LAYER_INDEX") &&
             strlen( $this->getVar("WFS_FILTER_LAYER_INDEX") ) > 0 )
        {
            // get the layer object
            $oLayer = $this->moMapObject->oMap->getlayer(
                                   $this->getVar("WFS_FILTER_LAYER_INDEX") );

            // build the filter string
            $szFilterString = $this->buildFilterString( 
                                      $this->getVar("WFS_FILTER"), $oLayer );

            // check for and delete annotation layer
            $nLayerCount = $this->moMapObject->oMap->numlayers;
            for ( $i=0; $i<$nLayerCount;  $i++ )
            {
                $oTempLayer = $this->moMapObject->oMap->getlayer($i);
                if ( $oTempLayer->name == 'ExpressionBuilderAnnotation'  )
                {
                    $oTempLayer->set( 'status', MS_DELETE );
                    break;
                }
            }
            
            // duplicate the layer
            $oAnnoLayer = ms_newLayerObj( $this->moMapObject->oMap, 
                                                            $oLayer );
            // reset orig layer name temporarily
            $oLayer->set( 'name', $oLayer->name.'tmp_backup' );
            
            // explcitly set to ON
            $oAnnoLayer->set( 'status', MS_ON );
            
            // set the template to an empty string to invalidate querying
            $oAnnoLayer->set( 'template', '' );
            
            // if layer type is WMS then convert to WFS
            if ( $oAnnoLayer->connectiontype == MS_WMS )
            {
                $oAnnoLayer->set( 'connectiontype', MS_WFS );
                $oAnnoLayer->set( 'connection', $this->getVar('WFS_FILTER_CONNECTION') );
                eval( '$oAnnoLayer->set( "type", '.$this->getVar('WFS_FILTER_LAYERTYPE').' );' );
            }
            
            // fix connection string to work around a caching problem
            $szTmpConnection = $oAnnoLayer->connection . "&ttt=1";
            $oAnnoLayer->set( 'connection', $szTmpConnection );             
            
            // apply the filter
            $oAnnoLayer->setmetadata("wfs_filter", $szFilterString );

            // check to see if the shared resource is specified and copy as necessary.
            if ( isset( $this->maSharedResourceWidgets["expressionbuilderresults"] ) )
            {
                // create the gml file to parse into a shared resource
                $szTmpGML = $oAnnoLayer->executeWFSGetfeature();

                // check if the file exists
                if ( file_exists( $szTmpGML ) )
                {
                    // read the file, parse it and pass it to the shared resource
                    $this->parseQueryResults( $szTmpGML, $oAnnoLayer->name, 
                                                                  $this->moMLT );
                    // delete the file
                    unlink( $szTmpGML );
                }
            } 
            
            // apply to map if necessary
            if ( $this->isVarSet("WFS_FILTER_APPLYTOMAP") &&
                 $this->getVar("WFS_FILTER_APPLYTOMAP") == 1 ) 
            {
                // get the sld according to the selected name
                if ( $this->getVar("WFS_FILTER_SLD") != 
                                                    'expressionbuilder_default' )
                {
                    // set the url to the sld
                    $szSLDURL = $this->getVar("WFS_FILTER_SLD_FILE");
                }
                else
                {
                    // default
                    if ( isset( $_SERVER['HTTPS'] ) && 
                                    strcasecmp( $_SERVER['HTTPS'], "on" ) == 0 )
                        $http = 'https://';
                    else
                        $http = 'http://';
                    
                    $szURI = $http.$_SERVER['HTTP_HOST'].
                                     dirname($_SERVER['PHP_SELF']);
                    
                    $szSLDURL = $szURI.'/widgets/sld/expressionbuilder.sld';
                }
                //echo $szSLDURL."<BR>";                                                
                // set the sld
                if ( isset( $szSLDURL ) )
                {
                    // check the sld for proper url
                    if ( substr( strtoupper( $szSLDURL ), 0, 4 ) == 'HTTP' )
                    {
                        // set annotation layer properties
                        $oAnnoLayer->set( 'name', 'ExpressionBuilderAnnotation' );
                        $oAnnoLayer->setmetadata( 'selected', '0' );
                        
                        // flag the session variable
                        $_SESSION['expr_build_filter_set'] = '1';
                        
                        // apply sld
                        //echo "applying $szSLDURL from layer ".$this->getVar('WFS_FILTER_SLD_NAMEDLAYER')."<BR>";
                        $oAnnoLayer->applysldurl( $szSLDURL, $this->getVar('WFS_FILTER_SLD_NAMEDLAYER') );
                    }
                    else 
                    {
                        // give error
                        $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                            $this->moMLT->get("29", 'Invalid SLD URL.  Please ensure '.
                            'that the URL is complete (not relative) and valid.' ) );
                    }
                }
                else 
                {
                    // delete temp anno layer
                    $oAnnoLayer->set( 'status', MS_DELETE );
                }
            }
            else 
            {
                // delete temp anno layer
                $oAnnoLayer->set( 'status', MS_DELETE );
            }
            
            // restore the layer's original name
            $oLayer->set( 'name', substr( $oLayer->name, 0, -10 ) );
        }
        
        // return success
        return true;
    }

    /**
     * GetJavascriptFunctions
     *
     * Build and return the array of functions needed in the
     * widget.
     */
    function GetJavascriptFunctions()
    {
        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        // set the wildcard etc
        if ( isset( $this->maParams["WILDCARD"] ) )
            $szTmpWildcard = urlencode( $this->maParams["WILDCARD"] );
        else
            $szTmpWildcard = "";
        if ( isset( $this->maParams["SINGLECHAR"] ) )
            $szTmpSinglChar = urlencode( $this->maParams["SINGLECHAR"] );
        else
            $szTmpSinglChar = "";
        if ( isset( $this->maParams["ESCAPE"] ) )
            $szTmpEscape = urlencode( $this->maParams["ESCAPE"] );
        else
            $szTmpEscape = "";

        // loop and get style names
        
        $szStyles = 'default (red points)@'.urlencode($_SESSION['gszCoreWebPath'].'/widgets/sld/expressionbuilder.sld').'@point@point|';
        $szStyles .= 'default (red lines)@'.urlencode($_SESSION['gszCoreWebPath'].'/widgets/sld/expressionbuilder.sld').'@line@line|';
        $szStyles .= 'default (red polygons)@'.urlencode($_SESSION['gszCoreWebPath'].'/widgets/sld/expressionbuilder.sld').'@polygon@polygon';
        
        if ( isset( $this->maszContents['STYLE'] ) )
        {
            $nCount = count( $this->maszContents['STYLE'] );
            for ( $i=0; $i<$nCount; $i++ )
            {
                // add to style list
                if (!isset($this->maszContents['STYLE'][$i]['NAME']))
                {
                    // set error handler and exit
                    $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                              $this->moMLT->get("34", "ERROR: ExpressionBuilder ".
                              "widget missing mandatory NAME attribute in ".
                              "STYLE declaration") );
                }
                else if (!isset($this->maszContents['STYLE'][$i]['SLD']))
                {
                    // set error handler and exit
                    $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                              $this->moMLT->get("35", "ERROR: ExpressionBuilder ".
                              "widget missing mandatory SLD attribute in ".
                              "STYLE declaration") );
                }
                else if (!isset($this->maszContents['STYLE'][$i]['NAMEDLAYER']))
                {
                    // set error handler and exit
                    $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                              $this->moMLT->get("36", "ERROR: ExpressionBuilder ".
                              "widget missing mandatory NAMEDLAYER attribute in ".
                              "STYLE declaration") );
                }
                else if (!isset($this->maszContents['STYLE'][$i]['LAYERTYPE']))
                {
                    // set error handler and exit
                    $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                              $this->moMLT->get("37", "ERROR: ExpressionBuilder ".
                              "widget missing mandatory LAYERTYPE attribute in ".
                              "STYLE declaration") );
                }
                else 
                {
                    $szStyles .= '|'.$this->maszContents['STYLE'][$i]['NAME'];
                    $szStyles .= '@'.urlencode( $this->maszContents['STYLE'][$i]['SLD'] );
                    $szStyles .= '@'.$this->maszContents['STYLE'][$i]['NAMEDLAYER'];
                    $szStyles .= '@'.$this->maszContents['STYLE'][$i]['LAYERTYPE'];
                }
            }
        }
        
        $aReturn = $this->moButton->GetJavascriptFunctions();//array();
        $this->moPopup->mszLink = $_SESSION['gszCoreWebPath']."/widgets/ExpressionBuilder/ExpressionBuilder.phtml?wildcard=".
           $szTmpWildcard."&singlechar=".
           $szTmpSinglChar."&escape=".
           $szTmpEscape.'&styles='.
           $szStyles."&selectedlayers=' + szSelectedLayers + ' ";
        $szJsFunctionName = "clickExpressionBuilder";
        $szButtonJS = $this->moPopup->DrawPublish();
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * popup a expression builder dialog
 */
function {$szJsFunctionName}()
{
    // set the var
    if ( {$this->mszHTMLForm}.SELECTED_LAYERS )
        szSelectedLayers = {$this->mszHTMLForm}.SELECTED_LAYERS.value;
    else
        szSelectedLayers = "";

    {$szButtonJS}
    return;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

        // add the apply javascript code to apply the filter
        $szJsFunctionName = "applyWFSFilter";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * Apply the WFS filter
 */
function {$szJsFunctionName}( szFilter, nLayerIndex, szExpressions, nApplyToMap,
                            szSLDName, szWFSConnection, szLayerType )
{
    // set the hidden form vars
    {$this->mszHTMLForm}.WFS_FILTER.value = escape( szFilter );
    {$this->mszHTMLForm}.WFS_FILTER_LAYER_INDEX.value = nLayerIndex;
    {$this->mszHTMLForm}.WFS_FILTER_EXPRESSIONLIST.value = szExpressions;
    {$this->mszHTMLForm}.WFS_FILTER_APPLYTOMAP.value = nApplyToMap;
    
    // split the sld name and file
    var aszTmpArray = szSLDName.split('@');
    
    {$this->mszHTMLForm}.WFS_FILTER_SLD.value = aszTmpArray[0];
    {$this->mszHTMLForm}.WFS_FILTER_SLD_FILE.value = aszTmpArray[1];
    {$this->mszHTMLForm}.WFS_FILTER_SLD_NAMEDLAYER.value =aszTmpArray[2];
    
    {$this->mszHTMLForm}.WFS_FILTER_CONNECTION.value = szWFSConnection;
    {$this->mszHTMLForm}.WFS_FILTER_LAYERTYPE.value = szLayerType;
    
    // submit the form
    if ( 1 == {$bCWCJSAPI} )
    {
        goCWCJSAPI.oMap.oApplication.UpdateExpressionBuilder( escape( szFilter ),
            nLayerIndex, szExpressions, nApplyToMap, aszTmpArray[0], szWFSConnection, 
            szLayerType, aszTmpArray[1] );
    }
    else
        {$this->mszHTMLForm}.submit();

    // return
    return;
}
EOT;

        $aReturn[$szJsFunctionName] = $szFunction;

        // add the apply javascript code to apply the filter
        $szJsFunctionName = "saveExpressions";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * Save Expressions
 */
function {$szJsFunctionName}( nLayerIndex, szExpressions )
{
    // set the hidden form vars
    {$this->mszHTMLForm}.WFS_FILTER_LAYER_INDEX.value = nLayerIndex;
    {$this->mszHTMLForm}.WFS_FILTER_EXPRESSIONLIST.value = szExpressions;
    
    // submit the form
    {$this->mszHTMLForm}.submit();

    // return
    return;
}
EOT;

        $aReturn[$szJsFunctionName] = $szFunction;
        
        // return the array of functions
        return $aReturn;
        
    }

    /**
     * DrawPublish
     *
     * Return the HTML code using the name in the map file and the
     * parameters of the CWC tag.
     */
    function DrawPublish()
    {
        if (!$this->mbVisible)
        {
            return "<!-- ExpressionBuilder popup widget hidden -->";
        }
        
        $szResult = $this->moButton->DrawPublish();

        return $szResult;
    }
    
/**
 * pareseQueryResults()
 *
 * Postcondition:  This function reads the give and parses it reporting any error
 *                 conditions and passes the results to a shared resource.
 * Precondition:   This function assumes the shared resource exists.
 *
 * @param $szFile string - The file to parse.
 * @param $szLayer string - Layer to parse.
 * @param $oMLT object - Language object to use.
 * @return boolean - true if successful, false if not;
 * @desc Reads and parses given file and passes results to shared resource and reports errors
 * 
 **/    
    function parseQueryResults( $szFile, $szLayer, $oMLT )
    {
        // int vars
        $data = "";
        
        // read the xml datasource
        $aszFile = file( $szFile );
        
        // check to see if an error was generated
        if ( !is_array( $aszFile ) || 
             strpos( strtoupper( $aszFile[0] ), '<SERVICEEXCEPTIONREPORT' ) !== false )
        {
            // give error
            if ( !is_array( $aszFile ) )
                $szServerDump = $aszFile;
            else 
            {
                $szServerDump = "";
                foreach(  $aszFile as $line )
                    $szServerDump .= $line;
            }
            
            // set error handler and exit
            $_SESSION['gErrorManager']->setError( ERR_WARNING, $oMLT->get("21", "WFS 'DescribeFeatureType' request failed.  ".
                            "The server returned the following: %1\$s", array( $szServerDump ) ) );
            return false;
        }
        
        // process url
        foreach( $aszFile as $line )
        {
            // process line
            $data .= $line;
        }
        //echo "data is $data<BR>";
        // parse
        $parser = xml_parser_create();
        xml_parser_set_option($parser,XML_OPTION_CASE_FOLDING,0);
        xml_parser_set_option($parser,XML_OPTION_SKIP_WHITE,1);
        xml_parse_into_struct($parser,$data,$values,$tags);
        xml_parser_free($parser);
    
        // loop through the structures and process the layer name
        foreach ($tags as $key=>$val) 
        {
            //echo "processing $key=>$val for layer $szLayer<BR>";
            // process layer name
            if ( strtoupper( $key ) == strtoupper( $szLayer ) ) 
            {
                //echo "here<BR>";
                $ranges = $val;
                // each contiguous pair of array entries are the 
                // lower and upper range for each layer definition
                $nCount = count($ranges);          
                for ($i=0; $i < $nCount; $i+=2) 
                {
                    $offset = $ranges[$i] + 1;
                    $len = $ranges[$i + 1] - $offset;
                    $this->parseElements( array_slice($values, $offset, $len ), $szLayer );
                }
            } 
            else 
            {
                // otherwise skip
                continue;
            }
        }
        
        // return 
        return true;
        
    // end parseQueryResults
    }
    
/**
 * parseElements()
 *
 * Postcondition:  This function parses the given array returns an array of 
 *                 attribute->type pairs.
 *
 * @param axValues Array - Mixed array of XML tags to process.
 * @param szLayer String - Layer name.
 * @return boolean - True if successful, false if not.
 * @desc Parses array and sets shared resource
 */
    function parseElements( $axValues, $szLayer ) 
    {   
        // init shared resource
        //echo "parseElements<BR>";
        if (!isset($this->maSharedResourceWidgets['expressionbuilderresults']->maszContents['RESULT'][$szLayer]))
        {
          //echo "making array<BR>";
          $this->maSharedResourceWidgets['expressionbuilderresults']->maszContents['RESULT'][$szLayer] = array();
        }
        //echo $szLayer."<BR>";
        //print_r($axValues);
        // loop through each item of the array
        $aResult = array();
        $nCount = count( $axValues );
        $j = 0;
        for ( $i=0; $i < $nCount; $i++ )
        {
            //echo "processing count $i<BR>";
            // only process all non "gml..." values
            if ( strtoupper( substr( $axValues[$i]["tag"], 0, 3 ) ) != "GML" )
            {
                //echo "in if ($i)<BR>";
                // add to shared resource
                $aResult[$j]['attribute'] = $axValues[$i]['tag'];
                if ( isset( $axValues[$i]['value'] ) )
                    $aResult[$j]['value'] = $axValues[$i]['value'];
                $j++;
            }
        }
        array_push( $this->maSharedResourceWidgets['expressionbuilderresults']->maszContents['RESULT'][$szLayer], $aResult );
        
        // return
        return true;
        
    // end parseElements() function
    }

/**
 * buildFilterString
 *
 * Postcondition:  This function takes the given expression and layer object
 *                 and creates a proper wfs filter string.
 *
 * @param $szExpression String - The expression string returned by the popup dialog    
 * @param $oLayer object - The layer to apply the expression to.
 * @return String - Filter string.
 * @desc Builds and returns a properly formatted wfs filter string.
 **/
    function buildFilterString( $szExpression, $oLayer )
    {
        // get projection
        $szProjection = trim( $oLayer->getprojection() );
        if ( strlen( $szProjection ) <= 0 )
            $szProjection = trim( $this->moMapObject->oMap->getprojection() );
        if ( strlen( $szProjection ) > 0 )
        {
            $szProjection = substr( $szProjection, 5 );
            $szProjection = str_replace( 'epsg', 'EPSG', $szProjection );
        }
        if ( substr( $szProjection, 0, 4 ) != 'EPSG' )
        {
            $szProjection = '';
        }
        
        // urldecode the filter if necessary
        if ( $szExpression != "BBOXONLY" )
        {
            $szDecodeFilter = urldecode( $szExpression );
            $szAndOpen = "<AND>";
            $szAndClose = "</AND>";
        }
        else
        {
            $szDecodeFilter = "";
            $szAndOpen = "";
            $szAndClose = "";
        }

        // check for AOI
        if ( isset( $_SESSION['ROIRENDERER'] ) &&
             is_array( $_SESSION['ROIRENDERER'] ) && 
             count( $_SESSION['ROIRENDERER'] ) > 0 )
        {
            // loop and build the filter string based on the ROI
            $bNotFirstExpression = false;
            $szBbox = '';
            foreach( $_SESSION['ROIRENDERER'] as $szTmpROI )
            {   
                // process based on type
                $szThisBbox = '';
                switch ( strtoupper($szTmpROI['type'] ) )
                {
                    case 'RECTANGLE':
                        $szThisBbox = buildRectFilter( $szTmpROI['aGeoCoords'], 
                                                                $szProjection );
                        break;
                    case 'CIRCLE':
                        $szThisBbox = buildCircleFilter( $szTmpROI['aGeoCoords'],
                            $szTmpROI['aGeoCoords'][2]-$szTmpROI['aGeoCoords'][0] );
                        break;
                    case 'POLYGON':
                        $szThisBbox = buildPolygonFilter( $szTmpROI['aGeoCoords'] );
                        break;
                    default:
                        $szBbox .= '';
                }
                
                // decide how to add to ROI
                if ( $bNotFirstExpression )
                {
                    switch( $szTmpROI['mode'] )
                    {
                        // additive
                        case 2:
                            $szBbox = '<OR>'.$szBbox.$szThisBbox.'</OR>';
                            break;
                        // subtractive
                        case 3:
                            $szBbox = '<AND>'.$szBbox.'<NOT>'.$szThisBbox.'</NOT></AND>';
                            break;
                    }
                }
                else
                    $szBbox .= $szThisBbox;
                
                // set flag
                $bNotFirstExpression = true;
            }
        }
        else
        {
            // use current extents as bbox
            $szBbox = buildRectFilter( 
                        array( $this->moMapObject->oMap->extent->minx,
                               $this->moMapObject->oMap->extent->miny,
                               $this->moMapObject->oMap->extent->maxx,
                               $this->moMapObject->oMap->extent->maxy ), 
                        $szProjection );
        }

        // return the filter string
        return $szAndOpen.$szBbox.$szDecodeFilter.$szAndClose;        

    // end of buildFilterString function
    }   
}

function buildRectFilter( $adCoords, $szProjection )
{
     return '<BBOX><PropertyName>Name</PropertyName>'.
                "<gml:Box srsName='".$szProjection."'><coordinates>".
                $adCoords[0].','.
                $adCoords[1].' '.
                $adCoords[2].','.
                $adCoords[3].
                '</coordinates></gml:Box></BBOX>';
}

function buildCircleFilter( $adCoords, $dDistance )
{
    return '<DWithin><PropertyName>Name</PropertyName><gml:Point>'.
           '<gml:coordinates>'.$adCoords[0].','.$adCoords[1].
           '</gml:coordinates></gml:Point><Distance '.
           "units='http://www.uomdict.com/uom.html#meters'>".
           $dDistance.'</Distance></DWithin>';
}

function buildPolygonFilter( $adCoords )
{
    $szReturn = '<Intersect><PropertyName>Name</PropertyName><gml:Polygon>'.
                '<gml:outerBoundaryIs><gml:LinearRing><gml:coordinates>';
                
    // loop and build list of coordinates
    $nCount = count( $adCoords );
    $szCoords = '';
    for( $i=0; $i<$nCount; $i=$i+2 )
    {
        $szCoords .= $adCoords[$i].','.$adCoords[$i+1].' ';
    }
    $szReturn .= trim( $szCoords );
    $szReturn .= '</gml:coordinates></gml:LinearRing></gml:outerBoundaryIs>'.
                 '</gml:Polygon></Intersect>';
    return $szReturn;
}
?>
