<?php
/**
 * Button Class
 *
 * @project     CWC2
 * @revision    $Id: Button.php,v 1.48 2004/07/08 15:16:28 pspencer Exp $
 * @purpose     Helper class to manage button attributes
 * @author      DM Solutions Group (sfournier@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/**
 * the button types.
 */ 
define ("CWCBUTTONTYPE_CLICK", 0);
define( "CWCBUTTONTYPE_TOGGLE", 1);
define( "CWCBUTTONTYPE_RADIO", 2);

include_once(COMMON."buttonizer/buttonizer.php" );

/**
 * CWCButton
 *
 * @desc CWCButton manages button attributes for other classes.
 */
class CWCButton
{
    var $mszOnClick = null;
    var $mxParams = null;
    var $mszToolSet = "null";
    var $mbCurrentTool = false;
    var $mszStyleResource = "";
    var $mnImageWidth = -1;
    var $mnImageHeight = -1;
    var $mszImageTip = "";
    var $mszImageName = "cwcimage_";
    var $maszImages = array();
    var $mszLabelAlign = "";
    var $mnType = CWCBUTTONTYPE_CLICK;
    
    /**
     * CWCButton
     *
     * @param oParent - the parent widget that contains the parsed
     * attribute info.
     */
    function CWCButton( &$oParent )
    {
        $this->moParent =& $oParent;
        $oParent->maAttributes["TOOLSET"] = new StringAttribute( "TOOLSET", false );
        $oParent->maAttributes["DEFAULT"] = new BooleanAttribute( "DEFAULT", false );
        $oParent->maAttributes["STYLERESOURCE"] = new StringAttribute( "STYLERESOURCE", false );
        $oParent->maAttributes["LABELALIGN"] = new StringAttribute( "LABELALIGN", false, array( "LEFT", "CENTER", "RIGHT" ) );
        $oParent->maAttributes["IMAGE"] = new StringAttribute( "IMAGE", false );
        $oParent->maAttributes["IMAGEWIDTH"] = new IntegerAttribute( "IMAGEWIDTH", false, 1 );
        $oParent->maAttributes["IMAGEHEIGHT"] = new IntegerAttribute( "IMAGEHEIGHT", false, 1 );
        $oParent->maAttributes["IMAGETIP"] = new StringAttribute( "IMAGETIP", false );
        $oParent->maAttributes["ONCLICK"] = new StringAttribute( "ONCLICK", false );
        $this->maszImages['normal'] = null;
        $this->maszImages['selected'] = null;
        $this->maszImages['hover'] = null;
        $this->maszImages['disabled'] = null;
    }

    /**
     * Set the onclick event of this button.  This can be provided in the widget tag
     * also, using the ONCLICK attribute
     *
     * @param szOnClick - the name of the javascript function to call.  It cannot
     * include the function parameter list or parentheses.
     */
    function SetOnClick( $szOnClick, $mxParams=null )
    {
        if ($szOnClick != "")
        {
            $this->mszOnClick = $szOnClick;
            if ($mxParams != null)
            {
                $this->mxParams = $mxParams;
            }
        }
        else
        {
            $this->mszOnClick = null;
            $this->mxParams = null;
        }
    }
    
    function SetType( $nType )
    {
        $this->mnType = $nType;
    }

    /**
     * initialize the default values from the parameters of the widget tag if
     * provided
     */
    function InitDefaults()
    {
        $oApp = GetChameleonApplication();
        
        if (isset($this->moParent->maParams["TOOLSET"]))
        {
            $this->mszToolSet = $this->moParent->maParams["TOOLSET"];
            if ($this->mnType == CWCBUTTONTYPE_CLICK)
                $this->mnType = CWCBUTTONTYPE_RADIO;
        }
        
        if (isset($this->moParent->maParams['DEFAULT']))
        {
            $this->mbCurrentTool = strcasecmp($this->moParent->maParams['DEFAULT'], 'true') == 0 ? true : false;
        }
        if (isset($this->moParent->maParams["STYLERESOURCE"]))
            $this->mszStyleResource = $this->moParent->maParams["STYLERESOURCE"];
            
        if (isset($this->moParent->maParams["LABELALIGN"]))
            $this->mszLabelAlign = strtolower( $this->moParent->maParams["LABELALIGN"] );

        if (isset($this->moParent->maParams["IMAGEWIDTH"]))
            $this->mnImageWidth = $this->moParent->maParams["IMAGEWIDTH"];

        if (isset($this->moParent->maParams["IMAGEHEIGHT"]))
            $this->mnImageHeight = $this->moParent->maParams["IMAGEHEIGHT"];

        if (isset($this->moParent->maParams["IMAGETIP"]))
            $this->mszImageTip = $this->moParent->maParams["IMAGETIP"];

        if (isset($this->moParent->maParams["ONCLICK"]))
            $this->mszOnClick = $this->moParent->maParams["ONCLICK"];
            
        $this->mszImageName .= $this->moParent->mnId;

        if (isset($this->moParent->maszContents['IMAGE']))
        {
            $nImages = count($this->moParent->maszContents['IMAGE']);
            for ($i=0; $i<$nImages; $i++)
            {
                $aImage = $this->moParent->maszContents['IMAGE'][$i];
                if (!isset($aImage['IMAGE']) && isset($this->moParent->maParams['IMAGE']))
                {
                    $aImage['IMAGE'] = $oApp->findFile( $this->moParent->maParams['IMAGE'] );
                }
                else if (isset($aImage['IMAGE']))
                {
                    $aImage['IMAGE'] = $oApp->findFile( $aImage['IMAGE'] );
                }
               
                $szState = (isset($aImage['STATE'])) ? strtolower($aImage['STATE']) : 'normal';
                
                if ($this->mszStyleResource != "") //use buttonizer to generate image
                {
                    $this->maszImages[$szState] = $this->GetImage( $szState, array_change_key_case( $aImage ) );
                }
                elseif( isset($aImage['IMAGE']) ) //assume a normal image URL
                {
                    $this->maszImages[$szState] = $oApp->fileSystemToURL($aImage['IMAGE']);
                }
            }
        }
        else if (isset($this->moParent->maParams['IMAGE']))
        {
            //only a 'normal' state for this image
            $szState = 'normal';
            $szImage = $oApp->findFile( $this->moParent->maParams['IMAGE'] );
            
            if ($this->mszStyleResource != "" )
            {
                $this->maszImages[$szState] = $this->GetImage( $szState, array( "image" => $szImage ) );
            }
            else
            {
                $this->maszImages[$szState] = $oApp->fileSystemToURL( $szImage );
            }
        }
    }
    
    /**
     * parse the URL and retrieve button states for groups if possible
     */
    function ParseURL()
    {
        if ($this->mszToolSet != "" && $this->mszToolSet != "null")
        {            
            if ($this->moParent->moFormVars->isVarSet('BUTTON_TOOLSET_'.$this->mszToolSet))
            {
                //echo $this->moParent->moFormVars->getVar('BUTTON_TOOLSET_'.$this->mszToolSet)." ";
                //echo $this->mszImageName." - ";
                if ($this->moParent->moFormVars->getVar('BUTTON_TOOLSET_'.$this->mszToolSet) == $this->mszImageName)
                {
                    //echo "set<BR>\n";
                    $this->mbCurrentTool = true;
                }
                else
                {
                    //echo "not set<BR>\n";
                    $this->mbCurrentTool = false;
                }
            }
        }
        return true;
    }
    
    /**
     * return an array of javascript functions used by the CWCButton,
     * currently none are required since everything has been moved to
     * cwc_button.js
     */
    function GetJavascriptFunctions()
    {
        return array();
    }
    
    /**
     * return an array of functions to be called when the page has finished
     * loading, currently none are required.
     *
     */
    function GetJavascriptOnLoadFunctions()
    {
        $aReturn = array();
        
        if ($this->mbCurrentTool)
        {
            $szFunction = 'CWCButtonInit_'.$this->moParent->mnId;
            if ($this->maszImages['normal'] != null && 
                ( $this->maszImages['selected'] != null ||
                  $this->maszImages['hover'] != null ||
                  $this->maszImages['disabled'] != null ) )
            {

                $szCode  = 'var '.$szFunction.' = goCWCButtonManager.GetButton( "'.$this->mszImageName.'" );'."\n";
                $szCode .= <<<EOT
{$szFunction}.bActive = true;
{$szFunction}.SetState( CWCBUTTONSTATE_SELECTED );
{$szFunction}.onClick( {$szFunction}, {$szFunction}.xValues );
if ({$szFunction}.oGroup != null)
{
    {$szFunction}.oGroup.ActivateButton( {$szFunction} );
}
EOT;
                
            }
            else
            {
                if (is_array($this->mxParams))
                {
                    $szParams = 'null, new Array(';
                    $szSep = '';
                    foreach ($this->mxParams as $szVal)
                    {
                        $szParams .= $szSep.'"'.$szVal.'"';
                        $szSep = ',';
                    }
                    $szParams .= ')';
                }
                else
                {
                    if(isset($this->mxParams) && $this->mxParams != "")
                        $szParams = 'null, "'.$this->mxParams.'"';
                    else
                        $szParams = "null, \"\"";
                }
                $szCode = $this->mszOnClick.'('.$szParams.');';
            }
            $aReturn[$szFunction] = $szCode;
        }
        
        return $aReturn;
    }

    /**
     * return an array of files to include.  Currently, just cwc_button.js is
     * required
     */
    function GetJavascriptIncludeFunctions()
    {
        $aReturn = array();
        
        if ($this->maszImages['normal'] != null && 
            ( $this->maszImages['selected'] != null ||
              $this->maszImages['hover'] != null ||
              $this->maszImages['disabled'] != null ) )
        {
            $szVar = 'cwc_dhtml.js';
            $aReturn[$szVar] = '<script language="JavaScript" src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_dhtml.js"></script>';
            $szVar = 'cwc_button.js';
            $aReturn[$szVar] = '<script language="JavaScript" src="'.$_SESSION['gszCoreWebPath'].
                            '/widgets/js/cwc_button.js"></script>';

            }
        
        return $aReturn;
    }
    /**
     * return an array of functions to call as the page loads.  Currently, this registers
     * the button with the button manager if necessary.
     *
     */
    function GetJavascriptInitFunctions()
    {
        $aReturn = array();
        
        if ($this->maszImages['normal'] != null && 
            ( $this->maszImages['selected'] != null ||
              $this->maszImages['hover'] != null ||
              $this->maszImages['disabled'] != null ) )
        {
            $szSelected = ($this->maszImages['selected'] != null) ? '"'.$this->maszImages['selected'].'"' : 'null';
            $szHover = ($this->maszImages['hover'] != null) ? '"'.$this->maszImages['hover'].'"' : 'null';
            $szDisabled = ($this->maszImages['disabled'] != null) ? '"'.$this->maszImages['disabled'].'"' : 'null';
            
            $szOnClick = ($this->mszOnClick != null) ? $this->mszOnClick : 'null';
            if ($this->mxParams != null)
            {
                if (is_array($this->mxParams))
                {
                    $szParams = 'new Array(';
                    $szSep = '';
                    foreach ($this->mxParams as $szVal)
                    {
                        $szParams .= $szSep.'"'.$szVal.'"';
                        $szSep = ',';
                    }
                    $szParams .= ')';
                }
                else
                {
                    $szParams = '"'.$this->mxParams.'"';
                }
            }
            else
            {
                $szParams = 'null';
            }
            
             $szCode = 'oButton =  new CWCButton('.
                '"'.$this->mszImageName.'", '.
                $this->mnType. ', '.
                '"'.$this->mszToolSet.'", '.
                $szOnClick . ', ' . $szParams . ', "'.
                $this->maszImages['normal'] . '", '.$szHover.', '.$szSelected .
                ', '. $szDisabled . ");\n";
             $szCode .= 'oButton.oForm = '.$this->moParent->mszHTMLForm.";\n";
             $szCode .= 'goCWCButtonManager.AddButton( oButton );'."\n";
            $aReturn[$this->mszImageName] = $szCode;
        }

        return $aReturn;
    }

    /**
     * return an array of javascript variables to register in global scope,
     * currently none are required.
     */
    function GetJavascriptVariables()
    {
        return array();
    }
    
    function GetHTMLHiddenVariables()
    {
        $aReturn = array();
        
        if ($this->mszToolSet != "" && strcasecmp($this->mszToolSet, "null") != 0 &&
            $this->maszImages['normal'] != null && 
            ( $this->maszImages['selected'] != null ||
              $this->maszImages['hover'] != null ||
              $this->maszImages['disabled'] != null ))
        {
            $szVariable = "BUTTON_TOOLSET_".$this->mszToolSet;
            $szTool="";
            if ($this->moParent->moFormVars->isVarSet($szVariable))
              $szTool = $this->moParent->moFormVars->getVar($szVariable);
            $szValue = "<INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"$szTool\">";
            $aReturn[$szVariable] = $szValue;

        }
        return $aReturn;
    }

    /**
     * DrawPublish
     *
     * Return the HTML code for this button.  
     */
    function DrawPublish( )
    {
       
        if ($this->maszImages['normal'] != null && 
            ( $this->maszImages['selected'] == null &&
              $this->maszImages['hover'] == null &&
              $this->maszImages['disabled'] == null ) )
        {
            $bNeedAnchor = true;
        }
        else
            $bNeedAnchor = false;
        $szResult = "";
        
        if ($bNeedAnchor)
        {
            if (is_array($this->mxParams))
            {
                $szParams = 'null, new Array(';
                $szSep = '';
                foreach ($this->mxParams as $szVal)
                {
                    $szParams .= $szSep.'"'.$szVal.'"';
                    $szSep = ',';
                }
                $szParams .= ')';
            }
            else
            {
                if(isset($this->mxParams) && $this->mxParams != "")
                    $szParams = 'null, "'.$this->mxParams.'"';
                else
                    $szParams = "null, \"\"";
            }
            
            if ($this->mszToolSet == "" || $this->mszToolSet == null || $this->mszToolSet == "null")
            {
                $szResult .= '<a href="#" onclick=\''.$this->mszOnClick.'('. $szParams .')\'>';
            }
            else
            {
                if ($this->mbCurrentTool)
                    $szChecked = " CHECKED";
                else
                    $szChecked = "";
                    
                if (isset($this->moParent->mszCmd))
                {
                    $szCmd = ' value="'.$this->mszImageName.'"';
                }
                else
                {
                    $szCmd = ' value="'.$this->mszImageName.'"';
                }
                $szResult .= '<input type="radio" name="BUTTON_TOOLSET_'.$this->mszToolSet.'" '.$szCmd.' onclick=\''.$this->mszOnClick.'('. $szParams .')\''.$szChecked.'>';
                /*
                 * TODO: I'd like to enable the following line and remove 
                 * $bNeedAnchor=false so that the user could click on the
                 * image to set the radio button but that turns out to be
                 * somewhat problematic.  You need to access the radio
                 * buttons as an array and search it for a radio button with
                 * the right value.  This can be difficult because in IE
                 * radio buttons that aren't grouped (i.e. there is only
                 * one of them) are not represented as arrays ... or at
                 * least, that was our experience from programming Studio.
                 */
                //$szResult .= '<a href="#" onclick=\''.$this->moParent->mszHTMLForm.'.BUTTON_TOOLSET_'.$this->mszToolSet.'.click(); '.$this->mszOnClick.'('. $szParams .');\'>';
                $bNeedAnchor = false;
            }
            
        }
        
        $szResult .= '<IMG NAME="'.$this->mszImageName.'" ID="'.$this->mszImageName.
                    '" SRC="'.$this->maszImages['normal'].'" ALT="'.
                    $this->mszImageTip.'" TITLE="'.$this->mszImageTip.
                    '" BORDER="0" WIDTH="'.$this->mnImageWidth.'" '.
                    'HEIGHT="'.$this->mnImageHeight.'">'; 

        if ($bNeedAnchor)
        {
            $szResult .= "</a>";
        }
        
        return $szResult;
    }
    
    /**
     * functions for making images from shared resources
     */
     
    /**
     * this function takes an image name and returns a URL.  If the image
     * is a SharedResource image then it will create a buttonized image
     * from the SharedResource settings and then cache the result.
     * @param szState the state of the image.  One of 'normal', 'selected', 'hover',
     *        or 'disabled'.  The image will get all the default settings from
     *        the shared resource, plus any state-specific settings from the
     *        appropriate 'state' entries.
     * @aImage an array of image settings.
     */
    function GetImage( $szState, $aImage )
    {
        
        //handle the simple cases first, if the image is in the session then
        //just use that value.  Side effect is that you will need to restart
        //your session to see changes ;)
        
        //turns out that multiple chameleon apps sharing the same session can
        //get confused about which button is which in the session.  To make
        //sure that we get the right button, use the template name as well
        
        $oApp = GetChameleonApplication();
        $szTemplate = $oApp->mszTemplateName;
        
        $szCacheName = $szTemplate.'_'.$this->mszImageName.'_'.$szState;
        if (isset($_SESSION[$szCacheName]))
        {
            if ($this->mnImageHeight == -1)
            {
                $this->mnImageHeight = $_SESSION[$this->mszStyleResource][$szState]['height'];
            }
            if ($this->mnImageWidth == -1)
            {
                $this->mnImageWidth = $_SESSION[$this->mszStyleResource][$szState]['width'];
            }
            return $_SESSION[$szCacheName];
        }
        
        $params = $this->moParent->maParams;
        
        //gather info about this SharedResource image, use session copy if available.
        if (isset($_SESSION[$this->mszStyleResource][$szState]))
        {
            $aParams = $_SESSION[$this->mszStyleResource][$szState];
        }
        else
        {
            if (isset($this->moParent->maSharedResourceWidgets[$this->mszStyleResource]))
            {
                $share = $this->moParent->maSharedResourceWidgets[$this->mszStyleResource]->maszContents;
                $aParams = $this->LoadSharedResource( $share, array(), $szState );
                if (!isset($_SESSION[$this->mszStyleResource]))
                {
                    $_SESSION[$this->mszStyleResource] = array();
                }
                $_SESSION[$this->mszStyleResource][$szState] = $aParams; //make a copy in the session.
            }
            else
            {
                //error ... no shared resource of this name
                $_SESSION['gErrorManager']->setError(ERR_WARNING,
                   "ERROR: Button.php was unable to find a shared resource named ".
                   $this->mszStyleResource." for widget ".get_class($this->moParent));
                return;            }
        }
        
        $aParams = array_merge( $aParams, $aImage );
        
        //these values override the sharedresource
        if ($this->mnImageHeight != -1)
        {
            $aParams['height'] = $this->mnImageHeight;
        }
        else
        {
            $this->mnImageHeight = $aParams['height'];
        }
        
        if ($this->mnImageWidth != -1)
        {
            $aParams['width'] = $this->mnImageWidth;
        }
        else
        {
            $this->mnImageWidth = $aParams['width'];
        }
        
        if ($this->mszLabelAlign != "")
        {
            $aParams['labelalign'] = $this->mszLabelAlign;
        }
        
        //the image parameter needs to be 'graphic' for the buttonizer
        if (isset($aParams['image']))
            $aParams['graphic'] = $oApp->findFile( $aParams['image'] );
         
        //these values don't have entries in the shared resource.
        if (!isset($aParams['label']))
        {
            if (isset( $params["LABEL"] ))
                $aParams["label"] = $params["LABEL"];
            else
                $aParams["label"] = "";
        }
                    
        //load platform specific behaviours
        $aParams["gd_module"] = "gd2";
        $aParams["freetype"] = "FreeType";
        
        //generate the cached image.
        $szImageFile = "b".md5( implode( "", $aParams ) ).".png";
        
        $szImagePath = $_SESSION['gszButtonCachePath'].$szImageFile;

        $szImageWebPath = $_SESSION['gszButtonCacheWebPath'].$szImageFile;
        
        //print_r($aParams);

        //cache the URL and generate the image, if required.
        if (!isset($aParams['usecache']) || (isset($aParams['usecache']) && $aParams['usecache']))
        {
            $_SESSION[$szCacheName] = $szImageWebPath;
            if (!is_file( $szImagePath ))
            {
                buttonize( $szImagePath, $aParams );
            }
        }
        else
        {
            buttonize( $szImagePath, $aParams );
        }
        
        return $szImageWebPath;
    }
    
    /**
     * load a shared resource into an array that contains buttonizer values
     * return the loaded array
     */
    function LoadSharedResource( $share, $aParams = array(), $szState )
    {
        $oApp = GetChameleonApplication();
        
        //process includes first
        if (isset($share['STATE']))
        {
            foreach( $share["STATE"] as $aszSR )
            {
                if($aszSR["VALUE"] == $szState)
                {
                    $aParams = $this->LoadSharedResource($aszSR["CHILDREN"][0], $aParams, $szState);
                }
            }
        }
        
        if (isset($share["IMAGEWIDTH"][0]["VALUE"]))
            $aParams["width"] = $share["IMAGEWIDTH"][0]["VALUE"];
        
        if (isset($share["IMAGEHEIGHT"][0]["VALUE"]))
            $aParams["height"] = $share["IMAGEHEIGHT"][0]["VALUE"];
        
        if (isset($share["TEXTBUTTONCOLOR"][0]["VALUE"]))
            $aParams["backgroundcolor"] = $share["TEXTBUTTONCOLOR"][0]["VALUE"];
        
        if (isset($share["BACKGROUNDIMAGE"][0]["VALUE"]))
            $aParams["backgroundgraphic"] = $oApp->findFile( $share["BACKGROUNDIMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONBORDER_TOPLEFT_IMAGE"][0]["VALUE"]))
            $aParams["border_tl_image"] = $oApp->findFile( $share["TEXTBUTTONBORDER_TOPLEFT_IMAGE"][0]["VALUE"] );
        
        if (isset($share["TEXTBUTTONBORDER_TOP_IMAGE"][0]["VALUE"]))
            $aParams["border_t_image"] = $oApp->findFile( $share["TEXTBUTTONBORDER_TOP_IMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONBORDER_TOPRIGHT_IMAGE"][0]["VALUE"]))
            $aParams["border_tr_image"] = $oApp->findFile( $share["TEXTBUTTONBORDER_TOPRIGHT_IMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONBORDER_RIGHT_IMAGE"][0]["VALUE"]))
            $aParams["border_r_image"] = $oApp->findFile( $share["TEXTBUTTONBORDER_RIGHT_IMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONBORDER_BOTTOMRIGHT_IMAGE"][0]["VALUE"]))
            $aParams["border_br_image"] = $oApp->findFile( $share["TEXTBUTTONBORDER_BOTTOMRIGHT_IMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONBORDER_BOTTOM_IMAGE"][0]["VALUE"]))
            $aParams["border_b_image"] = $oApp->findFile( $share["TEXTBUTTONBORDER_BOTTOM_IMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONBORDER_BOTTOMLEFT_IMAGE"][0]["VALUE"]))
            $aParams["border_bl_image"] =$oApp->findFile(  $share["TEXTBUTTONBORDER_BOTTOMLEFT_IMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONBORDER_LEFT_IMAGE"][0]["VALUE"]))
            $aParams["border_l_image"] = $oApp->findFile( $share["TEXTBUTTONBORDER_LEFT_IMAGE"][0]["VALUE"] );

        if (isset($share["TEXTBUTTONNUDGE"][0]["VALUE"]))
            $aParams["nudge"] = $share["TEXTBUTTONNUDGE"][0]["VALUE"];
        
        if (isset($share["TEXTBUTTONPADDING"][0]["VALUE"]))
            $aParams["padding"] = $share["TEXTBUTTONPADDING"][0]["VALUE"];
        
        if (isset($share["LABELFONT"][0]["VALUE"]))
            $aParams["labelfont"] = $oApp->findFile( $share["LABELFONT"][0]["VALUE"] );
        
        if (isset($share["LABELFONTSIZE"][0]["VALUE"]))
            $aParams["labelsize"] = $share["LABELFONTSIZE"][0]["VALUE"];
        
        if (isset($share["LABELCOLOR"][0]["VALUE"]))
            $aParams["labelcolor"] = $share["LABELCOLOR"][0]["VALUE"];
        
        if (isset($share["LABELALIGN"][0]["VALUE"]))
            $aParams["labelalign"] = $share["LABELALIGN"][0]["VALUE"];
        
        if (isset($share["LABELANTIALIAS"][0]["VALUE"]))
            $aParams["antialias"] = ($share["LABELANTIALIAS"][0]["VALUE"] == "true" )? true : false;
            
        if (isset($share["IMAGE"]))
            $aParams["graphic"] = $oApp->findFile( $share["IMAGE"][0]['VALUE'] );
            
        if (isset($share["USETEXTBUTTONCACHE"][0]["VALUE"]))
            $aParams['usecache'] = ($share["USETEXTBUTTONCACHE"][0]["VALUE"] == "true" )? true : false;
        
        return $aParams;
    }
}
?>
