<?php
/**
 * XMLObject class
 *
 * @project     CWC2
 * @revision    $Id: XMLObject.php,v 1.7 2004/01/14 19:51:47 sfournier Exp $
 * @purpose     XMLObject class
 * @author      DM Solutions Group (assefa@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * This class takes an XML document as a string and parses into a recursive
 * object structure.  The root XMLObject is the one you created and contains
 * no particular values.  Its children array contains the root XML objects
 * contained in the XML string passed to the constructor.
 *
 * Sample XML document:
 * <Employees version="1.0">
 *   <Employee Id="1">
 *     <Name>John</Name>
 *     <Salary Type="Annualy">20K</Salary>
 *   </Employee>
 *   <Employee Id="2">
 *     <Name>Ren</Name>
 *     <Salary Type="Weekly">2K</Salary>
 *   </Employee>
 * </MyRoot>
 *
 * Sample PHP code:
 *
 * $oXMLObj = new XMLObject( $szXMLDoc );
 *
 * $oEmployees = $oXMLObj[0];
 *
 * if ($oEmployees->version >= "1.0")
 * {
 *   foreach($Employees->children as $oEmployee)
 *   {
 *     echo $oEmployee->Id." ";
 *     echo "Name: ".$oEmployee->getNextChild("Name")->value."\n";
 *     $oSalary = $oEmployee->getNextChild("Salary");
 *     echo "Paid: ".$oSalary->value."(";
 *     echo $oSalary->Type.")";
 *   }
 * }
 * else
 *   echo "Wrong document version";
 *
 */

/**
 * a basic XML object.
 */
class XMLObject
{
    var $name = "XMLObject";
    var $value = "";
    var $_type = "";
    var $_cur_child_index = 0;
    var $_cur_child_name = "";
    var $children = array();

    /**
     * constructor, pass an XML string to populate it.
     * @param szXMLDoc the XML document string to parse.
     */
    function XMLObject( $szXMLDoc = "" )
    {
        if ($szXMLDoc != "")
        {
            $parser = xml_parser_create();
            xml_parser_set_option($parser, XML_OPTION_CASE_FOLDING, 0);
            xml_parser_set_option($parser, XML_OPTION_SKIP_WHITE,   1);
            xml_parse_into_struct($parser, $szXMLDoc, $aVals, $index);
            xml_parser_free($parser);

            $this->children = $this->parseValuesArray( $aVals );
        }
    }

    /**
     * internal function to add attributes to this object from an array
     * $attributes the array of values to add
     */
    function addAttributes( $attributes = array() )
    {
        if (is_array( $attributes ))
        {
            if (!isset($this->_attributes) || !is_array($this->_attributes))
                $this->_attributes = array();
            $this->_attributes = array_merge( $this->_attributes, $attributes );
            foreach( $attributes as $idx => $attr )
            {
                $this->{strtolower($idx)} = $attr;
            }
        }
        elseif( is_string( $attributes ) && is_string( $values ) )
        {
            $this->{strtolower($attributes)} = $values;
        }
    }

    /**
     * helper function to iterate through several children of the same
     * type name.  If szName is "" then this will return all children
     * in order.  When there are no more children of the given type
     * then the function will return "".
     * @param szName the type name to iterate through
     * @return an XMLObject for the next child or "" if not found.
     */
    function getNextChild( $szName )
    {
        //reset index if changing names
        if ($this->_cur_child_name != $szName)
        {
            $this->_cur_child_index = 0;
        }

        $this->_cur_child_name = $szName;

        if ($this->_cur_child_name == "")
        {
            if (isset($this->children[$this->_cur_child_index]))
                return $this->children[$this->_cur_child_index++];
            else
                return ""; //at the end.
        }
        else
        {
            $nChildren = count($this->children);
            for($i=$this->_cur_child_index; $i<$nChildren; $i++)
	    {
                if ($this->children[$i]->_type == $this->_cur_child_name ||
                    $this->children[$i]->name == $this->_cur_child_name)
                {
                    $this->_cur_child_index = $i+1;
                    return $this->children[$i];
                }
            }
        }

    }

    /**
     * recursively build an array of XMLObjects from an array returned by
     * xml_parse_into_struct
     * @param vals the values array
     * @param index the current index in the values array
     * @return an array of XMLObjects found in this vals array starting at
     * the current index.
     */
    function parseValuesArray(&$vals, $index = 0)
    {
        //children is the array of objects.  Each XML tag becomes an object
        //of type XMLChild with member variables for attributes and possibly
        //an array of children
        $GLOBALS["XMLObjectIndex"] = $index;

        $j = count($vals);

        while ($index < $j)
        {
            $k = $index++;

            // if current record is a closing tag, return the array of objects
            if ($vals[$k]['type'] == "close")
            {
                $GLOBALS["XMLObjectIndex"] = $index;
                return $this->children;
            }
            switch ($vals[$k]['type'])
            {
                // If current record is a complete tag or cdata then we create a new
                // XMLObject for it and add it to the children array
                case 'cdata':
                case 'complete':
                    $attributes = (isset($vals[$k]['attributes'])) ? $vals[$k]['attributes'] : array() ;
                    $o = new XMLObject( );
                    $o->_type = $vals[$k]['tag'];
                    $o->name = $vals[$k]['tag'];
                    $o->addAttributes( $attributes );
                    $o->value = (isset($vals[$k]['value'])) ? $vals[$k]['value'] : "";
                    array_push($this->children, $o);
                    //$index ++;
                    break;

                // If current record is an open tag
                case 'open':
                    $attributes = (isset($vals[$k]['attributes'])) ? $vals[$k]['attributes'] : array() ;
                    $o = new XMLObject( );
                    $o->_type = $vals[$k]['tag'];
                    $o->name = $vals[$k]['tag'];
                    $o->addAttributes( $attributes );
                    $o->value = (isset($vals[$k]['value'])) ? $vals[$k]['value'] : "";
                    $o->parseValuesArray($vals, $index);
                    $index = $GLOBALS["XMLObjectIndex"];
                    array_push($this->children, $o);
                break;
            }
        }
        return $this->children;
    }

    /**
     * dump this object out to stdout - debugging only
     */
    function dumpObject()
    {
        echo "<PRE>";
        print_r( $this );
        echo "</PRE>";
    }
}
?>
