<?php
/**
 * MapSession Module
 *
 * @project     PHP Wrapper Class
 * @revision    $Id: map_session.php,v 1.63 2003/10/10 13:39:08 sacha Exp $
 * @purpose     This file contains classes related to managing map sessions.
 *              There are two session types that can be instanciated, a read-
 *              only version and a read/write version.  Of the two, only the
 *              latter needs write access to the physical disk.
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2001, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */

/**
 * This class is the basic wrapper for the map file and provides basic file
 * i/o and state management.
 * It is assummed that the phpmapscript module is loaded prior to
 * instantiaiting this class.
 *
 * @author William A. Bronsema, C.E.T. (dev@dmsolutions.ca)
 *
 */
define("NOTSET", "ANNOTATIONLAYERNOTSET");

class MapSession extends Logger
{
    /**
     * The current session map file object (private).
     */
    var $oMap;

    /**
     * The current session map file (private).
     */
    var $szMapFile;

    /**
     * The temp directory to write the sessions to (private).
     */
    var $szTempDir;

    /**
     * Array of max extents (private).
     */
    var $adMaxExtents;

    /**
     * Array of Annotation Layers
     */
    var $aoAnnotation = array();

    /**
     *  Construct a new MapSession instance and initialize it.
     */
    function MapSession()
    {
        // initialize variables
        $this->oMap = null;
        $this->szMapFile = "";
        $this->szTempDir = "";
        $this->adMaxExtents = array();

        // set the scope for the logging functions
        $this->setScope( "MapSession" );

    // end constructor
    }

    /**
     * This function sets temp directory for the MapSession.
     *
     * @param szTempDir string - Path to the tenp directory.
     */
    function setTempDir($szTempDir)
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "setTempDir() starting" );
        $this->szTempDir = $szTempDir;

        //log exit
        $this->logFuncEnd( LOG_ALL, "setTempDir() done" );

    // end setTempDir function
    }

    /**
     * This function sets the optional maximum extents to be used to limit
     * zooming and panning.
     *
     * @param dMinX double - The min X extent to use.
     * @param dMinY double - The min Y extent to use.
     * @param dMaxX double - The max X extent to use.
     * @param dMaxY double - The max Y extent to use.
     * @return boolean - True if successful, False if not.
     **/
    function setMaxExtents($dMinX, $dMinY, $dMaxX, $dMaxY)
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "setMaxExtents() starting" );

        // check if the parameters are valid numbers and min is less than max
        if ( !( is_numeric($dMinX) && is_numeric($dMinY) &&
                is_numeric($dMaxX) && is_numeric($dMaxY) &&
                $dMinX < $dMaxX && $dMinY < $dMaxY))
        {
            // generate error and return
            $this->error( ERR_CRITICAL, "Invalid extents given to ."
                                            ."setMaxExtents() function." );
            return false;
        }

        // set the maxextent array
        $this->adMaxExtents["minX"] = $dMinX;
        $this->adMaxExtents["minY"] = $dMinY;
        $this->adMaxExtents["maxX"] = $dMaxX;
        $this->adMaxExtents["maxY"] = $dMaxY;

        //log exit
        $this->logFuncEnd( LOG_ALL, "setMaxExtents() done" );

    // end setMaxExtents function
    }

    /**
     * This function returns an associative array of extents if they were set
     * or false if they were not.  The indexes are as follows:
     *          array["minX"]
     *          array["minY"]
     *          array["maxX"]
     *          array["maxY"]
     *
     * @return mixed - Associative array of extents if set or false if not.
     **/
    function getMaxExtents()
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "getMaxExtents() starting" );

        // count the array items and if greater than 0 return
        if ( count( $this->adMaxExtents ) > 0 )
        {
            // return the array because extents are set
            $ReturnValue = $this->adMaxExtents;
        }
        else
        {
            // return failure
           //$this->error( ERR_CRITICAL, "Maximum extents are not set.");
           $ReturnValue = false;
        }

        //log exit
        $this->logFuncEnd( LOG_ALL, "getMaxExtents() done" );

        // return
        return $ReturnValue;

    // end getMaxExtents function
    }

    /**
     * This function opens a map file and sest oMap to a valid state.
     *
     * @param szMapFile string - Path and filename of the map to open.
     * @param szMapFilePath string - Path of the map to open. This is set
     *                               when a different path is used.
     *
     * @return boolean - True if successful, false if not.
     */
    function readMapFile($szMapFile, $szMapFilePath="")
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "readMapFile() starting" );

        // check to see if the map file exists
        if ( !file_exists($szMapFile) )
        {
            // log error
            $this->error( ERR_FILE_IO, "The map file[".$szMapFile.
                                                "] does not exist." );

            // return failure
            return false;
        }

        if ($szMapFilePath != "" && is_dir($szMapFilePath))
        {
            if (strncasecmp( "lmx.", strrev($szMapFile), 4 ) == 0)
            {
                $this->oMap = ms_newMapObj( "" );
                $this->oMap->loadMapContext( $szMapFile );
            }
            else
            {
                $this->oMap = ms_newMapObj($szMapFile, $szMapFilePath);
            }
        }
        else
        {
            // set the map object to be the phpmapscript map object
            $this->oMap = ms_newMapObj($szMapFile);
        }
        
        // check for errors
        if ($this->oMap == null)
        {
            // log error
            $this->error( ERR_FILE_IO, "Unable to access map [".$szMapFile.
                             "].  Check to see that the file exists and/or ".
                             "that the path and filename are spelled ".
                             "correctly" );

            // return failure
            return false;
        }

        // set the extents to validate them
        $this->oMap->setextent($this->oMap->extent->minx,
                               $this->oMap->extent->miny,
                               $this->oMap->extent->maxx,
                               $this->oMap->extent->maxy);

        // log successful open
        $this->log( LOG_VERBOSE, "readMapFile - Open map file[".$szMapFile.
                                                        "] was successful" );

        // record the map path and filename;
        $this->szMapFile = $szMapFile;

        //log exit
        $this->logFuncEnd( LOG_ALL, "readMapFile() done" );

        // return success
        return true;

    // end readMapFile function
    }

    /**
     * This function saves the current map object to the mapfile.
     *
     * @param szMapFile string - Path and filename of the map to write to.
     *
     * @return boolean - True if successful, false if not.
     */
    function writeMapFile($szMapFile)
    {
        $bResult = true;
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "writeMapFile() starting" );

        // check to see that there is a valid map file open
        if ($this->oMap == null)
        {
            // log error
            $this->error( ERR_FILE_IO, "No valid map object to write." );

            // return failure
            return false;
        }

        if ( file_exists( $szMapFile ) && !is_writable( $szMapFile ) )
        {
            $this->error( 0, "$szMapFile is not writeable" );
            $bResult = false;
        }
        else
        {
            if (strncasecmp( "lmx.", strrev($szMapFile), 4 ) == 0)
            {
                $this->oMap->saveMapContext( $szMapFilePath );
            }
            else
            {
            $this->oMap->save( $szMapFile );
            }
        }
        // log successful save
        $this->log( LOG_VERBOSE, "writeMapFile - Save map file[".$szMapFile.
                                                        "] was successful" );
        //log exit
        $this->logFuncEnd( LOG_ALL, "writeMapFile() done" );

        // return success
        return $bResult;

    // end writeMapFile function
    }

    /**
     * This function returns a pointer to the current open map object.
     *
     * @return object - A pointer to the current open map object or null.
     */
    function getMapObj()
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "getMapObj() starting" );

        //log exit
        $this->logFuncEnd( LOG_ALL, "getMapObj() done" );

        // return a pointer to the map object
        return $this->oMap;

    // end getMapObj function
    }

    /**
     * This function processes the template file as set in web object of the
     * current session's map object.  It returns the processed file as string.
     *
     * @param aszTag array - Optional associative array of user defined tags
     *              and their corresponding values to be processed.
     *              i.e. $aszTag["my_tag"] = "My tag's value". In this example
     *              all the tags [my_tag] will be replaced with the string "My
     *              tag's value".
     * @param nGenerateImages integer - This optional flag indicates how
     *              special tags are to be processed.  The special tags: [img],
     *              [scalebar], [ref], [legend] can be replaced with the
     *              appropriate URL if this flag is set to MS_TRUE.
     * @return mixed - A string representation(typically HTML) of the template
     *              file with all tags processed or false if failed.
     **/
    function processTemplate( $aszTag = array(), $nGenerateImages = MS_FALSE )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "processTemplate() starting" );

        // check to see that there is a valid map file open
        if ($this->oMap == null)
        {
            // log error
            $this->error( ERR_FILE_IO, "No valid map object to write." );

            // return failure
            return false;
        }

        // check to see if the the tag array is a valid array
        if ( !is_array( $aszTag ) )
        {
            // log error
            $this->error( ERR_FILE_IO, "Supplied Tag array is not a valid
                                                                    array." );

            // return failure
            return false;
        }

        //log exit
        $this->logFuncEnd( LOG_ALL, "processTemplate() done" );

        // call mapscript's proceestemplate
        return $this->oMap->processtemplate( $aszTag, $nGenerateImages );

    // end processTemplate function
    }

   /**
     * This function processes the query template file as set in each layer's
     * class object of the current session's map object.  It returns the
     * processed file as string.
     *
     * @param aszTag array - Optional associative array of user defined tags
     *              and their corresponding values to be processed.
     *              i.e. $aszTag["my_tag"] = "My tag's value". In this example
     *              all the tags [my_tag] will be replaced with the string "My
     *              tag's value".
     * @return mixed - A string representation(typically HTML) of the template
     *              file with all tags processed or false if failed.
     **/
    function processQueryTemplate( $aszTag = array() )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "processQueryTemplate() starting" );

        // check to see that there is a valid map file open
        if ($this->oMap == null)
        {
            // log error
            $this->error( ERR_FILE_IO, "No valid map object to write." );

            // return failure
            return false;
        }

        // check to see if the the tag array is a valid array
        if ( !is_array( $aszTag ) )
        {
            // log error
            $this->error( ERR_FILE_IO, "Supplied Tag array is not a valid
                                                                    array." );

            // return failure
            return false;
        }

        //log exit
        $this->logFuncEnd( LOG_ALL, "processQueryTemplate() done" );

        // call mapscript's proceesquerytemplate
        return $this->oMap->processquerytemplate( $aszTag );

    // end processQueryTemplate function
    }

    /**
     * This function processes the legend template file as set in the legend
     * object of the current session's map object.  It returns the processed
     * file as string.
     *
     * @param aszTag array - Optional associative array of user defined tags
     *              and their corresponding values to be processed.
     *              i.e. $aszTag["my_tag"] = "My tag's value". In this example
     *              all the tags [my_tag] will be replaced with the string "My
     *              tag's value".
     * @return mixed - A string representation(typically HTML) of the template
     *              file with all tags processed or false if failed.
     **/
    function processLegendTemplate( $aszTag = array() )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "processLegendTemplate() starting" );

        // check to see that there is a valid map file open
        if ($this->oMap == null)
        {
            // log error
            $this->error( ERR_FILE_IO, "No valid map object to write." );

            // return failure
            return false;
        }

        // check to see if the the tag array is a valid array
        if ( !is_array( $aszTag ) )
        {
            // log error
            $this->error( ERR_FILE_IO, "Supplied Tag array is not a valid
                                                                    array." );

            // return failure
            return false;
        }

        // process the template
        $szResult = $this->oMap->processlegendtemplate( $aszTag );

        //log exit
        $this->logFuncEnd( LOG_ALL, "processLegendTemplate() done" );
        return $szResult;

    // end processLegendTemplate function
    }

    /**
     * insertLayerCopyByType()
     *
     * Postcondition:  This function inserts a copy of the given layer object
     *                 into the cuurent mapfile and position it according to
     *                 it's type.
     * @param oSourceLayer object - Layer object to copy.
     * @return boolean - True if successful, false if not.
     * @desc Inserts a copy of the given layer based on type.
     */
    function insertLayerCopyByType( $oSourceLayer )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "insertLayerByType() starting" );

        // get the index at which to insert the layer
        $nInsertionIndex = $this->getInsertionIndexByType( $oSourceLayer->type );

        // copy the layer to a new object
        $oLayer = ms_newLayerObj( $this->oMap, $oSourceLayer );

        // move the layer as many iterations as required
        for ( $i=0; $i<$nInsertionIndex; $i++ )
            $this->oMap->movelayerup( $this->oMap->numlayers - 1 );

        //log exit
        $this->logFuncEnd( LOG_ALL, "insertLayerByType() done" );

        // return success
        return true;

    // end inxsertLayerByType() function
    }

    /**
     * getInsertionIndexByType()
     *
     * Postcondition:  This function returns the index at which to insert a
     *                 layer of the given type.
     * @param nMSType integer - Layer type to insert.
     * @return integer - Index at which to insert a layer.
     * @desc Returns the index at which to insert a layer of given type.
     */
    function getInsertionIndexByType( $nMSType )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "getTypeInsertionIndex() starting" );

        // get the layers in the order that they are drawn
        $anLayers = $this->oMap->getlayersdrawingorder();
        if ( !is_array( $anLayers ) ) $anLayers = array();
        $anLayers = array_reverse( $anLayers );

        // loop through the array until the correct type found
        $nCount = $this->oMap->numlayers;
        $nReturnIndex = 0;
        for( $i=0; $i<$nCount; $i++ )
        {
            // get next layer and check type
            $oLayer = $this->oMap->getlayer( $anLayers[$i] );
            if ( $this->rankLayer( $nMSType ) >= $this->rankLayer( $oLayer->type ) )
                break;
            else
                $nReturnIndex++;
        }

        //log exit
        $this->logFuncEnd( LOG_ALL, "getTypeInsertionIndex() done" );

        // return the index
        return $nReturnIndex;

    // end getTypeInsertionIndex() function
    }

    /**
     * rankLayer()
     *
     * Postcondition:  This function ranks layers according to type.
     * @param nMSType integer - Layer type to rank.
     * @return integer - Rank of layer.
     * @desc Rank layer according to type.
     */
    function rankLayer( $nMSType )
    {
        // return the corresponding rank
        switch ( $nMSType )
        {
        case MS_LAYER_QUERY:
            return 5;
            break;
        case MS_LAYER_ANNOTATION:
            return 4;
            break;
        case MS_LAYER_POINT:
            return 3;
            break;
        case MS_LAYER_LINE:
            return 2;
            break;
        case MS_LAYER_POLYGON:
            return 1;
            break;
        default:
            return 0;
        }
    // end rankLayer() function
    }

    function &addAnnotationLayer()
    {
        $oLayer =& new AnnotationLayer();
        $this->aoAnnotation[count($this->aoAnnotation)] =& $oLayer;

        return $oLayer;
    }


    function deleteAnnotationLayer($szAnnotationName)
    {
        $numAnnotations = count($this->aoAnnotation);
        for($i=0; $i<$numAnnotations; $i++)
        {
            if($this->aoAnnotation[$i]->name == $szAnnotationName ||
               $szAnnotationName == "")
            {
                unset($this->aoAnnotation[$i]);
                $this->aoAnnotation =& array_slice($this->aoAnnotation, 0);
                if(isset($_SESSION['aoAnnotation']) &&
                   isset($_SESSION['aoAnnotation'][$i]))
                {
                    unset($_SESSION['aoAnnotation'][$i]);
                    $_SESSION['aoAnnotation'] =&
                        array_slice($_SESSION['aoAnnotation'], 0);
                }

                if($szAnnotationName == "")
                {
                    $i = 0;
                    $numAnnotations = count($this->aoAnnotation);
                }
                else
                {
                    break;
                }
            }
        }
    }

    function resetAllAnnotationLayers()
    {
        deleteAnnotationLayer("");
    }

    /**
     * RestoreAnnotations
     */
    function restoreAnnotations()
    {
        if (isset($_SESSION['aoAnnotation']))
        {
            $this->aoAnnotation =& $_SESSION['aoAnnotation'];

            $nNumAnno = count($this->aoAnnotation);
            for($i=0;$i<$nNumAnno;$i++)
            {
                $oLayer = ms_newLayerObj($this->oMap);
                $this->aoAnnotation[$i]->sync($oLayer);
            }
        }
    }

// end MapSession class
}

/**
 * This class extends the base MapSession Class to allow a save and restore
 * state option.  This class is the read only version meaning that the save &
 * restore sessions will NOT wrote to the physical disk.  There is also a
 * read/write version of this class called MapSession_RW.
 * It is assummed that the phpmapscript module is loaded prior to
 * instantiaiting this class.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class MapSession_R extends MapSession
{
    /**
     *  Construct a new MapSession instance and initialize it.
     */
    function MapSession_R()
    {
        // call the constructor for the map session base class
        $this->MapSession();

        // set the scope for the logging functions
        $this->setScope( "MapSession_R" );

    // end constructor
    }

    /**
     * This function saves the current state of the map.
     *
     * @param - aszExcludeKeys array of state key to exclude
     *
     * @return string - A unique state ID or "" if failed.
     */
    function saveState($aszExcludeKeys = array())
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "saveState() starting" );

        // check to see that there is a valid map file open
        if ($this->oMap == null)
        {
            // log error
            $this->error( ERR_FILE_IO, "No valid map object to write." );

            // return failure
            return "";
        }

        // Save annotation layer in session.
        $_SESSION['aoAnnotation'] = $this->aoAnnotation;

        $anDrawingOrder = $this->oMap->getlayersdrawingorder();
        // remove annotation layers from map object
        foreach ( $anDrawingOrder as $nIndex )
        {
            // create the layer object
            $oLayer = $this->oMap->getlayer( $nIndex );;

            $bFind = false;
            for ($i=0;$i<count($this->aoAnnotation);$i++)
            {
                if ($oLayer->name == $this->aoAnnotation[$i]->name)
                {
                    $bFind = true;
                    break;
                }
            }

            if ($bFind)
            {
                $oLayer->set("status", MS_DELETE);
            }
        }

        // build the state id from the bbox, projection, and list of layers
        $szStateID = $this->buildStateID( $this->oMap, $aszExcludeKeys );

        // log the result
        $this->log( LOG_VERBOSE, "saveState new ID = ".$szStateID );

        //log exit
        $this->logFuncEnd( LOG_ALL, "saveState() done" );

        // return ID
        return $szStateID;

    // end the saveState function
    }

    /**
     * This function restores the state of a map file by applying the value
     * of the State ID to the map file.  If no map fil eis given then the
     * State ID will be applied to the current map.  If the state id is not
     * given but the map file is then the map file will be opened.
     *
     * @param szStateID - Optional unique state ID.
     * @param szMapFile - Optional mapfile to restore state to.
     * @param szMapFilePath - Optional mapfile path to restore state to.
     * @param bRestoreLayerState - Optionally turn on restoration of layer
     *        state from the state key.
     *
     * @return boolean - True if successful, false if not.
     */
    function restoreState( $szStateID = "", $szMapFile = "",
                           $szMapFilePath = "",
                           $bRestoreLayerState = true)
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "restoreState() starting" );

        // check if map file was supplied
        if ( $szMapFile != "" )
        {
            // open the requested map file
            if ( !$this->readMapFile($szMapFile, $szMapFilePath) )
                return false;
        }

        // check for a current map object
        if ( $this->oMap == "" )
        {
            // log error
            $this->error( ERR_NOTICE, "No Valid map object to restore to." );

            // return failure
            return "false";

        }

        // if no state given then done
        if ( $szStateID == "" )
        {
            // log finished
            $this->logFuncEnd( LOG_ALL, "restoreState() done" );
            return true;
        }

        // decode the stateid
        $szStateID = urldecode( $szStateID );

        // process the ID
        $aszElements = explode( "|", $szStateID);

        $aszKeys = array();

        foreach ($aszElements as $value)
        {
            $aszElement = explode( "=", $value, 2 );

	    if (count($aszElement) == 2)
            {
		$aszKeys[$aszElement[0]] = $aszElement[1];
	    }
        }

        if (isset($aszKeys["MAPSIZE"]) && $aszKeys["MAPSIZE"] != "" )
        {

            // set the mapsize if not empty
            // get the mapsize items
            $anMapsize = explode( "," , $aszKeys["MAPSIZE"] );
            $this->oMap->set( "width", $anMapsize[0]);
            $this->oMap->set( "height", $anMapsize[1]);

            // log
            $this->log( LOG_QUIET, "Set width and height [".$anMapsize[0]."x".$anMapsize[1]."]." );
        }

        if (isset($aszKeys["BBOX"]) && $aszKeys["BBOX"] != "" )
        {
            // separate out the extents
            $anExtents = explode(",",$aszKeys["BBOX"]);

            // log
            $this->log( LOG_QUIET, "Setting new extents...");

            // set the extents
            $this->oMap->setextent($anExtents[0],$anExtents[1],
                                   $anExtents[2],$anExtents[3]);

            // log current extent values
            $this->log( LOG_QUIET, "New extents: ".$this->oMap->extent->minx.
                                                  ", ".$this->oMap->extent->miny.
                                                  ", ".$this->oMap->extent->maxx.
                                                  ", ".$this->oMap->extent->maxy.
                                                  "." );
        }

        // set the projection if not empty
        if ( isset($aszKeys["SRS"]) && $aszKeys["SRS"] != "" )
        {
           // srs can contain "="
           $this->oMap->setprojection(implode( "\n", split( " ", $aszKeys["SRS"] ) ));
        }

        if (isset($aszKeys["LAYERS"]) && $bRestoreLayerState)
        {

            // split the string into an array
            if ( $aszKeys["LAYERS"] != "" )
            {
                // split
                $anLayers = split(",",$aszKeys["LAYERS"]);
                sort($anLayers);
                reset($anLayers);
            }
            else
            {
                // otherwise default
                $anLayers = array();
            }

            // initialize the array counter
            $j = 0;

            // loop through all the layers and set their values
            for ($i=0;$i<$this->oMap->numlayers;$i++)
            {
                // get layer object
                $oLayer = $this->oMap->getlayer($i);

                // check if the current layer matches the "ON" array
                if ( isset( $anLayers[$j] ) && $i == $anLayers[$j] )
                {
                    // log the check
                    $this->log(LOG_QUIET,"Layer ".$i." is ON.");

                    // turn the layer on
                    $oLayer->set("status",MS_ON);

                    // get next item in the array
                    $j++;
                }
                else
                {
                    // log the check
                    $this->log(LOG_QUIET,"Layer ".$i." is OFF.");

                    // turn the layer on
                    $oLayer->set("status",MS_OFF);

                }
            }
        }

        $this->restoreAnnotations();

        //log exit
        $this->logFuncEnd( LOG_ALL, "restoreState() done" );

        // return success
        return true;

    // end the restoreState function
    }

    /**
     * This function reads the given map object and builds a string to
     * to represent the state of the file in terms of:
     *  1) extents - key is "BBOX", values are space de-limited.
     *  2) projection - key is "SRS".
     *  3) map size - key is "MAPSIZE", values are comma de-limited.
     *  4) active layers - key is "LAYERS", values are comma de-limited.
     *
     * The three sections of the string will be separated by a "|".  Each
     * individual section will be in the following format:
     *      key=value (i.e. BBOX=0 0 100 100)
     *
     * @param oMap - Map file object to read.
     * @param aszExcludeKeys - Dont save Keys specified in array
     *
     * @return string - The state ID representing the current state if the
     *                  give map file.
     */
    function buildStateID( $oMap, $aszExcludeKeys = array() )
    {
        // initialize the return string
        $szReturn = "";

        // get the current extents
        if (!in_array("BBOX", $aszExcludeKeys))
            $szReturn = "BBOX=".$oMap->extent->minx.",".$oMap->extent->miny.",".
                                $oMap->extent->maxx.",".$oMap->extent->maxy;

        // get the current map projection
        if (!in_array("RSR", $aszExcludeKeys))
            $szReturn .= "|SRS=".urlencode($oMap->getProjection());

        // get the current map size
        if (!in_array("MAPSIZE", $aszExcludeKeys))
            $szReturn .= "|MAPSIZE=".$oMap->width.",".$oMap->height;

        // build the list of active layers
        if (!in_array("LAYERS", $aszExcludeKeys))
        {
            $szReturn .= "|LAYERS=";

            // get the drawing order
            $anDrawingOrder = $oMap->getlayersdrawingorder();

            // loop to build list
            foreach ( $anDrawingOrder as $nIndex )
            {
                // create the layer object
                $oLayer = $oMap->getlayer( $nIndex );

                // check it's status and add if on
                if ( $oLayer->status == MS_ON || $oLayer->status == MS_DEFAULT )
                {
                    $szReturn .= $oLayer->index.",";
                }
            }

            // remove the trailing ","
            $szReturn = substr($szReturn,0,strlen($szReturn)-1);
        }

        // return the string
        return $szReturn;

    }

// end MapSession_R class
}

/**
 * This class extends the base MapSession Class to allow a save and restore
 * state option.  This class is the read-write version meaning that the save &
 * restore sessions will write/read to the physical disk.  There is also a read-
 * only version of this class called MapSession_R
 * It is assummed that the phpmapscript module is loaded prior to
 * instantiaiting this class.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class MapSession_RW extends MapSession
{
    /**
     *  Construct a new MapSession instance and initialize it.
     */
    function MapSession_RW()
    {
        // call the constructor for the map session base class
        $this->MapSession();

        // set the scope for the logging functions
        $this->setScope( "MapSession_RW" );

    // end constructor
    }

    /**
     * This function saves the current state of the map.
     * It requires that the temp directory been set.
     *
     * @return string - A unique state ID or "" if failed.
     */
    function saveState()
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "saveState() starting" );

        // check to see that there is a valid map file open
        if ($this->oMap == null)
        {
            // log error
            $this->error( ERR_FILE_IO, "No valid map object to write." );

            // return failure
            return "";
        }

        // check to see if the temp directory has been set
        if ( !isset($this->szTempDir) )
        {
            // log error
            $this->error( ERR_FILE_IO, "The temp directory has not been set." );

            // return failure
            return "";
        }

        // check to see if the temp directory is a valid directory
        if ( !is_dir($this->szTempDir) )
        {
            // log error
            $this->error( ERR_FILE_IO, "The temp directory[".$this->szTempDir.
                             "] does not exist." );

            // return failure
            return "";
        }

        // get the unique state identifier
        $szStateID = time()."-".rand(1000,9999);

        // log the result
        $this->log( LOG_VERBOSE, "saveState new ID = ".$szStateID );

        // write the PrevStateID to the map object
        //$this->oMap->setmetadata( "PrevStateID", $this->szPrevStateID );

        // Save annotation layer in session.
        $_SESSION['aoAnnotation'] = $this->aoAnnotation;

        $anDrawingOrder = $this->oMap->getlayersdrawingorder();
        // remove annotation layers from map object
        foreach ( $anDrawingOrder as $nIndex )
        {
            // create the layer object
            $oLayer = $this->oMap->getlayer( $nIndex );;

            $bFind = false;
            for ($i=0;$i<count($this->aoAnnotation);$i++)
            {
                if ($oLayer->name == $this->aoAnnotation[$i]->sName)
                {
                    $bFind = true;
                    break;
                }
            }

            if ($bFind)
            {
                $oLayer->set("status", MS_DELETE);
            }
        }

        // write the map to a temp directory
        if ( !$this->writeMapFile( $this->szTempDir.$szStateID.".map" ) )
            return "";

        // update the previous state id
        //$this->szPrevStateID = $szStateID;

        $_SESSION['aoAnnotation'] = $this->aoAnnotation;

        //log exit
        $this->logFuncEnd( LOG_ALL, "saveState() done" );

        // return ID
        return $szStateID;

    // end the saveState function
    }

    /**
     * This function restores the state of a map file by opening the map file
     * in the temp directory corresponding to the supplied ID.  The restored
     * state becomes the current map object.
     * It requires that the temp directory been set.
     *
     * @param szStateID - Optional unique state ID.
     * @param szMapFile - Optional mapfile to restore state to.
     * @param szMapFilePath - Optional mapfile path to restore state to.
     *
     * @return boolean - True if successful, false if not.
     */
    function restoreState( $szStateID = "", $szMapFile = "",
                                                        $szMapFilePath = "" )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "restoreState() starting" );

        // check if map file was supplied
        if ( $szMapFile == "" )
            $szMapFile = $this->szMapFile;

        // check for a valid mapfile
        if ( $szMapFile == "" )
        {
            // log error
            $this->error( ERR_FILE_IO, "The map file has not been set." );

            // return failure
            return "false";

        }

        // if no state given then just open the map
        if ( $szStateID == "" )
        {
            // check for success
            if ( !$this->readMapFile($szMapFile, $szMapFilePath) )
                return false;
            else
                $this->logFuncEnd( LOG_ALL, "restoreState() done" );
                return true;
        }

        // check to see if the temp directory has been set
        if ( !isset($this->szTempDir) )
        {
            // log error
            $this->error( ERR_FILE_IO, "The temp directory has not been set." );

            // return failure
            return "false";
        }

        // check to see if the temp directory is a valid directory
        if ( !is_dir($this->szTempDir) )
        {
            // log error
            $this->error( ERR_FILE_IO, "The temp directory[".$this->szTempDir.
                             "] does not exist." );

            // return failure
            return "false";
        }

        // build the map names
        $szSessionMap = $this->szTempDir.$szStateID.".map";

        // check for a valid session map file
        if( !file_exists($szSessionMap) )
        {
            // log error
            $this->error( ERR_FILE_IO, "The requested map session file[".
                                        $szSessionMap."] does not exist." );

            // return failure
            return "false";
        }

        // attempt to open the map
        if ( !$this->readMapFile( $szSessionMap, $szMapFilePath ) )
            return false;

        // log the result
        $this->log( LOG_QUIET, "restoreState - successful for ID [".
                                                        $szStateID."]." );
        //log exit
        $this->logFuncEnd( LOG_ALL, "restoreState() done" );

        $this->restoreAnnotations();

        // return success
        return true;

    // end the restoreState function
    }

// end MapSession_RW class
}

/**
   Annotation class

   This class represent a Mapserver layer in a php structure.
*/
class AnnotationLayer
{
   var $numclasses = 0;
   var $index      = NOTSET;      /// (read-only)
   var $status     = NOTSET;     /// (MS_ON, MS_OFF, MS_DEFAULT or MS_DELETE)
   var $debug      = NOTSET;
   var $classitem  = NOTSET;
   var $name       = NOTSET;
   var $group      = NOTSET;
   var $data       = NOTSET;
   var $type       = NOTSET;
   var $tolerance  = NOTSET;
   var $toleranceunits = NOTSET;
   var $symbolscale    = NOTSET;
   var $minscale       = NOTSET;
   var $maxscale       = NOTSET;
   var $labelminscale  = NOTSET;
   var $labelmaxscale  = NOTSET;
   var $maxfeatures    = NOTSET;
   var $offsite       = NOTSET;
   var $annotate       = NOTSET;
   var $transform      = NOTSET;
   var $labelcache     = NOTSET;
   var $postlabelcache = NOTSET;
   var $labelitem      = NOTSET;
   var $labelsizeitem  = NOTSET;
   var $labelangleitem = NOTSET;
   var $tileitem       = NOTSET;
   var $tileindex      = NOTSET;
   var $header         = NOTSET;
   var $footer         = NOTSET;
   var $connection     = NOTSET;
   var $connectiontype = NOTSET;
   var $filteritem     = NOTSET;
   var $template       = NOTSET;
   var $styleitem      = NOTSET;
   var $projection     = NOTSET;
   var $transparency   = NOTSET;
   var $grid; /// GridObject
   var $aoClass         = array(); /// Array of ClassObject

   function AnnotationLayer()
   {
       $this->offsite = new ColorObject();
   }

   function sync(&$oLayer)
   {

       if ($this->status != NOTSET)       $oLayer->set("status", $this->status);
       if ($this->debug != NOTSET)        $oLayer->set("debug", $this->debug);
       if ($this->classitem != NOTSET)    $oLayer->set("classitem", $this->classitem);
       if ($this->name != NOTSET)         $oLayer->set("name", $this->name);
       if ($this->group != NOTSET)        $oLayer->set("group", $this->group);
       if ($this->data != NOTSET)         $oLayer->set("data", $this->data);
       if ($this->type != NOTSET)         $oLayer->set("type", $this->type);
       if ($this->tolerance != NOTSET)    $oLayer->set("tolerance", $this->tolerance);
       if ($this->toleranceunits != NOTSET)$oLayer->set("toleranceunits", $this->toleranceunits);
       if ($this->symbolscale != NOTSET)  $oLayer->set("symbolscale", $this->symbolscale);
       if ($this->minscale != NOTSET)     $oLayer->set("minscale", $this->inscale);
       if ($this->maxscale != NOTSET)     $oLayer->set("maxscale", $this->maxscale);
       if ($this->labelminscale != NOTSET)$oLayer->set("labelminscale", $this->labelminscale);
       if ($this->maxfeatures != NOTSET)  $oLayer->set("maxfeatures", $this->maxfeatures);
       if ($this->transform != NOTSET)    $oLayer->set("transform", $this->transform);
       if ($this->labelcache != NOTSET)   $oLayer->set("labelcache", $this->labelcache);
       if ($this->postlabelcache != NOTSET)$oLayer->set("postlabelcache", $this->postlabelcache);
       if ($this->labelitem != NOTSET)    $oLayer->set("labelitem", $this->labelitem);
       if ($this->labelsizeitem != NOTSET)$oLayer->set("labelsizeitem", $this->labelsizeitem);
       if ($this->labelangleitem != NOTSET)$oLayer->set("labelangleitem", $this->labelangleitem);
       if ($this->tileitem != NOTSET)     $oLayer->set("tileitem", $this->tileitem);
       if ($this->tileindex != NOTSET)    $oLayer->set("tileindex", $this->tileindex);
       if ($this->header != NOTSET)       $oLayer->set("header", $this->header);
       if ($this->footer != NOTSET)       $oLayer->set("footer", $this->footer);
       if ($this->connection != NOTSET)   $oLayer->set("connection", $this->connection);
       if ($this->connectiontype != NOTSET)$oLayer->set("connectiontype", $this->connectiontype);
       if ($this->filteritem != NOTSET)   $oLayer->set("filteritem", $this->filteritem);
       if ($this->template != NOTSET)     $oLayer->set("template", $this->template);
       if ($this->styleitem != NOTSET)    $oLayer->set("styleitem", $this->styleitem);
       if ($this->projection != NOTSET)   $oLayer->setprojection($this->projection);
       if ($this->transparency != NOTSET) $oLayer->set("transparency", $this->transparency);

       // grid
       // $this->brid->sync();

       // color
       $this->offsite->sync($oLayer->offsite);

       // classes
       for ($j=0;$j<$this->numclasses;$j++)
       {
           $oClass = ms_newClassObj($oLayer);

           $this->aoClass[$j]->sync($oClass);
       }
   }

   function set($szColName, $szValue)
   {
       eval("\$this->$szColName = \"$szValue\";");
   }

   function &addClass()
   {
       $this->numclasses++;
       $oClass =& new AnnotationLayerClass();
       $this->aoClass[count($this->aoClass)] =& $oClass;
       return $oClass;
   }
}

class AnnotationLayerClass

{
   var $name     = NOTSET;
   var $title    = NOTSET;
   var $type     = NOTSET;
   var $status   = NOTSET; /// (MS_ON, MS_OFF or MS_DELETE)
   var $minscale = NOTSET;
   var $maxscale = NOTSET;
   var $template = NOTSET;
   var $label    = NOTSET; // Labelobj
   var $numstyles= 0;
   var $aoStyle   = array();


   function AnnotationLayerClass()
   {
       $this->label = new LabelObject();
   }

   function set($szColName, $szValue)
   {
       eval("\$this->$szColName = \"$szValue\";");
   }

   function sync(&$oClass)
   {
       if ($this->name != NOTSET)       $oClass->set("name", $this->name);
       if ($this->title != NOTSET)      $oClass->set("title", $this->title);
       if ($this->type != NOTSET)       $oClass->set("type", $this->type);
       if ($this->status != NOTSET)     $oClass->set("status", $this->status);
       if ($this->minscale != NOTSET)   $oClass->set("minscale", $this->minscale);
       if ($this->maxscale != NOTSET)   $oClass->set("maxscale", $this->maxscale);
       if ($this->template != NOTSET)   $oClass->set("template", $this->template);

       $this->label->sync($oClass->label);

       for ($i=0;$i<$this->numstyles;$i++)
       {
           $oStyle = ms_newStyleObj($oClass);
           $this->aoStyle[$i]->sync($oStyle);
       }
   }

   function &addStyle()
   {
       $this->numstyles++;
       $oStyle =& new StyleObject();
       $this->aoStyle[count($this->aoStyle)] =& $oStyle;
       return $oStyle;
   }
}

class LabelObject
{
   var $font                   = NOTSET;
   var $type                   = NOTSET;
   var $color                 = NOTSET;
   var $outlinecolor          = NOTSET;
   var $shadowcolor           = NOTSET;
   var $shadowsizex            = NOTSET;
   var $shadowsizey            = NOTSET;
   var $backgroundcolor       = NOTSET;
   var $backgroundshadowcolor = NOTSET;
   var $backgroundshadowsizex  = NOTSET;
   var $backgroundshadowsizey  = NOTSET;
   var $size                   = NOTSET;
   var $minsize                = NOTSET;
   var $maxsize                = NOTSET;
   var $position               = NOTSET;
   var $offsetx                = NOTSET;
   var $offsety                = NOTSET;
   var $angle                  = NOTSET;
   var $autoangle              = NOTSET;
   var $buffer                 = NOTSET;
   var $antialias              = NOTSET;
   var $wrap                   = NOTSET;
   var $minfeaturesize         = NOTSET;
   var $autominfeaturesize     = NOTSET;
   var $mindistance            = NOTSET;
   var $partials               = NOTSET;
   var $force                  = NOTSET;

   function LabelObject()
   {
       $this->color = new ColorObject();
       $this->outlinecolor = new ColorObject();
       $this->shadowcolor = new ColorObject();

       $this->backgroundcolor = new ColorObject();
       $this->backgroundshadowcolor = new ColorObject();
   }

   function sync(&$oLabel)
   {
       $this->color->sync($oLabel->color);
       $this->outlinecolor->sync($oLabel->outlinecolor);
       $this->shadowcolor->sync($oLabel->shadowcolor);
       $this->backgroundcolor->sync($oLabel->backgroundcolor);
       $this->backgroundshadowcolor->sync($oLabel->backgroundshadowcolor);

       if ($this->font != NOTSET)                 $oLabel->set("font", $this->font);
       if ($this->type != NOTSET)                 $oLabel->set("type", $this->type);
       if ($this->shadowsizex != NOTSET)          $oLabel->set("shadowsizex", $this->shadowsizex);
       if ($this->shadowsizey != NOTSET)          $oLabel->set("shadowsizey", $this->shadowsizey);
       if ($this->backgroundshadowsizex != NOTSET)$oLabel->set("backgroundshadowsizex", $this->backgroundshadowsizex);
       if ($this->backgroundshadowsizey != NOTSET)$oLabel->set("backgroundshadowsizey", $this->backgroundshadowsizey);
       if ($this->size != NOTSET)                 $oLabel->set("size", $this->size);
       if ($this->minsize != NOTSET)              $oLabel->set("minsize", $this->minsize);
       if ($this->maxsize != NOTSET)              $oLabel->set("maxsize", $this->maxsize);
       if ($this->position != NOTSET)             $oLabel->set("position", $this->position);
       if ($this->offsetx != NOTSET)              $oLabel->set("offsetx", $this->offsetx);
       if ($this->offsety != NOTSET)              $oLabel->set("offsety", $this->offsety);
       if ($this->angle != NOTSET)                $oLabel->set("angle", $this->angle);
       if ($this->autoangle != NOTSET)            $oLabel->set("autoangle", $this->autoangle);
       if ($this->buffer != NOTSET)               $oLabel->set("buffer", $this->buffer);
       if ($this->antialias != NOTSET)            $oLabel->set("antialias", $this->antialias);
       if ($this->wrap != NOTSET)                 $oLabel->set("wrap", $this->wrap);
       if ($this->minfeaturesize != NOTSET)       $oLabel->set("minfeaturesize", $this->minfeaturesize);
       if ($this->autominfeaturesize != NOTSET)   $oLabel->set("autominfeaturesize", $this->autominfeaturesize);
       if ($this->mindistance != NOTSET)          $oLabel->set("mindistance", $this->mindistance);
       if ($this->partials != NOTSET)             $oLabel->set("partials", $this->partials);
       if ($this->force != NOTSET)                $oLabel->set("force", $this->force);
   }

   function set($szColName, $szValue)
   {
       eval("\$this->$szColName = \"$szValue\";");
   }
}

class StyleObject
{
    var $symbol          = NOTSET;
    var $symbolname      = NOTSET;
    var $size            = NOTSET;
    var $minsize         = NOTSET;
    var $maxsize         = NOTSET;
    var $offsetx         = NOTSET;
    var $offsety         = NOTSET;
    var $color           = NOTSET;
    var $backgroundcolor = NOTSET;
    var $outlinecolor    = NOTSET;

    function StyleObject()
    {
        $this->color = new ColorObject();
        $this->backgroundcolor = new ColorObject();
        $this->outlinecolor = new ColorObject();
    }

    function sync(&$oStyle)
    {
        if ($this->symbol != NOTSET)    $oStyle->set("symbol", $this->symbol);
        if ($this->symbolname != NOTSET)$oStyle->set("symbolname", $this->symbolname);
        if ($this->size != NOTSET)      $oStyle->set("size", $this->size);
        if ($this->minsize != NOTSET)   $oStyle->set("minsize", $this->minsize);
        if ($this->maxsize != NOTSET)   $oStyle->set("maxsize", $this->maxsize);
        if ($this->offsetx != NOTSET)   $oStyle->set("offsetx", $this->offsetx);
        if ($this->offsety != NOTSET)   $oStyle->set("offsety", $this->offsety);

        $this->color->sync($oStyle->color);
        $this->backgroundcolor->sync($oStyle->backgroundcolor);
        $this->outlinecolor->sync($oStyle->outlinecolor);
    }

   function set($szColName, $szValue)
   {
       eval("\$this->$szColName = \"$szValue\";");
   }

}

class ColorObject
{
    var $r = -1;
    var $g = -1;
    var $b = -1;

    function ColorObject()
    {
    }

    function set($szColName, $szValue)
    {
        eval("\$this->$szColName = \"$szValue\";");
    }

    function setRGB($r, $g, $b)
    {
        $this->r = $r;
        $this->g = $g;
        $this->b = $b;
    }

    function sync(&$oColor)
    {
        if ($this->r != -1 &&
            $this->g != -1 &&
            $this->b != -1)
            $oColor->setRGB($this->r,
                            $this->g,
                            $this->b);
    }
}
?>
