<?php
/**
 * MapQuery Module
 * 
 * @project     PHP Wrapper Class
 * @revision    $Id: map_query.php,v 1.10 2004/05/29 02:47:02 pspencer Exp $
 * @purpose     This file contains classes related to managing map queries. 
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2001, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 **/
 
/**
 * This class encapsulates the ability to perform a query and retrieve results.  
 * It will hide any differences due to source.
 * It is assummed that the phpmapscript module is loaded prior to 
 * instantiating this class.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class MapQuery extends Logger
{       
    /**
     * The current session object (private).
     */
    var $oSession;
    
    /**
     * The array of layer indexes to query (private).
     */
    var $anLayers;  
      
    /**
     * Construct a new MapQuery instance and initialize it.
     *
     * @param $oMap object - Map object.
     */
    function MapQuery( &$oSession )
    {
        // initialize a copy of the map object passsed to this class
        $this->oSession = &$oSession;
        $this->anLayers = array();
        
        // set the scope for the logging functions
        $this->setScope( "MapQuery" );
    
    // end constructor
    }
    
    /**
     * This function defines which layers are to be included in the query.
     * 
     * @param $xnLayers mixed - An array of layer indexes to query or just a 
     *                          single index.
     * @return boolean - True if successful, False if not.
     **/
    function setQueryLayers( $xnLayers )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "setQueryLayers() starting" );
    
        // check if parameter is set
        if ( !isset( $xnLayers ) || $xnLayers == "" ) 
        {
            // log error
            $this->error( ERR_CRITICAL, "xnLayers parameter has not been set ".
                                        "for the setQueryLayers() function.");
            
            return false;
        }
        
        // clear any previous layers
        $this->anLayers = array();
                
        // check if the xnLayers is an array
        if ( is_array( $xnLayers ) )
        {
            // just copy the array
            $this->anLayers = $xnLayers;
        }
        else
        {
            // otherwise it is single layer so add to array
            array_push( $this->anLayers, $xnLayers );
        }
        
        // return success
        return true;

        // log exit
        $this->logFuncEnd( LOG_ALL, "setQueryLayers() done" );
    
    // end setQueryLayers function      
    }
    
    /**
     * This function defines which layers are to be included in the query.
     * 
     * @param xszLayers mixed - An array of layer names to query or a single 
     *                          layer name.
     * @return boolean - True if successful, False if not.
     **/
    function setQueryLayersByName( $xszLayers )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "setQueryLayersByName() starting" );

        // define the map object
        $oMap = $this->oSession->oMap;
        
        // check if parameter is set
        if ( !isset( $xszLayers ) || $xszLayers == "" ) 
        {
            // log error
            $this->error( ERR_CRITICAL, "xszLayers parameter has not been ".
                            "set for the setQueryLayersByName() function.");
            
            return false;
        }
        
        // check if the map object is valid
        if ( !isset( $oMap ) || $oMap == "" ) 
        {
            // log error
            $this->error( ERR_CRITICAL, "There is no valid map object to set ".
                                                                "layers for.");
            
            return false;
        }       

        // clear any previous layers
        $this->anLayers = array();
        
        // check if the xszLayers is an array
        if ( is_array( $xszLayers ) )
        {
            // loop through each item get it's index and add to the array
            foreach( $xszLayers as $szLayer )
            {
                // create the layer object and get index
                $oLayer = $oMap->getLayerByName( $szLayer );
                
                // add the index to the array
                array_push( $this->anLayers, $oLayer->index );
            }
        }
        else
        {
            // create the layer object and get index
            $oLayer = $oMap->getlayerbyname( $xszLayers );
            
            // add the index to the array
            array_push( $this->anLayers, $oLayer->index );  
        }
                
        // log exit
        $this->logFuncEnd( LOG_ALL, "setQueryLayersByName() done" );
        
        // return success
        return true;
    
    // end setQueryLayersByName function        
    }
    
    /**
     * This function executes a point query at the given coordinates.
     * 
     * @param $nX integer - X click position in pixels.
     * @param $nY integer - Y click position in pixels.
     * @return mixed - QueryResults object or false if failed.
     **/
    function executePointQuery( $nX, $nY )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "executePointQuery() starting" );
        
        // create map object
        $oMap = $this->oSession->oMap;
        
        // check if layers have been set
        if ( count( $this->anLayers ) == 0 )
        {
            // log error
            $this->error( ERR_NOTICE, "No layers have been set for querying.");         
            return false;       
        }

        // check if the map object is valid
        if ( !isset( $oMap ) || $oMap == "" ) 
        {
            // log error
            $this->error( ERR_CRITICAL, "There is no valid map object to ".
                                                                "query.");          
            return false;
        }       

        // check for valid X and Y co-ordinates
        if ( !is_numeric( $nX ) || !is_numeric( $nY ) )
        {
            // log error
            $this->error( ERR_CRITICAL, "Invalid click point(s) given.");           
            return false;       
        }
        
        // define a query results object
        $oQueryResults = new QueryResults;
        
        // loop through each layer and perform a point query
        foreach( $this->anLayers as $nLayer )
        {
            // create a layer object
            $oLayer = $oMap->getlayer( $nLayer );
            
            // determine if this is a WMS layer
            if ( $oLayer->connectiontype == MS_WMS )
            {   
                // temp fix to prevent wms_sld_body from being passed to getfeature call
                $szSLDBody = $oLayer->getmetadata( 'wms_sld_body' );
                $oLayer->setmetadata( 'wms_sld_body', '' );
                $oLayer->removemetadata( 'wms_sld_body' );
                
                // get the results and surpress any errors
                $szWMSURL = @$oLayer->getWMSFeatureInfoURL($nX, $nY, 0,"MIME");
                
                // restore the value 
                $oLayer->setmetadata( 'wms_sld_body', $szSLDBody );
                
                // check the results
                if ( $szWMSURL != "" )
                {   
                    // set the result to be the URL string
                    $oQueryResults->addResultSet( $szWMSURL );
                }
                else
                {
                    // set the result
                    $oQueryResults->addResultSet( "No results found for ".
                                                "layer[".$oLayer->name."]." );
                }           
            }
            else
            {
                // run non WMS point query
                $nXGeo = $this->pixelToGeo($nX, 0,
                                           $oMap->width,
                                           $oMap->extent->minx,
                                           $oMap->extent->maxx);
                $nYGeo = $this->pixelToGeo($nY, 0,
                                           $oMap->height,
                                           $oMap->extent->miny,
                                           $oMap->extent->maxy, 1);

                // create new point object
                $oGeoPoint = ms_newPointObj();

                // set the geo co-ordinates
                $oGeoPoint->setXY($nXGeo, $nYGeo);

                // execute query - use "@" to suppress errors
                if (@$oMap->queryByPoint($oGeoPoint, MS_MULTIPLE, -1) ==
                                                                   MS_SUCCESS)
                { 
                    // get the query results
                    $xQueryResults = $this->processQuery( $oLayer );
                    
                    // add to the result set
                    $oQueryResults->addResultSet( $xQueryResults );
                }
                else
                {           
                    // not valid layer, add to result set
                    $oQueryResults->addResultSet( "No results found for layer".
                                                    "[".$oLayer->name."]." );                                                   
                }
            
            // end WMS layer check
            }
            
        // end for loop         
        }
                    
        // log exit
        $this->logFuncEnd( LOG_ALL, "executePointQuery() done" );
        
        // return the query object
        return $oQueryResults;
    
    // end executePointQuery function       
    }

    /**
     * This function executes a rectangle query at the given coordinates.
     * 
     * @return mixed - QueryResults object or false if failed.
     **/
    /**
     * This function executes a rectangle query at the given coordinates.
     * 
     * @param $nMinX integer - The minX pixel position.
     * @param $nMinY integer - The minY pixel position.
     * @param $nMaxX integer - The maxX pixel position.
     * @param $nMaxY integer - The maxY pixel position.
     * @return mixed - QueryResults object or false if failed.
     **/
    function executeRectQuery( $nMinX, $nMinY, $nMaxX, $nMaxY )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "executeRectQuery() starting" );
        
        // create map object
        $oMap = $this->oSession->oMap;
        
        // check if layers have been set
        if ( count( $this->anLayers ) == 0 )
        {
            // log error
            $this->error( ERR_NOTICE, "No layers have been set for querying.");         
            return false;       
        }

        // check if the map object is valid
        if ( !isset( $oMap ) || $oMap == "" ) 
        {
            // log error
            $this->error( ERR_CRITICAL, "There is no valid map object to ".
                                                                "query.");          
            return false;
        }       

        // check for valid X and Y co-ordinates
        if ( !is_numeric( $nMinX ) || !is_numeric( $nMinY ) || 
             !is_numeric( $nMaxX ) || !is_numeric( $nMaxY ) )
        {
            // log error
            $this->error( ERR_CRITICAL, "Invalid click point(s) given.");           
            return false;       
        }
        
        // define a query results object
        $oQueryResults = new QueryResults;
        
        // loop through each layer and perform a rectangle query
        foreach( $this->anLayers as $nLayer )
        {
            // create a layer object
            $oLayer = $oMap->getlayer( $nLayer );
            
            // determine if this is a WMS layer
            if ( $oLayer->connectiontype == MS_WMS )
            {
                // calculate the center of the rectangle
                $nXCenter = (($nMaxX - $nMinX)/2) + $nMinX;
                $nYCenter = (($nMaxY - $nMinY)/2) + $nMinY;   
                
                // temp fix to prevent wms_sld_body from being passed to getfeature call
                $szSLDBody = $oLayer->getmetadata( 'wms_sld_body' );
                $oLayer->setmetadata( 'wms_sld_body', '' );
                $oLayer->removemetadata( 'wms_sld_body' );

                // get the results and surpress any errors
                $szWMSURL = @$oLayer->getWMSFeatureInfoURL($nXCenter, $nYCenter, 
                                                                    0,"MIME");
                // restore the value
                $oLayer->setmetadata( 'wms_sld_body', $szSLDBody );
                
                // check the results
                if ($szWMSURL <> "")
                {
                    // set the result to be the URL string
                    $oQueryResults->addResultSet( $szWMSURL );
                }
                else
                {
                    // set the result
                    $oQueryResults->addResultSet( "No results found for ".
                                                "layer[".$oLayer->name."]." );
                }           
            }
            else
            {
                // run non WMS rect query
                $nMinXGeo = $this->pixelToGeo($nMinX, 0,
                                           $oMap->width,
                                           $oMap->extent->minx,
                                           $oMap->extent->maxx);
                $nMinYGeo = $this->pixelToGeo($nMaxY, 0,
                                           $oMap->height,
                                           $oMap->extent->miny,
                                           $oMap->extent->maxy,1);
                $nMaxXGeo = $this->pixelToGeo($nMaxX, 0,
                                           $oMap->width,
                                           $oMap->extent->minx,
                                           $oMap->extent->maxx);
                $nMaxYGeo = $this->pixelToGeo($nMinY, 0,
                                           $oMap->height,
                                           $oMap->extent->miny,
                                           $oMap->extent->maxy,1);

                // create new rectangle object
                $oGeoRect =  ms_newRectObj();

                // set the geo extents
                $oGeoRect->setextent($nMinXGeo,$nMinYGeo,$nMaxXGeo,$nMaxYGeo);

                // execute query - use "@" to suppress errors
                if (@$oMap->queryByRect( $oGeoRect ) == MS_SUCCESS)
                { 
                    // get the query results
                    $xQueryResults = $this->processQuery( $oLayer );
                    
                    // add to the result set
                    $oQueryResults->addResultSet( $xQueryResults );
                }
                else
                {           
                    // not valid layer, add to result set
                    $oQueryResults->addResultSet( "No results found for layer".
                                                    "[".$oLayer->name."]." );                                                   
                }
            
            // end WMS layer check
            }
            
        // end for loop         
        }
                    
        // log exit
        $this->logFuncEnd( LOG_ALL, "executeRectQuery() done" );
        
        // return the query object
        return $oQueryResults;
    
    // end executeRectQuery function        
    }
    
    /**
     * This function processes the given layer's query results.  It is assumed 
     * that a query has been performed on the layer prior to calling this 
     * function.
     * 
     * @param oLayer object - The layer object to process.
     * @return mixed - Array of results or text indicating no results found.
     **/
    function processQuery( $oLayer )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "processQuery() starting" );
        
        // create map object
        $oMap = $this->oSession->oMap;
        
        // initialize variables
        $axResults = array();
    
        // get the number of results
        $nNumResults = intval($oLayer->getNumResults());
        
        // return if no results
        if ( $nNumResults < 1 )
        {
            // log exit
            $this->logFuncEnd( LOG_ALL, "processQuery() done" );
            return "No results found for layer[".$oLayer->name."].";
        }
        
        // open the layer to make the shape properties available
        if ( strpos( strtoupper( ms_GetVersion() ), "VERSION 3.6") > 0 )
            $oLayer->open( $oMap->shapepath );
        else 
            $oLayer->open();

        // add element to results array for each result row
        for ($i=0; $i < $nNumResults; $i++)
        {
            // get next shape row
            $oResult = $oLayer->getResult($i);
            $oShape = $oLayer->getShape($oResult->tileindex,
                                                        $oResult->shapeindex);
    
            // push the row array onto the results array
            $aTmp = $oShape->values;
            $aTmp = array_merge( $aTmp , 
                array("SHAPE_INDEX" => $oResult->shapeindex, 
                      "TILE_INDEX" => $oResult->tileindex, 
                      "LAYER_INDEX" => $oLayer->index ) );
            array_push( $axResults, $aTmp );
        
        // end for loop   
        }
    
        // close the layer
        $oLayer->close();
    
        // log exit
        $this->logFuncEnd( LOG_ALL, "processQuery() done" );    
        
        // return the array
        return $axResults;
    
    // end processQuery function
    }

    /**
     * This function converts a pixel position to geocoded position.
     *
     * @param nPixPos double - The pixel position.
     * @param dfPixMin double - Minimum map pixel value.
     * @param dfPixMax double - Maximum map pixel value.
     * @param dfGeoMin double - Minimum map geocoded value.
     * @param dfGeoMax double - Maximum map geocoded value.
     * @param nInversePix integer - Optional flag to inverse , set to 1 for
     *                            Y pixel coordinates where UL > LR
     * @return double - Geocoded position.
     **/
    function pixelToGeo($nPixPos, $dfPixMin, $dfPixMax, $dfGeoMin, $dfGeoMax,
                         $nInversePix = "")
    {
        // calculate the geocoded & pixel width
        $dfWidthGeo = $dfGeoMax - $dfGeoMin;
        $dfWidthPix = $dfPixMax - $dfPixMin;
    
        // get ratio
        $dfPixToGeo = $dfWidthGeo / $dfWidthPix;
    
        // get difference
        if ($nInversePix == "")
            $dfDeltaPix = $nPixPos - $dfPixMin;
        else
            $dfDeltaPix = $dfPixMax - $nPixPos;
    
        // calculate
        $dfDeltaGeo = $dfDeltaPix * $dfPixToGeo;
        $dfPosGeo = $dfGeoMin + $dfDeltaGeo;
    
        // return value
        return $dfPosGeo;
    
    // end pixelToGeo function
    }   
      
// end MapQuery class    
}

/**
 * This class defines the structure and functions necessary to use the results 
 * returned from a query.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class QueryResults extends Logger
{       
    /**
     * The the number of result sets available (public).
     */
    var $nNumResults;
      
    /**
     * The the array of result sets (private).
     */
    var $axResultSets;

    /**
     * Construct a new QueryResults instance and initialize it.
     */
    function QueryResults()
    {
        // initialize variables
        $this->nNumResults = 0;
        $this->axResultSets = array();

        // set the scope for the logging functions
        $this->setScope( "QueryResults" );
        
    }
    
    /**
     * This function adds the supplied array of results to the array of 
     * results.
     * 
     * @param xResults mixed - Array of results or URL string (if WMS).
     * @return boolean - True if successful, False if not.
     **/
    function addResultSet( $xResults )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "addResultSet() starting" );
        
        // add the result set to the array
        $bob = array_push( $this->axResultSets, $xResults );
        
        // increment the result count
        $this->nNumResults++;

        // return success
        return true;
        
        // log exit
        $this->logFuncEnd( LOG_ALL, "addResultSet() done" );
    
    // end addResultSet function        
    }

    /**
     * This function returns the result set array values for the given index.
     * 
     * @param $nIndex integer - The index to return values for.
     **/
    function getResultSet( $nIndex )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "getResultSet() starting" );

        // check if there are results to return
        if ( $this->nNumResults == 0 )
        {
            // log error
            $this->error( ERR_NOTICE, "There are no result sets available.");
            return false;                       
        }
        
        // check the index to see if it is valid
        $bError = false;
        if ( is_integer( $nIndex ) )
        {
            // check range
            if ( $nIndex < 0 || $nIndex >= $this->nNumResults ) $bError = true;
        }
        else
        {
            // flag invlaid index
            $bError = true;
        }
        
        // check error flag
        if ( $bError )
        {
            // log error
            $this->error( ERR_CRITICAL, "The given index[".$nIndex.
                                                            "] is invalid.");
            return false;                       
        }
        
        // log exit
        $this->logFuncEnd( LOG_ALL, "getResultSet() done" );

        // return the requested result set
        return $this->axResultSets[ $nIndex ];  
    
    // end getResultSet function        
    }       
        
// end QueryResults class   
}
?>