<?php
/**
 * MapNavigator Module
 * 
 * @project     PHP Wrapper Class
 * @revision    $Id: map_navigator.php,v 1.19 2003/10/27 13:45:13 pspencer Exp $
 * @purpose     This file contains classes related to managing map navigation. 
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2001, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 
/*****************************************************************************
 * $Log: map_navigator.php,v $
 * Revision 1.19  2003/10/27 13:45:13  pspencer
 * added comment.
 *
 * Revision 1.18  2003/09/15 20:17:38  pspencer
 * added guard against reprojecting without sufficient information or if unnecessary
 *
 * Revision 1.17  2002/12/13 21:39:00  sacha
 * remove int operation on douvle val
 *
 * Revision 1.16  2002/12/12 21:20:19  sacha
 * copy centerscale reprojection function out the class since we need it out of it
 *
 * Revision 1.15  2002/12/10 20:51:14  sacha
 * added a function to reproject certer point from a rectangle
 *
 * Revision 1.14  2002/09/24 16:48:28  assefa
 * Correct pixel coorditaes setting before zoomrectangle is called.
 *
 * Revision 1.13  2002/05/10 18:25:20  pspencer
 * missing point obj defn
 *
 * Revision 1.12  2002/05/10 14:54:52  bronsema
 * Added zoomScale function.
 *
 * Revision 1.11  2002/03/20 03:42:12  bronsema
 * Changed zoomRectangle to always limit pixels to mapsize.
 *
 * Revision 1.10  2002/03/20 03:38:30  bronsema
 * Updated zoomRectangle to limit to pixels to map size if max extents are set.
 *
 * Revision 1.9  2002/03/15 14:05:49  bronsema
 * Updated the MapSession class to allow the set/get of maximum extents and
 * adjusted the map naviagation to be restricted to those extents if they are set.
 *
 * Revision 1.8  2002/03/13 15:21:56  bronsema
 * Test commit to debug
 *
 * Revision 1.7  2002/03/13 15:21:23  bronsema
 * Test commit to debug
 *
 * Revision 1.6  2002/03/13 15:16:02  bronsema
 * Test commit to debug
 *
 * Revision 1.5  2002/03/13 15:14:56  bronsema
 * Test commit to debug
 *
 * Revision 1.4  2002/03/05 04:12:05  bronsema
 * Finished construction & integration of the RW and R class extensions of 
 * MapSession.
 *
 * Revision 1.3  2002/03/04 18:46:35  bronsema
 * Updated to manage session objects consistently
 *
 * Revision 1.2  2002/02/28 20:51:21  bronsema
 * Fixed minor issues
 *
 * Revision 1.1  2002/02/28 19:30:53  bronsema
 * Initial creation
 *
 *****************************************************************************/

// define the pan directions
define( "NORTH"     , 0 );
define( "NORTH_EAST", 1 );
define( "EAST"      , 2 );
define( "SOUTH_EAST", 3 );
define( "SOUTH"     , 4 );
define( "SOUTH_WEST", 5 );
define( "WEST"      , 6 );
define( "NORTH_WEST", 7 );

/**
 * This class encapsulates the manipulation of extents of the map file by 
 * providing common navigation functions that hide the complexity.
 * It is assummed that the phpmapscript module is loaded prior to 
 * instantiaiting this class.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class MapNavigator extends Logger
{       
    /**
     * The current session object (private).
     */
    var $oSession;
  
    /**
     * Construct a new MapNavigator instance and initialize it.
     *
     * @param oSession object - MapSession class object.
     */
    function MapNavigator( &$oSession )
    {
        // initialize variables
        $this->oSession = &$oSession;
        
        // set the scope for the logging functions
        $this->setScope( "MapNavigator" );
    
    // end constructor
    }
    
    /**
     * This function is the basic zoom point function.  All other zoom related
     * functions will use this and the zoomRectngle function as the base.  
	 * If the map session's MaxExtents have been set then the zoom will be 
	 * limited accordingly.  
     *
     * @param nFactor integer - The zoom factor (negative is zoom out).
     * @param nX integer - The x pixel value.
     * @param nY integer - The y pixel value.
     */
    function zoomPoint( $nFactor, $nX, $nY )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "zoomPoint() starting" );
        
        // get the map object
        $oMap = $this->oSession->getMapObj();
    
        // create the extents rectangle object
        $oCurrentExt = ms_newrectobj();

        // set the rectangle objects extents
        $oCurrentExt->setextent($oMap->extent->minx, 
                                $oMap->extent->miny,
                                $oMap->extent->maxx,
                                $oMap->extent->maxy);

        // log current extent values
        $this->log( LOG_QUIET, "oMap extents: ".$oMap->extent->minx.
                                              ", ".$oMap->extent->miny.
                                              ", ".$oMap->extent->maxx.
                                              ", ".$oMap->extent->maxy.
                                              "." );

        // create new point object
        $oPixelPos = ms_newpointobj();

        // set the click position
        $oPixelPos->setxy( $nX, $nY );

		// create a max zoom rectangle object if max extents are set
		$adMaxExtents = $this->oSession->getMaxExtents();
		if ( is_array( $adMaxExtents ) )
		{
          	// create the max extents rectangle object
            $oMaxExt = ms_newrectobj();
    
            // set the rectangle objects extents
            $oMaxExt->setextent($adMaxExtents["minX"], 
                                $adMaxExtents["minY"],
                                $adMaxExtents["maxX"],
                                $adMaxExtents["maxY"]);
								
        	// log max extent values
        	$this->log( LOG_QUIET, "Set max extents to limit zoom: ".
											  $adMaxExtents["minX"].
                                     	 ", ".$adMaxExtents["minY"].
                                         ", ".$adMaxExtents["maxX"].
                                         ", ".$adMaxExtents["maxY"].
                                         "." );
										 
			// Record the current extents (if factor < -1) to check if the
			// zoomout had any effect on the extents.  If the zoomout had no
			// effect on the extents and the zoomout factor is greater than
			// 1 then we can assume the user was intending to zoom further,
			// but was limited due to max extents restriction.  Therefore we
			// zoom to maximum extents.
			if ( $nFactor < -1 )
			{
				// record extents
				$dTmpMinX = doubleval( $oMap->extent->minx );
				$dTmpMinY = doubleval( $oMap->extent->miny );
				$dTmpMaxX = doubleval( $oMap->extent->maxx );
				$dTmpMaxY = doubleval( $oMap->extent->maxy );
			}
										 
			// zoom as a point limited to max
       		$oMap->zoompoint( $nFactor, $oPixelPos, $oMap->width, 
                               $oMap->height, $oCurrentExt,$oMaxExt );
			
			// compare the new extents
			if ( $nFactor < -1 )
			{
				// check for a match
				if ( doubleval( $oMap->extent->minx ) == $dTmpMinX &&
					 doubleval( $oMap->extent->miny ) == $dTmpMinY &&
					 doubleval( $oMap->extent->maxx ) == $dTmpMaxX &&
					 doubleval( $oMap->extent->maxy ) == $dTmpMaxY )
				{
					// set the extents to the max because further zooming was
					// requested but nothing happened
					$oMap->setextent( $adMaxExtents["minX"], 
									  $adMaxExtents["minY"], 
									  $adMaxExtents["maxX"], 
									  $adMaxExtents["maxY"] );
					
					// log the event
					$this->log( LOG_VERBOSE, "Zoom exceeded max, extents ".
												"will be set to the max." );									  
				}					 
			}
										   
			// reslease the rectangle object
			$oMaxExt->free();
			
								
		}
		else
		{
        	// zoom as a point with no limit
        	$oMap->zoompoint($nFactor, $oPixelPos, $oMap->width, 
                               					$oMap->height, $oCurrentExt);
		}
		
        // log zoom
        $this->log( LOG_VERBOSE, "Zoom as a point by a factor of: ".
                                            $nFactor." was successful." );
        // log current extent values
        $this->log( LOG_QUIET, "Current extents: ".$oMap->extent->minx.
                                              ", ".$oMap->extent->miny.
                                              ", ".$oMap->extent->maxx.
                                              ", ".$oMap->extent->maxy.
                                              "." );        
        //release the objects
        $oPixelPos->free();
             
        // release the extent objects
        $oCurrentExt->free();
         
        // log exit
        $this->logFuncEnd( LOG_ALL, "zoomPoint() done" );
        
    // end zoomPoint function
    }  

    /**
     * This function is the basic zoom rectangle function.  All other zoom 
     * related functions will use this and the zoomPoint function as the base.
     *
     * @param nMinX integer - The minimum x pixel value.
     * @param nMinY integer - The minimum y pixel value.
     * @param nMaxX integer - The maximum x pixel value.
     * @param nMaxY integer - The maximum y pixel value.
     */
    function zoomRectangle( $nMinX, $nMinY, $nMaxX, $nMaxY )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "zoomRectangle() starting" );

        // get the map object
        $oMap = $this->oSession->getMapObj();

		// limit the pixels to the size of the map
        if ( $nMinX < 0 ) $nMinX = 0;
        if ( $nMinX > $oMap->width ) $nMinX = $oMap->width;
        if ( $nMaxX < 0 ) $nMaxX = 0;
        if ( $nMaxX > $oMap->width ) $nMaxX = $oMap->width;			
        if ( $nMinY < 0 ) $nMinY = 0;
        if ( $nMinY > $oMap->height ) $nMinY = $oMap->height;
        if ( $nMaxY < 0 ) $nMaxY = 0;
        if ( $nMaxY > $oMap->height ) $nMaxY = $oMap->height;						
    
        // create the extents rectangle object
        $oCurrentExt = ms_newrectobj();

        // set the rectangle objects extents
        $oCurrentExt->setextent($oMap->extent->minx, 
                                $oMap->extent->miny,
                                $oMap->extent->maxx,
                                $oMap->extent->maxy);

        // log current extent values
        $this->log( LOG_QUIET, "Current extents: ".$oMap->extent->minx.
                                              ", ".$oMap->extent->miny.
                                              ", ".$oMap->extent->maxx.
                                              ", ".$oMap->extent->maxy.
                                              "." );

        // create the rectangle pixel object
        $oPixelRect = ms_newrectobj();

        // set the pixel rectangle extent values
        $oPixelRect->setextent( $nMinX, $nMinY, $nMaxX, $nMaxY );
/* ==================================================================== */
/*      For screen coordinates, the Upper Lefet (UL) corner is at 0,0   */
/*      (for georefernce coordinates the 0,0 is at LL.                  */
/*       Thus the $oPixelRect should consider that when calling the     */
/*      zoomrectangle.                                                  */
/* ==================================================================== */
        if ($oPixelRect->minx >  $oPixelRect->maxx)
        {
            // Use *1 to avoid dfTmp becoming a reference
            // to $oPixelRect->minx with PHP4 !?!?!?
            $dfTmp = $oPixelRect->minx*1;
            $oPixelRect->set("minx",$oPixelRect->maxx);
            $oPixelRect->set("maxx",$dfTmp);
        }
        if ($oPixelRect->miny <  $oPixelRect->maxy)
        {
            $dfTmp = $oPixelRect->miny*1;
            $oPixelRect->set("miny", $oPixelRect->maxy);
            $oPixelRect->set("maxy", $dfTmp);
        }
        
        //zoom as a rectangle
        $oMap->zoomrectangle($oPixelRect, $oMap->width, 
                                   $oMap->height, $oCurrentExt);
        // log zoom
        $this->log( LOG_VERBOSE, "Zoom as a rectangle was successful." );
        
        // log current extent values
        $this->log( LOG_QUIET, "Current extents: ".$oMap->extent->minx.
                                              ", ".$oMap->extent->miny.
                                              ", ".$oMap->extent->maxx.
                                              ", ".$oMap->extent->maxy.
                                              "." );        
        
        //release the objects
        $oPixelRect->free();
             
        // release the extent object
        $oCurrentExt->free();
         
        // log exit
        $this->logFuncEnd( LOG_ALL, "zoomRectangle() done" );
        
    // end zoomRectangle function
    }  

    /**
     * This function is a simple zoom-in-by-factor function.
     *
     * @param nFactor integer - The zoom factor (absolute value of this 
     *                          number will be used).
     * @param nX integer - The optional X pixel. 
     * @param nY integer - The optional Y pixel.
     */
    function zoomIn( $nFactor, $nX = "", $nY = "" )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "zoomIn( ".abs($nFactor).
                                                            " ) starting" );
    
        // calculate the center pixel co-ordinates of the map if necessary
        if ( $nX == "" && $nY == "" )
        {
            $nX = $this->oSession->oMap->width/2;
            $nY = $this->oSession->oMap->height/2;
        }
        
        // zoom by factor
        $this->zoomPoint( abs($nFactor), $nX, $nY );
        
        // log exit
        $this->logFuncEnd( LOG_ALL, "zoomIn( ".abs($nFactor)." ) done" );
        
    // end zoomIn function
    }  

    /**
     * This function is a simple zoom-out-by-factor function.
     *
     * @param nFactor integer - The zoom factor (positive values will be 
     *                          converted to negative).
     * @param nX integer - The optional X pixel. 
     * @param nY integer - The optional Y pixel.
     */
    function zoomOut( $nFactor, $nX = "", $nY = "" )
    {
        // convert factor to negative
        $nFactor = 0 - abs( $nFactor );
    
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "zoomOut( ".$nFactor.
                                                            " ) starting" );        
        // calculate the center pixel co-ordinates of the map if necessary
        if ( $nX == "" && $nY == "" )
        {
            $nX = $this->oSession->oMap->width/2;
            $nY = $this->oSession->oMap->height/2;
        }
        
        // zoom by factor
        $this->zoomPoint( $nFactor, $nX, $nY );
        
        // log exit
        $this->logFuncEnd( LOG_ALL, "zoomOut( ".$nFactor." ) done" );
        
    // end zoomOut function
    }  

    /**
     * This function zooms to the scale given at the center of the map or
	 * at the given point.
     * 
     * @param $nScale interger - The scale to zoom to.
     * @param $nX integer - The optional x pixel position to center at.
     * @param $nY integer - The optional y pixel position to center at.
     **/
    function zoomScale( $nScale, $nX = "", $nY = "" )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "zoomScale( ".$nScale.
                                                            " ) starting" );
    
        // calculate the center pixel co-ordinates of the map if necessary
        if ( $nX == "" && $nY == "" )
        {
            $nX = $this->oSession->oMap->width/2;
            $nY = $this->oSession->oMap->height/2;
        }
        
        $oPixelPos = ms_newpointobj();
        
		//set the click position
        $oPixelPos->setxy($nX,$nY);

        // create new rectangle object
        $oRect = ms_newrectobj();
        
        // get current map extents
        $dMapMinX = $this->oSession->oMap->extent->minx;
        $dMapMinY = $this->oSession->oMap->extent->miny;
        $dMapMaxX = $this->oSession->oMap->extent->maxx;
        $dMapMaxY = $this->oSession->oMap->extent->maxy;

        // set the extents of the rectangle
        $oRect->setextent($dMapMinX, $dMapMinY, $dMapMaxX, $dMapMaxY);
        
        // call the zoomscale function
        $this->oSession->oMap->zoomscale($nScale, $oPixelPos, 
		  $this->oSession->oMap->width, $this->oSession->oMap->height, $oRect);
        // log exit
        $this->logFuncEnd( LOG_ALL, "zoomScale( ".$nScale." ) done" );
        
    // end zoomIn function
    }  	
	
    /**
     * This function will update the current map object's extents to cause a
     * pan movment in the direction indicated..
     *
     * @param nDirection integer - The pan direction (0-7) as defined at the 
     *                             top of this class.
     * @param nFactor integer - The optional pan factor.  It is the number 
     *                          of full map screens (either width or height) 
     *                          to pan.  The default value is 0.5. The absolute
     *                          value of the pan factor will be used.
     */
    function pan( $nDirection, $nFactor = 0.5 )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "pan() starting" );

        // get the map object
        $oMap = $this->oSession->getMapObj();
        
        // determine the center pixel co-ordinates of the map
        $nX = $oMap->width/2;
        $nY = $oMap->height/2;      
        
        // log the center co-ordinates
        $this->log( LOG_VERBOSE, "The co-ordinates of the center of the map ".
                                                    "are: ".$nX.", ".$nY.".");
        // determine the number of pixels to move in both directions
        $nDeltaX = abs( $nFactor ) * $oMap->width;
        $nDeltaY = abs( $nFactor ) * $oMap->height;
        
        // calculate the co-ordinates of the pan
        switch ($nDirection) {
            
            // North
            case 0:
                // move the center co-ordinates
                $nY = $nY - $nDeltaY;
                $this->log( LOG_VERBOSE, "Pan north - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;
            
            // North East
            case 1:
                // move the center co-ordinates
                $nX = $nX + $nDeltaX;
                $nY = $nY - $nDeltaY;
                $this->log( LOG_VERBOSE, "Pan northeast - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;

            // East
            case 2:
                // move the center co-ordinates
                $nX = $nX + $nDeltaX;
                $this->log( LOG_VERBOSE, "Pan north - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;
                
            // South East
            case 3:
                // move the center co-ordinates
                $nX = $nX + $nDeltaX;
                $nY = $nY + $nDeltaY;
                $this->log( LOG_VERBOSE, "Pan north - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;
                
            // South
            case 4:
                // move the center co-ordinates
                $nY = $nY + $nDeltaY;
                $this->log( LOG_VERBOSE, "Pan north - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;
                
            // South West
            case 5:
                // move the center co-ordinates
                $nX = $nX - $nDeltaX;
                $nY = $nY + $nDeltaY;
                $this->log( LOG_VERBOSE, "Pan north - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;
                
            // West
            case 6:
                // move the center co-ordinates
                $nX = $nX - $nDeltaX;
                $this->log( LOG_VERBOSE, "Pan north - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;
                
            // North West
            case 7:
                // move the center co-ordinates
                $nX = $nX - $nDeltaX;
                $nY = $nY - $nDeltaY;
                $this->log( LOG_VERBOSE, "Pan north - new map center is: ".
                                                        $nX.", ".$nY."." );
                break;

            // invalid direction
            default:
                // give error message
                $this->error( ERR_CRITICAL, "Invalid direction[".$nDirection.
                                               "] specified in pan() function" );
                // exit function
                return;
        } 
        
        // execute the pan by zooming with a factor of 1
        $this->zoomPoint( 1, $nX, $nY );

        // log exit
        $this->logFuncEnd( LOG_ALL, "pan() done" );
        
    // end pan function
    }  

    /**
     * This function modifies the extents to cause a recentering on the 
     * requested pixel co-ordinates.
     *
     * @param nX integer - The x pixel co-ordinate to recentre to.
     * @param nY integer - The y pixel co-ordinate to recentre to.
     
    * @return void
    * @desc recentre the map at the given coordinates
    */
    function recentre( $nX, $nY )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "recentre() starting" );

        // zoom with a factor of 1 to recentre
        $this->zoomPoint( 1, $nX, $nY );

        // log exit
        $this->logFuncEnd( LOG_ALL, "recentre() done" );
        
    // end recentre function
    }      

    /**
     * This reproject a bbox from one projection to an other
     * by using the central point of the bbox.
     *
     * @param   aszExtents - bbox to reproject
     * @param   nMapWidth  - Map width in pixel
     * @param   nMapHeight - Map height in pixel
     * @param   szProjFrom - From projection
     * @param   szProjTo   - To
     *
     * $return  array of aszExtents reprojected.
     */
    function reprojectExtentFromCenter($aszExtents, $nMapWidth, $nMapHeight, $szProjFrom, $szProjTo)
    {
    	if (trim($szProjFrom) == "" || trim($szProjTo) == "" || strcasecmp( $szProjFrom, $szProjTo ) == 0)
    		return $aszExtents;
    		
    	$oNewMap = ms_newMapObj("");
        $oNewMap->set("width", $nMapWidth);
        $oNewMap->set("height", $nMapHeight);

        $oNewMap->setProjection($szProjFrom, MS_TRUE);
        $oNewMap->setextent($aszExtents[0], $aszExtents[1],
                            $aszExtents[2], $aszExtents[3]);

        $nScale = doubleval($oNewMap->scale);
        
        $oOldProj = ms_newProjectionObj($szProjFrom);
        $oNewProj = ms_newProjectionObj($szProjTo);
        
        $nCenterX = ($oNewMap->extent->minx + $oNewMap->extent->maxx) / 2.0;
        $nCenterY = ($oNewMap->extent->miny + $oNewMap->extent->maxy) / 2.0;
        
        $oCenter = ms_newPointObj( );
        $oCenter->setXY( $nCenterX, $nCenterY );
        
        $oCenter->project( $oOldProj, $oNewProj );
        
        $oNewMap->setProjection( $szProjTo, 1 );
        
        $oNewMap->setExtent($oCenter->x - 1, $oCenter->y - 1,
                            $oCenter->x + 1, $oCenter->y + 1);
        
        $nWidth = $oNewMap->width;
        $nHeight = $oNewMap->height;
        
        $oPixelPos = ms_newPointObj();
        $oPixelPos->setXY( $nWidth/2, $nHeight/2 );
        
        $oNewMap->zoomScale($nScale, $oPixelPos, $nWidth, 
                            $nHeight, $oNewMap->extent);
        
        return array($oNewMap->extent->minx,$oNewMap->extent->miny,
                     $oNewMap->extent->maxx,$oNewMap->extent->maxy);
    }
// end MapNavigator class    
}

/**
 * This reproject a bbox from one projection to an other
 * by using the central point of the bbox.
 *
 * @param   aszExtents - bbox to reproject
 * @param   nMapWidth  - Map width in pixel
 * @param   nMapHeight - Map height in pixel
 * @param   szProjFrom - From projection
 * @param   szProjTo   - To
 *
 * $return  array of aszExtents reprojected.
 */
function reprojectExtentFromCenter($aszExtents, $nMapWidth, $nMapHeight, $szProjFrom, $szProjTo)
{
  $oNewMap = ms_newMapObj("");
  $oNewMap->set("width", $nMapWidth);
  $oNewMap->set("height", $nMapHeight);
  
  $oNewMap->setProjection($szProjFrom, MS_TRUE);
  $oNewMap->setextent($aszExtents[0], $aszExtents[1],
                      $aszExtents[2], $aszExtents[3]);
  
  $nScale = doubleval($oNewMap->scale);

  $oOldProj = ms_newProjectionObj($szProjFrom);
  $oNewProj = ms_newProjectionObj($szProjTo);
  
  $nCenterX = ($oNewMap->extent->minx + $oNewMap->extent->maxx) / 2.0;
  $nCenterY = ($oNewMap->extent->miny + $oNewMap->extent->maxy) / 2.0;
  
  $oCenter = ms_newPointObj( );
  $oCenter->setXY( $nCenterX, $nCenterY );
  
  $oCenter->project( $oOldProj, $oNewProj );
  
  $oNewMap->setProjection( $szProjTo, 1 );
  
  $oNewMap->setExtent($oCenter->x - 1, $oCenter->y - 1,
                      $oCenter->x + 1, $oCenter->y + 1);
  
  $nWidth = $oNewMap->width;
  $nHeight = $oNewMap->height;
  
  $oPixelPos = ms_newPointObj();
  $oPixelPos->setXY( $nWidth/2, $nHeight/2 );
  
  $oNewMap->zoomScale($nScale, $oPixelPos, $nWidth, 
                      $nHeight, $oNewMap->extent);
  
  return array($oNewMap->extent->minx,$oNewMap->extent->miny,
               $oNewMap->extent->maxx,$oNewMap->extent->maxy);
}
?>
