<?php
/**
 * @project     PHP Utils
 * @revision    $Id: drawmap.php,v 1.60 2004/06/06 17:10:37 pspencer Exp $
 * @purpose     This file outputs the current map file to sceen
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 **/

 /*****************************************************************************
 * The following parameters are REQUIRED when caling this page:
 *
 *      map_session_mode : integer
 *                          0 indicates the map session is read-only,
 *                          1 indicates the map session is read-write.
 *
 *      run_query : integer
 *                          0 indicates a query is NOT to be run,
 *                          1 indicates a query is to be run.
 *
 * The following parameters are OPTIONAL when caling this page:
 * (They MUST be set using session variables of the same name)
 *
 *      gszMapName : string
 *                          Path and filename to the mapfile.  This value can
 *                          be included in the URL if sessions are NOT used
 *                          OR can be set using a DEFINE statment as outlined
 *                          in the code below.
 *
 *      gszCurrentState : string
 *                          The map session current state key.  Defaults to ""
 *                          if not supplied.
 *
 *      gszMapPath : string
 *                          The path to where mapscript "thinks" the mapfile is
 *                          located.  Defaults to the path of the given mapfile
 *                          if not given.
 *
 *      gszPHPMapScriptModName : string
 *                          The mapscript module name.  Default to
 *                          "php_mapscript.so (or .dll)" if not given.
 *
 *      gszMapscriptModule : string
 *                          alternate for gszPHPMapScriptModName.
 *
 *      gszTmpMapPath : string
 *                          The path for the temporary map files in read-write
 *                          mode.  Defaults to the path of the given mapfile if
 *                          not given.
 *
 *      gszQueryLayers : string
 *                          Comma delimited string of layer indexes to query.
 *                          If not supplied and a query is called for then all
 *                          visible layers will be queried.
 *
 *      gszQueryCoords : string
 *                          String of co-ordinates in the following format:
 *                          minx,miny;maxx,maxy
 *                          This value MUST be supplied if a query is performed.
 *
 *****************************************************************************/

/* ============================================================================
 * Un-comment the following code if the mapfile is to be DEFINED instead of in
 * session or URL.
 * ========================================================================= */
//DEFINE( "MAPFILE", "./mymap.map" );
$bProfile = false;
$szProfileFile = '';
if (isset($_GET['profile']))
{
    if (!extension_loaded( 'xdebug' ))
    {
        dl( 'php_xdebug.'.PHP_SHLIB_SUFFIX );
    }
    
    $bProfile = extension_loaded( 'xdebug' );
}

if ($bProfile)
{
    $szProfileFile = $_GET['profile'];
    if ($szProfileFile != "")
        echo "profiling to ".$_GET['profile']."<BR>";
    else
        echo "profiling to page.<BR>";
    
    xdebug_start_profiling( $szProfileFile );
}

/* ============================================================================
 * Include necessary files
 * ========================================================================= */
include_once( "../session/session.php" );
include_once( "../logger/logfile.php" );
include_once( "../logger/error_manager.php" );
include_once( "../logger/logger.php" );
include_once( "./map_session.php" );
include_once( "./map_navigator.php" );
include_once( "./map_query.php" );

installSessionDirectoryHandler();

header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");    // Date in the past
header("Last-Modified: " . gmdate("D, d M Y H:i:s") . " GMT");
                                                     // always modified
header("Cache-Control: no-store, no-cache, must-revalidate");  // HTTP/1.1
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");                          // HTTP/1.0

/* ============================================================================
 * setup values
 * ========================================================================= */
// build an array with the HTTP GET or POST parameters
$http_form_vars = (count($_POST) > 0) ?
                  $_POST : ((count($_GET) > 0) ? $_GET : array() );

// determine current page location
$szURI = "http://".$_SERVER['HTTP_HOST'].dirname($_SERVER['PHP_SELF']);

// set max time out
set_time_limit(600);

/* ============================================================================
 * Determine the required values
 * ========================================================================= */
// default values
$szMapName = ( defined( "MAPFILE" ) ? MAPFILE : "?");
$szMapPath = "?";
$szMSModName = "?";
$szTmpMapPath = "?";
$szCurrentState = "?";
$szQueryLayers = "?";
$szQueryCoords = "?";
$oErrorManager = "?";
$szFatalErrorImage = "?";
$szGDVersion = "?";

$nMinX = "?";
$nMinY = "?";
$nMaxX = "?";
$nMaxY = "?";

$nWidth  = "?";
$nHeight = "?";

$bTransparent = "?";

$szMapId = "?";

// check URL if values not yet set
if ( $szMapName == "?" && isset($http_form_vars["gszMapName"]) )
    $szMapName = $http_form_vars["gszMapName"];
if ( $szMapPath == "?" && isset($http_form_vars["gszMapPath"]) )
    $szMapPath = $http_form_vars["gszMapPath"];
if ( $szFatalErrorImage == "?" && isset($http_form_vars["szFatalErrorImage"]) )
    $szFatalErrorImage = $http_form_vars["szFatalErrorImage"];

if ( $nMinX == "?" && isset($http_form_vars["nMinX"]) )
    $nMinX = $http_form_vars["nMinX"];
if ( $nMinY == "?" && isset($http_form_vars["nMinY"]) )
    $nMinY = $http_form_vars["nMinY"];
if ( $nMaxX == "?" && isset($http_form_vars["nMaxX"]) )
    $nMaxX = $http_form_vars["nMaxX"];
if ( $nMaxY == "?" && isset($http_form_vars["nMaxY"]) )
    $nMaxY = $http_form_vars["nMaxY"];

if ( $nWidth == "?" && isset($http_form_vars["nWidth"]) )
    $nWidth = $http_form_vars["nWidth"];
if ( $nHeight == "?" && isset($http_form_vars["nHeight"]) )
    $nHeight = $http_form_vars["nHeight"];

if ( $bTransparent == "?" && isset($http_form_vars["bTransparent"]))
    $bTransparent = $http_form_vars["bTransparent"];

if ( $szMapId == "?" && isset($http_form_vars["gszMapId"]))
    $szMapId = $http_form_vars["gszMapId"];

/*
 * REMOVED FOR SECURITY REASONS.
 *
 * TODO: replace with a globprefs include for non-session based apps if
 * required.
 *
if ( $szMSModName == "?" && isset($http_form_vars["gszPHPMapScriptModName"]) )
    $szMSModName = $http_form_vars["gszPHPMapScriptModName"];
if ( $szTmpMapPath == "?" && isset($http_form_vars["gszTmpMapPath"]) )
    $szTmpMapPath = $http_form_vars["gszTmpMapPath"];
if ( $szCurrentState == "?" && isset($http_form_vars["gszCurrentState"]) )
    $szCurrentState = $http_form_vars["gszCurrentState"];
if ( $szQueryLayers == "?" && isset($http_form_vars["gszQueryLayers"]) )
    $szQueryLayers = $http_form_vars["gszQueryLayers"];
if ( $szQueryCoords == "?" && isset($http_form_vars["gszQueryCoords"]) )
    $szQueryCoords = $http_form_vars["gszQueryCoords"];
*/
// determine if in session
$sess_name = "sid";

// only setup session if set
if ( isset( $http_form_vars[$sess_name] ) )
{
    initializeSession($sess_name);
    //print_r($_SESSION);
    // check for map name
    if ( $szMapName == "?" && isset( $_SESSION["gszMapName"] ) )
    {
        $szMapName = $_SESSION["gszMapName"];
    }

    // check for map path
    if ( $szMapPath == "?" && isset( $_SESSION["gszMapPath"] ) )
        $szMapPath = $_SESSION["gszMapPath"];

    // check for mapscript module name
    if ( $szMSModName == "?" && isset( $_SESSION["gszPHPMapScriptModName"] ) )
        $szMSModName = $_SESSION["gszPHPMapScriptModName"];

    // an alternate session variable name for mapscript module
    if ( $szMSModName == "?" && isset( $_SESSION["gszMapscriptModule"] ) )
        $szMSModName = $_SESSION["gszMapscriptModule"];

    // check for the temp mapfile path
    if ( $szTmpMapPath == "?" && isset( $_SESSION["gszTmpMapPath"] ) )
        $szTmpMapPath = $_SESSION["gszTmpMapPath"];

    // check for the map session key
    if ( $szCurrentState == "?" && isset( $_SESSION["gszCurrentState"] ) )
        $szCurrentState = $_SESSION["gszCurrentState"];

    // check for the query layers
    if ( $szQueryLayers == "?" && isset( $_SESSION["gszQueryLayers"] ) )
        $szQueryLayers = $_SESSION["gszQueryLayers"];

    // check for the query co-ordinates
    if ( $szQueryCoords == "?" && isset( $_SESSION["gszQueryCoords"] ) )
        $szQueryCoords = $_SESSION["gszQueryCoords"];

    // check for the error manager is set
    if ( $oErrorManager == "?" && isset( $_SESSION["gErrorManager"] ) )
        $oErrorManager =& $_SESSION["gErrorManager"];
}

// default the map path if not set
if ( $szMapPath == "?" ) $szMapPath = dirname( $szMapName );

// default mapscript module name as a last resort
if ( $szMSModName == "?" )
{
    // set according to OS
    $szMSModName = ( (PHP_OS == "WINNT" || PHP_OS == "WIN32") ?
                                    "php_mapscript.dll" : "php_mapscript.so" );
}

// default the temp map path as a last resort
if ( $szTmpMapPath == "?" ) $szTmpMapPath = dirname( $szMapName );

// default the statekey as a last resort
if ( $szCurrentState == "?" ) $szCurrentState = "";

// default the query co-ordinates if necessary
if ( $szQueryCoords == "?" ) $szQueryCoords = "0,0";

// check for trailing "/" in path names
$szTmpMapPath = str_replace("\\", "/", $szTmpMapPath);
if (substr($szTmpMapPath, -1) != "/") $szTmpMapPath .= "/";
$szMapPath = str_replace("\\", "/", $szMapPath);
if (substr($szMapPath, -1) != "/") $szMapPath .= "/";
$szMapName = str_replace("\\", "/", $szMapName);

// ensure that the mapscript module is loaded
if (!extension_loaded("MapScript")) dl($szMSModName);
$aMs = explode(" ", ms_GetVersion());

if (!extension_loaded( "gd" ))
{
    $szGDMod = "php_gd2".( (PHP_OS=="WINNT" || PHP_OS=="WIN32") ? ".dll" : ".so" );
    dl( $szGDMod );
}

/** DEBUG 
echo "MapName: $szMapName<br>";
echo "MapPath: $szMapPath<br>";
echo "ModuleName: $szMSModName<br>";
echo "TmpPath: $szTmpMapPath<br>";
echo "CurrentState: $szCurrentState<br>";
echo "QueryLayers: $szQueryLayers<br>";
echo "QueryCoords: $szQueryCoords<br>";
*/

/*
$aszLog = explode(".", basename($_SERVER["PHP_SELF"]));
$gLogFile = new LogFile( $szTmpMapPath.$aszLog[0]."_drawmap.log", LOG_TO_FILE, true );
$gLogFile->setMaxLogLevel( LOG_ALL );
$gErrorManager = new ErrorManager();
$oLog = new Logger( "General" );
$oLog->setLogFile( $gLogFile );
$oLog->setErrorManager( $gErrorManager );
*/


/* ============================================================================
 * Setup the map session & navigator classes
 * ========================================================================= */
if ( $http_form_vars["map_session_mode"] == 0 )
{
    // create a new map session object in read-only mode
    $oMapSession = new MapSession_R;
}
else
{
    // create a new map session object in read-write mode
    $oMapSession = new MapSession_RW;

    // set the temp directory for the map session
    $oMapSession->setTempDir( $szTmpMapPath );
}

// $oMapSession->setLogFile( $gLogFile );
if ($oErrorManager != "?")
    $oMapSession->setErrorManager( $oErrorManager );

/* ============================================================================
 * Re-instate the previous map state
 * ========================================================================= */
$oMapSession->restoreState( $szCurrentState, $szMapName, $szMapPath );

if ($oMapSession->oMap == null)
{
    //create a 1x1 image and return it.

    header("Content-type: image/png");
    imagepng( imagecreate( 1, 1 ) );
    exit;
}

if ($nMinX != "?" && $nMinY != "?" &&
    $nMaxX != "?" && $nMaxY != "?")
{
    $oMapSession->oMap->setextent($nMinX, $nMinY, $nMaxX, $nMaxY);
}

if ($nWidth != "?" && $nHeight != "?")
{
    $oMapSession->oMap->set("width", $nWidth);
    $oMapSession->oMap->set("height", $nHeight);
}

if (isset($http_form_vars["gszImgType"]) && $http_form_vars["gszImgType"] != "")
    $szImgType = $http_form_vars["gszImgType"];
else
if (isset($_SESSION["gszImgType"]) && $_SESSION["gszImgType"] != "")
    $szImgType = $_SESSION["gszImgType"];
else
    $szImgType = $oMapSession->oMap->imagetype;

$oMapSession->oMap->selectOutputFormat($szImgType);

if ($bTransparent != "?")
{
    if (strtoupper($bTransparent) == "TRUE")
        $oMapSession->oMap->set("transparent", MS_TRUE);
    else
        $oMapSession->oMap->set("transparent", MS_FALSE);
}

/* ============================================================================
 * Draw map
 * ========================================================================= */
//$old_error_handler = set_error_handler("userErrorHandler");
//$GLOBALS['gaErrors'] = array();

if (substr($aMs[2], 0, 3) >= "3.7")
{
    // Init error stack
    ms_ResetErrorList();
}

// run query if necessary
if ( isset($http_form_vars["run_query"]) &&
     $http_form_vars["run_query"] == 1 )
{
    // run query
    runQuery( $oMapSession, $szQueryCoords, $szQueryLayers, $szURI );

    // draw map with query
    $oImage = @$oMapSession->oMap->drawQuery();
}
//keymap
else if (isset($http_form_vars["REQUEST"]) &&
         (strcasecmp($http_form_vars["REQUEST"],"KEYMAP") == 0))
{
    if (isset($http_form_vars['nWidth']) &&
        isset($http_form_vars['nHeight']) &&
        isset($http_form_vars['szImage']) &&
        isset($http_form_vars['dfKeyMapMinX']) &&
        isset($http_form_vars['dfKeyMapMinY']) &&
        isset($http_form_vars['dfKeyMapMaxX']) &&
        isset($http_form_vars['dfKeyMapMaxY']) &&
        isset($http_form_vars['szOriginalProjection']) &&
        isset($http_form_vars['szColor']) &&
        isset($http_form_vars['szOutlineColor']) )
    {
        DrawKeyMap($oMapSession,
                   $http_form_vars['nWidth'],
                   $http_form_vars['nHeight'],
                   $http_form_vars['szImage'],
                   $http_form_vars['dfKeyMapMinX'],
                   $http_form_vars['dfKeyMapMinY'],
                   $http_form_vars['dfKeyMapMaxX'],
                   $http_form_vars['dfKeyMapMaxY'],
                   $http_form_vars['szOriginalProjection'],
                   $http_form_vars['szColor'],
                   $http_form_vars['szOutlineColor']);
        return;
    }
    elseif (isset( $_SESSION["KEYMAP_SRS"] ) )
    {
        /*
        echo "drawReferenceMap:<BR>";
        print_r($oMapSession->oMap);
        print_r($oMapSession->oMap->reference);
        */
        $oRef = $oMapSession->oMap->reference;
        DrawKeyMap( $oMapSession, $oRef->width, $oRef->height,
                    $oRef->image, $oRef->extent->minx, $oRef->extent->miny,
                    $oRef->extent->maxx, $oRef->extent->maxy,
                    $_SESSION["KEYMAP_SRS"],
                    ($oRef->color->red)." ".($oRef->color->green)." ".($oRef->color->blue),
                    ($oRef->outlinecolor->red)." ".($oRef->outlinecolor->green)." ".($oRef->outlinecolor->blue) );
        return;
    }
    else
    {
        /*
        echo "drawReferenceMap:<BR>";
        print_r($oMapSession->oMap);
        print_r($oMapSession->oMap->reference);
        */
        $oImage = @$oMapSession->oMap->drawReferenceMap();
    }
}
//scalebar
else if (isset($http_form_vars["REQUEST"]) &&
         strcasecmp($http_form_vars["REQUEST"],"SCALEBAR") == 0)
{
    $oImage = @$oMapSession->oMap->drawScaleBar();
}
else
{
/* -------------------------------------------------------------------- */
/*      Used for multi layers in cwc jsapi mode.                        */
/*      Draw a leyer if layer name is specified in the URL. Else        */
/*      draw the map. The layer is drawn even if the status is off.     */
/* -------------------------------------------------------------------- */
    if (isset($http_form_vars["LAYER"]) && $http_form_vars["LAYER"] != "")
    {
        $nId = intval($http_form_vars["LAYER"]);
        if ($nId >=0 && $nId < $oMapSession->oMap->numlayers)
        {
            $oLayer = $oMapSession->oMap->GetLayer($nId);
            if ($oLayer)
            {
                $nStatus = $oLayer->status;
                $nLabelCache = $oLayer->labelcache;
                $oLayer->set("status", MS_ON);
                $oLayer->set("labelcache", MS_OFF);
                $oImage = $oMapSession->oMap->prepareimage();
                $oLayer->draw($oImage);
                $oLayer->set("status",$nStatus);
                $oLayer->set("labelcache", $nLabelCache);

            }
        }
    }
    else 
    {
    // draw map
      $oImage = @$oMapSession->oMap->draw();
    }
}

if (substr($aMs[2], 0, 3) >= "3.7")
{
    $oError = ms_GetErrorObj();
    while($oError && $oError->code > 0)
    {
        switch($oError->code)
        {    
            case MS_NOERR:
                $szMsg = "No Error: ".$oError->message;
                break;

            case MS_IOERR:
                $szMsg = "MapServer IO Error: ".$oError->message;
                break;

            case MS_MEMERR:
                $szMsg = "MapServer Memory Allocation Error: ".$oError->message;
                break;

            case MS_TYPEERR:
                $szMsg = "MapServer Type Error: ".$oError->message;
                break;

            case MS_SYMERR:
                $szMsg = "MapServer Symbol Error: ".$oError->message;
                break;

            case MS_TYPEERR:
                $szMsg = "MapServer Regular Expression Error: ".$oError->message;
                break;

            case MS_TTFERR:
                $szMsg = "MapServer TrueType Font Error: ".$oError->message;
                break;

            case MS_DBFERR:
                $szMsg = "MapServer DBF Error: ".$oError->message;
                break;

            case MS_GDERR:
                $szMsg = "MapServer GD Error: ".$oError->message;
                break;

            case MS_IDENTERR:
                $szMsg = "MapServer IDENT Error: ".$oError->message;
                break;

            case MS_EOFERR:
                $szMsg = "MapServer EOF Error: ".$oError->message;
                break;

            case MS_PROJERR:
                $szMsg = "MapServer Projection Error: ".$oError->message;
                break;

            case MS_MISCERR:
                $szMsg = "MapServer Miscellaneous Error: ".$oError->message;
                break;

            case MS_CGIERR:
                $szMsg = "MapServer CGI Error: ".$oError->message;
                break;

            case MS_WEBERR:
                $szMsg = "MapServer Web Error: ".$oError->message;
                break;

            case MS_IMGERR:
                $szMsg = "MapServer Image Error: ".$oError->message;
                break;

            case MS_HASHERR:
                $szMsg = "MapServer HashTable Error: ".$oError->message;
                break;

            case MS_JOINERR:
                $szMsg = "MapServer Join Error: ".$oError->message;
                break;

            case MS_NOTFOUND:
                $szMsg = "MapServer Not Found (empty search result) Error: ".$oError->message;
                break;

            case MS_SHPERR:
                $szMsg = "MapServer Shapefile Error: ".$oError->message;
                break;

            case MS_PARSEERR:
                $szMsg = "MapServer Parser Error: ".$oError->message;
                break;

            case MS_SDEERR:
                $szMsg = "MapServer SDE Error: ".$oError->message;
                break;

            case MS_OGRERR:
                $szMsg = "MapServer OGR Error: ".$oError->message;
                break;

            case MS_QUERYERR:
                $szMsg = "MapServer Query Error: ".$oError->message;
                break;

            case MS_WMSERR:
                $szMsg = "MapServer WMS Error: ".$oError->message;
                break;

            case MS_WMSCONNERR:
                $szMsg = "MapServer WMS Connection Error: ".$oError->message;
                break;

            /*case MS_ORCALESPATIALERR:
                $szMsg = "MapServer Oracle Spatial Error: ".$oError->message;
                break;
            */
            case MS_WFSERR:
                $szMsg = "MapServer WFS Error: ".$oError->message;
                break;

            case MS_WFSCONNERR:
                $szMsg = "MapServer WFS Connection Error: ".$oError->message;
                break;

            /*case MS_CONTEXTERR:
                $szMsg = "MapServer Web Map Context Error: ".$oError->message;
                break;
            */
            case MS_HTTPERR:
                $szMsg = "MapServer HTTP Error: ".$oError->message;
                break;

            case MS_CHILDERR:
                $szMsg = "MapServer Child Array Error: ".$oError->message;
                break;

            case MS_WCSERR:
                $szMsg = "MapServer WCS Error: ".$oError->message;
                break;
            default:
                $szMsg = "Unrecognized error code ".$oError->code.", message is ".$oError->message;
        }
        
        // Skip errors http
        if ($oError->code != MS_HTTPERR && $oError->code > 0)
            $oErrorManager->setError(ERR_WARNING, "DrawMap: ". $szMsg);

        $oError = $oError->next();
    }
}

if ($oImage == MS_FALSE)
{
    if ($szFatalErrorImage != "?" &&
        ($fh = fopen($szFatalErrorImage, "r")) !== false)
    {
        switch(strtoupper(substr($szFatalErrorImage, -3)))
        {
        case "GIF":
            header("Content-type: image/gif");
            break;
        case "PNG":
        case "PNG24":
            header("Content-type: image/png");
            break;
        case "JPG":
            header("Content-type: image/jpeg");
            break;
        }

        fpassthru($fh);
        //exit;
    }
    /*
    else
    {
        $im = @imagecreate ($oMapSession->oMap->width, $oMapSession->oMap->height);
        $background_color = imagecolorallocate ($im, 255, 255, 255);
        $text_color = imagecolorallocate ($im, 233, 14, 91);
        ImageStringWrap($im, "", 10, 10, implode("", getErrors()),
                        $text_color, $oMapSession->oMap->width);
        header("Content-type: image/png");
        imagepng ($im);
        exit;
    }
    */
}
else
{
    /* =====================================================================
     * Draw points, lines, polygons for annotations layer if necessary
     * ================================================================== */
    if(isset($_SESSION["aoAnnotation"]))
    {
        $numAnnotations = count($_SESSION["aoAnnotation"]);
        for($i=0; $i<$numAnnotations; $i++)
        {
            if(isset($_SESSION["aoAnnotation"][$i]->name))
                $szName = $_SESSION["aoAnnotation"][$i]->name;
            else
                $szName = "__NOANNOTATION__";

            if($szMapId == "?" || 
               (substr($szName, (strlen($szMapId)*-1)) != $szMapId))
                continue;

            if(isset($_SESSION[$szName]))
            {
                switch($_SESSION["aoAnnotation"][$i]->type)
                {
                  case MS_LAYER_POINT:
                    if(!isset($_SESSION[$szName]["LongitudeField"]) || 
                       !isset($_SESSION[$szName]["LatitudeField"]))
                        break;

                    $szLat = $_SESSION[$szName]["LatitudeField"];
                    $szLon = $_SESSION[$szName]["LongitudeField"];
                    $aPoints = $_SESSION[$szName];
                    $numPoints = count($aPoints);
                    $aszPointKeys = array_keys($aPoints);

                    $oMap = $oMapSession->oMap; 
                    $poLayer = $oMap->getLayerByName($szName);

                    $poPoint = ms_newPointObj();
                    for($j=0; $j<$numPoints; $j++)
                    {
                        $key = $aszPointKeys[$j];
                        if(isset($aPoints[$key][$szLon]) && 
                           isset($aPoints[$key][$szLat]))
                        {
                            $poPoint->setXY($aPoints[$key][$szLon], 
                                            $aPoints[$key][$szLat]);
                            if(isset($aPoints[$key]["TXT"]))
                                $szText = $aPoints[$key]["TXT"];
                            else
                                $szText = $key+1;
                            $poPoint->draw($oMap, $poLayer, $oImage, 
                                           "dynamic point class", $szText);
                        }
                    }
                    break;
                  case MS_LAYER_ANNOTATION:
                    if(!isset($_SESSION[$szName]["LongitudeField"]) || 
                       !isset($_SESSION[$szName]["LatitudeField"]))
                        break;
                    $szLat = $_SESSION[$szName]["LatitudeField"];
                    $szLon = $_SESSION[$szName]["LongitudeField"];
                    $aPoints = $_SESSION[$szName];
                    $numPoints = count($aPoints);
                    $aszPointKeys = array_keys($aPoints);
                
                    $oMap =& $oMapSession->oMap; 

                    $poLayer = $oMap->getLayerByName($szName);
                                   
                    for($j=0; $j<$numPoints; $j++)
                    {
                        $key = $aszPointKeys[$j];
                        if(isset($aPoints[$key][$szLon]) && 
                           isset($aPoints[$key][$szLat]))
                        {
                            $poLine = ms_newLineObj();
                            $poShape = ms_newShapeObj(MS_SHAPE_POINT);
                            $poLine->addXY($aPoints[$key][$szLon], 
                                           $aPoints[$key][$szLat]);
                            if(isset($aPoints[$key]["TXT"]))
                                $szText = $aPoints[$key]["TXT"];
                            else
                                $szText = $key+1;
                            $poShape->add($poLine);
                            $poShape->set("text", $szText);
                            $poLayer->addFeature($poShape);
                        }
                    }
                    $poLayer->draw($oImage);
                    $oMap->drawLabelCache($oImage);
                    break;
                  case MS_LAYER_LINE:
                    if(!isset($_SESSION[$szName]["LongitudeField"]) || 
                       !isset($_SESSION[$szName]["LatitudeField"]))
                        break;
                    $szLat = $_SESSION[$szName]["LatitudeField"];
                    $szLon = $_SESSION[$szName]["LongitudeField"];
                    $aPoints = $_SESSION[$szName];
                    $numPoints = count($aPoints);

                    $oMap = $oMapSession->oMap; 
                    $poLayer = $oMap->getLayerByName($szName);

                    $poLine = ms_newLineObj();
                    $poShape = ms_newShapeObj(MS_SHAPE_LINE);

                    for($j=0; $j<$numPoints; $j++)
                    {
                        if(isset($aPoints[$j][$szLon]) && 
                           isset($aPoints[$j][$szLat]))
                        {
                            $poLine->addXY($aPoints[$j][$szLon], 
                                           $aPoints[$j][$szLat]);
                        }
                    }
                    $poShape->add($poLine);
                    $poLayer->addFeature($poShape);

                    $poLayer->draw($oImage);
                    break;
                  default:
                    break;
                }
            }
        }
    }

    /* =====================================================================
     * Draw points from URL if necessary
     * ================================================================== */
    if(isset($http_form_vars["AddPointsLat"]) && 
       $http_form_vars["AddPointsLat"] != ""  &&
       isset($http_form_vars["AddPointsLon"]) && 
       $http_form_vars["AddPointsLon"] != ""  && 
       isset($http_form_vars["AddPointsIcon"])&& 
       $http_form_vars["AddPointsIcon"] != "")
    {
        $aszLat = explode("|", $http_form_vars["AddPointsLat"]);
        $aszLon = explode("|", $http_form_vars["AddPointsLon"]);
        $aszIcon = explode("|", $http_form_vars["AddPointsIcon"]);
        $aszText = explode("|", $http_form_vars["AddPointsText"]);
        $szText = "";

        $numPoints = count($aszLat);
        if($numPoints != count($aszLon))
            $numPoints = -1;

        $aszLayerNames = $oMapSession->oMap->getAllLayerNames();
        if(isset( $http_form_vars["AddPointsLayer"]) && 
           $http_form_vars["AddPointsLayer"] != "")
        {
            $poLayer=$oMapSession->oMap->getLayerByName(
                $http_form_vars["AddPointsLayer"]);
            $poClass = $poLayer->getClass(0);
            $poStyle = $poClass->getStyle(0);
        }
        else if(in_array("__DRAWMAP_ADDPOINT", $aszLayerNames))
        {
            $poLayer=$oMapSession->oMap->getLayerByName("__DRAWMAP_ADDPOINT");
            $poClass = $poLayer->getClass(0);
            $poStyle = $poClass->getStyle(0);
        }
        else
        {
            $poLayer = ms_newLayerObj($oMapSession->oMap);
            $poLayer->set("name", "__DRAWMAP_ADDPOINT");
            $poLayer->set("status", MS_ON);
            $poLayer->set("type", MS_LAYER_POINT);
            $poLayer->set("labelcache", MS_OFF);
            $poClass = ms_newClassObj($poLayer);
            $poClass->set("name", "dynamic point class");
            $poStyle = ms_newStyleObj($poClass);
            $poStyle->set("symbolname", $aszIcon[0]);
            $poClass->label->set("font", "arial-bold");
            $poClass->label->set("type", "TRUETYPE");
            $poClass->label->set("size", 8);
            $poClass->label->set("position", 8);
            $poClass->label->color->setRGB(255, 255, 255);
            if($aszIcon[0] == "circle")
            {
                $poStyle->set("size", 15);
                $poStyle->color->setRGB(0, 0, 0);
            }
        }

        $poPoint = ms_newPointObj();
        for($i=0; $i<$numPoints; $i++)
        {
            if(isset($aszIcon[$i]) && $aszIcon[$i]!="")
                $poStyle->set("symbolname", $aszIcon[$i]);
            if(isset($aszText[$i]))
                $szText = $aszText[$i];
            $poPoint->setXY($aszLon[$i], $aszLat[$i]);
            $poPoint->draw($oMapSession->oMap, $poLayer, $oImage, 
                           "dynamic point class", $szText);
        }
    }

    /* ============================================================================
     * Determine image support
     * ========================================================================= */
    //there is a problem in MS3.7 with MS_GIF not being defined because the output
    //stuff has all changed.  Hack for now if not defined
    if (!defined( "MS_GIF" ))
        define( "MS_GIF", 0 );
    // check gif first
    $aMs = explode(" ", ms_GetVersion());
    if (substr($aMs[2], 0, 3) < "3.7")
    {
        if (strpos( ms_GetVersion(), "OUTPUT=GIF") > 0 )
        {
            // gif is supportted
            $nImageType = MS_GIF;
        }
        elseif(strpos( ms_GetVersion(), "OUTPUT=PNG") > 0 )
        {
            // png is supportted
            $nImageType = MS_PNG;
        }
        else
        {
            // default to JPEG
            $nImageType = MS_JPEG;
        }

        // set the output type
        switch ($nImageType)
        {
        case MS_GIF:
            header("Content-type: image/gif");
            break;
        case MS_PNG:
            header("Content-type: image/png");
            break;
        case MS_JPEG:
            header("Content-type: image/jpeg");
            break;
        }
        // output map
        $url = $oImage->saveImage("", $nImageType, 0, 0, -1);
    }
    else
    {
        // set the output type
        switch (strtoupper($oMapSession->oMap->imagetype))
        {
        case "GIF":
            header("Content-type: image/gif");
            break;
        case "PNG":
        case "PNG24":
            // set the output type
            header("Content-type: image/png");
            break;
        case "JPEG":
            header("Content-type: image/jpeg");
            break;
        }

        // output map
        $url = $oImage->saveImage("");
    }
}


if ($bProfile)
{
    echo "<BR>profiling complete, dumping results<BR>";
    xdebug_dump_function_profile( 1 );
    xdebug_stop_profiling();
    //xdebug_dump_function_profile( 0 );
}

/**
 * This function takes the given query parameters and executes a query on the
 * given map session's map object.
 *
 * @param $oMapSession object - Map Session object.
 * @param $szQueryCoords string - Delimited string of co-ordinates.
 * @param $szQueryLayers string - Comma delimited string of layer indexes.
 * @return boolean - True if successful, False if not.
 **/
function runQuery( $oMapSession, $szQueryCoords, $szQueryLayers )
{
    // create a new map query object
    $oMapQuery = new MapQuery( $oMapSession );

    // process coordinates
    $aszMinMax = explode( ";", $szQueryCoords );
    $adMin = explode( ",", $aszMinMax[0] );
    $adMax = explode( ",", $aszMinMax[1] );

    // initialize var(s)
    $aszLayers = array();
    $oMap = $oMapSession->oMap;

    // check to see if the list of layers has been supplied
    if ( $szQueryLayers != "?" )
    {
        // get array of layers
        $aszLayers = explode( $szQueryLayers );

        // loop and get the name of each layer
        for ( $i=0; $i<$nCount; $i++ )
        {
            // get layer
            $oLayer = $oMap->getLayer( $aszLayers[$i] );

            // update array to be name
            $aszLayers[$i] = $oLayer->name;
        }
    }
    else
    {
        // loop through all layers and add to list of layers to query
        // NOTE:  Layers will only have results generated for them if they
        // have a "template" value set.  The template value does not have
        // to be valid, it could be as simple as "ttt".
        $nCount = $oMap->numlayers;
        for ( $i=0; $i<$nCount; $i++ )
        {
            // get layer name
            $oLayer = $oMap->getLayer( $i );

            // only add if on
            if ( $oLayer->status == MS_ON || $oLayer->status == MS_DEFAULT )
            {
                array_push( $aszLayers, $oLayer->name );
            }
        }
    }

    // commit
    $oMapQuery->setQueryLayersByName( $aszLayers );

    // execute query
    if ( ( is_numeric( $adMax[0] ) && is_numeric( $adMax[1] ) ) &&
         ( $adMax[0] != $adMin[0] && $adMax[1] != $adMin[1] ) ) {
        $oResultSet = $oMapQuery->executeRectQuery( $adMin[0],
                                    $adMin[1],$adMax[0], $adMax[1] );
    }
    else {
        $oResultSet = $oMapQuery->executePointQuery( $adMin[0],
                                                        $adMin[1] );
    }

    // return success
    return true;
}

function ImageStringWrap($image, $font, $x, $y, $text, $color, $maxwidth)
{
    $fontwidth = ImageFontWidth($font);
    $fontheight = ImageFontHeight($font);

    if ($maxwidth != NULL)
    {
        $maxcharsperline = floor($maxwidth / $fontwidth);
        $text = wordwrap($text, $maxcharsperline, "\n", 1);
    }

    $lines = explode("\n", $text);
    while (list($numl, $line) = each($lines))
    {
        ImageString($image, $font, $x, $y, $line, $color);
        $y += $fontheight;
    }
}

function DrawKeyMap($oMapSession, $nWidth, $nHeight, $szImage, $dfKeyMapMinX,
                    $dfKeyMapMinY, $dfKeyMapMaxX, $dfKeyMapMaxY,
                    $szOriginalProjection, $szColor="", $szOutlineColor="")
{
        $oMapSession->oMap->reference->set("image", $szImage);
        $oMapSession->oMap->reference->set("status", MS_ON);
        $oMapSession->oMap->reference->set("width", $nWidth);
        $oMapSession->oMap->reference->set("height", $nHeight);

        if (strncmp( $szImage, ".", 1 ) == 0)
        {
            
            $szImage = dirname($GLOBALS["szMapName"])."/".$szImage;
        }

        $szOriginalExtents = $dfKeyMapMinX.",".
                             $dfKeyMapMinY.",".
                             $dfKeyMapMaxX.",".
                             $dfKeyMapMaxY;
 /* -------------------------------------------------------------------- */
/*      Here is what is done :                                          */
/*        - Get the current extents of the map and reproject It to      */
/*      the keymap projection                                           */
/*        - use these new projected rectangle to draw the keymap red    */
/*      bbox manually on top of the keymap image.                       */
/* -------------------------------------------------------------------- */
        $aMapExtents[0] = $oMapSession->oMap->extent->minx;
        $aMapExtents[1] = $oMapSession->oMap->extent->miny;
        $aMapExtents[2] = $oMapSession->oMap->extent->maxx;
        $aMapExtents[3] = $oMapSession->oMap->extent->maxy;
        
        if (stristr( $szOriginalProjection, "epsg" ) && ! stristr( $szOriginalProjection, "init" ))
            $szOriginalProjection = "init=".strtolower($szOriginalProjection);

        $oNav = new MapNavigator( $oMapSession->oMapSession );
        $adfKeyMapBBoxExtents = $oNav->reprojectExtentFromCenter(
                                     $aMapExtents, $oMapSession->oMap->width,
                                     $oMapSession->oMap->height, $oMapSession->oMap->getProjection(),
                                     $szOriginalProjection);
        $szMapProj = $oMapSession->oMap->getProjection();
        if ($szMapProj != $szOriginalProjection )
        {   
            $oProjIn = ms_newProjectionObj($szMapProj);
            $oProjOut = ms_newProjectionObj( $szOriginalProjection );
            $oMapSession->oMap->extent->project( $oProjIn, $oProjOut );
        }

        $adfKeyMapBBoxExtents[0] = $oMapSession->oMap->extent->minx;
        $adfKeyMapBBoxExtents[1] = $oMapSession->oMap->extent->miny;
        $adfKeyMapBBoxExtents[2] = $oMapSession->oMap->extent->maxx;
        $adfKeyMapBBoxExtents[3] = $oMapSession->oMap->extent->maxy;
                                     
        if (!file_exists($szImage))
        {
            $_SESSION['gErrorManager']->setError(ERR_WARNING,
                                                 "ERROR: Key map image ($szImage) was not found in KeyMap.widget.php.");
        }
        else //keymap image found
        {
/* -------------------------------------------------------------------- */
/*      try to guess the format of the image using the extension. I     */
/*      thought the imagecreatefromgd would open any gd file but It     */
/*      does not work.                                                  */
/*      Note : there is a gd function exif_imagetype which is           */
/*      supposed to work for version php4.3.0 but was not (at least     */
/*      on the windows binaries downloaded from php site)               */
/* -------------------------------------------------------------------- */
            //var_dump(gd_info());
            $szImage = trim($szImage);
            $szFormat = "";

            //is gif ?
            if (imagetypes() & IMG_GIF)
            {
                $szFormat = stristr($szImage, "gif");
                if ($szFormat != "")
                  $szFormat = "gif";
            }


            //is png
            if (imagetypes() & IMG_PNG)
            {
                if ($szFormat == "")
                {
                    $szFormat = stristr($szImage, "png");
                    if ($szFormat != "")
                      $szFormat = "png";
                }
            }

            //is jpeg
            if (imagetypes() & IMG_JPG)
            {
                if ($szFormat == "")
                {
                    $szFormat = stristr($szImage, "jpg");
                    if ($szFormat == "")
                      $szFormat = stristr($szImage, "jpeg");

                    if ($szFormat != "")
                      $szFormat = "jpeg";
                }
            }

            if ($szFormat == "")
            {
                $szValidFormats = "";
                if (imagetypes() & IMG_GIF)
                  $szValidFormats .= "GIF (file_name.gif)";
                if (imagetypes() & IMG_PNG)
                {
                    if ($szValidFormats != "")
                      $szValidFormats .= " , ";
                    $szValidFormats .= "PNG (file_name.png)";
                }
                if (imagetypes() & IMG_JPG)
                {
                    if ($szValidFormats != "")
                      $szValidFormats .= " , ";
                    $szValidFormats .= "JEPG (file_name.jpg)";
                }

                $_SESSION['gErrorManager']->setError(ERR_WARNING,
                                    "ERROR: Could not determine the format of the keymap image : " .$szImage . ". Valid formats are" . $szValidFormats);
            }
            else //format found
            {
                if ($szFormat == "png")
                  $image = imagecreatefrompng($szImage);
                else if ($szFormat == "gif")
                  $image = @imagecreatefromgif($szImage);
                else
                  $image = imagecreatefromjpeg($szImage);

/* -------------------------------------------------------------------- */
/*      get the values in pixels for the rectable bbox.                 */
/* -------------------------------------------------------------------- */
                //the offset from the left edge of the keymap in geo
                //coords
                $dfDeltaGeoX = $adfKeyMapBBoxExtents[0] - $dfKeyMapMinX;

                //the offset from the bottom edge of the keymap in geo
                //coords
                $dfDeltaGeoY = $adfKeyMapBBoxExtents[1] - $dfKeyMapMinY;

                //minx, min y of the bbox in pixel position
                /*
                $dfDeltaPixX = abs($dfDeltaGeoX * $nWidth /
                  ($dfKeyMapMaxX - $dfKeyMapMinX));
                $dfDeltaPixY = abs($dfDeltaGeoY * $nHeight /
                  ($dfKeyMapMaxY - $dfKeyMapMinY));
                */
                //the original code above applied an abs() function to the
                //location, don't know who did this 
                $dfDeltaPixX = $dfDeltaGeoX * $nWidth /
                  ($dfKeyMapMaxX - $dfKeyMapMinX);
                $dfDeltaPixY = $dfDeltaGeoY * $nHeight /
                  ($dfKeyMapMaxY - $dfKeyMapMinY);

                //width, height of the bbox in pixel position
                $dfWidthPixBBox = abs(($nWidth *
                                    ($adfKeyMapBBoxExtents[2] -
                                     $adfKeyMapBBoxExtents[0]))/
                                    ($dfKeyMapMaxX - $dfKeyMapMinX));
                $dfHeightPixBBox = abs(($nHeight *
                                    ($adfKeyMapBBoxExtents[3] -
                                     $adfKeyMapBBoxExtents[1]))/
                                    ($dfKeyMapMaxY - $dfKeyMapMinY));

                $szDebug =  <<<EOT
szOriginalProjection = $szOriginalProjection<BR>
szOriginalExtents = $szOriginalExtents<BR>
szProjection = $szMapProj<BR>
adfKeyMapBBoxExtents[0] = {$adfKeyMapBBoxExtents[0]}<BR>
adfKeyMapBBoxExtents[1] = {$adfKeyMapBBoxExtents[1]}<BR>
dfKeyMapMinX = {$dfKeyMapMinX}<BR>
dfKeyMapMinY = {$dfKeyMapMinY}<BR>
dfDeltaGeoX = {$dfDeltaGeoX}<BR>
dfDeltaGeoY = {$dfDeltaGeoY}<BR>
dfDeltaPixX = {$dfDeltaPixX}<BR>
dfDeltaPixY = {$dfDeltaPixY}<BR>
dfWidthPixBBox = {$dfWidthPixBBox}<BR>
dfHeightPixBBox = {$dfHeightPixBBox}<BR>
EOT;

                //echo $szDebug;

                $nWhite = ImageColorAllocate($image, 255, 255, 255);


/* -------------------------------------------------------------------- */
/*      set the color :                                                 */
/*                                                                      */
/*   - if the "COLOR" parameter is defined, we will use it to           */
/*      draw a filled rectangle.                                        */
/*   - if not and the "OUTLINECOLOR" is defined we will use it          */
/*      to draw a rectangle.                                            */
/*   - if none is defined we will use the red color to draw a rectangle.*/
/* -------------------------------------------------------------------- */
                $nRectColor = -1;
                $bDrawFillRectangle = 0;
                if ($szColor != "")
                {
                    $aColors = explode (" ", $szColor);
                    if ($aColors[0] >= 0 && $aColors[1] >= 0 && $aColors[2] >= 0 &&
                        $aColors[0] <= 255 && $aColors[1] <= 255 && $aColors[2] <= 255)
                    {
                        $nRectColor = ImageColorAllocate($image, $aColors[0], $aColors[1], $aColors[2]);
                        $bDrawFillRectangle = 1;

                        //keep the value for the JSAPI

                        $aOrigColor[0] = $aColors[0];
                        $aOrigColor[1] = $aColors[1];
                        $aOrigColor[2] = $aColors[2];

                    }
                }

                if ($nRectColor < 0)
                {
                    if ($szOutlineColor != "")
                    {
                        $szColor = $szOutlineColor;
                        $aColors = explode (" ", $szColor);
                        if ($aColors[0] >= 0 && $aColors[1] >= 0 && $aColors[2] >= 0 &&
                            $aColors[0] <= 255 && $aColors[1] <= 255 && $aColors[2] <= 255)
                        {

                            $nRectColor = ImageColorAllocate($image, $aColors[0], $aColors[1], $aColors[2]);
                        }

                        $aOrigOutlineColor[0] = $aColors[0];
                        $aOrigOutlineColor[1] = $aColors[1];
                        $aOrigOutlineColor[2] = $aColors[2];

                    }
                }
                if ($nRectColor < 0)
                {
                    $nRectColor = ImageColorAllocate($image, 255, 0, 0);
                }


/* -------------------------------------------------------------------- */
/*      set the colors to -1 so the reference object won't draw the bbox.*/
/* -------------------------------------------------------------------- */

                $oMapSession->oMap->reference->color->setRGB(-1, -1, -1);
                $oMapSession->oMap->reference->outlinecolor->setRGB(-1, -1, -1);

/* -------------------------------------------------------------------- */
/*      if the size of the box is less than 3 pixels, draw a cross.     */
/* -------------------------------------------------------------------- */
                if ($dfWidthPixBBox <=3 && $dfHeightPixBBox <=3)
                {
                    $dfMiddleX = ($dfDeltaPixX + ($dfDeltaPixX + $dfWidthPixBBox))/2;
                    $dfMiddleY = (($nHeight - $dfDeltaPixY) + ($nHeight - ($dfDeltaPixY + $dfHeightPixBBox)))/2;
                    imageline($image, ($dfMiddleX -7), $dfMiddleY,
                              ($dfMiddleX -2), $dfMiddleY, $nRectColor);
                    imageline($image, ($dfMiddleX +2), $dfMiddleY,
                              ($dfMiddleX +7), $dfMiddleY, $nRectColor);

                    imageline($image, ($dfMiddleX), ($dfMiddleY -7),
                              ($dfMiddleX), ($dfMiddleY -2), $nRectColor);
                    imageline($image, ($dfMiddleX), ($dfMiddleY +2),
                              ($dfMiddleX), ($dfMiddleY +7), $nRectColor);

                }
                else
                {
/* -------------------------------------------------------------------- */
/*   if the color is set draw a filled rectangle, else draw a rectangle.*/
/* -------------------------------------------------------------------- */
                    $x1 = intval($dfDeltaPixX);
                    $x2 = intval($dfDeltaPixX + $dfWidthPixBBox);
                    $y1 = intval(($nHeight - ($dfDeltaPixY + $dfHeightPixBBox)));
                    $y2 = intval(($nHeight - $dfDeltaPixY));
                    //echo "x1=$x1, y1=$y1, x2=$x2, y2=$y2, nHeight=$nHeight, dfDeltaPixY=$dfDeltaPixY, dfHeightPixBBox=$dfHeightPixBBox<BR>";
                    if ($bDrawFillRectangle)
                    {
                        imagefilledrectangle($image, $x1, $y1, $x2, $y2, $nRectColor);
                    }
                    else
                    {
                        imagerectangle($image, $x1, $y1, $x2, $y2, $nRectColor);
                    }
                }

                if ($szFormat == "png")
                  imagepng($image);
                else if ($szFormat == "gif")
                  imagegif($image);
                else
                  imagejpeg($image);

            } //end else format found

        }//end else //keymap image found

/* -------------------------------------------------------------------- */
/*      draw the image and output it.                                   */
/* -------------------------------------------------------------------- */
        //$imgObj = $oMapSession->oMap->drawReferenceMap();

        //$szWebImage = $imgObj->saveWebImage();

/* ==================================================================== */
/*      Temp solution for JS API. Set the values and use the            */
/*      drawmap.php functions to draw the keymap.                       */
/* ==================================================================== */
        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
        {
            $oMapSession->oMap->reference->set("image", realpath($szImage));

            if (isset($aOrigColor) && count($aOrigColor) == 3)
            {
                $oMapSession->oMap->reference->color->setRGB(
                                 $aOrigColor[0], $aOrigColor[1],
                                 $aOrigColor[2]);
            }
            if (isset($aOrigOutlineColor) && count($aOrigOutlineColor) == 3)
            {
                $oMapSession->oMap->reference->outlinecolor->setRGB(
                                 $aOrigOutlineColor[0],
                                 $aOrigOutlineColor[1],
                                 $aOrigOutlineColor[2]);
            }

            $oMapSession->oMap->reference->extent->setextent(
                                 $dfKeyMapMinX, $dfKeyMapMinY,
                                 $dfKeyMapMaxX, $dfKeyMapMaxY);
        }
}
?>
