<?php
/**
 * MapBrowser application
 *
 * @project     MapLab
 * @revision    $Id: server_data_manager.php,v 1.31 2004/03/31 20:51:07 sfournier Exp $
 * @purpose     This file contains server data management functions.
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*****************************************************************************
 * $Log: server_data_manager.php,v $
 * Revision 1.31  2004/03/31 20:51:07  sfournier
 * removed useles condition
 *
 * Revision 1.30  2004/03/05 16:12:27  pspencer
 * switch to getcapabiltiies and version when requesting capabilities
 *
 * Revision 1.29  2003/05/12 14:47:37  sacha
 * coma delimited for formatlist
 *
 * Revision 1.28  2002/11/29 15:08:45  sacha
 * fixed some common directory issues
 *
 * Revision 1.27  2002/11/27 19:06:13  pspencer
 * modified to assist in debugging.
 *
 * Revision 1.26  2002/11/25 18:22:46  sacha
 * fixed image path when specified
 *
 * Revision 1.25  2002/11/25 18:11:14  sacha
 * Fixed some image path
 *
 * Revision 1.24  2002/11/21 22:42:12  sacha
 * added a base directory for image
 *
 * Revision 1.23  2002/11/20 14:44:40  julien
 * the bbox_id in the capab.dbf file must -1 if there's no bbox
 *
 * Revision 1.22  2002/11/19 21:04:46  julien
 * Add legendurl height, width and format for the mapcontext in the style.dbf
 *
 * Revision 1.21  2002/10/28 16:48:07  sacha
 * remove some typo
 *
 * Revision 1.20  2002/10/28 14:51:37  sacha
 * Removed a debug info I forgot.
 *
 * Revision 1.19  2002/10/25 15:30:10  sacha
 * added functionality to layer browser.
 *
 * Revision 1.17  2002/10/11 15:11:35  sacha
 * Display icon legend when available.
 *
 * Revision 1.16  2002/10/10 18:39:08  julien
 * Add abstract and style and
 *
 * Revision 1.15  2002/10/08 20:53:12  pspencer
 * fixed an oversight when saving the new server record (getmap url gets overwritten
 *
 * Revision 1.14  2002/10/08 19:33:17  pspencer
 * fixed a couple of bugs with using multiple urls
 *
 * Revision 1.13  2002/10/08 18:52:26  pspencer
 * split capabilities into two fields
 *
 * Revision 1.12  2002/10/07 19:23:51  assefa
 * Add extraclable layers suuport.
 *
 * Revision 1.11  2002/10/04 18:27:45  pspencer
 * Modified error message for when the server returns an invalid document
 *
 * Revision 1.10  2002/09/24 14:16:37  sacha
 * ensure that dbase module is loaded.
 *
 * Revision 1.9  2002/08/15 17:21:27  bronsema
 * Upped max execution default time out to 300
 *
 * Revision 1.8  2002/08/15 14:16:15  bronsema
 * Fixed parseCapabilities to not over write the user defined name of the server
 *
 * Revision 1.7  2002/07/08 19:54:40  pspencer
 * get version 1.1.1 by default
 *
 * Revision 1.5  2002/07/05 14:56:43  bronsema
 * Added "&SERVICE=WMS" to getCapabilties
 *
 * Revision 1.4  2002/06/26 16:52:03  assefa
 * Check if string is empty beofore calling file_exists.
 *
 * Revision 1.3  2002/06/26 13:11:42  sacha
 * Fixe a problem when refreshing. Loosin server name.
 *
 * Revision 1.2  2002/06/25 20:48:35  sacha
 * Fixed a bug when adding a new server.
 *
 * Revision 1.1  2002/06/17 18:15:12  sacha
 * moved php wms dialogs and component to php_utils.
 *
 * Revision 1.10  2002/06/13 14:10:27  sacha
 * Change copywrite
 *
 * Revision 1.9  2002/06/07 13:42:37  pspencer
 * updated refresh of servers to use service name if a user name is not provided
 *
 * Revision 1.8  2002/06/06 01:46:33  bronsema
 * Added MaxExecutionTime paramater.
 *
 * Revision 1.7  2002/05/28 14:27:34  pspencer
 * refresh servers on add
 *
 * Revision 1.6  2002/05/27 21:08:17  pspencer
 * updated layer id handling
 *
 * Revision 1.4  2002/05/24 23:07:56  pspencer
 * preserve path to wmsparse
 *
 * Revision 1.3  2002/05/22 18:45:04  pspencer
 * get version 1.0.1 capabilities only
 *
 * Revision 1.2  2002/05/22 03:47:28  bronsema
 * Added zoom to layer functionality and fixed minor javascript issues
 *
 * Revision 1.1  2002/05/16 16:52:27  bronsema
 * Initial addition
 *
 *****************************************************************************/

if ( defined( "COMMON" ) && is_dir( COMMON ) )
{
    // check for closing "\" or "/"
    if ( substr( COMMON, strlen( COMMON )- 1, 1 ) == "\\" ||
         substr( COMMON, strlen( COMMON ) - 1, 1 ) == "/" )
    {
        include_once( COMMON."logger/logger.php");
        include_once( COMMON."logger/error_manager.php");
        include_once( COMMON."phpwms/xpath.class.php");
        include_once( COMMON."phpwms/dbf.php");
    }
    else
    {
        include_once( COMMON."/logger/logger.php");
        include_once( COMMON."/logger/error_manager.php");
        include_once( COMMON."/phpwms/xpath.class.php");
        include_once( COMMON."/phpwms/dbf.php");
    }
}
elseif (file_exists("../logger/logger.php"))
{
    include_once("../logger/logger.php");
    include_once("../logger/error_manager.php");
    include_once("../phpwms/xpath.class.php");
    include_once("../phpwms/dbf.php");
}
else
{
    include_once("./logger/logger.php");
    include_once("./logger/error_manager.php");
    include_once("./phpwms/xpath.class.php");
    include_once("./phpwms/dbf.php");
}

class ServerDataManager extends Logger
{
    var $oWMSDatabase;
    var $nMaxExecutionTime;

    function ServerDataManager($oWMSDatabase, $nMaxExecutionTime=300)
    {
        parent::Logger("ServerDataManager");

        $this->oWMSDatabase = $oWMSDatabase;
        $this->nMaxExecutionTime = $nMaxExecutionTime;
    }

    /**
     * This function opens the server database and returns the rows as an array
     *
     * @param $szDataPath string - The path to the data directory.
     * @return array - Server info as an associative array.
     **/
    function buildServersList( $szDataPath )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin buildServersList() function. " );

        // get the pointer to the servers database object
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // initialize the return array
        $axReturn = array();

        // loop through each record and return an array of data
        $nrecs = dbase_numrecords( $db );
        for ($i = 1; $i <= $nrecs; $i++ )
        {
            // get the array of data
            $axReturn[$i] = dbase_get_record_with_names( $db, $i );

            // get the capabilities file date/status
            $szCapabilitiesXML = $szDataPath.
            $this->normalizeString( trim( $axReturn[$i]["capab_url"] ) ).".xml";

            // check if the file exists
            if ( file_exists( $szCapabilitiesXML ) )
            {
                // give time of last modified
                $axReturn[$i]["file_status"] = date( "l F j, Y, g:i a",
                                                     filemtime( $szCapabilitiesXML ) );
            }
            else
            {
                // file does not exist
                $axReturn[$i]["file_status"] = "Never";
            }
        }

        // close the database
        dbase_close( $db );

        // log the array
        $this->log( LOG_QUIET, $axReturn );

        // log function exit
        $this->logFuncEnd( LOG_VERBOSE, "End buildServersList() function. " );

        // return the results array
        return $axReturn;

        // end buildServersList function
    }

    /**
     * This function adds a server to the servers database.
     *
     * @param $nMaxID integer - The current maximum server ID in use.
     * @param $szName string - The name of the server.
     * @param $szURL string - The url of the server.
     * @param $szComment string - The comment.
     * @return integer - the id of the server just added or false if it failed.
     **/
    function addServer( $nMaxID, $szName, $szURL, $szComment )
    {
        global $gaExtraFields;
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin addServer() function. " );

        // get the pointer to the servers database object
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // build the row array
        $axServerRecord["server_id"] = $nMaxID + 1;
        $axServerRecord["capab_url"] = $szURL;
        $axServerRecord["map_url"] = $szURL;
        $axServerRecord["version"] = "";
        $axServerRecord["formats"] = "";
        $axServerRecord["name"] = $szName;
        $axServerRecord["svc_title"] = "";
        $axServerRecord["comment"] = $szComment;
        $axServerRecord["con_status"] = 1;

        if (isset($gaExtraFields[DB_SERVER]) &&
            is_array($gaExtraFields[DB_SERVER]) &&
            count($gaExtraFields[DB_SERVER]) > 0)
        {
            foreach($gaExtraFields[DB_SERVER] as $field)
                $axServerRecord[$field[0]] = "";
        }

        // add the record
        dbase_add_record( $db, array_values($axServerRecord) );

        // pack the database
        dbase_pack( $db );

        // close the database
        dbase_close( $db );

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End addServer() function. " );

        // return success
        return $axServerRecord["server_id"];

        // end addServer function
    }

    /**
     * This function updates the given server with the specified infoormation.
     *
     * @param $nID integer - The server ID.
     * @param $szName string - The new name of the server.
     * @param $szURL string - The new URL.
     * @param $szComment string - The new comment.
     * @return boolean - True if successful, false if not.
     **/
    function updateServer( $nID, $szName, $szURL, $szComment )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin updateServer() function. " );

        // get the pointer to the servers database object
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // loop through the dbase table and look for a match
        $nNumRecords = intval( dbase_numrecords( $db ) );
        for ( $i=1; $i<=$nNumRecords; $i++ )
        {
            // get the next row
            $axRow = dbase_get_record_with_names( $db, $i );

            // check if the id matches
            if ( $axRow["server_id"] == $nID )
            {
                // found - remove deleted field
                array_pop( $axRow );

                // update array
                $axRow["capab_url"] = $szURL;
                if ( trim( $szName ) != "" )
                    $axRow["name"] = $szName;
                else
                    $axRow["name"] = $axRow["svc_title"];
                $axRow["comment"] = $szComment;

                // update database
                dbase_replace_record( $db, array_values( $axRow ), $i );

                // exit loop
                break;
            }
        }

        // pack the database
        dbase_pack( $db );

        // close the database
        dbase_close( $db );

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End updateServer() function. " );

        // return success
        return true;

        // end updateServer function
    }

    /**
     * This function sets the given server's status to the value given.
     *
     * @param $nID integer - The ID of the server to update.
     * @param $nStatus integer - The status value to update to.
     * @return boolean - True if successful, false if not.
     **/
    function setServerStatus( $nID, $nStatus )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin setServerStatus() function. " );

        // get the pointer to the servers database object
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // loop through the dbase table and look for a match
        $nNumRecords = intval( dbase_numrecords( $db ) );
        for ( $i=1; $i<=$nNumRecords; $i++ )
        {
            // get the next row
            $axRow = dbase_get_record_with_names( $db, $i );

            // check if the id matches
            if ( $axRow["server_id"] == $nID )
            {
                // update the con_status
                $this->oWMSDatabase->setInfo( $db, $i, "con_status", $nStatus );

                // exit loop
                break;
            }
        }

        // pack the database
        dbase_pack( $db );

        // close the database
        dbase_close( $db );

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End setServerStatus() function. " );

        // return success
        return true;

        // end setServerStatus function
    }

    /**
     * This function removes a server from the server database and all refernces
     * to it.
     *
     * @param $nID integer - The server ID.
     * @param $szDataPath string - The path to the data directory.
     * @return boolean - True if successful, False if not.
     **/
    function removeServer( $nID, $szDataPath )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin removeServer() function. " );

        // flush the server
        if (!$this->flushServer( $nID, $szDataPath )) return false;

        // get the pointer to the servers database object
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // loop through the dbase table and look for a match
        $nNumRecords = intval( dbase_numrecords( $db ) );
        for ( $i=1; $i<=$nNumRecords; $i++ )
        {
            // get the next row
            $axRow = dbase_get_record_with_names( $db, $i );

            // check if the id matches
            if ( $axRow["server_id"] == $nID )
            {
                // remove
                dbase_delete_record( $db, $i );

                // exit loop
                break;
            }
        }

        // pack the database
        dbase_pack( $db );

        // close the database
        dbase_close( $db );

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End removeServer() function. " );

        // return success
        return true;

        // end removeServer function
    }

    /**
     * This function clears out data related to a server record and invalidate
     * the treemenu file.
     *
     * @param $nServerID Integer - The server ID.
     * @param $szDataPath string - The path to the server data files.
     * @return boolean - True if successful, false if not.
     **/
    function flushServer( $nServerID, $szDataPath )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin flushServer() function. " );

        // get the pointer to the server database
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        //get the server record number
        $rec_num = $this->oWMSDatabase->find_record( $db, "server_id", $nServerID );

        // check if found
        if ( $rec_num == false )
        {
            // give error and return
            $this->error( ERR_WARNING, "Server ID [".$nServerID
                          ."] was not found." );
            return false;
        }

        // get the server record and compute associated file names
        $rec = dbase_get_record_with_names( $db, $rec_num );
        $szCapabilitiesXML = $szDataPath.$this->normalizeString( trim( $rec["capab_url"] ) ).
        ".xml";
        $szSRSFile = $szDataPath."s".trim( $rec["server_id"] )."srs.txt";
        $szAbstractFile = $szDataPath."a".trim( $rec["server_id"] )."abstract.txt";

        // done with the server dbf
        dbase_close( $db );

        // remove XML & SRS file & Abstract file
        if (file_exists( $szCapabilitiesXML )) unlink ($szCapabilitiesXML);
        if (file_exists( $szSRSFile )) unlink ($szSRSFile);
        if (file_exists( $szAbstractFile )) unlink ($szAbstractFile);

        // remove entries from the capabilities, bbox and style dbf files
        $dbCapab = $this->oWMSDatabase->getDB( DB_CAPABILITIES );
        if ( $dbCapab == false )
        {
            $this->error( ERR_WARNING,"Error getting capabilities db file." );
            return false;
        }
        $dbBbox = $this->oWMSDatabase->getDB( DB_BBOX );
        if ( $dbBbox == false )
        {
            $this->error( ERR_WARNING,"Error getting bbox db file." );
            return false;
        }

        $dbStyle = $this->oWMSDatabase->getDB( DB_STYLE );
        if ( $dbStyle == false )
            $this->error( ERR_WARNING,"Warning style db file dont exist." );

        // get the number of records and loop through
        $nCapab = dbase_numrecords( $dbCapab );

        for ($i=1; $i<=$nCapab; $i++)
        {
            $rec = dbase_get_record_with_names( $dbCapab, $i );

            if ($rec['deleted'] == 1) continue;

            if ( $rec["server_id"] == $nServerID)
            {
                // delete all bbox_ids associated with the layer
                $bbox_id = $rec["bbox_id"];
                while ( $bbox_id != -1 )
                {
                    $bbox_rec_num = $this->oWMSDatabase->find_record( $dbBbox, "bbox_id", $bbox_id );
                    if ($bbox_rec_num != "")
                    {
                        $bbox_rec = dbase_get_record_with_names($dbBbox, $bbox_rec_num);
                        $bbox_id = $bbox_rec["next_id"];
                        dbase_delete_record( $dbBbox, $bbox_rec_num );
                    }
                    else
                        $bbox_id = -1;
                }

                // delete all style_ids associated with the layer
                if (isset($res["style_id"]))
                {
                    $style_id = $rec["style_id"];
                    while ( $style_id != -1 )
                    {
                        $style_rec_num = $this->oWMSDatabase->find_record( $dbStyle, "style_id", $style_id );
                        $style_rec = dbase_get_record_with_names($dbStyle, $style_rec_num);
                        $style_id = $style_rec["next_id"];
                        dbase_delete_record( $dbStyle, $style_rec_num );
                    }
                }

                // delete the layer
                dbase_delete_record( $dbCapab, $i );
            }
        }

        // pack the dbs and close them
        dbase_pack( $dbCapab );
        dbase_close( $dbCapab );
        dbase_pack( $dbBbox );
        dbase_close ($dbBbox );
        dbase_pack( $dbStyle );
        dbase_close ($dbStyle );

        // delete the treemenu
        if (file_exists( $szDataPath."server.treemenu" ))
        unlink( $szDataPath."server.treemenu" );

        if (file_exists( $szDataPath."server.treemenu.all" ))
            unlink( $szDataPath."server.treemenu.all" );

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End flushServer() function. " );

        // return success
        return true;

        // end of flushServer function
    }

    /**
     * This function refreshs a server entry by removing all related records
     * and re-downloading/parsing the capabilities file.
     *
     * @param $nServerID Integer - The server ID.
     * @param $szDataPath string - The path to the server data files.
     * @param $szWMSParseFile strign - The path and filename of the wmsparser
     *                                 executable.
     * @param $bUseLocal - If set to true, the XML capab is not retrieve from
     *                     server.
     *
     * @return boolean - True if successful, false if not.
     **/
    function refreshServer( $nServerID, $szDataPath, $szWMSParseFile, $bUseLocal=false )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin refreshServer() function. " );
        // flush the server
        if (!$this->flushServer( $nServerID, $szDataPath )) return false;
        // get the pointer to the servers database
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // get the server record number
        $rec_num = $this->oWMSDatabase->find_record( $db, "server_id", $nServerID );

        // check success
        if ( $rec_num == false )
        {
            // close db
            dbase_close( $db );
            $this->error( ERR_WARNING,"Unable to locate server ID [$nServerID]" );
            return false;
        }

        // get the record
        $rec = dbase_get_record_with_names( $db, $rec_num );

        // close db
        dbase_close( $db );

        // set the online resources & capabilities vars
        $szOnlineResource = trim($rec['capab_url']);
        $szCapabilitiesXML = $szDataPath.$this->normalizeString($szOnlineResource).".xml";

        /* ============================================================================
         * Fetch capabilities.
         * ========================================================================= */
        if (!$bUseLocal)
            if (!$this->getCapabilities( $szOnlineResource, $szCapabilitiesXML ))
            {
                // give error
                $this->error( ERR_WARNING, "An error occurred fetching capabilities." );
                return false;
            }

        /* ============================================================================
         * Parse capabilities.
         * ========================================================================= */
        if (!$this->parseCapabilities( $szCapabilitiesXML, $szOnlineResource, $nServerID,
                                $szWMSParseFile, $szDataPath ) )
        {
            // give error
            $this->error( ERR_WARNING, "An error occurred parsing capabilities." );
            return false;
        }

        /* ============================================================================
         * Update the server record if necessary.
         * ========================================================================= */
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // loop through the dbase table and look for a match
        $nNumRecords = intval( dbase_numrecords( $db ) );
        for ( $i=1; $i<=$nNumRecords; $i++ )
        {
            // get the next row
            $axRow = dbase_get_record_with_names( $db, $i );

            // check if the id matches
            if ( $axRow["server_id"] == $nServerID )
            {
                if ( trim( $axRow["name"] ) == "" )
                {
                    $axRow["name"] = $axRow["svc_title"];
                    array_pop($axRow); //remove 'deleted' item
                    // update database
                    dbase_replace_record( $db, array_values( $axRow ), $i );
                }

                // exit loop
                break;
            }
        }

        // pack the database
        dbase_pack( $db );

        // close the database
        dbase_close( $db );
        // log function end
        $this->logFuncEnd( LOG_QUIET, "End refreshServer() function. " );

        // return success
        return true;

        // end refreshServer function
    }

    /**
     * This function executes a getcapabilities call against the CubeStor
     * server at the url and the results are  saved an xml file.
     *
     * @param $szOnlineResource string - The URL to fetch from.
     * @param $szCacheFname string - The xml file to write to.
     * @return boolean - True if successful, false if not.
     **/
    function getCapabilities( $szOnlineResource, $szCacheFname )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin getCapabilities() function. " );

        // check for the request/capabilities query item in the url
        if (! stristr($szOnlineResource, "request") ||
            ! stristr($szOnlineResource, "getcapabilities"))
        {
            // add REQUEST=GetCapabilities to the URL
            if (!strchr($szOnlineResource, "?"))
            {
                // URL contains no '?', i.e. no QUERY_STRING part yet
                $szOnlineResource .= "?REQUEST=getcapabilities&version=1.1.1&SERVICE=WMS";
            }
            elseif (substr($szOnlineResource, -1) == "?" ||
                    substr($szOnlineResource, -1) == "&")
            {
                // URL ends with ? or & ... ready to append params
                $szOnlineResource .= "REQUEST=getcapabilities&version=1.1.1&SERVICE=WMS";
            }
            else
            {
                // URL contains a QUERY_STRING but does not end with &
                $szOnlineResource .= "&REQUEST=getcapabilities&version=1.1.1&SERVICE=WMS";
            }
        }

        // read capabilities via HTTP and write to cache file
        set_time_limit( $this->nMaxExecutionTime );

        // open a pointer to the input file
        if (false === ($fpIn = fopen($szOnlineResource, "r")))
        {
            // log error
            $this->error( ERR_WARNING, "Could not access $szOnlineResource");
            return false;
        }

        // open a pointer to the receiving file
        if (!($fpOut = fopen($szCacheFname, "w")))
        {
            // log error
            $this->error( ERR_WARNING, "Could not open $szCacheFname");
            return false;
        }

        // copy the file line by line
        while($fpIn && !feof($fpIn))
        {
            // build buffer
            $line = fgets($fpIn, 4096);

            // rewrite encoding tag
            $line = str_replace( "encoding=\"UTF-8\"", "encoding=\"ISO-8859-1\"",
                                 $line );
            // write the buffer to file
            fwrite($fpOut, $line, strlen($line));
        }

        // force a buffer write
        fflush( $fpOut );

        // close the files
        fclose($fpIn);
        fclose($fpOut);

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End getCapabilities() function. " );

        // return success
        return true;

        // end getCapabilities function
    }

    /**
     * This function parses an xml capabilities file and populate dbase files
     * from it.  This function is based on wmslist.php created by Daniel with
     * modifications for the new treemenu format.
     *
     * @param $szCapabilitiesFile string - XML capability file to use as input.
     * @param $szOnlineResource string - The URL to parse capabilities from.
     * @param $nServerID integer - The server ID.
     * @param $szWMSParseFile string - The path and filename to the wmsparse
     *                                 executable file
     * @param $szServerDataPath string - The directory in which all the data files
     *                                    reside.
     * @return boolean - True if successful, false if not.
     **/
    function parseCapabilities( $szCapabilitiesFile, $szOnlineResource, $nServerID,
                                $szWMSParseFile, $szServerDataPath )
    {
        global $xml;
        global $gaExtraFields;

        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin parseCapabilities() function. " );

        // extend the execution time
        set_time_limit( $this->nMaxExecutionTime );

        /* ============================================================================
         * Use fast parser
         * ========================================================================= */

        // check for the fast parser
        if (strlen(trim($szWMSParseFile)) > 0 && file_exists( $szWMSParseFile ))
        {
            // touch the critical DB files to make sure they exist
            $db = $this->oWMSDatabase->getDB( DB_SERVER );
            dbase_pack( $db );
            dbase_close( $db );
            $db = $this->oWMSDatabase->getDB( DB_CAPABILITIES );
            dbase_pack( $db );
            dbase_close( $db );
            $db = $this->oWMSDatabase->getDB( DB_BBOX );
            dbase_pack( $db );
            dbase_close( $db );
            $db = $this->oWMSDatabase->getDB( DB_STYLE );
            dbase_pack( $db );
            dbase_close( $db );

            // save the current working directory
            $curdir = getcwd();
            //we are changing directories so preserve the path to wmsparse
            $szWMSParseFile = realpath($szWMSParseFile);
            chdir( $szServerDataPath );

            // build the parse command to execute
            $cmd = $szWMSParseFile." ".basename($szCapabilitiesFile)." ".
            basename(DB_SERVER)." ".basename(DB_CAPABILITIES)." ".
            basename(DB_BBOX)." ".basename(DB_STYLE);

            $cmd .= " s".$nServerID."srs.txt ";

            $cmd .= "a".$nServerID."abstract.txt ".$nServerID;

            // execute the command
            flush();
            exec( $cmd, $a, $retval );

            // reset to the previous directory
            chdir( $curdir );

            // check the return value
            if ($retval != 0)
            {
                // log errors
                $this->error( ERR_WARNING, "Parse returned $retval" );
                $this->error( ERR_WARNING, "The output was:" );

                // loop and log errors
                foreach( $a as $l )
                {
                    $this->error( ERR_WARNING, "$l" );
                }
                return false;
            }
            else
            {

                // log function end
                $this->logFuncEnd( LOG_QUIET, "End parseCapabilities() function. " );

                // return success
                return true;
            }
        }

        /* ============================================================================
         * Use slow parse method
         * ========================================================================= */

        // create an XML object for the XML file
        $xml = new XPath($szCapabilitiesFile);

        // WMS Protocol Version
        $szVersion = "1.0.0"; // Assume 1.0.0 by default
        $aszPath = $xml->evaluate("/WMT_MS_Capabilities[@version]");
        if (count($aszPath))
        {
            $szVersion = $xml->getAttributes($aszPath[0], "version");
        }

        // WMS OnlineResource URL
        if ( $szOnlineResource == "" )
        {
            $szOnlineResource = "__onlineresource_not_found__";
            if (($aszPath = $xml->evaluate("//GetCapabilities/DCPType/".
                                           "HTTP/Get/OnlineResource"))
                && count($aszPath))
            {
                // WMS 1.1.x
                $szOnlineResource = $xml->getAttributes($aszPath[0], "xlink:href");
            }
            else if (($aszPath = $xml->evaluate("//Capabilities/DCPType/HTTP/Get"))
                     && count($aszPath))
            {
                // WMS 1.0.x
                $szOnlineResource = $xml->getAttributes($aszPath[0],
                                                        "onlineResource");
            }
        }
        // GetMap Formats
        $szGetMapFormats = "";
        if (($aszPath = $xml->evaluate("//Capability/Request/GetMap/Format"))
            && count($aszPath))
        {
            // WMS 1.1
            foreach ( $aszPath as $thisFormat )
            $szGetMapFormats .= $xml->getData($thisFormat).",";
        }
        else if (($aszPath = $xml->evaluate("//Capability/Request/Map/".
                                            "Format/descendant::*"))
                 && count($aszPath))
        {
            // WMS 1.0.x
            foreach ( $aszPath as $thisFormat )
            $szGetMapFormats .= $xml->nodeName($thisFormat).",";
        }

        //GetMap url
        $szGetMapURL = $szOnlineResource;
        if (($aszPath = $xml->evaluate("//GetMap/DCPType/".
                                       "HTTP/Get/OnlineResource"))
            && count($aszPath))
        {
            // WMS 1.1.x
            $szGetMapURL = $xml->getAttributes($aszPath[0], "xlink:href");
        }
        else if (($aszPath = $xml->evaluate("//Map/DCPType/HTTP/Get"))
                 && count($aszPath))
        {
            // WMS 1.0.x
            $szGetMapURL = $xml->getAttributes($aszPath[0],
                                                    "onlineResource");
        }

        // Service Title (human readable server name)
        $aszPath = $xml->evaluate("//Service/Title");
        $szServerTitle = "untitled";
        if (count($aszPath))
        {
            $szServerTitle = $xml->getData($aszPath[0]);
        }

        //now record the info as a server record in the server database
        $db = $this->oWMSDatabase->getDB( DB_SERVER );
        if ($db === false)
        {
            $this->error( ERR_WARNING, "Cannot access server database." );
            return false;
        }

        //echo "getmap url is $szGetMapURL";

        //this is the basic server record
        $rec = array();
        $rec['server_id'] = 0;
        $rec['capab_url'] = $szOnlineResource;
        $rec['map_url'] = $szGetMapURL;
        $rec['version'] = $szVersion;
        $rec['formats'] = $szGetMapFormats;
        $rec['name'] = $szServerTitle;
        $rec['svc_title'] = $szServerTitle;
        $rec['comment'] = "";
        $rec['con_status'] = 0;

        if (isset($gaExtraFields[DB_SERVER]) &&
            is_array($gaExtraFields[DB_SERVER]) &&
            count($gaExtraFields[DB_SERVER]) > 0)
        {
            foreach($gaExtraFields[DB_SERVER] as $field)
                $rec[$field[0]] = "";
        }

        // check for a duplicate on url
        $rec_num = $this->oWMSDatabase->find_record( $db, "capab_url", $szOnlineResource );

        //update old record if it exists
        if ($rec_num != false)
        {
            $rec_exist = dbase_get_record_with_names( $db, $rec_num );
            $rec['server_id'] = $rec_exist['server_id'];
            $rec['capab_url'] = $rec_exist['capab_url'];
            $rec['comment'] = $rec_exist['comment'];
            $rec['con_status'] = $rec_exist['con_status'];
            $rec['name'] = $rec_exist['name'];

            if (isset($gaExtraFields[DB_SERVER]) &&
                is_array($gaExtraFields[DB_SERVER]) &&
                count($gaExtraFields[DB_SERVER]) > 0)
            {
                foreach($gaExtraFields[DB_SERVER] as $field)
                    if (isset($res_exist[$field[0]]))
                        $rec[$field[0]] = $rec_exist[$field[0]];
            }

            dbase_replace_record( $db, array_values($rec), $rec_num );
        }
        else //create a new record, auto-increment server_id field.
        {
            $nrecs = dbase_numrecords( $db );
            if ($nrecs === 0)
                $rec['server_id'] = 1;
            else
            {
                $nServerID = $this->oWMSDatabase->findMaxValue( $db, 'server_id' );
                if ($nServerID === false)
                    $nServerID = 0;
                else
                    $nServerID++;

                $rec['server_id'] = $nServerID;
            }
            dbase_add_record( $db, array_values($rec) );
        }

        //remove old SRS and Abstract file if not appending
	//        if ( !$bAppend )
        {
            if (file_exists( "s".trim(strval($rec['server_id']))."srs.txt" ))
                unlink( "s".trim(strval($rec['server_id']))."srs.txt" );
            if (file_exists( "a".trim(strval($rec['server_id']))."abstract.txt" ))
                unlink("a".trim(strval($rec['server_id']))."abstract.txt");
        }

        // Look for the root Layer... must be present and must be unique.
        $aszPath = $xml->evaluate("//Capability[1]/Layer[1]");

        //find next layerid
        $hDB = $this->oWMSDatabase->getDB( DB_CAPABILITIES );
        if ( $hDB === false )
        {
            $this->error( ERR_WARNING, "Failed to open capabilities database.");
            return false;
        }
        $nLayerID = $this->oWMSDatabase->findMaxValue( $hDB, "layer_id" );
        if ($nLayerID === false)
            $nLayerID = 1;
        else
            $nLayerID++;
        dbase_close( $hDB );

        $this->log( LOG_VERBOSE, "starting to add layers at id $nLayerID" );

        if (count($aszPath))
        {
            // Recursively scan all layers starting with the root
            $res =  $this->addCapability( -1, $nLayerID, $aszPath[0], $rec['server_id'], "",
                                          0, 0, 0, 0, 0, -1, $szServerDataPath, "."/*,""*/);
        }
        else
        {
            $this->error( ERR_WARNING, "The server returned an invalid response.\nPlease try again later or contact your administrator." );
            dbase_close( $db );
            return false;
        }

        // close the database
        dbase_close( $db );

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End parseCapabilities() function. " );

        // return success
        return true;

        // end of parseCapabilities function
    }

    /**
     * create a treemenu file from the current state of the databases.
     *
     * @param szTreemenuFile string the destination file
     *
     * @return bool true if it succeeded, false otherwise
     */
    function createTreemenu( $szTreemenuFile )
    {
        global $http_form_vars, $_SERVER;

    if (isset($http_form_vars['common']))
    {
      $szBasePath = $http_form_vars['common']."/phpwms/";
    }
    else
      $szBasePath = dirname($_SERVER['PHP_SELF'])."/";

        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin createTreemenu() function. " );

        // create an array for "connected" servers
        $anConnected = array();

        // create an array for each server
        $db = $this->oWMSDatabase->getDB( DB_SERVER );
        $n_rec = dbase_numrecords( $db );
        $a_servers = array();
        for ( $i=1; $i<=$n_rec; $i++ )
        {
            $rec = dbase_get_record_with_names( $db, $i );

            // only include the "connected" servers
            if ( $rec["con_status"] == 1 )
            {
                // add the server to list of connected
                array_push( $anConnected, $rec["server_id"] );

                // add to the array
                $s_id = $rec['server_id'];
                $a_servers[ $s_id ] = array();
                $a_servers[ $s_id ][0][0] = "@server_id=$s_id";
                $a_servers[ $s_id ][0][1] = $rec['name'];
                $a_servers[ $s_id ][0][2] = '';
                $a_servers[ $s_id ][0][3] = '.';
                $a_servers[ $s_id ][0][4] = $szBasePath.'images/tree_server.gif';
            }
        }
        dbase_close( $db );

        //enumerate the capabilities and put the info into the appropriate server
        //array
        $db = $this->oWMSDatabase->getDB( DB_CAPABILITIES );
        $n_rec = dbase_numrecords( $db );

        //keep track of the current and last position in the arrays for each server
        $position = array();
        $last_position = array();
        $a_styles = "";
        for ($i=1; $i<=$n_rec; $i ++ )
        {
            $rec = dbase_get_record_with_names( $db, $i );

            // only process "connected" servers
            if ( in_array( $rec['server_id'], $anConnected ) )
            {
                // group everything by server
                $s_id = $rec['server_id'];

                //see if this server has been set before, if not, initialize last_pos
                //(pos will be initialized anyway)
                if (!isset( $position[$s_id] ))
                {
                    $last_position[$s_id] = count( $a_servers[$s_id] );
                    $position[$s_id] = count( $a_servers[$s_id] );
                }
                //convenience vars for lazy typer.
                $pos = $position[$s_id];
                $last_pos = $last_position[$s_id];

                $a_servers[$s_id][$pos] = array();
                $a_servers[$s_id][$pos][0] = "@layer_id=".trim($rec['layer_id']);
                $a_servers[$s_id][$pos][1] = trim($rec['name']);
                $a_servers[$s_id][$pos][2] = trim($rec['title']);
                $a_servers[$s_id][$pos][3] = ".".trim($rec['depth']);

                // If there's a style take the first
                // legend url for the tree image.
                if (isset($rec['style_id']) && $rec['style_id'] != -1)
                {
                    if (!is_array($a_styles))
                    {
                        $a_styles = array();
                        $db_style = $this->oWMSDatabase->getDB( DB_STYLE );
                        $n_rec_style = dbase_numrecords( $db_style );
                        for ($j=1; $j<=$n_rec_style; $j++)
                        {
                            $rec_style = dbase_get_record_with_names( $db_style, $j );
                            $a_styles[$rec_style['style_id']] = $rec_style['legendurl'];
                        }

                        dbase_close($db_style);
                    }

                    if (isset($a_styles[$rec['style_id']]) &&
                        $a_styles[$rec['style_id']] != "")
                    {
                        $a_servers[$s_id][$pos][4] = trim($a_styles[$rec['style_id']]);
                    }
                    else
                        $a_servers[$s_id][$pos][4] = $szBasePath."images/tree_layer.gif";
                }
                else
                    $a_servers[$s_id][$pos][4] = $szBasePath."images/tree_layer.gif";

               //check to see if this is a child of the last layer
                //and change the icon if it is
                if ($last_pos != $pos && $a_servers[$s_id][$last_pos][3] != ".")
                {
                    $last_depth = $a_servers[$s_id][$last_pos][3];
                    if ( strlen($a_servers[$s_id][$pos][3]) > strlen($last_depth) )
                        $a_servers[$s_id][$last_pos][4] = $szBasePath."images/tree_layers.gif";
                }


                //remember last position and increment current position for this server
                $last_position[$s_id] = $position[$s_id];
                $position[$s_id] ++;
            }
        }

        // okay, now we have to go through the arrays and build the treemenu file
        if ( file_exists( $szTreemenuFile ) )
            unlink( $szTreemenuFile );

        $fp = fopen( $szTreemenuFile, "w+" );
        //   for ( $i=0; $i<count( $a_servers ); $i++ )
        foreach ( $anConnected as $i )
        {
            for ( $j=0; $j<count( $a_servers[$i] ); $j++ )
            {
                $entry  = $a_servers[$i][$j][3]."|";
                $entry .= $a_servers[$i][$j][4]."|";
                //if name is not empty, add a callback
                if ( $a_servers[$i][$j][1] != "")
                {
                    $entry .= $a_servers[$i][$j][0];

                    if ( $a_servers[$i][$j][2] != "" )
                        $entry .= "&&".$a_servers[$i][$j][2];
                    else
                        $entry .= "&&".$a_servers[$i][$j][1];

                }
                else //otherwise just put in the name
                    $entry .= $a_servers[$i][$j][2];
                fwrite( $fp, $entry."\n" );
            }
        }

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End createTreemenu() function. " );

        // return success
        return true;

        // end createTreemenu function
    }

    /**
     * Recursively scan the tree of layers and generate db records for it.
     *
     * @param hDB integer handle to the database or -1 if not created yet
     * @param nLayerID integer the ID for the new layer.
     * @param aszPath string the layer to scan
     * @param nServerID integer index of a server record associated with this layer
     * @param szSRS string
     * @param ll_minx integer
     * @param ll_miny integer
     * @param ll_maxx integer
     * @param ll_maxy integer
     * @param bb_id integer
     * @param style_id integer
     * @param szDepth string
     * @param szAbstract string
     * @return integer the ID of the last layer added, or false if the add failed.
     *         The ID may not be the same value that was passed in if layers are
     *         added recursively.
     */

    function addCapability( $hDB, $nLayerID, $szLayerPath, $nServerID, $szSRS, $ll_minx,
                            $ll_miny, $ll_maxx, $ll_maxy, $bb_id, $style_id, $szDataPath,
                            $szDepth="."/*, $szAbstract*/)
    {
        global $xml;
        global $gaExtraFields;

        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin addCapability() function. " );

        // set the flag
        $bCloseDB = false;

        // check if set
        if ($hDB == -1)
        {
            // open the database
            $hDB = $this->oWMSDatabase->getDB( DB_CAPABILITIES );
            if ( $hDB === false )
            {
                $this->error( ERR_WARNING, "Failed to open capabilities database.");
                return false;
            }

            // set flag
            $bCloseDB = true;
        }

/* -------------------------------------------------------------------- */
/*                Retreive extractable layers info.                     */
/* -------------------------------------------------------------------- */
        if (defined("EXTRACTABLE_LAYER") && EXTRACTABLE_LAYER == true)
        {
            $szLayersPath = "//VendorSpecificCapabilities/OASIS_montaj/Extract/ExtractableLayers/ExtractableLayer";
            $aszPathExtract = $xml->evaluate($szLayersPath);


            $nExtractCount = count($aszPathExtract);
            for ($i=0; $i < $nExtractCount; $i++)
            {
                $szName =  $xml->getAttributes($aszPathExtract[$i], "name");
                $szType =  $xml->getAttributes($aszPathExtract[$i], "type");

                $aExtractLayers[$i][0] = trim($szName);

                $aExtractLayers[$i][1] = trim($szType);

                //Extarct the formats
                $szPathFormats = $szLayersPath."[".($i+1)."]/ExtractFormat";
                $aszPathFormats = $xml->evaluate($szPathFormats);
                $nFormats = count($aszPathFormats);
                $szFormats = "";
                for ($j=0; $j<$nFormats; $j++)
                {
                    if ($j == 0)
                        $szFormats .= $xml->getAttributes($aszPathFormats[$j], "name");
                    else
                    {
                        $szFormats .=";";
                        $szFormats .= $xml->getAttributes($aszPathFormats[$j], "name");
                    }
                }
                $aExtractLayers[$i][2] = $szFormats;
            }
        }

/* -------------------------------------------------------------------- */
/*   Retreive extractable layers info.                                  */
/* -------------------------------------------------------------------- */


        // Retrieve information about the layer.

        // Layer Name
        $szName = "";
        $aszPath = $xml->evaluate($szLayerPath."/Name[1]");

        if (count($aszPath))
            $szName = $xml->getData($aszPath[0]);

        // Layer Title - Mandatory (we'll still set a default just in case)
        $szTitle = "Untitled";
        $aszPath = $xml->evaluate($szLayerPath."/Title[1]");

        if (count($aszPath))
            $szTitle  = $xml->getData($aszPath[0]);

        // Queriable layer?   0/1
        $szQueryable = $xml->getAttributes($szLayerPath, "queryable");

        // List of SRS
        // We append this layer's SRS to its parent's SRS
        // __TODO__ we should check to avoid duplicate SRS in the
        // resulting string
        //
        // Before WMS 1.1.1, there was a single SRS tag with all SRS' separated
        // by spaces.  Starting with WMS1.1.1, this is deprecated and there
        // is an individual SRS tag for every SRS.
        $layerSRS = "";
        if (($aszPath = $xml->evaluate($szLayerPath."/SRS"))
            && count($aszPath))
        {
            foreach ( $aszPath as $thisSRS )
            $layerSRS .= $xml->getData($thisSRS)." ";
        }

        if ( $layerSRS != "" )
        {
            // change '\n' in '\'.'n' and '\' in '\'.'\'
            $aSearch = array("/\\\s*/", "/\n/", "/\r/");
            $aReplace = array("\\\\\\\\", "\\n", "\\r");
            $layerSRS=preg_replace($aSearch,$aReplace,$layerSRS);

            //add the SRS to the SRS flat file and append the line number to
            //szSRS for child layers
            $szSRSFile =  $szDataPath."s".trim(strval($nServerID))."srs.txt";

            // Count number of SRS entries until now
            // Note that this could be optimized by carrying the number of SRS
            // in a var somewhere.
            $i = 0;
            if (file_exists($szSRSFile))
            {
                $fp = fopen( $szSRSFile,"r");
                //scan lines until we get to the end
                do
                {
                    $buffer = fgets($fp, 4096);
                    if (strlen($buffer)) $i++;
                } while (!feof ($fp));
                fclose($fp);
            }

            // Reopen file for write and append SRS string
            $fp = fopen( $szSRSFile,"a");
            fwrite( $fp, $layerSRS."\n" );
            if ( $szSRS != "" )
            {
                $szSRS .= ",$i";
            }
            else
            $szSRS = "$i";
            fclose( $fp );
        }

        if (defined("LAYER_ABSTRACT") && LAYER_ABSTRACT == true)
        {
            // List of Abstracts
            $layerAbstract = "";
            $aszPath = $xml->evaluate($szLayerPath."/Abstract");

            if (count($aszPath))
                $layerAbstract  = $xml->getData($aszPath[0]);

            if ( $layerAbstract != "")
            {
                // change '\n' in '\'.'n' and '\' in '\'.'\'
                $aSearch = array("/\\\s*/", "/\n/", "/\r/");
                $aReplace = array("\\\\\\\\", "\\n", "\\r");
                $layerAbstract=preg_replace($aSearch,$aReplace,$layerAbstract);

                //add the Abstract to the Abstract flat file and append the
                //line number to szAbstract for child layers
                $szAbstractFile =  $szDataPath."a".trim(strval($nServerID)).
                "abstract.txt";

                // Count number of Abstract entries until now
                // Note that this could be optimized by carrying the number of
                // Abstract in a var somewhere.
                $i = 0;
                if (file_exists($szAbstractFile))
                {
                    $fp = fopen( $szAbstractFile,"r");
                    //scan lines until we get to the end
                    do
                    {
                        $buffer = fgets($fp, 4096);
                        if (strlen($buffer)) $i++;
                    } while (!feof ($fp));
                    fclose($fp);
                }

                // Reopen file for write and append Abstract string
                $fp = fopen( $szAbstractFile,"a");
                fwrite( $fp, $layerAbstract."\n" );
                /*if ( $szAbstract != "" )
                {
                    $szAbstract .= ",$i";
                }
                else*/

                $szAbstract = "$i";

                fclose( $fp );
            }
        }

        // LatLonBoundingBox
        // If not set the inherit the parent's LLBBOX
        $aszPath = $xml->match($szLayerPath."/LatLonBoundingBox[1]");
        if ( count($aszPath) > 0)
        {
            $ll_minx = $xml->getAttributes($aszPath[0], "minx");
            $ll_miny = $xml->getAttributes($aszPath[0], "miny");
            $ll_maxx = $xml->getAttributes($aszPath[0], "maxx");
            $ll_maxy = $xml->getAttributes($aszPath[0], "maxy");
        }

        // BoundingBox (maybe more than one!)
        // If not set the inherit the parent's BBOX.  If it is set, add records
        // to the bb dbase.
        $bb_id = -1;
        $aszPath = $xml->match($szLayerPath."/BoundingBox");
        if (count($aszPath))
        {
            $bb_db = $this->oWMSDatabase->getDB( DB_BBOX );
            if ( $bb_db === false )
            {
                $this->error( ERR_WARNING, "Cannot open bbox database." );
                dbase_close( $hDB );
                return false;
            }
        //figure out the id of the first record(used in layer record).
            $nrecs = dbase_numrecords( $bb_db );
            $bb_id = 0;
            for ($i = 1; $i <= $nrecs; $i++ )
            {
                $rec = dbase_get_record_with_names( $bb_db, $i );
                $bb_id = max( $rec['bbox_id'], $bb_id );
            }
            $bb_id ++;
            $rec_id = $bb_id;
            for ( $i=0; $i<count( $aszPath ); $i++ )
            {
                $bb_SRS = $xml->getAttributes( $aszPath[$i], "SRS" );
                $bb_minx = $xml->getAttributes($aszPath[$i], "minx");
                $bb_miny = $xml->getAttributes($aszPath[$i], "miny");
                $bb_maxx = $xml->getAttributes($aszPath[$i], "maxx");
                $bb_maxy = $xml->getAttributes($aszPath[$i], "maxy");

                if ( $i == count( $aszPath ) - 1)
                    $next_id = -1;
                else
                    $next_id = $rec_id + 1;
                $rec_new = array( $rec_id ++, $bb_SRS, $bb_minx, $bb_miny,
                                  $bb_maxx, $bb_maxy , $next_id );

                if (isset($gaExtraFields[DB_BBOX]) &&
                    is_array($gaExtraFields[DB_BBOX]) &&
                    count($gaExtraFields[DB_BBOX]) > 0)
                {
                    foreach($gaExtraFields[DB_BBOX] as $field)
                        $rec_new[$field[0]] = "";
                }

                dbase_add_record( $bb_db, $rec_new );
            }
            dbase_close( $bb_db );
        }


        if (defined("LAYER_STYLE") && LAYER_STYLE == true)
        {
            // Styles (maybe more than one!)
            // If it is set, add records to the style dbase.
            $aszPath = $xml->match($szLayerPath."/Style");
            if (count($aszPath))
            {
                $style_db = $this->oWMSDatabase->getDB( DB_STYLE );
                if ( $style_db === false )
                {
                    $this->error( ERR_WARNING, "Cannot open style database." );
                    dbase_close( $hDB );
                    return false;
                }

                //figure out the id of the first record(used in layer record).
                $nrecs = dbase_numrecords( $style_db );
                $style_id = 0;
                for ($i = 1; $i <= $nrecs; $i++ )
                {
                    $rec = dbase_get_record_with_names( $style_db, $i );
                    $style_id = max( $rec['style_id'], $style_id );
                }

                $style_id ++;
                $rec_id = $style_id;
                $numStyle = count( $aszPath );
                for ( $i=0; $i<$numStyle; $i++ )
                {
                    $style_name = $xml->getData( $aszPath[$i]."/Name" );
                    $style_title = $xml->getData( $aszPath[$i]."/Title" );

                    if (($aszTempPath = $xml->evaluate($aszPath[$i].
                                        "/LegendURL/OnlineResource"))
                        && count($aszTempPath))
                    {
                        // WMS 1.1.x
                        $style_legendurl = $xml->getAttributes($aszTempPath[0],
                                                           "xlink:href");
                        if(($aszLegendPath = $xml->evaluate($aszPath[$i].
                                                            "/LegendURL"))
                           && count($aszLegendPath))
                        {
                            $style_legendurl_height =
                                $xml->getAttributes($aszLegendPath[0],
                                                    "height");
                            $style_legendurl_width =
                                $xml->getAttributes($aszLegendPath[0],
                                                    "width");
                        }
                        else
                        {
                            $style_legendurl_height = "";
                            $style_legendurl_width = "";
                        }
                        $style_legendurl_format = $xml->getData($aszPath[$i].
                                                         "/LegendURL/Format");
                    }
                    else if (($aszTempPath = $xml->evaluate($aszPath[$i].
                                         "/LegendURL")) && count($aszTempPath))
                    {
                        // WMS 1.0.x
                        $style_legendurl = $xml->getData($aszTempPath[0]);
                        if(($aszLegendPath = $xml->evaluate($aszPath[$i].
                                                            "/LegendURL"))
                           && count($aszLegendPath))
                        {
                            $style_legendurl_height =
                                $xml->getAttributes($aszLegendPath[0],
                                                    "height");
                            $style_legendurl_width =
                                $xml->getAttributes($aszLegendPath[0],
                                                    "width");
                            $style_legendurl_format =
                                $xml->getAttributes($aszLegendPath[0],
                                                    "format");
                        }
                        else
                        {
                            $style_legendurl_height = "";
                            $style_legendurl_width = "";
                            $style_legendurl_format = "";
                        }
                    }
                    else
                    {
                        $style_legendurl = "";
                        $style_legendurl_height = "";
                        $style_legendurl_width = "";
                        $style_legendurl_format = "";
                    }

                    if (($aszTempPath = $xml->evaluate($aszPath[$i].
                                         "/StyleSheetURL/OnlineResource"))
                        && count($aszTempPath))
                    {
                        // WMS 1.1.x
                        $style_stylesheeturl = $xml->getAttributes(
                                                     $aszTempPath[0],
                                                        "xlink:href");
                    }
                    else if (($aszTempPath = $xml->evaluate($aszPath[$i].
                                    "/StyleSheetURL")) && count($aszTempPath))
                    {
                        // WMS 1.0.x
                        $style_stylesheeturl = $xml->getData($aszTempPath[0]);
                    }
                    else
                        $style_stylesheeturl = "";

                    if (($aszTempPath = $xml->evaluate($aszPath[$i].
                                              "/StyleURL/OnlineResource"))
                        && count($aszTempPath))
                    {
                        // WMS 1.1.x
                        $style_styleurl = $xml->getAttributes($aszTempPath[0],
                                                "xlink:href");
                    }
                    else if (($aszTempPath = $xml->evaluate($aszPath[$i].
                                                "/StyleURL")) && count($aszTempPath))
                    {
                        // WMS 1.0.x
                        $style_styleurl = $xml->getData($aszTempPath[0]);
                    }
                    else
                        $style_styleurl = "";

                    if ( $i == count( $aszPath ) - 1)
                        $next_id = -1;
                    else
                        $next_id = $rec_id + 1;

                    $rec_new = array( $rec_id ++, $style_name, $style_name,
                                      $style_legendurl,
                                      intval( $style_legendurl_height ),
                                      intval( $style_legendurl_width ),
                                      $style_legendurl_format,
                                      $style_stylesheeturl,
                                      $style_styleurl, $next_id );

                    if (isset($gaExtraFields[DB_STYLE]) &&
                        is_array($gaExtraFields[DB_STYLE]) &&
                        count($gaExtraFields[DB_STYLE]) > 0)
                    {
                        foreach($gaExtraFields[DB_STYLE] as $field)
                            $rec_new[$field[0]] = "";
                    }

                    dbase_add_record( $style_db, $rec_new );
                }
                dbase_close( $style_db );
            }
        }



/* -------------------------------------------------------------------- */
/*      extractable and type                                            */
/* -------------------------------------------------------------------- */
        if (defined("EXTRACTABLE_LAYER") && EXTRACTABLE_LAYER == true)
        {
            $nExtractLayers = count($aExtractLayers);
            $nExtractable = 0; //not extractable
            $szType = " "; //not set
            $szFormat = " ";
            for ($i=0; $i<$nExtractLayers; $i++)
            {
                $ttt = trim($aExtractLayers[$i][0]);
                if (strcmp(trim($szName), $ttt) == 0)
                {
                    $nExtractable = 1;
                    $szType = $aExtractLayers[$i][1];
                    $szFormat = $aExtractLayers[$i][2];
                    break;
                }
            }

            //Note : we duplicate the URL for each layer but It was easier
            //code structure wise than trying to add it the server dbf.
            $szDataExtractPath = "//VendorSpecificCapabilities/OASIS_montaj/Extract/DCPType/HTTP/Get";
            $aszDataExtractPath = $xml->evaluate($szDataExtractPath);

            $szExtractDataURL = $xml->getAttributes($aszDataExtractPath[0], "onlineResource");
        }

/* -------------------------------------------------------------------- */
/*      End of extract                                                  */
/* -------------------------------------------------------------------- */

        $rec_new = array( $nLayerID, $nServerID, $szName, $szTitle,
                          $szSRS);
        //add layer to the database
        if (defined("LAYER_ABSTRACT") && LAYER_ABSTRACT == true)
            array_push($rec_new, $szAbstract);

        $rec_new = array_merge($rec_new, array($ll_minx, $ll_miny,
                         $ll_maxx, $ll_maxy, $bb_id));

        if (defined("LAYER_STYLE") && LAYER_STYLE == true)
            array_push($rec_new, $style_id);

        $rec_new = array_merge($rec_new, array($szDepth, $szQueryable));

        if (defined("EXTRACTABLE_LAYER") && EXTRACTABLE_LAYER == true)
            $rec_new = array_merge($rec_new, array($nExtractable, $szType,
                                                   $szFormat, $szExtractDataURL));

        if (isset($gaExtraFields[DB_CAPABILITIES]) &&
            is_array($gaExtraFields[DB_CAPABILITIES]))
        {
            foreach ($gaExtraFields[DB_CAPABILITIES] as $field)
            {
                array_push($rec_new, "");
            }
        }

        dbase_add_record( $hDB, $rec_new );

        // Select all layers.
        $layers = $xml->evaluate($szLayerPath."/Layer");

        if (count($layers) == 0)
            return $nLayerID;  // Leaf node.

        // Run through all layers and dump info.
        foreach ( $layers as $thislayer )
        {
            $nLayerID = $this->addCapability($hDB, $nLayerID + 1, $thislayer,
                         $nServerID, $szSRS, $ll_minx, $ll_miny, $ll_maxx,
                         $ll_maxy, $bb_id,$style_id,$szDataPath, $szDepth."."/*,$szAbstract*/);

            if ( $nLayerID === false)
            {
                $this->error( ERR_WARNING, "addCapability failed on $thislayer." );
                dbase_close( $hDB );
                return false;
            }
        }

        if ($bCloseDB)
            dbase_close( $hDB );

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End addCapability() function. " );

        // return success
        return $nLayerID;

        // end addCapability function
    }

    /**
     * This function tests the availablility of a server.
     *
     * @param $szOnlineResource string - the URL of the server to contact.
     * @return boolean - True if available, false if not.
     **/
    function testServer( $nServerID )
    {
        // log function start
        $this->logFuncStart( LOG_VERBOSE, "Begin testServer() function. " );

        // get the pointer to the servers database
        $db = $this->oWMSDatabase->getDB( DB_SERVER );

        // get the server record number
        $rec_num = $this->oWMSDatabase->find_record( $db, "server_id", $nServerID );

        // check success
        if ( $rec_num == false )
        {
            // close db
            dbase_close( $db );
            $this->error( ERR_WARNING,"Unable to locate server ID [$nServerID]" );
            return false;
        }

        // get the record
        $rec = dbase_get_record_with_names( $db, $rec_num );

        // close db
        dbase_close( $db );

        // set the online resources & capabilities vars
        $szOnlineResource = trim($rec['capab_url']);

        // check for the request/capabilities query item in the url
        if (! stristr($szOnlineResource, "request") ||
            ! stristr($szOnlineResource, "capabilities"))
        {
            // add REQUEST=GetCapabilities to the URL
            if (!strchr($szOnlineResource, "?"))
            {
                // URL contains no '?', i.e. no QUERY_STRING part yet
                $szOnlineResource .= "?REQUEST=Capabilities";
            }
            elseif (substr($szOnlineResource, -1) == "?" ||
                    substr($szOnlineResource, -1) == "&")
            {
                // URL ends with ? or & ... ready to append params
                $szOnlineResource .= "REQUEST=Capabilities";
            }
            else
            {
                // URL contains a QUERY_STRING but does not end with &
                $szOnlineResource .= "&REQUEST=Capabilities";
            }
        }

        // read capabilities via HTTP and write to cache file
        set_time_limit( $this->nMaxExecutionTime );


        // open a pointer to the input file
        if (!($fpIn = fopen($szOnlineResource, "r")))
        {
            // test failed
            $bReturn = false;
        }
        else
        {
            // test succeeded
            $bReturn = true;
            fclose( $fpIn );
        }

        // log function end
        $this->logFuncEnd( LOG_QUIET, "End testServer() function. " );

        // return status
        return $bReturn;

        // end testServer function
    }

    /**
     * This function replaces all special characters in the given string.
     *
     * @param szString string - The string to convert.
     *
     * @return string converted
     */
    function normalizeString($szString)
    {
        // Normalize string by replacing all special characters
        // e.g.    "http://my.host.com/cgi-bin/mywms?"
        // becomes "http___my_host_com_cgi_bin_mywms_"
        return preg_replace("/(\W)/", "_", $szString);
    }
}
?>
