<?php
/**
 * Multi-Lingual Translator Module
 *
 * @project     PHP Utilities
 * @revision    $Id: mlt.php,v 1.17 2003/10/27 13:44:23 pspencer Exp $
 * @purpose     This file contains functions to support multiple language
 *              translations in php generated documents.
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2001, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*****************************************************************************
 * $Log: mlt.php,v $
 * Revision 1.17  2003/10/27 13:44:23  pspencer
 * use strtoupper when loading resources to ensure that we can find them.
 *
 * Revision 1.16  2003/10/21 01:30:25  pspencer
 * bug fix for vsprintf
 *
 * Revision 1.15  2003/06/19 02:15:37  pspencer
 * change to warning if missing a value.
 *
 * Revision 1.14  2003/05/09 17:06:38  sacha
 * Sacha: display an error when no resourcefile exist
 *
 * Revision 1.13  2003/05/08 14:52:02  sacha
 * trim the dbf MLT
 *
 * Revision 1.12  2003/04/17 15:34:59  bronsema
 * changed all instances of ERROR_CRITICAL to ERR_CRITICAL
 *
 * Revision 1.11  2003/03/18 14:14:03  sacha
 * Iptimize the code and made some cleaning extra tabs.
 *
 * Revision 1.10  2003/03/14 19:58:17  sacha
 * Replace all non alpha numeric char to _
 *
 * Revision 1.9  2002/12/20 13:40:30  bronsema
 * Updated to load dbase module if necessary
 *
 * Revision 1.8  2002/09/27 12:27:25  sacha
 * Force collumn name in dbf uppercase.
 *
 * Revision 1.7  2002/07/05 02:47:33  bronsema
 * Changed dbase_open to read-only
 *
 * Revision 1.6  2002/05/10 15:09:41  pspencer
 * added support for defining location of php utils directories through COMMON define
 *
 * Revision 1.5  2002/04/02 20:33:46  sacha
 * Fixed documentation.
 *
 * Revision 1.4  2002/03/18 13:14:48  bronsema
 * Updated to fixe minor bugs.  Added small utility to produce dbf file to use as
 * default database.
 *
 * Revision 1.3  2002/03/15 16:25:33  bronsema
 * Overhauled the MLT class as per discussions and created a dBase sub class.
 *
 *
 *****************************************************************************/

/**
 * This class provides the functionality to support any number of languages
 * in a php generated document.
 * This class extends the logger class for error reporting and logging.  It
 * is assumed that the logger class file has been included prior to
 * instanciating the Multi-Ligual Translator class.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */

/**
 * Check for a defined COMMON variable containing the absolute path.  If not
 * found then check ../ then ./ else failure.
 */
if ( defined( "COMMON" ) && is_dir( COMMON ) )
{
    // check for closing "\" or "/"
    if ( substr( COMMON, strlen( COMMON )- 1, 1 ) == "\\" ||
         substr( COMMON, strlen( COMMON ) - 1, 1 ) == "/" )
        include_once( COMMON."logger/logger.php");
    else
        include_once( COMMON."/logger/logger.php");
}
elseif (file_exists("../logger/logger.php"))
{
    include_once("../logger/logger.php");
}
else
{
    include_once("./logger/logger.php");
}

class MLT extends Logger
{
    /**
     * The current resource language (private).
     */
    var $szLanguage;

    /**
     * The current array of values to use for translation (private).
     */
    var $aszMLT;

    /**
     * Construct a new MLT instance and initialize it.
     *
     * @param szLanguage string - The language to return in the get function.
     **/
    function MLT( $szLanguage )
    {
        // initialize variables
        $this->setLanguage( $szLanguage );
        $this->aszMLT = array();

        // set the scope for the logging functions
        $this->setScope( "MLT" );

    // end constructor
    }

    /**
     * This function sets the language to return.
     *
     * @param szLanguage integer - The language to return.
     */
    function setLanguage( $szLanguage )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "setLanguage() starting" );

        // change language
        $this->szLanguage = preg_replace("/(\W)/", "_", $szLanguage);

        //log exit
        $this->logFuncEnd( LOG_ALL, "setLanguage() done" );

    // end setLanguage function
    }

    /**
     * This function returns the value for the key requested in the language
     * specified.
     *
     * @param $szKey string - The key of the resource to return.
     * @param $szDefault string - Optional string to return if function fails.
     * @param $aszParams array - List of parameters that, when supplied,
     *                          will cause the resource string to be passed
     *                          through the vsprintf function using those
     *                          parameters.
     * @return string - The resource requested or the default string on
     *                          failure.
     **/
    function get( $szKey, $szDefault = "", $aszParams = array() )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "get() starting" );

        // check if the language has been set
        if ( !isset($this->szLanguage) )
        {
            // log notice
            $this->log( LOG_QUIET, "Language not set.  Cannot translate ".
                        "requested key[".$szKey."].  The default string[".
                        $szDefault."] will be returned" );

            // return default
            $this->logFuncEnd( LOG_ALL, "get() done" );
            return $szDefault;
        }

        // check for resource in array
    if (isset($this->aszMLT[$szKey][$this->szLanguage]))
            $szResource = $this->aszMLT[$szKey][$this->szLanguage];
    else
        $szResource = $szDefault;
   
    // process
    if ( $szResource != "" )
    {
        // check if needs to pass through vsprintf
        if ( count( $aszParams ) > 0 )
        {
            // pass through the vsprintf function
            eval('$szResource = vsprintf( $szResource, $aszParams );');
        }
    }
    else
    {
        // not found so default
        $szResource = $szDefault;
    }

    // log exit and return results
    $this->logFuncEnd( LOG_ALL, "get() done" );
    return $szResource;

    // end get function
    }

// end MLT class
}

/**
 * This class extends the MLT class by allowing the loading of dBASE specific
 * resources.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class MLTdBase extends MLT
{
    var $mszResourceFile;

    /**
     * Construct a new MLTdBase instance and initialize it.
     *
     * @param szLanguage string - The language to return in the get function.
     **/
    function MLTdBase( $szLanguage )
    {
        // ensure that the dbase mdoule is loaded
        if (!extension_loaded("dbase")) dl("php_dbase.dll");

        // call the MLT constructor
        $this->MLT( strtoupper($szLanguage) );

    // end constructor
    }

    function get( $szKey, $szDefault = "", $aszParams = array() )
    {
        $bRes = parent::get( $szKey, $szDefault, $aszParams);

        if ($bRes !== false)
            return trim($bRes);
        else
        {
            $this->error(ERR_WARNING, "Error getting index $szKey in resource file ".$this->mszResourceFile.". Returning default value ($szDefault).");
        }
    }

    /**
     * This function populates the aszMLT array with all resources in the
     * requested language(s).
     *
     * @param $szName string - The path and filename of the dBase file to load.
     * @param $mLang mixed - Optional language(s) to load.  If not given then
     * the current language will be used.  This parameter can be a string or
     * an array.
     * @return boolean - True if successful, False if not.
     **/
    function loadResource( $szName, $mLang = "" )
    {
    $this->mszResourceFile = $szName;

        // log entry
        $this->logFuncStart( LOG_VERBOSE, "loadResource() starting" );

        // check if there is a valid language to use
        if ( $mLang == "" )
        {
            // check if current language is set
            if ( $this->szLanguage == "" )
            {
        // give error and return
                $this->error( ERR_CRITICAL, "No langauge is set." );
                return false;
            }
        }

        // try to open the database (surpress error if open fails)
        $db = @dbase_open( $szName, 0 );

        // check for failure
        if ( $db == false )
        {
            // log error and return
            $this->error( ERR_CRITICAL, "Unable to open dBase file[".$szName.
                          "]." );
            return false;
        }

        if ($mLang != "" && !is_array($mLang))
        {
            $mLang = preg_replace("/(\W)/", "_", $mLang);
        }
        elseif ($mLang != "")
        {
            foreach ($mLang as $szKey => $szLang)
            {
                $mLang[$szKey] = preg_replace("/(\W)/", "_", $szLang);
            }
        }

        // loop through each record and populate the array
        for ( $i=1; $i <= dbase_numrecords( $db ); $i++ )
        {
            // get next record
            $aRec = dbase_get_record_with_names($db, $i);

            // skip if this record is "deleted"
            if ( $aRec["deleted"] != 1 )
            {
                // check if the language was supplied
                if ( $mLang == "" )
                {
                    // use current language
                    $this->aszMLT[trim($aRec["KEY"])][strtoupper($this->szLanguage)] =
                                         $aRec[strtoupper($this->szLanguage)];
                }
                elseif ( is_array( $mLang ) )
                {
                    foreach ( $mLang as $szLang )
                    {
                        $this->aszMLT[trim($aRec["KEY"])][strtoupper($szLang)] =
                                                 $aRec[strtoupper($szLang)];
                    }
                }
                else
                {
                    // use the given language
                    $this->aszMLT[trim($aRec["KEY"])][$mLang] = $aRec[strtoupper($mLang)];
                }

                // end deleted record check
            }

            // end for loop
        }

        // log the array
        $this->log( LOG_QUIET, $this->aszMLT );

        // log exit
        $this->logFuncEnd( LOG_ALL, "loadResource() done" );

        // return success
        return true;

        // end loadResource function
    }

// end of MTLdBase class
}
?>
