<?php
/**
 * Logger Module
 * 
 * @project     PHP Wrapper Class
 * @revision    $Id: logger.php,v 1.14 2003/06/10 14:10:03 sacha Exp $
 * @purpose     this file contains classes related to output logging 
 * @author      Paul Spencer (spencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*****************************************************************************
 * $Log: logger.php,v $
 * Revision 1.14  2003/06/10 14:10:03  sacha
 * Replaced all <? php tag to <?php
 *
 * Revision 1.13  2002/12/20 14:36:46  sacha
 * Check for session variable
 *
 * Revision 1.12  2002/08/22 15:11:09  sacha
 * Set a default paramater to logger constructor to avoid warning.
 *
 * Revision 1.11  2002/06/13 15:10:48  sacha
 * Check if var is set.
 *
 * Revision 1.10  2002/06/13 14:56:16  sacha
 * Var is not supported in finction.
 *
 * Revision 1.9  2002/06/13 14:53:07  sacha
 * Change copyright info and declare some vars.
 *
 * Revision 1.8  2002/06/05 20:39:41  sacha
 * Check if error manager is a object instead of isset.
 *
 * Revision 1.7  2002/04/16 19:43:39  sacha
 * Acces global variables instead of members.
 *
 * Revision 1.6  2002/02/28 00:58:45  bronsema
 * Tweaked some comments and functions
 *
 * Revision 1.5  2002/02/23 19:41:09  pspencer
 * added new functions, fixed a couple of bugs, updated documentation
 *
 * Revision 1.4  2002/02/22 20:15:03  bronsema
 * no message
 *
 * Revision 1.3  2002/02/22 16:31:49  bronsema
 * Updated various functions.  Modified log and error functions in logger to recieve logfile and error manager objects by reference.
 *
 * Revision 1.2  2002/02/22 03:53:33  bronsema
 * Updated various functions
 *
 * Revision 1.1  2002/02/21 18:18:49  bronsema
 * Under Construction
 *
 * Revision 1.2  2002/02/20 14:50:13  pspencer
 * no message
 *
 *****************************************************************************/
 
// define default log levels

// LOG_OFF     - turn off all logging output.
// LOG_QUIET   - minimal log reporting (important functions only).
// LOG_VERBOSE - verbose log reporting (all functions).
// LOG_ALL     - very verbose log reporting.
//               (all functions and actions within functions).
define( "LOG_OFF", 0 );
define( "LOG_QUIET", 1 );
define( "LOG_VERBOSE", 2 );
define( "LOG_ALL", 3);

 
/**
 * Logger encapsulates a capability to output messages to a log file
 * 
 * It optionally restricts output based on a logging level.
 *
 * Output is timestamped and tagged with a scope name to make it easy
 * to follow a verbose log file
 *
 * Log levels are used to output debugging messages throughout
 * an application and can conveniently be turned off when deploying the
 * application. The following can be used in a class constructor to
 * accomplish this.
 *
 * $this->Logger( "myClassName" );
 * $this->log( LOG_VERBOSE, "Class was constructed." );
 *
 * Logger can also be used to report errors:
 *
 * $this->error( MY_ERROR, "MyError occurred in MyFunction" );
 *
 * By default, Logger uses two global variables, gLogFile and gErrorManager
 * to use for logging and error message reporting.  If either isn't set
 * then those calls will be ignored.  You may set them explicitly also using
 * the setLogFile and setErrorManager functions.
 *
 * Two functions are provided for structuring the log file for easier reading.
 * Calling logFuncStart and logFuncEnd will cause all intermediate log() calls
 * to be indented in the log file.  This is nested across functions and 
 * produces a much more readable output in most cases.  Indent level is
 * controlled in the logfile object.
 *
 * Logging levels are application-specific.  A general set of logging levels
 * is defined in this file as follows:
 *
 * // LOG_OFF     - turn off all logging output.
 * // LOG_QUIET   - minimal log reporting (important functions only).
 * // LOG_VERBOSE - verbose log reporting (all functions).
 * // LOG_ALL     - very verbose log reporting.
 * //               (all functions and actions within functions).
 *  define( "LOG_OFF", 0 );
 *  define( "LOG_QUIET", 1 );
 *  define( "LOG_VERBOSE", 2 );
 *  define( "LOG_ALL", 3);
 *
 * You may define your own levels, but be aware that level 0 is always OFF and
 * no messages will be logged at this level, and negative level values are not
 * allowed.
 *
 * @author Paul Spencer (spencer@dmsolutions.ca)
 */
class Logger
{
    /**
     * The LogFile object to log to
     */
    var $oLogFile;
    
    /**
     * The ErrorManager object to report errors to
     */
    var $oErrorManager;
    
    /**
     * The scope name to write out with log messages 
     */
    var $szScopeName;
    
    /**
     *  Construct a new Logger instance and initialize it for the
     *  given scope.
     *
     *  @param szScopeName string - The name of the scope (i.e. Class name).
     */
    function Logger( $szScopeName="NoScopeDefine" )
    {
        global $gLogFile, $gErrorManager;
                       
        //initialize instance variables
        $this->szScopeName = $szScopeName;
        
        //initialize logfile object
        if ( isset( $gLogFile ) )
            $this->setLogFile( $gLogFile );
        else
        if ( isset( $_SESSION['gLogFile'] ) )
            $this->setLogFile( $_SESSION['gLogFile'] );
        else
            $this->oLogFile = null;
            
        //initialize the error object
        if ( isset( $gErrorManager ) )
            $this->setErrorManager( $gErrorManager );
        else
        if ( isset( $_SESSION['gErrorManager'] ) )
            $this->setErrorManager( $_SESSION['gErrorManager'] );
        else
            $this->oErrorManager = null;
    }

    /**
     * Set the object to use for logging
     * 
     * @param oLogFile object - The log file object to use.
     */
    function setLogFile( &$oLogFile )
    {
        $this->oLogFile = &$oLogFile;
    }

    /**
     * Set the object to use for errors.
     * 
     * @param oErrorManager object - The error manager object to use.
     */
    function setErrorManager( &$oErrorManager )
    {
        $this->oErrorManager = &$oErrorManager;
    }

    /**
     * Set the scope of the logger
     *
     * @param szScopeName string - The new scope value.
     */
    function setScope( $szScopeName )
    {
        $this->szScopeName = $szScopeName;
    }

    /**
     * Output a message to the log file at the given nLevel. Note that nLevel
     * must be greater than 0, this is by design since nLevel = 0 is no 
     * debugging and less than 0 is not allowed at all
     *
     * @param nLevel integer - The log level of this message.
     * @param mMessage mixed - The message to log
     */
    function log ( $nLevel, $mMessage )
    {
        global $gLogFile, $gErrorManager;
        $szMessage = array();
        
        //make sure the log file object is valid and the level is valid
        if ( !isset($gLogFile) || $nLevel < 1 )
            return;

        //allow an array to be passed in and print it nicely using
        //the output buffering functions
        if ( is_array($mMessage) )
        {
            ob_start();
            print_r( $mMessage );
            $szMessage = "Array Contents:\n".ob_get_contents();
            ob_end_clean();
        }
        else
            $szMessage = "".$mMessage;
            
        // set the message
        $szMessage = $this->szScopeName." : ".$szMessage;
        
        // log the message
        $gLogFile->logMessage($nLevel, $szMessage);
    }
     
    /**
     * Log an error message and set the global error state.
     *    
     * @param nErrorType integer - The error type indicator.
     * @param szMessage string - The message to record.
     */
    function error ( $nErrorType, $szMessage )
    {
        // set the error message if the error object is valid
        if ( isset($this->oErrorManager) && is_object($this->oErrorManager) )
            $this->oErrorManager->setError($nErrorType, $szMessage);
        
        //and log it at the quietest level (1)
        $szMessage = "Error $nErrorType - ".$szMessage;
        $this->log( 1, $szMessage );
    }
    
    function logFuncStart( $nLevel, $szMessage )
    {
        global $gLogFile, $gErrorManager;
        
        if ( !isset($gLogFile) )
            return;
        
        $this->log( $nLevel, $szMessage );
        
        $gLogFile->indent();
    }
    
    function logFuncEnd( $nLevel, $szMessage )
    {
        global $gLogFile, $gErrorManager;
        
        if ( !isset($gLogFile) )
            return;
            
        $gLogFile->dedent();
        $this->log( $nLevel, $szMessage );
        
    }
}

?>
