<?php
/**
 * ErrorManager Module
 *
 * @project     PHP Wrapper Class
 * @revision    $Id: error_manager.php,v 1.13 2002/07/04 15:38:06 pspencer Exp $
 * @purpose     this file contains classes related to managing errors
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*****************************************************************************
 * $Log: error_manager.php,v $
 * Revision 1.13  2002/07/04 15:38:06  pspencer
 * fixed PHP_SELF ref
 *
 * Revision 1.12  2002/06/13 14:56:16  sacha
 * Var is not supported in finction.
 *
 * Revision 1.11  2002/06/13 14:53:07  sacha
 * Change copyright info and declare some vars.
 *
 * Revision 1.10  2002/06/10 14:06:02  sacha
 * Change the way errors are handled.
 *
 * Revision 1.9  2002/06/05 20:38:31  sacha
 * Added a way to trap php error.
 *
 * Revision 1.8  2002/05/29 15:29:17  pspencer
 * missing global for PHP_SELF
 *
 * Revision 1.7  2002/05/23 19:43:34  pspencer
 * added a context to the error object that will remember PHP_SELF for the page that initially contained the error
 *
 * Revision 1.6  2002/03/06 16:37:39  bronsema
 * Fixed minor problem with the getError() function.
 *
 * Revision 1.5  2002/02/28 19:03:49  bronsema
 * no message
 *
 * Revision 1.4  2002/02/28 15:35:56  bronsema
 * Updated comments for doc formatting purposes
 *
 * Revision 1.3  2002/02/28 15:34:44  bronsema
 * Added error constants
 *
 * Revision 1.2  2002/02/21 18:18:22  bronsema
 * Adjusted comments for documentation
 *
 * Revision 1.1  2002/02/21 16:04:13  bronsema
 * Initial Creation
 *****************************************************************************/

// define the core set of error types
define( "ERR_NOTICE", 0 );
define( "ERR_WARNING", 1 );
define( "ERR_CRITICAL", 2 );
define( "ERR_FILE_IO", 3 );

/**
 * The ErrorManager class will provide error status by creating and
 * and maintaining error objects.
 *
 * It is intended to be instantiated globally.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class ErrorManager
{
    /**
     * The error message count (public).
     */
    var $nErrorCount;

    /**
     * The array of error messages (private).
     */
    var $aszErrorList;
    
    var $mszOldErrorHandler;
    var $mnErrorLevel;
    
    /**
     *  Construct a new ErrorManager instance and initialize it.
     */
    function ErrorManager()
    {
        // initialize variables
        $this->aszErrorList = array();
        $this->nErrorCount = 0;

        $this->mszOldErrorHandler = "";
    // end constructor
    }

    /**
     * Reset the error array.
     */
    function reset()
    {
        $this->aszErrorList = array();
        $this->nErrorCount = 0;

    // end reset function
    }

    /**
     * Set the error type and message.
     *
     * @param nErrorType integer - The type of error.
     * @param szMessage string - The error message to set.
     */
    function setError($nErrorType, $szMessage)
    {
        // get the size of the array
        $nSize = count($this->aszErrorList);

        // create new error object
        $oError = new Error($nErrorType, $szMessage);

        // push the error onto the array
        array_push($this->aszErrorList,$oError);

        // update the count
        $this->nErrorCount = $nSize + 1;

    // end setError function
    }

    /**
     * Function to return the last error entry.
     *
     * @return mixed - The error object or FALSE if no error
     */
    function getLastError()
    {
        // set the error members
        if ($this->nErrorCount > 0)
        {
            // get last error entry
            return $this->aszErrorList[$this->nErrorCount-1];
        }
        else
        {
            // otherwise return false
            return false;
        }

    // end getLastError function
    }

    /**
     * Function to return and remove the last error entry.
     *
     * @return mixed - The error object or FALSE if no error
     */
    function popLastError()
    {
        // set the error members
        if ($this->nErrorCount > 0)
        {
            // get last error entry
            $this->nErrorCount--;
            return array_pop($this->aszErrorList);
        }
        else
        {
            // otherwise return false
            return false;
        }

    // end popLastError function
    }

    /**
     * Function to return the requested error.
     *
     * @param nIndex integer - The index of the error to return.
     *
     * @return mixed - The error object or FALSE if no error
     */
    function getError($nIndex)
    {
        // set the error members
        if ($this->nErrorCount > 0 && is_numeric($nIndex) )
        {
            // get last error entry
            return $this->aszErrorList[$nIndex];
        }
        else
        {
            // otherwise return false
            return false;
        }

    // end getLastError function
    }
    
    /**
     * Trap all error messages
     */
    function setErrorHandler($szErrorHandler="")
    {
        $this->mnErrorLevel = error_reporting(0);

        $this->mszErrorHandler = $szErrorHandler;

        set_error_handler("errorHandler");
    }
    
    function restoreErrorHandler()
    {
        error_reporting($this->mnErrorLevel);

        restore_error_handler();
    }
    
// end ErrorManager class
}

/**
 * The Error class will manage simple error objects
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class Error
{
    /**
     * The error type to report (public).
     */
    var $nErrorType;

    /**
     * The current error message (public).
     */
    var $szMessage;

    /**
     * The context (original PHP page) that caused the error
     */
    var $szContext;

    /**
     *  Construct a new Error instance and initialize it.
     *
     *  @param nErrorType integer - The type of error.
     *
     *  @param szMessage string - The error message.
     */
    function Error($nErrorType, $szMessage)
    {
        // initialize error items
        $this->nErrorType = $nErrorType;
        $this->szMessage = $szMessage;
        $this->szContext = $_SERVER["PHP_SELF"];
    }
    
// end Error class
}

function errorHandler ($errno, $errmsg, $filename, $linenum, $vars) 
{
    global $gErrorManager;
    
    // timestamp for the error entry
    $dt = date("Y-m-d H:i:s (T)");

    // define an assoc array of error string
    // in reality the only entries we should
    // consider are 2,8,256,512 and 1024
    $errortype = array (
                1   =>  "Error",
                2   =>  "Warning",
                4   =>  "Parsing Error",
                8   =>  "Notice",
                16  =>  "Core Error",
                32  =>  "Core Warning",
                64  =>  "Compile Error",
                128 =>  "Compile Warning",
                256 =>  "User Error",
                512 =>  "User Warning",
                1024=>  "User Notice"
                );
    // set of errors for which a var trace will be saved
    $user_errors = array(E_USER_ERROR, E_USER_WARNING, E_USER_NOTICE);
    
    $err = "<errorentry>\n";
    $err .= "\t<datetime>".$dt."</datetime>\n";
    $err .= "\t<errornum>".$errno."</errornum>\n";
    $err .= "\t<errortype>".$errortype[$errno]."</errortype>\n";
    $err .= "\t<errormsg>".$errmsg."</errormsg>\n";
    $err .= "\t<scriptname>".$filename."</scriptname>\n";
    $err .= "\t<scriptlinenum>".$linenum."</scriptlinenum>\n";

    if (in_array($errno, $user_errors))
        $err .= "\t<vartrace>".wddx_serialize_value($vars,"Variables")."</vartrace>\n";
    $err .= "</errorentry>\n\n";
    
    // for testing
    // echo $err;

    if (function_exists($gErrorManager->mszErrorHandler))
    {
        eval($gErrorManager->mszErrorHandler."(\$errno, \$errmsg, \$filename, \$linenum, \$vars);");
    }
    else
        echo $err;
}
?>
