<?php
/**
 * AppContext is a utility class that can be used to read configuration
 * information from an xml file.  It requires the logger utilites.
 *
 * @author Paul Spencer (spencer@dmsolutions.ca)
 * @project MapEdit
 * @revision $Id: appcontext.php,v 1.11 2002/11/07 16:13:40 sacha Exp $
 * @purpose manage application configuration through XML files
 * @copyright Copyright DM Solutions Group Inc 2002 All Rights Reserved
 */

/**
 * Check for a defined COMMON variable containing the absolute path.  If not 
 * found then check ../ then ./ else failure.
 */
if ( defined( "COMMON" ) && is_dir( COMMON ) )
{
    // check for closing "\" or "/"
    if ( substr( COMMON, strlen( COMMON )- 1, 1 ) == "\\" ||
         substr( COMMON, strlen( COMMON ) - 1, 1 ) == "/" )
        include_once( COMMON."logger/logger.php");
    else
        include_once( COMMON."/logger/logger.php");
}
elseif (file_exists("../logger/logger.php"))
{
    include_once("../logger/logger.php");
}
else
{
    include_once("./logger/logger.php");
} 

/**
 * AppContext is a utility class that allows an application to access
 * configuration information stored in an xml file. The format of the
 * xml file is simple:
 *
 * <app-context> <!--top level tag-->
 *   <include-context> <!--section to include sub context-->
 *     <context-name> <!--sub context name-->
 *     </context-name>
 *     <context-file> <!--sub context filename-->
 *     </context-file>
 *   </include-context>
 *   <!-- repeat for all sub context-->
 * 
 *   <context-param> <!--start a parameter-->
 *     <param-name>my-param</param-name>
 *     <param-value>some value</param-value>
 *     <description>This value does something for the app</description>
 *   </context-param>
 *   <!--repeat for all parameters-->
 * </app-context>
 *
 * The AppContext is initialized optionally passing a configuration file
 * name.  A configuration file can also be parsed explicity by calling
 * parseContextFile().  This allows multiple files to be parsed into the
 * same context.
 *
 * Values are retrieved from the AppContext by calling getContextValue and
 * getContextDescription with a parameter name (case-insensitive).  If the
 * parameter is found, the value/description is returned as a string.
 * Otherwise, false is returned and an error is reported to the global error
 * manager through the logger interface error().
 */
class AppContext extends Logger
{
    /**
     * an array to hold the context
     */
    var $aContext;
    
    /**
     * Sub context array
     */
    var $aoContext;
    
    /**
     * Context file name
     */
    var $szContextFile;
    
    /**
     * Context name
     */
    var $szContextName;
    
    
    /**
     * instantiate a new AppContext on a ContextFile.  If the context
     * file is omitted or invalid, a default empty context is
     * constructed anyway and an error will be reported through the
     * logger error interface
     *
     * @param szContextFile the context file to read.
     *
     * @return a new AppContext instance
     */
    function AppContext( $szContextFile = "", $szContextName = "main")
    {
        //initial base class with scope name
        $this->Logger( "AppContext" );
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "AppContext( $szContextFile ) called" );

        //initialize context array
        $this->aContext = array();
        
        //initialize sub context array
        $this->aoContext = array();        

        $this->szContextFile = $szContextFile;
        $this->szContextName = $szContextName;
        
        //parse the file if set
        if ( $szContextFile != "" )
            $this->parseContextFile( $szContextFile );

        //log exit
        $this->logFuncEnd( LOG_ALL, "AppContext( $szContextFile ) returning." );
    }

    /**
     * parse a context file into the internal context structure
     *
     * @param szContextFile the file to parse
     *
     * @return this function returns false if an error occurred and will set
     *         the global error with the reason
     */
    function parseContextFile( $szContextFile )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "parseContextFile( $szContextFile ) starting" );

        //sanity check on existence of file
        if ( !file_exists($szContextFile) )
        {
            $this->error( 1, "missing file $szContextFile" );
            return false;
        }

        //open a file pointer
        if ( !($fp = @fopen( $szContextFile, "r" )) )
        {
            $this->error( 1, "fopen failed on $szContextFile (is it locked?)" );
            return false;
        }

        //read the file
        $szContext = @fread( $fp, filesize($szContextFile) );

        //close the file pointer
        if ( !fclose( $fp ) )
        {
            $this->error( 1, "fclose failed on $szContextFile (fp is $fp)" );
            return false;
        }

        //create the parser
        $oParser = xml_parser_create( );
        $this->log( LOG_ALL, "created parser context" );

        xml_parser_set_option( $oParser, XML_OPTION_CASE_FOLDING, 0 );
        xml_parser_set_option( $oParser, XML_OPTION_SKIP_WHITE, 1 );


        //parse the file into two arrays that are the structure of the doc
        xml_parse_into_struct( $oParser, $szContext, $values, $tags );
        $this->log( LOG_ALL, "parsed XML into context array" );
        $this->log( LOG_ALL, $tags );
        $this->log( LOG_ALL, $values );

        xml_parser_free( $oParser );
        $this->log( LOG_ALL, "free'd parser instance" );

        //process the tags/values ... xml_parse_into_struct returns two
        //arrays with a reasonably complex interaction.  This code and
        //parseContextParam are based on the example in the PHP documentation
        //for xml_parse_into_struct
        foreach( $tags as $key => $val )
        {
            if ( strcasecmp($key, "include-context") == 0)
            {
                $this->log( LOG_ALL, "processing sub context" );
                
                $ranges = $val;
                for ( $i = 0; $i < count($ranges); $i += 2)
                {
                    $offset = $ranges[$i] + 1;
                    $len = $ranges[$i+1] - $offset;
                    $tdb = $this->parseContextParam(
                                  array_slice( $values, $offset, $len ));
                    $this->aoContext[strtolower($tdb['context-name'])] = new AppContext($tdb['context-file'], $tdb['context-name']);
                }
            }
            
            if ( strcasecmp($key, "context-param") == 0)
            {
                $this->log( LOG_ALL, "processing key $key" );

                $ranges = $val;
                for ( $i = 0; $i < count($ranges); $i += 2)
                {
                    $offset = $ranges[$i] + 1;
                    $len = $ranges[$i+1] - $offset;
                    $tdb = $this->parseContextParam(
                                  array_slice( $values, $offset, $len ));
                    $this->aContext[strtolower($tdb['param-name'])] = $tdb;
                }
            }
        }

        $this->log( LOG_VERBOSE, "parsing complete" );
        $this->log( LOG_ALL, $this->aContext );
        //log exit
        $this->logFuncEnd( LOG_ALL, "parseContextFile( $szContextFile ) done" );
        return true;
    }

    /**
     * PRIVATE - parse an individual parameter out of the arrays
     * returned by xml_parse_into_struct
     *
     * @param mValues an array of values to parse
     *
     * @result mixed, an array of tag=value pairs
     */
    function parseContextParam( $mValues)
    {
        //log entry
        $this->logFuncStart( LOG_ALL, "parseContextParam( $mValues ) starting" );

        $result = array();
        //implementation
        for ( $i=0; $i < count( $mValues ); $i++ )
        {
            //value might be empty
            if ( !isset( $mValues[$i]["value"]) )
                $mValues[$i]["value"] = "";
            $result[strtolower($mValues[$i]["tag"])] =
                                    trim($mValues[$i]["value"]);
            $this->log( LOG_ALL, strtolower($mValues[$i]["tag"]).
                                  "=".trim($mValues[$i]["value"]));
        }

        //log exit
        $this->logFuncEnd( LOG_ALL, "parseContextParam( $mValues ) done" );
        return $result;
    }
    
    /**
     * return the value of a context parameter from the parameter's name
     *
     * @param szParam the name of the parameter to query
     *
     * @return mixed, the string value or false if not set
     */
    function getContextValue( $szParam, $szDefault=false )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "getContextValue( $szParam ) starting" );
        $result = false;

        //implementation
        if ( isset($this->aContext[strtolower($szParam)]))
        {
            $result = $this->aContext[strtolower($szParam)]['param-value'];
            $this->log( LOG_ALL, "$szParam value is $result" );
        }
        else
        {
            // Check all sub context
            foreach ($this->aoContext as $oContext)
            {
                $result = $oContext->getContextValue( $szParam );
                if ($result !== false)
                    break;
            }
        }

        if ($result === false)
        {
            $this->error( 1, "$szParam not found. Returning default value.");
            $result = $szDefault;
        }

        //log exit
        $this->logFuncEnd( LOG_ALL, "getContextValue( $szParam  ) done" );
        return $result;
    }

    /**
     * return the description of a context parameter from it's name
     *
     * @param szParam the name of the parameter to retrieve the description
     *        for
     *
     * @return string, the description or false if not set
     */
    function getContextDescription( $szParam )
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE,
                        "getContextDescription( $szParam ) starting" );
        $result = false;

        //implementation
        if ( isset($this->aContext[strtolower($szParam)]))
        {
            $result = $this->aContext[strtolower($szParam)]['description'];
            $this->log( LOG_ALL, "$szParam value is $result" );
        }
        else
        {
             // Check all sub context
            foreach ($this->aoContext as $oContext)
            {
                $result = $oContext->getContextDescription( $szParam );
                if ($result !== false)
                    break;
            }
        }
        
        if ($result === false)
            $this->error( 1, "$szParam not found");
        
        //log exit
        $this->logFuncEnd( LOG_ALL,
                            "getContextDescription( $szParam  ) done" );
        return $result;
    }
    /**
     * return all the parameter names that are currently defined in this
     * context
     *
     * @return mixed an array of strings
     */
    function getParamNames()
    {
        //log entry
        $this->logFuncStart( LOG_VERBOSE, "getParamNames(  ) starting" );

        //implementation
        $result = array_keys( $this->aContext );

        //log exit
        $this->logFuncEnd( LOG_ALL, "getParamNames(   ) done" );
        return result;
    }
    
    /**
     * Overwrite existing parameter value
     *
     * @param szKey Param name to overwrite
     * @param szValue ew param value
     *
     * @return TRUE if success
     **/
    function setContextValue($szKey, $szValue)
    {
        $this->logFuncStart(LOG_VERBOSE, "setContextValue($szKey, $szValue) starting.");
        
        $bRes = false;
        
        if (!isset($this->aContext[strtolower($szKey)]))
        {
            // Check all sub context
            foreach ($this->aoContext as $szContextKey => $oContext)
            {
                $bRes = $this->aoContext[$szContextKey]->setContextValue( $szKey, $szValue );
                
                if ($bRes == true)
                    break;
            }

            if ($bRes == false)
                $this->error(1, "$szKey not found");
        }
        else
        {
            $this->aContext[strtolower($szKey)]['param-value'] = $szValue;
            $bRes = true;
        }
        
        $this->logFuncEnd(LOG_ALL, "setContextValue() done.");
        
        return $bRes;
    }
    
    /**
     * Overwrite existing parameter description
     *
     * @param szKey Param name to overwrite
     * @param szDescription New param description
     *
     * @return TRUE if success.
     **/
    function setContextDescription($szKey, $szDescription)
    {
        $this->logFuncStart(LOG_VERBOSE, "setContextDescription($szKey, $szDescription) starting.");
        
        $bRes = false;
        
        if (!isset($this->aContext[strtolower($szKey)]))
        {
            // Check all sub context
            foreach ($this->aoContext as $szContextKey => $oContext)
            {
                $bRes = $this->aoContext[$szContextKey]->setContextDescription( $szKey, $szDescription );
                if ($bRes !== false)
                    break;
            }

            if ($bRes == false)
                $this->error(1, "$szKey not found");
        }
        else
        {
            $this->aContext[strtolower($szKey)]['description'] = $szDescription;
            $bRes = true;
        }
        
        $this->logFuncEnd(LOG_ALL, "setContextDescription() done.");
        
        return $bRes;
    }

    /**
     * Add a new context parameter
     *
     * @param $szKey Param name
     * @param $szValue Param value
     * @param $szDescription Param description
     *
     * @return TRUE if success
     **/
    function addContextParameter($szKey, $szValue, $szDescription="")
    {
        $this->logFuncStart(LOG_VERBOSE, "addContextParameter($szKey, $szValue, $szDescription) starting.");
        
        $bRes = true;
        
        if (isset($this->aContext[strtolower($szKey)]))
        {
            $this->error(1, "$szKey already exist");
            
            $bRes = false;
        }
        else        
        {
            if ($szKey == "")
            {
                $this->error(1, "$szKey is null");
            
                $bRes = false;
            }
            else
                array_push($this->aContext, array('param-name' => $szKey, 
                                                  'param-value'=> $szValue,
                                                  'description'=> $szDescription));
        }
        
        $this->logFuncEnd(LOG_ALL, "addContextParameter() done.");
        
        return $bRes;        
    }

    /**
     * Remove a existing param from context
     *
     * @param $szKey Param name to remove
     *
     * @return TRUE if success
     **/
    function deleteContextParameter($szKey)
    {
        $this->logFuncStart(LOG_VERBOSE, "deleteContextParameter($szKey) starting.");
        
        $bRes = true;
        
        if (!isset($this->aContext[strtolower($szKey)]))
        {
            $this->error(1, "$szKey not found");
            
            $bRes = false;
        }
        else
        {
            $i = 0;
            foreach($this->aContext as $szTmpKey => $aParam)
            {
                if ($szTmpKey == $szKey)
                    break;
                else
                    $i++;
            }
            
            array_splice($this->aContext, $i, 1);
        }
        
        $this->logFuncEnd(LOG_ALL, "deleteContextParameter() done.");
        
        return $bRes;                
    }

    /**
     * Save to a file
     *
     * @param $szFileName
     *
     * @return TRUE if success
     **/
    function saveAppContext($szFileName="", $bSaveSubContext=false)
    {
        $this->logFuncStart(LOG_VERBOSE, "saveAppContext($szFileName) starting.");

        $bRes = true;

        $szXMLOutput = "<app-context>\n";

        foreach($this->aoContext as $oContext)
        {
            if ($bSaveSubContext)
                $oContext->saveAppContext();

            $szXMLOutput.= "    <include-context>\n";

            $szXMLOutput.= "        <context-name>".$oContext->szContextName."</context-name>\n";
            $szXMLOutput.= "        <context-file>".$oContext->szContextFile."</context-file>\n";
            
            $szXMLOutput.= "    </include-context>\n\n";            
        }
        
        foreach($this->aContext as $szKey => $aParam)
        {
            $szXMLOutput.= "    <context-param>\n";
            
            $szXMLOutput.= "        <param-name>$szKey</param-name>\n";
            $szXMLOutput.= "        <param-value>".$aParam['param-value']."</param-value>\n";
            $szXMLOutput.= "        <description>".$aParam['description']."</description>\n";
            
            $szXMLOutput.= "    </context-param>\n\n";
        }
        
        $szXMLOutput.= "</app-context>";

        if ($szFileName == "")
            $szFileName = $this->szContextFile;
        
        $hContextFile = fopen( $szFileName, "w" );

        if ($hContextFile === false)
        {
            $this->error(1, "Cant open file $szFileName");
            
            $bRes = false;            
        }
        else
        {
            fwrite($hContextFile, $szXMLOutput);
            fclose($hContextFile);
        }

        $this->logFuncEnd(LOG_ALL, "saveAppContext() done.");
        
        return $bRes;
    }
}
?>
