<?php
/**
 * UI Template Parser
 *
 * @project     CWC2
 * @revision    $Id: TemplateParser.php,v 1.11 2004/05/13 14:16:17 pspencer Exp $
 * @purpose     Parse UI Template HTML file
 * @author      DM Solutions Group (assefa@dmsolutions.ca,spencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */



/**
 * TemplateParser provides parsing functions for extracting a set of elements
 * from a text file and substituting them back into the text file after some
 * arbitrary manipulation.
 *
 * Typical useage might be:
 *
 * $oParser = new TemplateParser()
 * $oParser->SetPattern( $myPattern );
 * $oParser->SetTemplate( $myTemplate );
 * $nElements = $oParser->Parse();
 * for ($i=0; $i<$nElements; $i++)
 * {
 *    $szElement = $oParser->GetElement( $i );
 *    $szElement = "My Element ->".$szElement." <- My Element";
 *    $oParser->SetElement( $i, $szElement );
 * }
 * $oParser->ProcessElements();
 * echo $oParser->GetTemplate();
 *
 * this would replace all elements that match $myPattern with the element
 * wrapped in some text
 */
 class TemplateParser
{
    /**
     * the contents of the template before, during and after parsing and
     * elements substitution
     */
    var $mszTemplate;

    /**
     * array of elements found by the parser
     */
    var $maszElements;
    
    /**
     * number of elements found by parser
     */
    var $mnElements;

    /**
     * array of CWC placeholders
     */
    var $maszPlaceholders;

    /**
     * the RegExp pattern to use for finding elements
     */
    var $mszPattern = "";
    
    /**
     * the object to load a template from.  The object must
     * define a GetTemplate method, an isVarSet method and
     * a getVar method.
     */
    var $moTemplateLoader;

    /**
     * PRIVATE
     * member variable used to count replacements done through ProcessPatternReplacement
     */
    var $mnPPRCnt = 0;
    
    /**
     * PRIVATE
     * member variable used to track the current index in ProcessPatternReplacement
     */
    var $mnPPRIndex = 0;
    
    /**
     * PRIVATE
     * member variable used to track what the current replacement is in ProcessPatternReplacement
     */
    var $mszPPRReplacement = "";
    
    /**
     * Constructor to build a new TemplateParser object.  Optionally
     * pass a pattern and an object to get the template from
     * @param szPattern the pattern to use
     * @param oTemplateLoader an object that defines a function called GetTemplate
     *        that actually returns the template to be parsed.  It also defines
     *        getVar and isVarSet for templates with variable replacements.
     */
    function TemplateParser( &$oTemplateLoader, $szPattern = "" )
    {
        $this->maszElements = array();
        $this->maszPlaceholders = array();
        $this->mszPattern = $szPattern;
        $this->moTemplateLoader =& $oTemplateLoader;
        
        if (!is_object( $this->moTemplateLoader ))
        {
            echo "invalid oTemplateLoader specified in TemplateParser.<BR>".
                 "oTemplateLoader must be an object (and it isn't)!<BR>";
            exit;
        }
        
        if (!method_exists( $this->moTemplateLoader, "GetTemplate" ))
        {
            echo "invalid oTemplateLoader specified in TemplateParser.<BR>".
                 "oTemplateLoader must define a GetTemplate method!<BR>";
            exit;
        }
        
        if (!method_exists( $this->moTemplateLoader, "isVarSet" ))
        {
            echo "invalid oTemplateLoader specified in TemplateParser.<BR>".
                 "oTemplateLoader must define a isVarSet method!<BR>";
            exit;
        }
        
        if (!method_exists( $this->moTemplateLoader, "getVar" ))
        {
            echo "invalid oTemplateLoader specified in TemplateParser.<BR>".
                 "oTemplateLoader must define a getVar method!<BR>";
            exit;
        }
        
    }

    /**
     * Set the template parsing pattern, a regular expression pattern
     * compatible with preg_replace_callback.
     * @param $szPattern the pattern to use
     */
    function SetPattern( $szPattern )
    {
        $this->mszPattern = $szPattern;
    }

    /**
     * Set the template to be parsed by the parser.  This prepares the file for 
     * parsing but doesn't actually parse it.
     * @param $szTemplateName the filename of the template to parse
     * @return boolean true if the file loaded, false if an error happened
     */
    function SetTemplate(&$szTemplate)
    {
        $this->mszTemplate =& $szTemplate;
        return true;
    }

    /**
     * parse the current template contents into a set of elements
     * @return the number of elements found in the template
     */
    function Parse()
    {
        // First, include all files to include.
        $nPos = strpos($this->mszTemplate, "[#");
        while ($nPos !== false)
        {
            $nEndPos = strpos($this->mszTemplate, "#]");

            if ($nEndPos === false)
                break;

            $szTemplateName = substr($this->mszTemplate, $nPos+2, $nEndPos-$nPos-2);

            // Make a copy of the current teplate
            $szTemplate = $this->mszTemplate;
            
            $this->mszTemplate =& str_replace("[#$szTemplateName#]", 
                                  $this->moTemplateLoader->GetTemplate($szTemplateName), 
                                  $szTemplate);

            $nPos = strpos($this->mszTemplate, "[#");
        }

        // Then, replace all [$something$] to the correct url var value
        $nPos = strpos($this->mszTemplate, "[$");
        while($nPos !== false)
        {
            $nEndPos = strpos($this->mszTemplate, "$]", $nPos);
            $szTag = substr($this->mszTemplate, $nPos, $nEndPos-$nPos+2);
            $szVal = strtoupper(substr($szTag, 2, -2));
            
            //use ! at the beginning of a variable name to prevent decoding
            // i.e [$!myvar$] will substitute myvar without decoding
            $bDecode = true;
            if (substr($szVal, 0, 1) == "!")
            {
                $szVal = substr($szVal, 1 );
                $bDecode = false;
            }
            if ($this->moTemplateLoader->isVarSet($szVal))
            {
                $theVal = $this->moTemplateLoader->getVar($szVal);
                if ($bDecode)
                    $theVal = urldecode( $theVal );
                $this->mszTemplate =& str_replace($szTag, $theVal, $this->mszTemplate);
            }   
            $nPos = strpos($this->mszTemplate, "[$", $nPos+2);
        }

        if (strlen($this->mszTemplate) == 0)
            return 0;

        $this->maszElements = array();
        $this->maszPlaceholders = array();

        $this->mszTemplate = preg_replace_callback( $this->mszPattern,
                    array( &$this, "ParseCallback" ), $this->mszTemplate );

        $this->mnElements = count( $this->maszElements );
        return $this->mnElements;
    }

    /**
     * parser callback function to handle each tag as it is found
     * @param $aszMatch an array of matched parameters, use index 0
     * @return string the value to replace the tag with
     */
    function ParseCallback( $aszMatch )
    {
        $n = array_push($this->maszElements, $aszMatch[0]) - 1;
        $m = array_push($this->maszPlaceholders, "<TemplateParser ID=\"$n\"/>") - 1;
        if ($n != $m)
        {
            echo "TemplateParser: ParseCallback: array length mismatch (maszElements has $n and maszPlaceholders has $m).<BR>";
        }
        return $this->maszPlaceholders[$m];
    }

    /**
     * process widget tags back into the template using simple string replacements
     * on arrays of strings
     */
    function ProcessElements()
    {
        $this->mszTemplate = str_replace( $this->maszPlaceholders,
                                $this->maszElements, $this->mszTemplate );
    }

    /**
     * process arbitrary pattern replacements in the template
     * @param $szPattern the preg_replace compatible pattern to search for
     * @param $szReplacement the string to put in place of the pattern
     * @param $nIndex Index of the pattern instance to replace. -1 mean ALL.
     * @see http://www.php.net/preg_replace
     */
    function ProcessPatternReplacement( $szPattern, $szReplacement, $nIndex=-1 )
    {
        $this->mnPPRCnt = 0;
        $this->mnPPRIndex = $nIndex;
        $this->mszPPRReplacement = $szReplacement;

        if ($nIndex == -1)
            $this->mszTemplate = preg_replace( $szPattern, $szReplacement, $this->mszTemplate );
        else
            $this->mszTemplate = preg_replace_callback( $szPattern, array(&$this, "ProcessPatternReplacementCallback"),
                                                        $this->mszTemplate);
    }
 
    function ProcessPatternReplacementCallback($matches)
    {
        $this->mnPPRCnt++;

        if( $this->mnPPRCnt == $this->mnPPRIndex)
            return $matches[0].$this->mszPPRReplacement;
        else
            return $matches[0];
    }

    /**
     * process arbitrary replacements in the template
     * @param $szSearch the string to search for
     * @param $szReplacement the string to put in place of the search string
     */
    function ProcessStringReplacement( $szSearch, $szReplacement )
    {
        $this->mszTemplate = str_replace( $szSearch, $szReplacement, $this->mszTemplate );
    }

    /**
     * return the number of widget tags
     * @return int the number of widget tags
     */
    function NumElements()
    {
        return $this->mnElements;
    }

    /**
     * get an element by ID
     * @param $nID the id of the element to get
     * @return string the element at index $nID
     */
    function GetElement( $nID )
    {
        return $this->maszElements[$nID];
    }

    /**
     * set an element by ID
     * @param $nID the id of the element to set
     * @param $szElement the text to set for the element
     */
    function SetElement( $nID, $szElement )
    {
        $this->maszElements[$nID] = $szElement;
    }

}

/**
 * this is the template parser instance used by Chameleon.  It should probably
 * be defined in chameleon.php rather than here ... just lazy
 */
class CWCTemplateParser extends TemplateParser
{
    function CWCTemplateParser( $oObject )
    {
        parent::TemplateParser( $oObject, "/(<Cwc2[^>]*\/>)|(<Cwc2[^>]*>.*<\/CWC2>)/iUs" );
    }
}
?>
