<?php
/**
 * ZoomFactor Widget Class
 *
 * @project     CWC2
 * @revision    $Id: ZoomFactor.widget.php,v 1.2 2004/04/24 22:00:05 pspencer Exp $
 * @purpose     Display a Zoom Factor to control how far we zoom in/out
 *              on point zooms.
 * @author      DM Solutions Group (assefa@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../Label.php");

/**
 * ZoomFactor
 * 
 * @desc Display a widget that controls the amount to zoom in/out
 * when performing point zooms.
 */
class ZoomFactor extends CWCWidget
{
    var $mnDefaultFactor = 2;
    var $moLabel;
    var $mnCurrentFactor = "";
    var $mnMinimum = "";
    var $mnMaximum = "";
    var $mnDefaultTextFieldSize = 2;

    /**
     * ZoomFactor
     *
     * Constctor method for the ZoomFactor widget.
     */
    function ZoomFactor()
    {
        $this->mszLanguageResource = str_replace("\\","/",dirname(__FILE__))."/ZoomFactor.dbf";

        // invoke constructor of parent
        parent::CWCWidget();

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The ZoomFactor widget allows the user to enter a zoom factor that is used
when the user zooms in at a point.  This is ignored if the user zooms to
a rectangular region.  It contains the ability to restrict the zoom factor
to a range of values.
EOT;

        $this->moLabel = new CWCLabel( $this );
        $this->maAttributes["MINIMUMFACTOR"] = new IntegerAttribute( "MINIMUMFACTOR", false );
        $this->maAttributes["MAXIMUMFACTOR"] = new IntegerAttribute( "MAXIMUMFACTOR", false );
        $this->maAttributes["DEFAULTFACTOR"] = new IntegerAttribute( "DEFAULTFACTOR", false );
        $this->maAttributes["TEXTFIELDSIZE"] = new IntegerAttribute( "TEXTFIELDSIZE", false, 1 );
    }

    /**
     * GetJavascriptFunctions
     * 
     * Return the Javacriptfunctions needed by the widget.
     */
    function GetJavascriptFunctions()
    {
        $nMin = $this->mnMinimum;
        if (isset($this->maParams["MINIMUMFACTOR"]))
            $nMin = $this->maParams["MINIMUMFACTOR"];
        $nMax = $this->mnMaximum;
        if (isset($this->maParams["MAXIMUMFACTOR"]))
            $nMax = $this->maParams["MAXIMUMFACTOR"];

        $nZoomFactor = $this->mnCurrentFactor;
        if ($nZoomFactor == "")
        {
            $nZoomFactor = $this->GetValue( "DefaultFactor",
                                            $this->mnDefaultFactor );
        }

        $szJsMinimum = "";
        if ($nMin != "")
        {
            $szMinMsg = trim($this->moMLT->get("0", "Minimum Zoom Factor is"));
            $szJsMinimum = <<<EOT
    // check for Minimum value
    if (obj.value < {$nMin} )
    {
        obj.value = {$nMin};
        alert( "{$szMinMsg} " + {$nMin} );
    }
EOT;
        }

        $szJsMaximum = "";
        if ($nMax != "")
        {
            $szMaxMsg = trim($this->moMLT->get("1", "Maximum Zoom Factor is"));
            $szJsMaximum = <<<EOT
    // check for Maximum Value
    if (obj.value > {$nMax} )
    {
        obj.value = {$nMax};
        alert( "{$szMaxMsg} " + {$nMax} );
    }
EOT;
        }

        $szJsFunctionName = "changeZoomFactor";
        $szValueMsg = trim($this->moMLT->get("2", "Value entered is not an integer"));
        $szFunction = <<<EOT
function {$szJsFunctionName}(obj)
{
    nDefault = {$nZoomFactor};
    if (!validateInteger( obj.value ))
    {
        alert( "{$szValueMsg}" );
        obj.value = nDefault;
    }
{$szJsMinimum}
{$szJsMaximum}
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

        $szJsFunctionName = "validateInteger";
        $szFunction = <<<EOT
function {$szJsFunctionName}( nValue )
{
    return !isNaN( Number(nValue) );
}
EOT;

        $aReturn[$szJsFunctionName] = $szFunction;
        return $aReturn;
    }

    /**
     * ParseURL
     *
     * Look for the ZOOM_IN command and if founddo the zooming
     * according to the other NAV paramaters.
     */
    function  ParseURL()
    {
        if ($this->isVarSet( "ZOOMFACTOR" ))
            $this->mnCurrentFactor = $this->getVar( "ZOOMFACTOR" );
            
        // return success
        return true;
    }

    /**
     * DrawPublish
     * 
     * Return the HTML code using the name in the map file and the
     * parameters of the CWC tag.
     */
    function DrawPublish()
    {
        if (!$this->mbVisible)
            return "<!-- Zoom Factor is hidden -->";

        $nSize = $this->mnDefaultTextFieldSize;
        if (isset($this->maParams["TEXTFIELDSIZE"]))
            $nSize = intval($this->maParams["TEXTFIELDSIZE"]);

        $nZoomFactor = $this->mnDefaultFactor;
        if (isset($this->maParams["DEFAULTFACTOR"]))
            $nZoomFactor = strtoupper($this->maParams["DEFAULTFACTOR"]);
        if ($this->isVarSet( "ZOOMFACTOR" ))
            $nZoomFactor = strtoupper($this->getVar( "ZOOMFACTOR" ));

        $szResult = "<input type=\"text\" name=\"ZOOMFACTOR\"".
            " size=\"$nSize\" value=\"$nZoomFactor\"".
            " onChange=\"changeZoomFactor(this)\">";
        $szResult = $this->moLabel->DrawPublish( $szResult );

        return $szResult;
    }
}
?>
