<?php
/**
 * Base Widget class
 *
 * @project     CWC2
 * @revision    $Id: Widget.php,v 1.51 2004/07/08 15:58:56 pspencer Exp $
 * @purpose     Base Widget class
 * @author      DM Solutions Group (assefa@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

define("PRIORITY_LAST","0");
define("PRIORITY_LOW","1");
define("PRIORITY_MEDIUM","2");
define("PRIORITY_HIGH","3");
define("PRIORITY_SUPER","4");
define("PRIORITY_GOD","5");

define( "PRIORITY_MINIMUM", PRIORITY_LAST );
define( "PRIORITY_MAXIMUM", PRIORITY_GOD );

define( "ONEVENT_UNDEFINED", "-1");
define( "ONEVENT_FATAL", "0");
define( "ONEVENT_WARNING", "1");

define("AUTO_PROJECTION_UNIT","9001");

include_once( COMMON.'/xml_utils/XMLObject.php' );

/**
 *  class  CWCWidget
 *
 * @description Base widget class
 */
class CWCWidget
{
    var $mszName = "";
    var $moMapObject;
    var $moURLArray; /* to be deprecated in favour of moFormVars */
    var $moFormVars;
    var $mszCWCTag="";
    var $mnParams = 0;
    var $maParams = array();
    var $mnId = 0;
    var $mszHTMLForm = "document.forms[0]"; //TODO : set form with functions
    var $mnPriority = PRIORITY_MEDIUM;
    var $mszContent = ""; //the contents between > and </CWC2> in raw form
    var $maszContents = array(); //contents of mszContent parsed as per
                                 //ParseContents
    var $maAttributes = array();
    var $mbVisible = true;
    var $mbEnabled = true;
    var $mbInitDefaults = false;

    var $mnFormIndex = 0;

    var $maSharedResourceWidgets = array();

    var $mszLanguageResource = "";

    var $moMLT = "";
    var $moCommonMLT = "";

    var $moMap;
    var $moApp;

    var $mszWidgetName = "Widget";

    var $maszErrorMsg = array();
    
    /**
     * CWCWidget
     *
     * Base CWC Widget.
     */
    function CWCWidget()
    {
        $this->moApp =& GetChameleonApplication();

        $this->maAttributes["NAME"] = new StringAttribute( "NAME", false);
        $this->maAttributes["TYPE"] = new StringAttribute( "TYPE", true, array( strtoupper(get_class($this))));
        $this->maAttributes["VISIBLE"] = new BooleanAttribute( "VISIBLE", false);
        $this->maAttributes["ENABLED"] = new BooleanAttribute( "ENABLED", false);

        // set the description for this widget
        $this->szWidgetDescription = "CWCWidget is the most basic type of ".
                        "widget. It provides the basic functionality of a ".
                        "widget and defines the standard API that widgets ".
                        "must implement in order to function correctly with ".
                        "the CWC service instance. All widgets represent ".
                        "themselves in the template using the CWCWidget tag ".
                        "and set the type and other attributes as necessary. ";

        // create a new MLT object
        if (!isset($_SESSION['gszCurrentLanguage']))
            $lang = "en-CA";
        else
            $lang = $_SESSION['gszCurrentLanguage'];

        if ($this->mszLanguageResource != "" )
        {
            $this->moMLT = new MLTdBase( $lang );
            $this->moMLT->setErrorManager( $_SESSION["gErrorManager"] );

            // load the resourcre file into memory
            $this->moMLT->loadResource($this->mszLanguageResource);
        }

        static $oCommonMLT = "";

        // create a new common MLT object
        $oCommonMLT = new MLTdBase( $lang );
        $oCommonMLT->setErrorManager( $_SESSION["gErrorManager"] );

        // load common resources
        $oCommonMLT->loadResource(CHAMELEON_PATH."/common.dbf");

        $this->moCommonMLT =& $oCommonMLT;
    }

    /**
     * ValidateAttributes
     *
     *
     */
    function ValidateAttributes( )
    {
        $bValid = true;
        foreach( $this->maAttributes as $szAttr => $oValidator )
        {
            if ($oValidator->mbMandatory && !isset($this->maParams[$szAttr]))
            {
                echo "<!-- Missing required attribute $szAttr in ".$this->maParams["TYPE"]." -->\n";
                $bValid = false;
                $_SESSION['gErrorManager']->setError(ERR_WARNING,
                "ERROR: missing required attribute $szAttr in ".$this->maParams["TYPE"]);
            }
            elseif (isset( $this->maParams[$szAttr] ) && !$oValidator->Validate( $this->maParams[$szAttr] ))
            {
                echo "<!-- Invalid attribute $szAttr in ".$this->maParams["TYPE"]." -->\n";
                $bValid = false;
                $_SESSION['gErrorManager']->setError(ERR_WARNING,
                "ERROR: Invalid attribute $szAttr in ".$this->maParams["TYPE"].", value is ".$this->maParams[$szAttr]);
            }
        }
        return $bValid;
    }


    /**
     * SetMap($oMapObject)
     *
     * Set the map object.
     *
     * @param oMApObject
     */
    function SetMap(&$oMapObject)
    {
        $this->moMapObject =& $oMapObject;
    }


    /**
     * SetSharedResourceWidgets
     *
     * SetSharedResourceWidgets
     *
     * @param aSharedResourceWidgets By reference
     */
    function SetSharedResourceWidgets(&$aSharedResourceWidgets)
    {
        $this->maSharedResourceWidgets =& $aSharedResourceWidgets;
    }


    /**
     * SetURL($oURLObject)
     *
     * Set the URLObject.
     *
     * @param oURLArray By reference
     */
    function SetURL(&$oFormVars, &$oURLArray)
    {
        
        if (!$this->mbInitDefaults)
        {
            $_SESSION['gErrorManager']->setError(ERR_WARNING,
            "ERROR: InitDefaults didn't call base class InitDefaults in widget ".get_class( $this ));
            
            return false;
        }
        
        $this->moURLArray =& $oURLArray;
        $this->moFormVars =& $oFormVars;

        return $this->ParseURL();
    }

    /**
     * return true if a var is set in the form, false otherwise
     */
    function isVarSet( $szVar )
    {
        // check if moFormVars is set first
        if ( !isset( $this->moFormVars ) )
        {
            return false;
        }
        
        // otherwise passthru to the function
        return $this->moFormVars->isVarSet( $szVar );
    }
    
    /**
     * return value of a form var or false if not set
     */
    function getVar( $szVar )
    {
        // check if moFormVars is set first
        if ( !isset( $this->moFormVars ) )
        {
            return false;
        }        
        
        return $this->moFormVars->getVar( $szVar );
    }
    
    /**
     * set a value in the form vars
     */
    function setVar( $szVar, $szValue )
    {
        return $this->moFormVars->setVar( $szVar, $szValue );
    }

    /**
     * ParseURL
     *
     * This method should be redefined by the widgets.
     */
    function ParseURL()
    {
        return true;
    }

    /**
     * SetUniqueId
     *
     * Unique id associated with the widget. Can be used to append
     * to function names/varaibles to guarantee uniquness.
     *
     * @param $nId
     */
    function SetUniqueId($nId)
    {
        $this->mnId = $nId;
    }

    /**
     * InitUsingTag($szTag)
     *
     * Parse the tag and extract all the parameters.
     *
     * @param $szTag
     */
    function InitUsingTag(&$szTag)
    {
        $bResult = true;
        $this->mszCWCTag = $szTag;
        $this->ParseTag();
        //validate by default
        if (!isset($GLOBALS["gbValidateTemplate"]) || strcasecmp($GLOBALS["gbValidateTemplate"], "false") != 0)
            $bResult = $this->ValidateAttributes();
        return $bResult;
    }

    /**
     * InitDefaults
     *
     * per-widget initialization to initialize default values from
     * attributes etc.
     */
     function InitDefaults()
     {
        if (isset($this->maParams["VISIBLE"]))
            $this->mbVisible = ((strcasecmp($this->maParams["VISIBLE"], "TRUE")==0) ? true : false);

        if (isset($this->maParams["ENABLED"]))
            $this->mbEnabled = ((strcasecmp($this->maParams["ENABLED"], "TRUE")==0) ? true : false);

        if (isset($this->maParams["NAME"]))
            $this->mszName = $this->maParams["NAME"];

        if (isset($this->maParams["FORMINDEX"]))
            $this->mnFormIndex = $this->maParams["FORMINDEX"];

        $this->mszHTMLForm = "document.forms[".$this->mnFormIndex."]";
        
        $this->mbInitDefaults = true;
        
        return;
     }

    /**
     * ParseAttributes( $szAttributes )
     *
     * parse a string of attributes name="attribute" into an array with
     * names as keys and attributes as values
     *
     * @param $szAttributes
     */
    function ParseAttributes( $szAttributes )
    {
        $aszAttribs = array();
        $szRegExp = "/(.+)=\"(.*)\"/U";

        $nResults = preg_match_all( $szRegExp, $szAttributes, $aszParams);
  
        $aszKeys = array_values( $aszParams[1] );
        $aszValues = array_values( $aszParams[2] );

        for($i=0;$i<count($aszKeys);$i++)
        {
            $szName = trim($aszKeys[$i]);
            $szValue = $this->CWCHtmlEntities($aszValues[$i]);
            $aszAttribs[strtoupper($szName)] = $szValue;
        }

        return $aszAttribs;
    }


    /**
     * ParseTag()
     *
     * Parse a CWC tag and extract the parameters and the values.
     */
    function ParseTag()
    {
        if (strlen($this->mszCWCTag) > 5 &&
            (strncasecmp("<CWC2", $this->mszCWCTag, 5) == 0))
        {
            $szTag = trim($this->mszCWCTag);

            //first look for a complex tag
            if( stristr( $szTag, "</CWC2>" ) === false )
            {
                $iEndTag = strpos($szTag, "/>");
            }
            else
            {
                $iEndTag = strpos( $szTag, ">" );
                $iStartContents = $iEndTag + 1;
                $iEndContents = strrpos( $szTag, "<" ) - 1;
                $this->mszContent = substr( $szTag, $iStartContents, $iEndContents - $iStartContents );
            }
            if ($iEndTag === false)
              return;

            $szParams = substr($szTag, 5, ($iEndTag -5));
            $szParams = trim($szParams);

            $this->maParams = array_merge($this->ParseAttributes( $szParams ), $this->maParams);
        }

        if (trim($this->mszContent) != "")
            $this->ParseContents();
    }

    /**
     * ParseContents()
     *
     * Parse the contents of a <CWC2 ...>contents</CWC2> tag
     * assuming that the contents conform to the following
     * formatting:
     * <contenttype param1="value" param2="value"/>
     * <contenttype param1="value" param2="value"/>
     *
     * into a member variable array structured as follows:
     *
     * maszContent[contenttype][index][param] = value
     *
     * different content types are grouped by contenttype and then
     * by index of appearance.  Each parameter is given a key in a
     * subarray with the value being parsed from the tag.
     */
    function ParseContents()
    {
        
        $szContents = trim($this->mszCWCTag);
        if (strlen($szContents) <= 0)
            return;
            
        $oXML = new XMLObject( $szContents );
        if (!isset($oXML->children[0]))
        {
            //$oXML->dumpObject();
            //error parsing contents of tag as an XML structure.
        }
        else
        {
            $this->maszContents = $this->ProcessContentChildren( $oXML->children[0]->children );
        }
        return true;
                
    }
    
    function ProcessContentChildren( $aXMLObj  )
    {
        if (!is_array( $aXMLObj))
            return;
            
        $aContents = array();
        foreach($aXMLObj as $oXMLObj)
        {
            //$oXMLObj->dumpObject();
            $szIndex = strtoupper($oXMLObj->_type);
            if (!isset($aContents[$szIndex]))
            {
                $aContents[$szIndex] = array();
            }
            
            $idx = array_push( $aContents[$szIndex], array_change_key_case($oXMLObj->_attributes, CASE_UPPER) ) - 1;
            if (isset($oXMLObj->value))
            {
                $aContents[$szIndex][$idx]["_CONTENTS"] = $oXMLObj->value;
            }
            
            $nChildren = count($oXMLObj->children);
            if ($nChildren > 0)
            {
                $aContents[$szIndex][$idx]["CHILDREN"] = array();
                array_push( $aContents[$szIndex][$idx]["CHILDREN"], $this->ProcessContentChildren( $oXMLObj->children) );
            }
        }    
        return $aContents;
    }

    /**
     * GetValue($szParameter,$szDefault)
     *
     * Get the value using the parameter name or return the default
     * value if not found.
     *
     * @param $szParameter
     * @param $szDefault Optional. Default ""
     */
    function GetValue($szParameter, $szDefault = "")
    {
        $szResult = $szDefault;
        if (isset($this->maParams[strtoupper($szParameter)]))
            $szResult = $this->maParams[strtoupper($szParameter)];

        return $szResult;
    }

    /**
     * SetValue( $szParameter, $szValue )
     *
     * sets the parameter to value and creates it if necessar
     *
     * @param szParameter - the value to set
     * @param szValue - the value to place in that parameter
     */
    function SetValue( $szParameter, $szValue )
    {
        $this->maParams[strtoupper($szParameter)] = $szValue;
    }

    /**
     * DrawPublish()
     *
     * return an empty string. Should be redefined for Widgets
     * returning HTML code.
     */
    function DrawPublish()
    {
        $szReturn = "";

        foreach ($this->maszErrorMsg as $szErrorMsg)
            $szReturn .= $szErrorMsg."<br>\n";

        return $szReturn;
    }
    
    /**
     * GetJavascriptFunctions
     *
     * Returns an empty  string. Should be redefined for Widgets
     * returning Javascript code.
     */
    function GetJavascriptFunctions()
    {
        return "";
    }

    /**
     * GetJavascriptOnLoadFunctions
     *
     * Returns an empty  string. Should be redefined for Widgets
     * returning Javascript code.
     */
    function GetJavascriptOnLoadFunctions()
    {
        return "";
    }


    /**
     * GetJavascriptOnLoadFunctions
     *
     * Returns an empty  string. Should be redefined for Widgets
     * returning Javascript code.
     */
    function GetJavascriptOnMouseMoveFunctions()
    {
        return "";
    }


    function GetJavascriptOnMouseUpFunctions()
    {
        return "";
    }

    function GetJavascriptOnMouseDownFunctions()
    {
        return "";
    }

    /**
     * GetJavascriptInitFunctions
     *
     * Returns an array that contains the initialization for the javascript
     * button if necessary
     */
    function GetJavascriptInitFunctions()
    {
        return "";
     }

    /**
     * GetJavascriptIncludeFunctions
     *
     * Returns an empty  string.
     */
    function GetJavascriptIncludeFunctions()
    {
        return "";
    }

    /**
     * GetJavascriptVariables
     *
     * Returns an empty  string. Should be redefined for Widgets
     * returning Javascript global code
     */
    function GetJavascriptVariables()
    {
        return "";
    }

    /**
     * GetHTMLHiddenVariables
     *
     * Returnrs an empty  string. Should be redefined for Widgets
     * returning hidden variables
     */
    function GetHTMLHiddenVariables()
    {
        //make sure that the session id makes it into the output page.
        $szVariable = "sid";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"".session_id()."\">";
        return array($szVariable => $szValue);
    }

    function SetHTMLForm($szForm)
    {
        $this->mszHTMLForm = $szForm;
    }

    function GetSample()
    {
        $szResult = "&lt;CWC2 ";

        foreach( $this->maAttributes as $oAttr )
        {
            $szResult .= " ".$oAttr->getSample();
        }

        $szResult .= "/&gt;";
        return $szResult;
    }

    /**
     * ReprojectAuto
     *
     * Utility function to do the reprojection for AUTO
     * projections.
     *
     * @param szNewProj Optional. Default ""
     */
    function ReprojectAuto($szNewProj = "")
    {
        $oMap = $this->moMapObject->oMap;
        $szMapProj = $oMap->getProjection();

        if ($szNewProj == "")
          $szProj = $szMapProj;
        else
          $szProj = $szNewProj;

        if(strncasecmp(trim($szProj), "AUTO", 4) ==0)
        {
          //echo "ReprojectAuto() : initial= " . $szProj . "<br>\n";

            $nCenterX = ($oMap->extent->minx + $oMap->extent->maxx) / 2.0;
            $nCenterY = ($oMap->extent->miny + $oMap->extent->maxy) / 2.0;
            $oCenter = ms_newPointObj( );
            $oCenter->setXY( $nCenterX, $nCenterY );

            $oProj = ms_newProjectionObj($szMapProj);

            $oLatLonProj = ms_newProjectionObj("proj=latlong");
            $oCenter->project($oProj, $oLatLonProj);
            //was not able to use the latlon object : php error ?????
            //$oCenter->project($oProj, $oMap->latlon);

            $aProjElem = split(",",$szProj);
            $nProjElem = count($aProjElem);
            if ($nProjElem == 1)
            {
                $szProj =
                  $szProj.",".AUTO_PROJECTION_UNIT.",".$oCenter->x.",".$oCenter->y;
            }
            else if ($nProjElem == 4) //valid AUTO projection
            {
                $szProj =
                  $aProjElem[0].",".AUTO_PROJECTION_UNIT.",".$oCenter->x.",".$oCenter->y;
            }
            else
              $szProj = "";

            $oProj = ms_newProjectionObj($szProj);

            if ($szProj != "")
            {
                //echo "ReprojectAuto() : final= " . $szProj . "<br>\n";
                $oMap->setProjection( $szProj, MS_TRUE );
            }
        }
    }

    function ProcessEvent($szEvent)
    {
        if (isset($this->maszContents["ONEVENT"]))
        {
            foreach ($this->maszContents["ONEVENT"] as $aEvent)
            {
                $nErrorLevel = ONEVENT_WARNING;

                if (strcasecmp($aEvent['EVENT'], $szEvent)==0)
                {
                    if (isset($aEvent['ERRORLEVEL']))
                      $nErrorLevel = $aEvent['ERRORLEVEL'];

                    if (isset($aEvent['URL']))
                    {
                        // Then, replace all [something] to the correct url var value
                        $nPos = strpos($aEvent['URL'], "[$");
                        while($nPos !== false)
                        {
                            $nEndPos = strpos($aEvent['URL'], "$]", $nPos);
                            $szTag = substr($aEvent['URL'], $nPos, $nEndPos-$nPos+2);

                            if ($this->isVarSet( strtoupper(substr($szTag, 2, -2)) ))
                                $aEvent['URL'] =& str_replace($szTag, 
                                        $this->getVar( strtoupper(substr($szTag, 2, -2)) ),
                                        $aEvent['URL']);
                            $nPos = strpos($aEvent['URL'], "[", $nPos+2);
                        }
                        header("Location: ".$aEvent['URL']);

                        return ONEVENT_FATAL;
                    }
                    else
                    if (isset($aEvent['TEMPLATE']))
                    {
                        $this->moApp->CWCSetTemplate($aEvent['TEMPLATE']);
                        echo "<!-- redirecting to (".$aEvent['TEMPLATE'].") template. -->\n\n";

                        return ONEVENT_FATAL;
                    }
                    else
                    if (isset($aEvent['TEXT']))
                    {
                        array_push($this->maszErrorMsg, $aEvent['TEXT']);

                        return $nErrorLevel;
                    }
                }
            }

        }
        return ONEVENT_UNDEFINED;
    }

    function GetEvent($szEvent)
    {
        $aReturn = array();

        if(isset($this->maszContents["ONEVENT"]) &&
           is_array($this->maszContents["ONEVENT"]))

        {
            foreach($this->maszContents["ONEVENT"] as $oEvent)
            {
                if (isset($oEvent["EVENT"]) && $oEvent["EVENT"] == $szEvent)
                array_push($aReturn, $oEvent);
            }
        }

        return $aReturn;
    }

    function CWCHtmlEntities($szString)
    {
        $szString = str_replace(array("&amp;", "&#039;", "&gt;", "&lt;", "&quot;"),
                                array( "&",    "'",      ">",    "<",    "\""), $szString);

        return $szString;
    }
}

/**
 * Attribute
 *
 * @desc Attribute validators
 */
class Attribute
{
    var $mbMandatory;
    var $mszName;

    function Attribute( $szName, $bMandatory )
    {
        $this->mszName = $szName;
        $this->mbMandatory = $bMandatory;
    }
}

class StringAttribute extends Attribute
{
    var $maValues = array();

    function StringAttribute( $szName, $bMandatory, $aValues = array() )
    {
        $this->Attribute( $szName, $bMandatory );
        $this->maValues = $aValues;
    }

    function Validate( $szValue )
    {
        $bResult = true;
        if (count($this->maValues) > 0)
        {
            return (in_array( strtoupper($szValue), $this->maValues ));
        }
        else
            return true;
    }

    function GetSample()
    {
        if (count($this->maValues) > 0)
        {
            $szValue = implode( "|", $this->maValues );
        }
        else
            $szValue = "string";
        if ($this->mszName == "TYPE")
        {
            $szValue = "&quot;$szValue&quot;";
        }
        else
        {
            $szValue = "&quot;[$szValue]&quot;";
        }
        $szResult = "$this->mszName=$szValue";
        if ($this->mbMandatory)
            $szResult = "<font color=#FF3333>$szResult</font>";

        return $szResult;
    }
}

class BooleanAttribute extends StringAttribute
{
    function BooleanAttribute( $szName, $bMandatory )
    {
        $this->StringAttribute( $szName, $bMandatory, array( "TRUE", "FALSE" ) );
    }
}

define( "PHP_MINIMUM_INTEGER", -2147483647 );
define( "PHP_MAXIMUM_INTEGER", 2147483647 );
define( "PHP_MINIMUM_FLOAT", -1.8e308 );
define( "PHP_MAXIMUM_FLOAT", 1.8e308 );

class IntegerAttribute extends Attribute
{
    var $mnMin;
    var $mnMax;

    function IntegerAttribute( $szName, $bMandatory, $nMin = PHP_MINIMUM_INTEGER, $nMax = PHP_MAXIMUM_INTEGER )
    {
        $this->Attribute( $szName, $bMandatory );
        $this->mnMin = $nMin;
        $this->mnMax = $nMax;
    }

    function Validate( $nValue )
    {
        if (!is_numeric($nValue))
            return false;

        if (is_float($nValue))
            return false;

        return ($nValue >= $this->mnMin && $nValue <= $this->mnMax);
    }

    function GetSample()
    {
        $szValue = "&quot;[";
        if ($this->mnMin > PHP_MINIMUM_INTEGER)
        {
            $szValue .= $this->mnMin." < ";
        }
        $szValue .= "integer";
        if ($this->mnMax < PHP_MAXIMUM_INTEGER )
        {
            $szValue .= " < ".$this->mnMax;
        }
        $szValue .= "]&quot;";
        $szResult = "$this->mszName=$szValue";
        if ($this->mbMandatory)
            $szResult = "<font color=#FF3333>$szResult</font>";
        return $szResult;
    }
}

class FloatAttribute extends Attribute
{
    var $mfMin;
    var $mfMax;

    function FloatAttribute( $szName, $bMandatory, $fMin = PHP_MINIMUM_FLOAT, $fMax = PHP_MAXIMUM_FLOAT)
    {
        $this->Attribute( $szName, $bMandatory );
        $this->mfMin = $fMin;
        $this->mfMax = $fMax;
    }

    function Validate( $fValue )
    {
        if (!is_numeric($fValue))
            return false;

        return ($fValue >= $this->mfMin && $fValue <= $this->mfMax);
    }

    function GetSample()
    {
        $szValue = "&quot;[";
        if ($this->mfMin > PHP_MINIMUM_FLOAT)
        {
            $szValue .= $this->mfMin." < ";
        }
        $szValue .= "integer";
        if ($this->mfMax < PHP_MAXIMUM_FLOAT )
        {
            $szValue .= " < ".$this->mfMax;
        }
        $szValue .= "]&quot;";
        $szResult = "$this->mszName=$szValue";
        if ($this->mbMandatory)
            $szResult = "<font color=#FF3333>$szResult</font>";
        return $szResult;
    }
}

$GLOBALS['gaHexValues'] = array( "0", "1", "2", "3", "4", "5", "6", "7",
                      "8", "9", "A", "B", "C", "D", "E", "F" );

class HexColorAttribute extends Attribute
{
    function HexColorAttribute( $szName, $bMandatory )
    {
        $this->Attribute( $szName, $bMandatory );
    }

    function Validate( $szValue )
    {
        if (strlen( $szValue ) < 6 || strlen( $szValue ) > 7)
        {
            return false;
        }
        if (strlen($szValue) == 7 && $szValue[0] != "#")
        {
            return false;
        }
        if ($szValue[0] == "#")
            $szValue = substr( $szValue, 1 );
        for( $i=0; $i<6; $i++)
        {
            if (!in_array( strtoupper($szValue[$i]), $GLOBALS["gaHexValues"] ))
                return false;
        }
        return true;
    }

    function GetSample()
    {
        $szValue = "&quot;[[#]RRGGBB]&quot;";
        $szResult = "$this->mszName=$szValue";
        if ($this->mbMandatory)
            $szResult = "<font color=#FF3333>$szResult</font>";
        return $szResult;
    }
}

class RGBColorAttribute extends Attribute
{
    function RGBColorAttribute( $szName, $bMandatory )
    {
        $this->Attribute( $szName, $bMandatory );
    }

    function Validate( $szValue )
    {
        $aszRGB = explode( " ", $szValue );

        if (count($aszRGB) != 3)
            return false;

        for ($i=0; $i<3; $i++)
            if (!is_numeric($aszRGB[$i]) || $aszRGB[$i] < -1 || $aszRGB[$i] > 255 )
                return false;

        return true;
    }


    function GetSample()
    {
        $szValue = "&quot;[RRR GGG BBB]&quot;";
        $szResult = "$this->mszName=$szValue";
        if ($this->mbMandatory)
            $szResult = "<font color=#FF3333>$szResult</font>";
        return $szResult;
    }
}
