<?php
/**
 * WMSCache Widget class
 *
 * @project     GeoInnovations 2002 - MapLab enhancements
 * @revision    $Id: WMSCache.widget.php,v 1.3 2004/05/04 02:00:30 pspencer Exp $ 
 * @purpose     WMSCache Widget class puts the contents of cached WMS 
 *              capabilities into a shared resource.
 * @author      DM Solutions Group (spencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2003, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


include_once(dirname(__FILE__)."/../Widget.php");
include_once(COMMON."/phpwms/dbf.php");

/**
 * WMSCache Widget
 *
 * @desc a widget that extracts the contents of the WMS cache and inserts it
 *       into a SharedResource.  The widget can either list the servers, or
 *       list the layers in one server.
 */
class WMSCache extends CWCWidget
{
    /* the name of the shared resource to dump the results into */
    var $mszSharedResourceName;

    /* the location of the cached WMS data */
    var $mszWMSCacheDirectory;

    /* control listing of servers */
    var $mbListServer = true;

    /* control listing of layers from 'active' server */
    var $mbListLayers = false;

    /* the server id to list layers for, or the first server if not set.
     * This value is set by setting an HTML hidden variable called
     * WMSCACHE_<SharedResourceName> to a server id
     */
    var $mnActiveServer = "";

    /* the image to publish for servers */
    var $mszServerImage = "images/tree_servers.gif";

    /* the image to publish for theme branches (closed) */
    var $mszThemeImage = "images/tree_layers.gif";

    /* the image to publish for theme branches (open) */
    var $mszThemeOpenImage = "images/tree_layers.gif";

    /* the imaget to publish for layers */
    var $mszLayerImage = "images/tree_servers.gif";

    /**
     * WMSCache
     *
     * Constructor method for the WMSCache widget.
     */
    function WMSCache()
    {
        // invoke constructor of parent
        parent::CWCWidget();

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The WMSCache widget gets the contents (servers and layers) and places it inside
a SharedResource widget in a format suitable for accessing via a javascript-
enabled SharedResource.  It can either list the servers or list the layers
available in a single server.
EOT;

        $this->maAttributes["SHAREDRESOURCENAME"] = 
            new StringAttribute( "SHAREDRESOURCENAME", true );
        $this->maAttributes["WMSCACHEDIRECTORY"] = 
            new StringAttribute( "WMSCACHEDIRECTORY", true );
        $this->maAttributes["LISTSERVERS"] = 
            new BooleanAttribute( "LISTSERVERS", false );
        $this->maAttributes["LISTLAYERS"] = 
            new BooleanAttribute( "LISTLAYERS", false );
        $this->maAttributes["SERVERIMAGE"] =
            new StringAttribute( "SERVERIMAGE", false );
        $this->maAttributes["THEMEIMAGE"] =
            new StringAttribute( "THEMEIMAGE", false );
        $this->maAttributes["THEMEOPENIMAGE"] =
            new StringAttribute( "THEMEOPENIMAGE", false );
        $this->maAttributes["LAYERIMAGE"] =
            new StringAttribute( "LAYERIMAGE", false );

    }

    /**
     * initialize respectable defaults
     */
    function InitDefaults()
    {
        parent::InitDefaults();

        if (isset($this->maParams["SHAREDRESOURCENAME"]))
            $this->mszSharedResourceName = 
                $this->maParams["SHAREDRESOURCENAME"];
        if (isset($this->maParams["WMSCACHEDIRECTORY"]))
        {
            if (isset( $_SESSION['gszServerDataPath'] ))
            {
                $this->mszWMSCacheDirectory = $_SESSION['gszServerDataPath'];
            }
            else
            {
                $this->mszWMSCacheDirectory = $this->maParams["WMSCACHEDIRECTORY"];
            }
            
            //make sure to check if this is a relative path
            if (strcasecmp(substr($this->mszWMSCacheDirectory, 0, 1), "/") == 0 ||
                strcasecmp(substr($this->mszWMSCacheDirectory, 0, 1), "\\") == 0 ||
                strcasecmp(substr($this->mszWMSCacheDirectory, 1, 2), ":") == 0 )
            {
                //absolute, do nothing
            }
            else
            {
                //relative, make absolute from template path
                $oChameleon = GetChameleonApplication();
                $szTemplatePath = dirname($oChameleon->mszTemplateName);
            }                  
        }
        else
        if (isset( $_SESSION['gszServerDataPath'] ))
        {
            $this->mszWMSCacheDirectory = $_SESSION['gszServerDataPath'];
        }
        else
            $this->mszWMSCacheDirectory = realpath($_SESSION['gszCorePath']."/data");         
            
        if (isset($this->maParams["LISTSERVERS"]))
        {
            if (strcasecmp($this->maParams["LISTSERVERS"], "true" ) == 0 )
            {
                $this->mbListServers = true;
            }
            else if (strcasecmp($this->maParams["LISTSERVERS"], "false" ) == 0)
            {
                $this->mbListServers = false;
            }
        }
        if (isset($this->maParams["LISTLAYERS"]))
        {
            if (strcasecmp($this->maParams["LISTLAYERS"], "true" ) == 0 )
            {
                $this->mbListLayers = true;
            }
            else if (strcasecmp($this->maParams["LISTLAYERS"], "false" ) == 0)
            {
                $this->mbListLayers = false;
            }
        }
        if (isset($this->maParams["SERVERIMAGE"]))
            $this->mszServerImage = $this->maParams["SERVERIMAGE"];
        if (isset($this->maParams["THEMEIMAGE"]))
            $this->mszThemeImage = $this->maParams["THEMEIMAGE"];
        if (isset($this->maParams["THEMEOPENIMAGE"]))
            $this->mszThemeOpenImage = $this->maParams["THEMEOPENIMAGE"];
        if (isset($this->maParams["LAYERIMAGE"]))
            $this->mszLayerImage = $this->maParams["LAYERIMAGE"];

    }

    /**
     * return an array of HTML variables to include in the page
     */
    function GetHTMLHiddenVariables()
    {
        $aReturn = array();

        $szVariable = $this->mszSharedResourceName;

        $aReturn[$szVariable] = '<INPUT TYPE="HIDDEN" NAME="'.$szVariable.
            '" VALUE="'.$this->mnActiveServer.'">';

        return $aReturn;
    }

    /**
     * Connect to the WMS cache and create the necessary entries in the 
     * shared resource.
     */
    function ParseURL()
    {
        if (!is_dir( $this->mszWMSCacheDirectory ))
        {
            //ERROR: invalid WMS Cache Directory
            return false;
        }

        $srName = $this->mszSharedResourceName;
        
        if ($this->isVarSet( $srName ))
        {
            $this->mnActiveServer = trim($this->getVar( $srName ));

        }
        
        //create a new WMSDatabase object using the widget's configured path
        //echo "wms cache dir is ".$this->mszWMSCacheDirectory."<BR>";
        $oWMSDatabase = new WMSDatabase( $this->mszWMSCacheDirectory );
        
        //get the servers
        $hDB = $oWMSDatabase->getDB( DB_SERVER );
        $nRec = dbase_numrecords( $hDB );
        $axServers = array();

        for ( $i=1; $i<=$nRec; $i++ )
        {   
            $aRec = dbase_get_record_with_names( $hDB, $i );

            /* make the first server active by default */
            if ($this->mnActiveServer == "" && $i == 1)
            {
                $this->mnActiveServer = trim($aRec['server_id']);
            }
            /* add the server if we are listing servers or
             * we are listing layers and this is the active server
             */
            if ($this->mbListServers || 
                ( $this->mbListLayers && 
                  $this->mnActiveServer == trim($aRec['server_id'])))
            {
                // add to the array
                $nServerId = trim($aRec['server_id']);
                $axServers[$nServerId] = array();
                $axServers[$nServerId]['server_id'] = $nServerId;
                $axServers[$nServerId]['name'] = trim($aRec['name']);
                $axServers[$nServerId]['title'] = trim($aRec['svc_title']);
                $axServers[$nServerId]['current_depth'] = "";
                $axServers[$nServerId]['icon'] = $this->mszServerImage;
                $axServers[$nServerId]['iconopen'] = $this->mszServerImage;
                $axServers[$nServerId]['capab_url'] = trim($aRec['capab_url']);
                $axServers[$nServerId]['map_url'] = trim($aRec['map_url']);
                $axServers[$nServerId]['version'] = trim($aRec['version']);
                $sxServers[$nServerId]['srs'] = "";
                $aszFormats = explode( ",", trim($aRec['formats']));
                $szDefaultFormat = $aszFormats[0]; 
                //but try to find one that matches the current image type
                foreach( $aszFormats as $szFormat )
                {       
                    if (stristr( $szFormat, $_SESSION["gszImgType"]) !== false)
                    {
                        $szDefaultFormat = $szFormat;
                        break;
                    }
                }
                $axServers[$nServerId]['format'] = $szDefaultFormat;
                $axServers[ $nServerId ]['layers'] = array();
            }
        }
        dbase_close( $hDB );
        
        if ($this->mbListLayers)
        {
            //now process the layers.
            $hDB = $oWMSDatabase->getDB( DB_CAPABILITIES );
            $nRec = dbase_numrecords( $hDB );
            $szCurrentLayer = "";
            $nCurrentServerId = "";
            $aAbs = array();
            $aSRSs = array();
            $szAbFile = "";
            $szSRSFile = "";
            $dbStyle = $oWMSDatabase->getDB( DB_STYLE );
            // check for failure on opening
            if ( $dbStyle === false )
            {
                // failed so give message and return
                $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                               "Failed to open style file [".DB_STYLE."].");
                return false;
            }
            for ($i=1; $i<=$nRec; $i ++ )
            {
                $aRec = dbase_get_record_with_names( $hDB, $i );
                $nServerId = trim($aRec["server_id"]);
            
                if ($nCurrentServerId != $nServerId)
                {
                    $szCurrentLayer = "$nServerId";
                    $nCurrentServerId = $nServerId;
                }
                
                if (isset($axServers[$nServerId]))
                {
                    $axLayer = array();
                    $axLayer["layer_id"] = trim($aRec["layer_id"]);
                    $axLayer["name"] = trim($aRec["name"]);
                    $axLayer["title"] = trim($aRec["title"]);
                    $axLayer["abstract"] = "no abstract";
            if (isset($aRec["style_id"]))
                        $nStyleID = trim($aRec["style_id"]);
                    $axLayer["icon"] = $this->mszThemeImage;
                    $axLayer["iconopen"] = $this->mszThemeOpenImage;
                    $axLayer["bbox"] = $aRec['ll_minx'].",".
                             $aRec['ll_miny'].",".
                             $aRec['ll_maxx'].",".
                             $aRec['ll_maxy'];
                    
                    if (isset($aRec["style_id"]) && $nStyleID != -1)
                    {
                        // put the record into an associative array
                        $axStyleRec = dbase_get_record_with_names( $dbStyle, 
                                                                   $nStyleID );
                        $axLayer["icon"] = trim($axStyleRec['legendurl']);
                        $szLayer["iconopen"] = $axLayer["icon"];
                    }
                    
                    
                    if (isset($aRec["abstractid"]))
                    {
                        $ab_id = trim($aRec['abstractid']);
                        if ($ab_id != -1 && $ab_id != "")
                        {
                            $szFile = $this->mszWMSCacheDirectory.
                                "/a".$nServerId."abstract.txt";
                            if (strcasecmp($szAbFile, $szFile) != 0 && file_exists( $szFile ))
                            {
                                $aAbs = file( $szFile );
                                $szAbFile = $szFile;
                            }
                            if (isset($aAbs[$ab_id]))
                            {
                                $axLayer["abstract"] = trim($aAbs[$ab_id]);
                            }
                        }    
                    }

                    /*
                     * this is a performance hack.  We *should* track SRS
                     * by layer.  But this takes longer to process and
                     * longer to send to the client, esp for cubewerk servers
                     * so we just find the first SRS and use that for all
                     * layers (since they will support this SRS anyway
                     */
                    if ((!isset($axServers[$nServerId]['srs']) || 
                         $axServers[$nServerId] == "") &&
                        isset($aRec["srs_ids"]) && 
                        trim($aRec["srs_ids"]) != "" &&
                        trim($aRec["srs_ids"]) != "-1" )
                    {
                        $szSRS = "";
                        $szFile = $this->mszWMSCacheDirectory."/s".$nServerId.
                            "srs.txt";
                        if (strcasecmp($szSRSFile, $szFile) != 0)
                        {
                            $aSRSs = file($szFile);
                            $szSRSFile = $szFile;
                        }
                        $anSRS = explode(",", trim($aRec["srs_ids"]));
                        foreach ($anSRS as $nSRS)
                        {       
                            if (isset($aSRSs[$nSRS]))
                                $szSRS .= " ".$aSRSs[$nSRS];
                        }
                        $axServers[$nServerId]['srs'] = $szSRS;
                    }
                    $axLayer["layers"] = array( "LAYER" => array() );
                    $szLayerDepth = ".".trim($aRec["depth"]);
                    $nLastDot = strrpos( $szCurrentLayer, "." );

                    $szParentLayer = $szCurrentLayer;
                    
                    if (strlen( $szLayerDepth ) > 
                        strlen($axServers[$nServerId]["current_depth"]))
                    {
                        //add to previous layer, descend into current layer
                        $szCurrentLayer = $szParentLayer.".".
                            $axLayer["layer_id"];
                    }
                    else if (strlen( $szLayerDepth ) < 
                             strlen($axServers[$nServerId]["current_depth"]))
                    {
                        //move up a layer, move current layer up one.
                        $nLastDot = strrpos( $szParentLayer, "." );
                        if ($nLastDot !== false)
                            $szParentLayer = substr( $szParentLayer, 0, 
                                                     $nLastDot );
                        $nLastDot = strrpos( $szParentLayer, "." );
                        if ($nLastDot !== false)
                            $szParentLayer = substr( $szParentLayer, 0, 
                                                     $nLastDot );
                        $szCurrentLayer = $szParentLayer.".".
                            $axLayer["layer_id"];
                    }
                    else 
                    {
                        //process at same level as previous layer, 
                        //current layer doesn't change
                        $nLastDot = strrpos( $szParentLayer, "." );
                        if ($nLastDot !== false)
                            $szParentLayer = substr( $szParentLayer, 0, 
                                                     $nLastDot );
                        $szCurrentLayer = $szParentLayer.".".
                            $axLayer["layer_id"];
                    }   
                                        
                    $axServers[$nServerId]['current_depth'] = $szLayerDepth;

                    //add the layer to the servers array in the right place for
                    // a SharedResource to interpret
                    $aS =& $axServers;
                    $aszSplit = explode( ".", $szCurrentLayer );
                    $nSplit = count($aszSplit) - 1;
                    for ($j=0; $j<$nSplit; $j++)
                    {
                        $aS =& $aS[$aszSplit[$j]]['layers']['LAYER'];
                    }
                    $aS[$axLayer['layer_id']] = $axLayer;
                }
            }
            dbase_close( $hDB );
            dbase_close( $dbStyle );
        }
        
        $sr = array( "WMSSERVER" => $axServers );
        if (!isset($this->maSharedResourceWidgets[$srName]))
        {
            $this->maSharedResourceWidgets[$szName] = new SharedResource();
            $this->maSharedResourceWidgets[$srName]->mszName = $srName;
        }
        $this->maSharedResourceWidgets[$srName]->maszContents = $sr;
        
        return true;
    }
}
?>