<?php
/**
 * ProjectionSelector Widget Class
 *
 * @project     CWC2
 * @revision    $Id: ProjectionSelector.widget.php,v 1.7 2004/07/08 17:49:08 pspencer Exp $
 * @purpose     Display a projection selector dialog box.
 * @author      DM Solutions Group (assefa@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../Button.php");
include_once(dirname(__FILE__)."/../Popup.php");

/**
 * ProjectionSelector
 *
 * @desc Display a widget that pops up a projection selection widget.
 */
class ProjectionSelector extends CWCWidget
{
    var $moButton;
    var $moPopup;

    /**
     * ProjectionSelector
     *
     * Constctor method for the ProjectionSelector
     */
    function ProjectionSelector()
    {
        // invoke constructor of parent
        parent::CWCWidget();

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The ProjectionSelector widget allows the user to select a new projection
for the current context from those defined in a common resource called
"projection".  The structure of the common resource is: &ltprojection
name="human name" srs="epsg code"/&gt;.  This structure is shared with the
projection label widget if used.
EOT;

        $this->moButton = new CWCButton( $this );

        $this->moPopup = new CWCPopup( $this );
        $this->moPopup->mszLink = $_SESSION['gszCoreWebPath']."/widgets/ProjectionSelector/ProjectionSelectorPopup.phtml";
    }

    function InitDefaults()
    {
        parent::InitDefaults();
        //this widget should never belong to a toolset
        $this->maParams["TOOLSET"] = "";
        $this->moButton->InitDefaults();
        $this->moButton->SetOnClick("clickProjectionSelectorPopup");
    }
    
    function GetJavascriptVariables()
    {
        if ($this->mbVisible)
            return $this->moButton->GetJavascriptVariables();
        else return array();
    }

    function GetJavascriptInitFunctions()
    {
        if ($this->mbVisible)
            return $this->moButton->GetJavascriptInitFunctions();
        else return array();
    }

    function GetJavascriptIncludeFunctions()
    {
        if ($this->mbVisible)
            $aReturn = $this->moButton->GetJavascriptIncludeFunctions();
        else
            $aReturn = array();
        return $aReturn;
    }


    /**
     * GetJavascriptFunctions
     *
     * Build and return the array of functions needed in the
     * widget.
     */
    function GetJavascriptFunctions()
    {
        if (isset($this->maSharedResourceWidgets["CWCJSAPI"]))
          $bCWCJSAPI = 1;
        else
          $bCWCJSAPI = 0;

        $aReturn = array();
        $szJsFunctionName = "clickProjectionSelectorPopup";
        $szButtonJS = $this->moPopup->DrawPublish();
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * popup a Projection Selector dialog
 */
function {$szJsFunctionName}()
{
    {$szButtonJS}
    return;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;
        $szValidProj = "";
        if (isset($this->maSharedResourceWidgets["projection"]->maszContents["PROJECTION"]))
        {
            $szSep = "";
            foreach($this->maSharedResourceWidgets["projection"]->maszContents["PROJECTION"] as $aProj )
            {
                if (isset($aProj["SRS"]))
                {
                    $szValidProj .= $szSep."new Array(".
                                    "\"".$aProj["NAME"]."\", ".
                                    "\"".$aProj["SRS"]."\")";
                    $szSep = ", ";
                }
            }
        }
        $szJsFunctionName = "applyProjection";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * apply the user's projection selection
 */
function {$szJsFunctionName}( szProj )
{
    aszValidProj = getProjectionList();
    //validate szProj
    bValid = false;
    for (i=0;i<aszValidProj.length;i++)
    {
        if (aszValidProj[i][1] == szProj)
        {
            bValid = true;
            break;
        }
    }
    if (bValid)
    {
        if ($bCWCJSAPI)
        {
            goCWCJSAPI.oMap.SetProjection( szProj );
            goCWCJSAPI.UpdateProjection();
        }
        else
        {
            {$this->mszHTMLForm}.MAP_PROJECTION.value = szProj;
            {$this->mszHTMLForm}.submit();
        }
    }
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

        if (is_object( $this->moMapObject ))
        {
            $szProj = $this->moMapObject->oMap->getProjection();
        }
        else
        {
            $szProj = "init=EPSG:None";
        }
        $szProj = substr( $szProj, 5 );
                $szJsFunctionName = "getProjectionList";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}()
 * return the list of valid projections
 */
function {$szJsFunctionName}()
{
    return new Array( {$szValidProj} );
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

        $szJsFunctionName = "getCurrentProjection";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}()
 * return the current projection
 */
function {$szJsFunctionName}()
{
    return "{$szProj}";
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

        if ($bCWCJSAPI)
        {
            $nValidProj = count($this->maSharedResourceWidgets["projection"]->maszContents["PROJECTION"]);
            $szStrValidProj="";
            if ($nValidProj > 0)
            {
                $szStrValidProj = "goCWCJSAPI.oMap.validprojections = new Array(".$nValidProj.");\n";
                $iTmp =0;
                foreach($this->maSharedResourceWidgets["projection"]->maszContents["PROJECTION"] as $aProj )
                {       
                    $szStrValidProj .= "goCWCJSAPI.oMap.validprojections[".$iTmp."]='".$aProj["SRS"]."';\n";
                    $szStrValidProj .= "goCWCJSAPI.oMap.validprojectionNames[".$iTmp."]='".$aProj["NAME"]."';\n";
                    $iTmp++;
                }
            }
            $szJsFunctionName = "ProjSelectWSetJSAPIProjList";
            $szFunction = <<<EOT
/**
 * {$szJsFunctionName}()
 * return the current projection
 */
function {$szJsFunctionName}()
{
    {$szStrValidProj}
}
EOT;
            $aReturn[$szJsFunctionName] = $szFunction;
        }
            
        return $aReturn;
    }

    /**
     * return an array of javascript functions needed by Scalebar widget
     * and called when the page is loaded. 
     * @return array of name = function values
     */
     function GetJavascriptOnLoadFunctions()
     {
        if ($this->mbVisible)
            $aReturn = $this->moButton->GetJavascriptOnLoadFunctions();
        else
            $aReturn = array();

         if (isset($this->maSharedResourceWidgets["CWCJSAPI"]) &&
             isset($this->maSharedResourceWidgets["projection"]->maszContents["PROJECTION"]))
         {
             $szJsFunctionName = "ProjSelectWSetJSAPIProjList";
             $szFunction = "$szJsFunctionName();\n";
             $aReturn[$szJsFunctionName] = $szFunction;
         }

         return $aReturn;
     }
     
    /**
     * GetHTMLHiddenVariables
     *
     * Return HTML hidden variables.
     */
    function GetHTMLHiddenVariables()
    {
        $aReturn = $this->moButton->GetHTMLHiddenVariables();
        
        if (is_object($this->moMapObject))
            $szProj = $this->moMapObject->oMap->getProjection();
        else
            $szProj = "init=EPSG:None";
        $szProj = substr( $szProj, 5 );
        $szVariable = "MAP_PROJECTION";
        $szValue = " <INPUT TYPE=HIDDEN NAME=$szVariable VALUE=\"\">";
        $aReturn[$szVariable] = $szValue;
        return $aReturn;
    }

    /**
     * ParseURL
     *
     * Look for the MAP_PROJECTION being different from the
     * current projection
     */
    function  ParseURL()
    {
        $oMap = &$this->moMapObject->oMap;

        $szOriginalProjection = trim( $oMap->getMetaData( "original_projection" ));
        if ($szOriginalProjection == "" && $oMap->getprojection() != "")
        {
            $oMap->setMetaData( "original_projection", $oMap->getprojection() );
        }
        $szOriginalProjection = trim( $oMap->getMetaData( "original_projection" ));

        $szProj = "";
        if ($this->isVarSet( "MAP_PROJECTION" ))
            $szProj = $this->getVar( "MAP_PROJECTION" );
        if ($szProj != "")
        {
            $szCurrentProjection = $this->moMapObject->oMap->getProjection();
            $szCurrentProjection = substr( $szCurrentProjection, 5 );
            if (strcasecmp( $szProj, $szCurrentProjection ) != 0)
            {
                // Check all layers. If no projection use the current one.
                for( $i=0; $i< $oMap->numlayers; $i++)
                {
                    $oLayer = $oMap->getLayer( $i );
                    if ($oLayer->getProjection() == "")
                    {
                        $oLayer->setProjection($szOriginalProjection);
                    }
                }
              
                $aTmpExtent = array($oMap->extent->minx,$oMap->extent->miny,
                                    $oMap->extent->maxx,$oMap->extent->maxy);
                $szTmpProjection = $oMap->getProjection();

                $oMap->setProjection($szTmpProjection, MS_TRUE);
                $oMap->setExtent( $aTmpExtent[0], $aTmpExtent[1],
                                  $aTmpExtent[2], $aTmpExtent[3] );

                $nScale = intval($oMap->scale);

                $oOldProj = ms_newProjectionObj($szTmpProjection);

                $nCenterX = ($oMap->extent->minx + $oMap->extent->maxx) / 2.0;
                $nCenterY = ($oMap->extent->miny + $oMap->extent->maxy) / 2.0;

                $oCenter = ms_newPointObj( );
                $oCenter->setXY( $nCenterX, $nCenterY );


                if (strncasecmp(trim($szProj), "AUTO", 4) ==0)
                {
                    $this->ReprojectAuto($szProj);
                    return;
                }
                
                //assuming it is an epsg
                $szNewProj = "init=".strtolower($szProj);

                $oNewProj = ms_newProjectionObj( $szNewProj );


                $oCenter->project( $oOldProj, $oNewProj );

                $oMap->setProjection( $szNewProj, MS_TRUE );

                $oMap->setExtent( $oCenter->x - 1, $oCenter->y - 1,
                                  $oCenter->x + 1, $oCenter->y + 1 );

                $oPixelPos = ms_newPointObj();
                $nWidth = $oMap->width;
                $nHeight = $oMap->height;
                $oPixelPos->setXY( $nWidth/2, $nHeight/2 );

                $oMap->zoomScale( $nScale, $oPixelPos, $nWidth, $nHeight, $oMap->extent );
            }
        }

        return true;
    }

    /**
     * DrawPublish
     *
     * Return the HTML code using the name in the map file and the
     * parameters of the CWC tag.
     */
    function DrawPublish()
    {
        if (!$this->mbVisible)
            return "<!-- Projection Selector popup widget hidden -->";

        $szResult = $this->moButton->DrawPublish();

        return $szResult;
    }
}
?>
