<?php
/**
 * CWC application
 *
 * @project     CWC
 * @revision    $Id: production_draw_legend.php,v 1.2 2004/06/05 13:38:26 pspencer Exp $
 * @purpose     This is a utility page that contains the code necessary to draw
 *              a legend for print production
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2001, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*****************************************************************************
 * $Log: production_draw_legend.php,v $
 * Revision 1.2  2004/06/05 13:38:26  pspencer
 * bug 440: removed session variables related to gd module, free type.
 *
 * Revision 1.1  2004/04/23 16:52:05  pspencer
 * moved production_draw_legend.php to PrintWidget/production_draw_legend.php
 *
 * Revision 1.17  2004/03/30 20:25:23  jlacroix
 * Don't always use the same layer name in legend.
 *
 * Revision 1.16  2004/03/30 18:25:01  jlacroix
 * Fix undefined variables notice (maptools bug 326)
 *
 * Revision 1.15  2003/10/27 20:46:19  sfournier
 * Overwrite main branch with 1.1 stuff
 *
 * Revision 1.12.2.2  2003/10/15 21:04:57  pspencer
 * fixes to work with Chameleon 1.1
 *
 * Revision 1.12.2.1  2003/08/05 15:08:33  zak
 * removed tests requiring wms layers and metadata in map
 * (bug 2161)
 *
 * Revision 1.12  2003/04/25 19:42:31  sacha
 * Initialize a var
 *
 * Revision 1.11  2003/04/23 17:29:41  pspencer
 * use GD version from session.
 *
 * Revision 1.10  2003/04/17 19:10:51  sacha
 * Change the preview to reflect the exact map size from the main page.
 *
 * Revision 1.9  2003/03/27 15:40:50  sacha
 * Use wms style when drawing legend icon
 *
 * Revision 1.8  2003/02/24 19:32:34  sacha
 * Don't display layer temp_points in legend. Also fixed a few potential bugs in add point
 *
 * Revision 1.7  2003/01/30 13:49:58  sacha
 * Changed the session management
 *
 * Revision 1.6  2003/01/14 19:49:29  sacha
 * Fix a parsing font file bug
 *
 * Revision 1.5  2003/01/09 18:32:09  bronsema
 * Added ability to delete some temp image files that get created for the legend
 *
 * Revision 1.4  2003/01/09 18:19:44  bronsema
 * Updated to draw icon properly after bug was fixed in mapscript of not
 * being able to use an image file as a symbol directly
 *
 * Revision 1.3  2003/01/08 16:43:12  bronsema
 * Updated code to remove the dot that was appearing on labels with no
 * symbol
 *
 * Revision 1.2  2003/01/07 20:25:08  bronsema
 * Updated function header comments to the doxygen style
 *
 * Revision 1.1  2003/01/07 18:33:45  bronsema
 * Added legend
 *
 *****************************************************************************/
/*****************************************************************************
 * define the necessary default values to be used in the function
 *****************************************************************************/
// define the gif sample directory
define( "GIF_SAMPLE", "http://www2.dmsolutions.ca/cgi-bin/gifsample/" );

// define the default gif draw icon
define( "DEFAULT_ICON", GIF_SAMPLE."20/15?color=CCCCCC&".
                                    "outline=000000&corner=1" );

// define the blank gif draw icon to use for the layer icon
define( "BLANK_ICON", GIF_SAMPLE."20/15?color=000000&symbolid=6&symbolsize=4");

$gdModule = "php_gd2";

// ensure that the gd module is loaded
if (PHP_OS == "WINNT" || PHP_OS == "WIN32")
{
    if (!extension_loaded("gd")) dl($gdModule.".dll");
}
else
{
    if (!extension_loaded("gd")) dl($gdModule.".so");
}

/**
 * Postcondition:  This function builds a list of layers to show in the legend.
 *
 * @param oMap object - Map file object.
 * @return array - List of layer indexes to draw in the legend or empty array.
 **/
function getLegendList( $o_map )
{
    // initialize arrays
    $an_legend_layers = array();
    $asz_group_names = array();

    // get array of all group names
    $asz_tmp_names = $o_map->getAllGroupNames();

    // create the array with group names as the key
    if (is_array($asz_tmp_names))
    {
        // loop to build array
        foreach($asz_tmp_names as $name)
        {
            // create array item and set to false
            $asz_group_names[$name] = 0;
        }
    }

    // get all layers in map
    $asz_all_layers = $o_map->getAllLayerNames();

    // check if any layers
    if (!is_array($asz_all_layers)) return array();

    // loop and build the array of layer names to include in the legend
    foreach ($asz_all_layers as $sz_layer)
    {
        // only add the the array if not in a group or 1st in group
        $o_layer = $o_map->getLayerByName($sz_layer);

        // check to see if in a group
        $sz_group = $o_layer->group;

        // add if necessary
        if ($sz_group != "")
        {
            // check to see if this group has been added
            if ($asz_group_names[$sz_group] == 0)
            {
                // group has not been added so add
                 if  ($o_layer->status == MS_ON)
                        array_push($an_legend_layers,$o_layer->index);

                 // set flag
                 $asz_group_names[$sz_group] = -1;
            }
        }
        else
        {
            // just add it
            if ($o_layer->status == MS_ON)
                        array_push($an_legend_layers,$o_layer->index);
        }
    }

    // reverse the array to show drawing order
    $an_reversed = array_reverse($an_legend_layers);

    // return the array
    return $an_reversed;

// end get_legend_list function
}

/**
 * Postcondition:  This function creates a legend image object according to the
 *                 legend settings in map file and passed parameters.
 *
 * @param o_map object - The map object to read.
 * @param an_layers array - Layer indexes to draw in the legend.
 * @param n_font_size integer - Font size to draw the legend at.
 * @param sz_fontfile string - Path and filename of the fonts.txt file.
 * @param sz_font_name string - Name of the font to use as listed in the
 *        fonts.txt file.
 * @param d_scale_factor double - Optional value to scale legend by.
 *
 * @return object - Handle for the image object.
*/
function getLegendImage($o_map, $an_layers, $n_font_size, $sz_font, $sz_font_name, 
                        $d_scale_factor = 1)
{
    // create default image on error
    $o_default_image = $o_map->prepareImage();

    // check if valid array
    if (!is_array($an_layers)) return $o_default_image;

    // get the map settings
    $n_keyspace_x = $d_scale_factor * ($o_map->legend->keyspacingx);
    $n_keyspace_y = $d_scale_factor * ($o_map->legend->keyspacingy);
    $n_keysize_x = $d_scale_factor * ($o_map->legend->keysizex);
    $n_keysize_y = $d_scale_factor * ($o_map->legend->keysizey);

    // apply scale factor to supplied font size
    $n_font_size = $d_scale_factor * ($n_font_size);

    // get the font file to use
    //$sz_font = $sz_font_file; //get_map_font($sz_font_name, $sz_fontfile);
    //echo $sz_font." ".$sz_font_name."<BR>";
    // initialize the variables
    $n_max_width = 0;
    $n_max_height = 0;
    $n_num_leg_lines = 0;

    // loop through each element of the array and process settings
    for ($i=0;$i<count($an_layers);$i++)
    {
        // create layer object
        $o_layer = $o_map->getlayer($an_layers[$i]);

        // check for valid layer returned
        if (!is_object($o_layer)) return $o_default_image;

        // if not set the check for group name
        $sz_layer_title = "";
        // check group name
        $sz_layer_title = $o_layer->group;

        // if not set then default to the layer name
        if ($sz_layer_title == "")
        {
            // set to layer name
            $sz_layer_title = $o_layer->name;
        }

        $szCurrentStyle = $o_layer->getmetadata("wms_style");
        $sz_layer_icon = "";


        if ($szCurrentStyle != "")
        {
            // In mapserver 4.1 the href of the legendurl is in 
            // wms_style_$szCurrentStyle_legendurl_href
            // in 4.0 it will be the fourth element (space separated) of the  
            // wms_style_$szCurrentStyle_legendurl
            $sz_layer_icon = $o_layer->getmetadata("wms_style_".$szCurrentStyle."_legendurl_href");

            if($sz_layer_icon == "")
            {
                $aStyle = explode(" ", 
             $o_layer->getmetadata("wms_style_".$szCurrentStyle."_legendurl"));

                if(is_array($aStyle) && isset($aStyle[3]) && $aStyle[3] != "")
                    $sz_layer_icon = $aStyle[3];
            }
        }

        // check if blank icon should be displayed
        if ($sz_layer_icon == "" || $o_layer->numclasses > 0 )
        {
            // set icon to the blank icon as set in globprefs.php
            $sz_layer_icon = BLANK_ICON;
        }

        // Read image to file
        if (!($fpIn = fopen($sz_layer_icon, "rb")))
        {
            trigger_error("ERROR: Could not access create icon $sz_layer_icon");
        }

        if (!($fpOut = fopen($_SESSION['gszTmpPath']."tmp_icon_".$i.".gif" ,
                                                                        "wb")))
        {
            trigger_error("ERROR: Could not open icon");
            //return FALSE;
        }

        while($fpIn && !feof($fpIn))
        {
            $line = fread($fpIn, 4096);
            fwrite($fpOut, $line);
        }

        fclose($fpIn);
        fclose($fpOut);

        // add the title to the title array
        $asz_legend_items[$i][0]["label"] = $sz_layer_title;
        $asz_legend_items[$i][0]["icon"] = $_SESSION['gszTmpPath'].
                                                        "tmp_icon_".$i.".gif";
        $asz_legend_items[$i][1] = array();
        $asz_legend_items[$i][2] = array();

        // increment the counter for the layer and then add the num classes
        $n_num_leg_lines += $o_layer->numclasses + 1;

        //echo "n_font_size: $n_font_size<br>";
        //echo "sz_font: $sz_font<br>";
        //echo "sz_layer_title: $sz_layer_title<br>";

        // process the label widths to determine the max
        $an_text_bbox = imagettfbbox ($n_font_size, 0, $sz_font,
                                      $sz_layer_title);

        //echo "an_text_bbox: ";
        //foreach( $a as $k => $v ) echo "[$k] = $v ";
        //echo "<br>";
        // get width and height of text in pixels
        $n_text_width = abs($an_text_bbox[2]-$an_text_bbox[0]);
        $n_text_height = abs($an_text_bbox[5]-$an_text_bbox[1]);

        $n_text_width = max( $n_text_width, strlen( $sz_layer_title ) * $n_font_size / 3 );

        // check if this text string is the widest or highest yet
        $n_max_width = max($n_text_width,$n_max_width);
        $n_max_height = max($n_text_height,$n_max_height);

        // Loop through the number of classes and check their titles too
        for ($j=0;$j<$o_layer->numclasses;$j++)
        {
            // get the class object
            $o_class = $o_layer->getclass($j);

            // add the class name to the array
            $asz_legend_items[$i][1][$j] = $o_class->name;

            // get the image object
            $asz_legend_items[$i][2][$j] =
                                    $o_class->createLegendIcon($n_keysize_x,
                                                               $n_keysize_y);

            // process the label widths to determine the max
            $an_text_bbox = imagettfbbox ($n_font_size, 0, $sz_font,
                                          $o_class->name);

            // get width and height of text in pixels
            $n_text_width = abs($an_text_bbox[2]-$an_text_bbox[0]);
            $n_text_height = abs($an_text_bbox[5]-$an_text_bbox[1]);

            // add extra check to fix any discrepancies caused by the
            // imagettfbbox() function
            $n_text_width = max( $n_text_width, strlen( $sz_layer_title ) *
                                 $n_font_size / 3 );

            // check if this text string is the widest or highest yet
            $n_max_width = max($n_text_width,$n_max_width);
            $n_max_height = max($n_text_height,$n_max_height);
        }
    }

    // take the maximum height to be the larger of the icon and text
    $n_max_height = max($n_max_height,$n_keysize_y);

    // calculate the overall width of the legend
    $n_legend_width = ($n_keyspace_x * 12) + $n_max_width + $n_keysize_x +10;

    // calculate the overall height
    $n_legend_height = $n_num_leg_lines * ($n_max_height + $n_keyspace_y);

    // record the cuurent map colour and size
    $tmp_map_colour_r = $o_map->imagecolor->red;
    $tmp_map_colour_g = $o_map->imagecolor->green;
    $tmp_map_colour_b = $o_map->imagecolor->blue;
    $tmp_map_width = $o_map->width+0; // leave the +0 to force a copy of the var
    $tmp_map_height = $o_map->height+0;

    // set the map colour to white
    $o_map->imagecolor->setRGB(255,255,255);

    // set the height and width
    $o_map->set("width",$n_legend_width);
    $o_map->set("height",$n_legend_height);

    // create canvas
    $o_img_canvas = $o_map->prepareImage();

    // return the values to previous
    $o_map->imagecolor->setRGB($tmp_map_colour_r,
                               $tmp_map_colour_g,
                               $tmp_map_colour_b);
    $o_map->set("width",$tmp_map_width);
    $o_map->set("height",$tmp_map_height);

    // create annotation layer
    $o_anno_layer = get_temp_anno_layer($o_map, $n_font_size, $sz_font_name);

    // create temp point layer to draw the icons
    $o_point_layer = get_temp_point_layer($o_map);

    // create class object for icon
    $o_point_class = $o_point_layer->getclass(0);

    // create style object
    $o_point_style = ms_newStyleObj( $o_point_class );

    // initialize x & y poition variables
    $n_pix_x = $n_keyspace_x;
    $n_pix_y = $n_keyspace_y;

    // loop to draw each legend record and paste onto canvas image
    for ($i=0;$i<count($asz_legend_items);$i++)
    {

        // show the image
        print_icon($o_map, $o_point_layer, $o_point_style, $o_img_canvas,
                   $asz_legend_items[$i][0]["icon"], $n_pix_x,
                   $n_pix_y, $d_scale_factor);

        // delete the temp icon
        if (file_exists($asz_legend_items[$i][0]["icon"]))
                        unlink($asz_legend_items[$i][0]["icon"]);

        // display layer title as point annotation
        print_label($o_map, $o_anno_layer, $o_img_canvas,
                    $asz_legend_items[$i][0]["label"], $n_pix_x + (22 * $d_scale_factor),
                    $n_pix_y);

        // init counter
        $j=0;

        // loop through classes
        foreach($asz_legend_items[$i][1] as $class)
        {
            // set the y position
            $n_pix_y += $n_max_height + 2;

            // print the icon
            $o_img_canvas->pasteImage( $asz_legend_items[$i][2][$j], -1,
                                       ($n_pix_x * 3) + 14 , $n_pix_y );

            // print the label
            print_label($o_map, $o_anno_layer, $o_img_canvas,
                        $class, ($n_pix_x * 4) + 14 + $n_keysize_x, $n_pix_y);

            // increment the counter
            $j++;
        }

        // update the y position variable
        $n_pix_y += $n_max_height + $n_keyspace_y;

    }

    // delete temp layers
    del_temp_anno_layer($o_map);
    del_temp_point_layer($o_map);

    // return the image object
    return $o_img_canvas;

// end get_legend_image function
}

/**
 * Function to print labels on given image using map object and temp annotation
 * layer.
 *
 * @param o_map object - The map object to use.
 * @param o_anno_layer object - Annotation layer object to use.
 * @param o_image object - Image to paste the label onto.
 * @param sz_label string - Label to print.
 * @param n_x_pos integer - x poisiton of label.
 * @param n_y_pos integer - y poisiton of label.
 * @return void.
 **/
function print_label($o_map, $o_anno_layer, $o_image, $sz_label,
                     $n_x_pos, $n_y_pos)
{
    // create new point object
    $o_point = ms_newPointObj();

    //position
    $o_point->setXY($n_x_pos,$n_y_pos);

    // draw the point onto the map
    $o_point->draw($o_map, $o_anno_layer, $o_image, 0,$sz_label);

// end print_label function
}

/**
 * Function to print icon on given image using map object and temp point
 * layer
 *
 * @param o_map object - The map object to use.
 * @param o_point_layer object - Annotation layer object to use.
 * @param o_point_style object - Style object to use for styling the class.
 * @param o_image object - Image to paste the label onto.
 * @param sz_icon string - Path and filename of the icon to paste.
 * @param n_x_pos integer - x poisiton of label.
 * @param n_y_pos integer - y poisiton of label.
 * @param d_scale_factor double - Scale factor to apply to size.
 * @return void.
 **/
function print_icon($o_map, $o_point_layer, $o_point_style, $o_image, $sz_icon,
                     $n_x_pos, $n_y_pos, $d_scale_factor)
{
    // create style object
    $o_point_style->color->setRGB( 0,0,0 );
    $o_point_style->set("maxsize"      , 30    );
    $o_point_style->set("size"         , 15    );
    $o_point_style->set("symbolname"   , $sz_icon );

    // create new point object
    $o_point = ms_newPointObj();

    //position
    $o_point->setXY($n_x_pos + (10 * $d_scale_factor),
                    $n_y_pos + (4 * $d_scale_factor));

    // draw the point onto the map
    $nStatus = $o_point->draw($o_map, $o_point_layer, $o_image, 0,"");

    // free the resources
    //$o_point->free;

// end print_icon function
}

/**
 * Function to create temp annotation layer and class and returns layer object.
 *
 * @param o_map object - The map object to read.
 * @param n_font_size integer - Font size to draw the legend at.
 * @param sz_msfont string - Name of the font to use as listed in the fonts.txt
 *                           file.
 * @return object - Handle for the layer object.
 **/
function get_temp_anno_layer($o_map, $n_font_size, $sz_msfont)
{
    // set flag
    $b_layer_exists = false;

    // loop through each layer
    for ($i=0;$i<$o_map->numlayers;$i++)
    {
        // create layer object
        $o_anno_layer = $o_map->getlayer($i);

        // check name
        if ($o_anno_layer->name == "DMSG_annotation_layer")
        {
            // flag as found
            $b_layer_exists = true;
            break;
        }

    }

    // create layer if not found
    if (!$b_layer_exists) $o_anno_layer = ms_newLayerObj($o_map);

    // set the properties of the layer
    $o_anno_layer->set("name", "DMSG_annotation_layer");
    $o_anno_layer->set("type", MS_LAYER_ANNOTATION);
    $o_anno_layer->set("status", 1);
    $o_anno_layer->set("transform", MS_FALSE);
    $o_anno_layer->set("labelitem", "annotation");
    $o_anno_layer->set("labelcache", MS_OFF);

    // set flag
    $b_class_exists = false;

    // loop through the classes and check if anno class exists
    for ($i=0;$i<$o_anno_layer->numclasses;$i++)
    {
        // create class object
        $o_anno_class = $o_anno_layer->getclass($i);

        // check the name
        if ($o_anno_class->name == "DMSG_anno_class")
        {
            // flag as found and exit loop
            $b_class_exists = true;
            break;
        }
    }

    // create class object if necessary
    if (!$b_class_exists) $o_anno_class = ms_newClassObj($o_anno_layer);

    // create style object
    $o_style = ms_newStyleObj( $o_anno_class );

    // set the class colour
    $o_style->color->setRGB( -1,-1,-1 );

    // set class properties
    //$o_anno_class->set("color", $n_class_colour    );
    $o_anno_class->set("name" , "DMSG_anno_class"  );

    // set the label properties
    $o_anno_class->label->color->setRGB( 0,0,0 );
    //$o_anno_class->label->set("color"        , $n_class_colour    );
    //$o_anno_class->label->set("size"         , $n_font_size       );
    $o_anno_class->label->set("size"         , 8       );
    $o_anno_class->label->set("font"         , $sz_msfont         );
    $o_anno_class->label->set("type"         , MS_TRUETYPE        );
    $o_anno_class->label->set("position"     , MS_LR              );
    $o_anno_class->label->set("partials"     , MS_FALSE           );

    // return the layer object
    return $o_anno_layer;

// end get_temp_anno_layer function
}

/**
 * Function to create temp point layer and class and returns layer object
 *
 * @param o_map object - The map object to read.
 * @return object - Handle for the layer object.
 **/
function get_temp_point_layer($o_map)
{
    // set flag
    $b_layer_exists = false;

    // loop through each layer
    for ($i=0;$i<$o_map->numlayers;$i++)
    {
        // create layer object
        $o_point_layer = $o_map->getlayer($i);

        // check name
        if ($o_point_layer->name == "DMSG_point_layer")
        {
            // flag as found
            $b_layer_exists = true;
            break;
        }

    }

    // create layer if not found
    if (!$b_layer_exists) $o_point_layer = ms_newLayerObj($o_map);

    // set the properties of the layer
    $o_point_layer->set("name", "DMSG_point_layer");
    $o_point_layer->set("type", MS_LAYER_POINT);
    $o_point_layer->set("status", 1);
    $o_point_layer->set("transform", MS_FALSE);
    $o_point_layer->set("labelitem", "point");
    $o_point_layer->set("labelcache", MS_OFF);

    // set flag
    $b_class_exists = false;

    // loop through the classes and check if point class exists
    for ($i=0;$i<$o_point_layer->numclasses;$i++)
    {
        // create class object
        $o_point_class = $o_point_layer->getclass($i);

        // check the name
        if ($o_point_class->name == "DMSG_point_class")
        {
            // flag as found and exit loop
            $b_class_exists = true;
            break;
        }
    }

    // create class object if necessary
    if (!$b_class_exists) $o_point_class = ms_newClassObj($o_point_layer);

    // create style object
    //$o_style = ms_newStyleObj( $o_point_class );

    // set the class colour
    //$o_style->color->setRGB( 0,0,0 );

    // set class properties
    $o_point_class->set("name" , "DMSG_point_class"  );

    // return the layer object
    return $o_point_layer;

// end get_temp_point_layer function
}

/**
 * Function to set the status if the temp annotation layer to MS_DELETE.
 *
 * @param o_map object - The map object to read.
 * @return void.
 **/
function del_temp_anno_layer($o_map)
{
    //  create the layer object
    $o_layer = $o_map->getlayerbyname("DMSG_annotation_layer");

    // set the status
    $o_layer->set("status",MS_DELETE);

// end del_temp_anno_layer function
}

/**
 * Function to set the status if the temp point layer to MS_DELETE.
 *
 * @param o_map object - The map object to read.
 * @return void.
 **/
function del_temp_point_layer($o_map)
{
    //  create the layer object
    $o_layer = $o_map->getlayerbyname("DMSG_point_layer");

    // set the status
    $o_layer->set("status",MS_DELETE);

// end del_temp_point_layer function
}

/**
 * Function to get the path and filename of the font as it corresponds to
 * the name of the mapfile font given.
 *
 * @param sz_font_name string - Name of the map file font to check.
 * @param sz_fontfile string - Path to the fonts.txt file.
 * @return mixed - Path and file name to the font file to use or false.
 **/
function get_map_font( $sz_font_name, $sz_fontfile )
{
    // get the font to use
    $asz_font_list = ReadStringList( $sz_fontfile );

    // check if results
    if (!is_array($asz_font_list)) return false;

    // loop through the array and read results
    foreach($asz_font_list as $sz_list)
    {
        $szFont = preg_replace("/\s+/", "|", trim($sz_list));
        $asz_fonts = explode("|", $szFont);

    if (is_array($asz_fonts) && count($asz_fonts)==2 &&
        stristr($sz_font_name, $asz_fonts[0]))
    {
            // return font
            return dirname( $sz_fontfile )."/".trim($asz_fonts[1]);
    }
    }
    return "";
// end get_map_font function
}

/**
 * Reads the given textfile to an array.
 *
 * @param szfilename string - The textfile to read.
 * @return mixed - Array or false if unsuccessful.
 **/
function ReadStringList($szfilename)
{
    // define array variable
    $list = array();

    // attempt to open the file
    if (!($fp = fopen($szfilename, "r")))
    {
        // error occurred
        trigger_error("ReadStringList(): Could not open $szfilename");

        // return failure
        return FALSE;
    }

    // file is open so parse each entry
    while($fp && !feof($fp))
    {
        // get next line
        $line = fgets($fp, 4096);

        // trim() strips off whitespaces and \r and \n
        $line = trim($line);
        if (strlen($line) > 0)
            array_push($list, $line);
    }

    // close the file
    fclose($fp);

    // return the array
    return $list;
}

/**
 * Write the given array to the specified textfile.
 *
 * @param szfilename string - The textfile to write to.
 * @param aszlist array - the array write.
 * @return boolean - Flag to indicate success.
 **/
function WriteStringList($szfilename, $aszlist)
{
    // open the file
    if (!($fp = fopen($szfilename, "w")))
    {
        // open failed so give error
        trigger_error("WriteStringList(): Could not open $szfilename");

        // return failure
        return FALSE;
    }

    // loop through the array an dwrite to file
    foreach ($aszlist as $line)
    {
        // write entry to file
        fputs($fp, $line."\n");
    }

    // close the file
    fclose($fp);

    // return success
    return TRUE;
}

/**
 * Draw the legend for the map onto an image at a given position
 * it also draws a neatline around the legend.
 *
 * @param oMap object - The map to get the legend.
 * @param oImage object - The image to draw the legend onto.
 * @param oLegendImage object - The legend image to draw.
 * @param nX integer - The left edge of the legend.
 * @param nY integer - The top edge of the legend.
 * @return mixed - The image object passed in.
 **/
function draw_legend( $oMap, $oImage, $oLegendImage , $nX, $nY)
{
    // get the legend object
    $oLegend = $oMap->legend;

     // set the outline colour, used again for the neatline.
    //$nColor = $oMap->addColor( 0, 0, 0 );

    // set status and transparency
    $oLegend->set("status",MS_ON); //was MS_EMBED, now MS_ON
    $oLegend->set("transparent",MS_FALSE);

    //get the size of the map and legend for validation
    $nMapWidth = $oImage->width;
    $nMapHeight = $oImage->height;
    $nLegendWidth = $oLegendImage->width;
    $nLegendHeight = $oLegendImage->height;

    //legend must fit in map size
    // if ($nLegendWidth > $nMapWidth || $nLegendHeight > $nMapHeight)
    //    return false;

    //top & left validation
    if ($nX < 0) $nX = 0;
    if ($nY < 0) $nY = 0;

    //bottom & right validation
    if ( ($nX + $nLegendWidth) > $nMapWidth )
        $nX = $nMapWidth - $nLegendWidth;
    if ( ($nY + $nLegendHeight) > $nMapHeight )
        $nY = $nMapHeight - $nLegendHeight;

    //paste the legend onto the image at the desired location with
    //no transparent colour.
    $oImage->pasteImage( $oLegendImage, -1, $nX, $nY );

    //draw a neatline around the legend
    $oLayer = ms_newLayerObj($oMap);
    $oLayer->set("name", "temp_layer");
    $oLayer->set("type", MS_LAYER_LINE);
    $oLayer->set("status", MS_OFF);
    $oLayer->set("transform", MS_FALSE);
    $oClass = ms_newClassObj($oLayer);

    // set the class colour
    //$oClass->set("color", $nColor);
    // create style object
    $o_style = ms_newStyleObj( $oClass );

    // set the class colour
    $o_style->color->setRGB( 0,0,0 );

    // create new rectangle object
    $oRect = ms_newRectObj();

    // set to dimensions of the legend
    $oRect->setExtent($nX, $nY,
                      $nX + $nLegendWidth - 1,
                      $nY + $nLegendHeight-1);

    // draw the rectangle onto the map
    $oRect->draw($oMap, $oLayer, $oImage, 0, "");

    // delete temp layer
    $oLayer->set("status", MS_DELETE);

// end draw_legend function
}

?>
