<?php

$bFoundAttributes = false;

/*

   Processing Logic:
  
   Based on layer connection type and layer type.  Only works if layer type
   is one of:
   
   MS_LAYER_POINT
   MS_LAYER_LINE
   MS_LAYER_POLYGON
   
   We find attributes by doing a layer->open() and getting the first shapeObj
   then looking at the shapeObj's values array.
   
   Special case for WMS and WFS connection types, we can get the information from
   the WFS server (in the WMS case, we try a DescribeLayer request to find out
   about associated WFS)

   layer types

   MS_LAYER_POINT
   MS_LAYER_LINE
   MS_LAYER_POLYGON
   MS_LAYER_RASTER
   MS_LAYER_ANNOTATION
   MS_LAYER_QUERY
   MS_LAYER_CIRCLE
   MS_LAYER_GRATICULE

   connection types
   
   MS_INLINE
   MS_SHAPEFILE
   MS_TILED_SHAPEFILE
   MS_SDE
   MS_OGR
   MS_TILED_OGR
   MS_POSTGIS
   MS_WMS
   MS_ORACLESPATIAL
   MS_WFS
   MS_GRATICULE
   MS_MYGIS
 */

function GetLayerAttributes( $oLayer )
{
    $aszAttributes = array();
    
    if (//$oLayer->type == MS_LAYER_RASTER &&
        $oLayer->connectiontype == MS_WMS)
    {
        //get the layer name from metadata if possible to avoid the problems
        //with context layers being renamed
        $szLayerName = $oLayer->getMetaData( "wms_name" );
        if ($szLayerName == "")
            $szLayerName = $oLayer->name;
            
        // get WFS the connection string
        $szWFSConnection = getWFSConnection( $oLayer->connection, $szLayerName );

        // check for false (meaning no associated WFS)
        if ( $szWFSConnection != "" )
        {
            $szWFSConnection = fixWFSURL( $szWFSConnection, $szLayerName );

            // get the attribute types
            $aszAttributes['name'] = $oLayer->name;
            $aszAttributes['index'] = $oLayer->index;
            $aszAttributes['onlineresource'] = $szWFSConnection;
            $aszAttributes['minx'] = "";
            $aszAttributes['miny'] = "";
            $aszAttributes['maxx'] = "";
            $aszAttributes['maxy'] = "";
            $aszAttributes['fields'] = getAttributeTypes( $szWFSConnection.
                        'request=describefeaturetype', $szLayerName );
        }
    }
    elseif ($oLayer->type == MS_LAYER_POINT ||
        $oLayer->type == MS_LAYER_LINE ||
        $oLayer->type == MS_LAYER_POLYGON )
    {
    
        // process according to layer type
        switch( $oLayer->connectiontype )
        {
            case MS_WFS:
                //get the layer name from metadata if possible to avoid the problems
                //with context layers being renamed
                $szLayerName = $oLayer->getMetaData( "wms_name" );
                if ($szLayerName == "")
                    $szLayerName = $oLayer->name;
                    
                // record the connection string
                $szWFSConnection = $oLayer->connection;
                    // check the the WFS online resource contains the necessary params
                $szWFSConnection = fixWFSURL( $szWFSConnection, $szLayerName );
    
                // get the attribute types
                $aszAttributes['name'] = $oLayer->name;
                $aszAttributes['index'] = $oLayer->index;
                $aszAttributes['onlineresource'] = $szWFSConnection;
                $aszAttributes['minx'] = "";
                $aszAttributes['miny'] = "";
                $aszAttributes['maxx'] = "";
                $aszAttributes['maxy'] = "";
                $aszAttributes['fields'] = getAttributeTypes( $szWFSConnection.
                                'request=describefeaturetype', $szLayerName );
                
    
                break;
             
    
            default:
                // open the layer
                $oLayer->open();
                
                //handle tiled shapefiles correctly, otherwise they don't get
                //extents or attributes
                if ($oLayer->connectiontype == MS_TILED_SHAPEFILE ||
                    $oLayer->tileindex != "")
                {
                    //first tile is sufficient, assume that all attributes are the same
                    //in every tile (which is reasonable)
                    $nTile = 0;
                }
                else
                {
                    //use -1 in getShape for non-tiled sources
                    $nTile = -1;
                }
                // read the first line of the datasource to get
                // name, index, and fields
                $oShape = $oLayer->getShape($nTile, 0 );
                $aszAttributes['name'] = $oLayer->name;
                $aszAttributes['index'] = $oLayer->index;
                $oRect = $oShape->bounds;
                $aszAttributes['minx'] = $oRect->minx;
                $aszAttributes['miny'] = $oRect->miny;
                $aszAttributes['maxx'] = $oRect->maxx;
                $aszAttributes['maxy'] = $oRect->maxy;
                
                foreach($oShape->values as $key => $value)
                {
                    $aszAttributes['fields'][$key] = "string";
                }
                break;           
        }
    }
    
    return $aszAttributes;
}

/*
 * return the SRS value (ESPG:xxxx) for a given layer in a WFS server
 * by grabbing the capabilities of the server, then looking for any
 * SRS tag (since all layers are in the same SRS :)
 */
function getWFSSRS( $szXmlUrl )
{
    // read the xml datasource
    if (function_exists( 'file_get_contents' ))
    {
        $data = file_get_contents( $szXmlUrl );
    }
    else
    {
        $data = implode( '', file($szXmlUrl) );
    }
    
    if ( stristr( $data, '<SERVICEEXCEPTIONREPORT' ) !== false )
        return "";
    
    // parse
    $parser = xml_parser_create();
    xml_parser_set_option($parser,XML_OPTION_CASE_FOLDING,0);
    xml_parser_set_option($parser,XML_OPTION_SKIP_WHITE,1);
    xml_parse_into_struct($parser,$data,$values,$tags);
    xml_parser_free($parser);

    return $values[$tags['SRS'][0]]['value'];
}

/**
 * getAttributeTypes()
 *
 * Postcondition:  This function processes the given XML file returns an array
 *                 of attribute names and types.
 *
 * @param $szXmlUrl string - URL to the XML document to parse.
 * @return mixed - Array of results or a string of the server error results.
 * @desc Reads XML datasource and returns array of att->type pairs.
 */
function getAttributeTypes( $szXmlUrl, $szLayerName ) 
{
    // int vars
    $data = "";
    $axReturn = "";
    
    // read the xml datasource
    $aszFile = file( $szXmlUrl );

    // combine results
    if ( is_array( $aszFile ) )
    {
        foreach( $aszFile as $line )
            $data .= $line;
    }

    // check to see if an error was generated
    if ( !is_array( $aszFile ) || 
         strpos( strtoupper( $data ), '<SERVICEEXCEPTIONREPORT' ) !== false )
        return array();
    
    // parse
    $parser = xml_parser_create();
    xml_parser_set_option($parser,XML_OPTION_CASE_FOLDING,0);
    xml_parser_set_option($parser,XML_OPTION_SKIP_WHITE,1);
    xml_parse_into_struct($parser,$data,$values,$tags);
    xml_parser_free($parser);

    // loop through the structures and process the SEQUENCE
    foreach ($tags as $key=>$val) 
    {
        // process ROWSET
        if (strtoupper( $key ) == "SEQUENCE") 
        {
            $ranges = $val;
            // each contiguous pair of array entries are the 
            // lower and upper range for each SEQUENCE definition
            $nCount = count($ranges);          
            for ($i=0; $i < $nCount; $i+=2) 
            {
                $offset = $ranges[$i] + 1;
                $len = $ranges[$i + 1] - $offset;
                $axReturn = parseElements( array_slice($values, $offset, $len) );
            }
        } 
        else 
        {
            // otherwise skip
            continue;
        }
    }
    
    // return
    return $axReturn;
    
// end getAttributeTypes() function    
}

/**
 * parseElements()
 *
 * Postcondition:  This function parses the given array returns an array of 
 *                 attribute->type pairs.
 *
 * @param axValues Array - Mixed array of XML tags to process.
 * @return array - Array of att->type pairs if successful or empty array if not.
 * @desc Parses array and returns array of att->type pairs.
 */
function parseElements( $axValues ) 
{   
    // init vars
    $aszReturn = array();
    
    // loop through each item of the array
    $nCount = count( $axValues );
    for ( $i=0; $i < $nCount; $i++ )
    {
        // only process "name" and "type" values
        if ( strtoupper( $axValues[$i]["tag"] ) == "ELEMENT" )
        {
            // add to array
            if ( isset(  $axValues[$i]["attributes"]["name"] ) && 
                 $axValues[$i]["attributes"]["type"] )
            {
               $aszReturn[$axValues[$i]["attributes"]["name"]] = 
                                            $axValues[$i]["attributes"]["type"];
            }
            
            // check for the layer type
            $szLayerType = '';
            if ( isset( $axValues[$i]["attributes"]["ref"] ) )
                $szLayerType = strtoupper( $axValues[$i]["attributes"]["ref"] );
            elseif ( isset( $axValues[$i]["attributes"]["type"] ) && 
                     strtoupper( substr( $axValues[$i]["attributes"]["type"], 0, 4 ) ) == 'GML:' )
                $szLayerType = strtoupper( $axValues[$i]["attributes"]["type"] );
            
            // check for the type
            if (  strlen( $szLayerType ) > 0 )
            {
                // check if poly
                if ( strpos( $szLayerType, 'POLYGON' ) !== false )
                    $aszReturn['LayerType'] = 'MS_LAYER_POLYGON';
                elseif ( strpos( $szLayerType, 'BOX' ) !== false )
                    $aszReturn['LayerType'] = 'MS_LAYER_POLYGON';                    
                elseif ( strpos( $szLayerType, 'LINE' ) !== false )
                    $aszReturn['LayerType'] = 'MS_LAYER_LINE';
                elseif ( strpos( $szLayerType, 'POINT' ) !== false )
                    $aszReturn['LayerType'] = 'MS_LAYER_POINT';                   
            }
        }
    }
    
    // return array
    return $aszReturn;
    
// end parseElements() function
}


/**
 * getWFSConnection()
 *
 * Postscript:  This function reads the given WMS connection and returns the WFS 
 *              online resource.
 *
 * @param $szWMSConnection string - WMS connection string.
 * @return string - WFS online resource if successful, false if failed.
 * @desc Parses given connection string and returns WFS online resource.
 */
function getWFSConnection( $szWMSConnection, $szLayerName )
{
    // get the url to perform a describelayer on
    $szUpperCase = strtoupper( $szWMSConnection );

    // check last char for ? or &
    if ( substr( $szWMSConnection, -1 ) != "?" &&
         substr( $szWMSConnection, -1 ) != "&" )
    {
        // check for ?
        if ( strpos( $szWMSConnection, "?" ) === false )
        {
            $szWMSConnection .= "?";
        }
        else
            $szWMSConnection .= "&";
    }

    // check for and add VERSION
    if ( strpos( $szUpperCase, "VERSION=" ) === false )
        $szWMSConnection .= "VERSION=1.1.0&";

    // check for and add SERVICE
    if ( strpos( $szUpperCase, "SERVICE=" ) === false )
        $szWMSConnection .= "SERVICE=WMS&";

    // add LAYERS
    if ( strpos( $szUpperCase, "LAYERS=" ) ===  false )
        $szWMSConnection .= "LAYERS=".$szLayerName."&";

    // add REQUEST
    $szWMSConnection .= "REQUEST=DESCRIBELAYER";

    // perform a describe layer
    return getWFSResource( $szWMSConnection );
    
// end getWFSConnection function
}

/**
 * getWFSResource()
 *
 * Postscript:  This function reads the given url and returns the WFS online
 *              resource.
 *
 * @param $szURL string - URL to the GML source.
 * @return string - WFS online resource if successful, false if failed.
 * @desc Parses given url and returns WFS online resource.
 */
function getWFSResource( $szURL )
{
    // init vars
    $data = "";
    $szReturn = "";
    
    // read the xml datasource
    $aszFile = file( $szURL );

    // check to see if an error was generated
    if ( !is_array( $aszFile ) ||
         strtoupper( substr( $aszFile[0], 0, 23 ) ) == "<SERVICEEXCEPTIONREPORT" )
    {
        return false;
    }

    // process url
    foreach( $aszFile as $line )
    {
        // check each line until "<LayerDescription" is found
        if ( strpos( strtoupper( $line ) ,
                                "<LAYERDESCRIPTION" ) !== false )
        {
            // loop through and extract the wfs param
            $nCount = strlen( $line );
            $bOpen = false;
            for ( $i=0; $i<$nCount; $i++ )
            {
                // check for the closing "
                if ( $bOpen )
                {
                    // look for the closing "
                    if ( substr( $line, $i, 1 ) != "\"" )
                        $szReturn .= substr( $line, $i, 1 );
                    else
                        break;
                }
                else
                {
                    // check for first occurance of " wfs="
                    if ( substr( strtoupper( $line ), $i, 5 )
                                                == " WFS=" )
                    {
                        // flag as open and advance
                        $bOpen = true;
                        $i += 5;
                    }
                }
            }
        }
    }

    // return true
    return $szReturn;

// end getWFSResource function
}

/**
 * fixWFSURL()
 *
 * Postscript:  This function adds any missing WFS parameters to a WFS connection.
 *
 * @param $szConnection string - Connection string.
 * @param $szLayerName string -  Layer name to add.
 * @return string - The updated url.
 * @desc Checks given WFS connection and adds the necessary parameters.
 */
function fixWFSURL( $szConnection, $szLayerName )
{
    // convert the url to uppercase
    $szUpperCase = strtoupper( $szConnection );
    
    // check last char for ? or &
    if ( substr( $szConnection, -1 ) != "?" &&
         substr( $szConnection, -1 ) != "&" )
    {
        // check for ?
        if ( strpos( $szConnection, "?" ) === false )
        {
            $szConnection .= "?";
        }
        else
            $szConnection .= "&";
    }    
    
    // check for and add VERSION
    if ( strpos( $szUpperCase, "VERSION=" ) === false )
        $szConnection .= "VERSION=1.0.0&";

    // check for and add SERVICE
    if ( strpos( $szUpperCase, "SERVICE=" ) === false )
        $szConnection .= "SERVICE=WFS&";

    // add TYPENAME
    if ( strpos( $szUpperCase, "TYPENAME=" ) ===  false )
        $szConnection .= "TYPENAME=".$szLayerName."&";
    
    // return
    return $szConnection;

// end fixWFSURL function
}

?>