<?php
/**
 * Geolink Widget Class
 *
 * @project     CWC2
 * @revision    $Id: Geolink.widget.php,v 1.2 2004/04/24 21:58:23 pspencer Exp $
 * @purpose     Display a Geolink Widget
 * @author      William A. Bronsema, C.E.T. (dev@dmsolutions.ca)
 *              DM Solutions Group 
 * @copyright
 * <b>Copyright (c) 2003, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

include_once(dirname(__FILE__)."/../Widget.php");
include_once(dirname(__FILE__)."/../Label.php");

/**
 * This widget provides a selection of layers to geocode based on selected 
 * inputs.
 */
class GeolinkWidget extends CWCWidget
{
    var $moLabel;

    /**
     * build a new quick zoom widget
     */
    function GeolinkWidget()
    {
        //$this->mszLanguageResource = str_replace("\\","/",dirname(__FILE__))."/QuickZoom.dbf";

        // invoke constructor of parent
        parent::CWCWidget();

        $this->mnPriority = PRIORITY_HIGH;

        // set the description for this widget
        $this->szWidgetDescription = <<<EOT
The Geolink widget allows the user to add geolinked data to the current map
by selecting the desired subregion from a series of drop-down menus.
EOT;

        $this->moLabel = new CWCLabel( $this );

        //  create custom parameters for the widget
        $this->maAttributes["GDAS"] = new StringAttribute( "GDAS", true );
        $this->maAttributes["GEOLINKINGSERVICE"] = new StringAttribute( "GEOLINKINGSERVICE", true );
    }
    
    /**
     * GetJavascriptOnLoadFunctions
     *
     */
    function GetJavascriptOnLoadFunctions()
    {
        // after the page has loaded then populate the global javascript 
        // arrays with the correct dropdown information
        $szJsFunctionName = "globalTagValueArrays()";
        $aReturn[$szJsFunctionName] = $szJsFunctionName; 
        return $aReturn;
    }   
    
    /**
     * GetJavascriptVariables
     */    
    function GetJavascriptVariables()
    {
        $aReturn = array();
        $szVariableName = "gaDropDownValues";
        $szVariable = "var $szVariableName = new Array();";
        $aReturn[$szVariableName] = $szVariable;

        return $aReturn;
    }

    /**
     * get the javascript functions for this widget
     */
    function GetJavascriptFunctions()
    {
        parent::GetJavascriptFunctions();
        
        $aReturn = array();
        
/* ============================================================================
 * Add the framework dropdown change functions
 * ========================================================================= */
        $szJsFunctionName = "changeFramework";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the Framework value is changed.
 */
function {$szJsFunctionName}(obj)
{
    // set the hidden textbox value
    {$this->mszHTMLForm}.SEL_GEOLINK_FRAME.value = obj.options[obj.selectedIndex].text;
    
    // clear the previous list
    {$this->mszHTMLForm}.DATASET_DD.length = 0;
    
    // create the dataset option list
    // loop and match the selected attribute name
    for( i=0; i<gaDropDownValues.length; i++ )
    {
        // check for a match on the framework
        if ( gaDropDownValues[i][0] == obj.options[obj.selectedIndex].text )
        {
            // loop through all available datasets
            for( j=0; j<gaDropDownValues[i][1].length; j++ )
            {
                // add to option list
                opt = new Option( gaDropDownValues[i][1][j][0], gaDropDownValues[i][1][j][0], true, true );
                eval("{$this->mszHTMLForm}.DATASET_DD.options["+ j +"]=opt;");
            }
            
            // found so exit loop
            break;
        }
    }
 
    // loop and select the correct item
    bSelectedSrc = 0;
    for ( i=0;i<{$this->mszHTMLForm}.DATASET_DD.length;i++ )
    {
        // check for a match
        if ( {$this->mszHTMLForm}.DATASET_DD.options[i].value == {$this->mszHTMLForm}.SEL_GEOLINK_DS.value )
        {    
            {$this->mszHTMLForm}.DATASET_DD.options[i].selected = true;
            changeDataset( {$this->mszHTMLForm}.DATASET_DD );
            bSelectedSrc = 1;
            break;
        }
    }
    
    // select the first item if nothing selected
    if ( bSelectedSrc == 0 && {$this->mszHTMLForm}.DATASET_DD.length > 0 )
    {
        {$this->mszHTMLForm}.DATASET_DD.options[0].selected = true;
        changeDataset( {$this->mszHTMLForm}.DATASET_DD );
    }    
    
    return;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;
        
/* ============================================================================
 * Add the dataset dropdown change functions
 * ========================================================================= */        
        $szJsFunctionName = "changeDataset";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the dataset value is changed.
 */
function {$szJsFunctionName}(obj)
{
    // update the hidden form variable
    {$this->mszHTMLForm}.SEL_GEOLINK_DS.value = obj.options[obj.selectedIndex].text;
    
    // clear the previous list
    {$this->mszHTMLForm}.GEOLINK_ATT.length = 0;
    
    // create the source option list
    // loop and match the selected dataset name
    for( i=0; i<gaDropDownValues.length; i++ )
    {
        // check for a match on the framework
        oAttObj = {$this->mszHTMLForm}.GEOLINK_FRAME;
        if( gaDropDownValues[i][0] == oAttObj.options[oAttObj.selectedIndex].text )
        {
            // loop through all available datasets
            for( j=0; j<gaDropDownValues[i][1].length; j++ )
            {
                // check for match on dataset
                if( gaDropDownValues[i][1][j][0] == obj.options[obj.selectedIndex].text )
                {
                    // loop for each attribute and add to option
                    for ( k=0;k<gaDropDownValues[i][1][j][1].length;k++ )
                    {
                        opt = new Option( gaDropDownValues[i][1][j][1][k], gaDropDownValues[i][1][j][1][k], true, true );
                        eval("{$this->mszHTMLForm}.GEOLINK_ATT.options["+ k +"]=opt;");
                    }
                    
                    // found so exit loop
                    break;
                }
            }
            
            // found so exit loop
            break;
        }
    }
 
    // loop and select the correct item
    bSelectedFld = 0;
    for ( i=0;i<{$this->mszHTMLForm}.GEOLINK_ATT.length;i++ )
    {
        // check for a match
        if ( {$this->mszHTMLForm}.GEOLINK_ATT.options[i].value == {$this->mszHTMLForm}.SEL_GEOLINK_ATT.value )
        {    
            {$this->mszHTMLForm}.GEOLINK_ATT.options[i].selected = true;
            changeAttribute( {$this->mszHTMLForm}.GEOLINK_ATT );
            bSelectedFld = 1;
            break;
        }
    }
    
    // select the first item if nothing selected
    if ( bSelectedFld == 0 && {$this->mszHTMLForm}.GEOLINK_ATT.length > 0 )
    {
        {$this->mszHTMLForm}.GEOLINK_ATT.options[0].selected = true;
        changeAttribute( {$this->mszHTMLForm}.GEOLINK_ATT );
    }     
   
    return;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction; 
        
/* ============================================================================
 * Add the attribute dropdown change functions
 * ========================================================================= */
        $szJsFunctionName = "changeAttribute";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * called when the attribute value is changed.
 */
function {$szJsFunctionName}(obj)
{
    // update the hidden form variable
    {$this->mszHTMLForm}.SEL_GEOLINK_ATT.value = obj.options[obj.selectedIndex].text;
    return;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;

/* ============================================================================
 * Build the javascript array code
 * ========================================================================= */
        // build the array for the attribute tag values
        if ( isset( $this->mszContent ) && 
             !isset( $_SESSION["szJavascript"] ) )
        {
            // build request capabilities url
            $szRequest = $this->maParams["GDAS"]."?request=getCapabilities";
            
            // read and parse the GDAS getCap doc
            $data = "";
            $aszFile = file( $szRequest );
            
            // check the first few chars to see if a doc was returned
            if ( !is_array( $aszFile ) || 
                 strtoupper( substr( $aszFile[0], 0, 18 ) ) != "<GDAS_CAPABILITIES" )
            {
                // give error
                if ( !is_array( $aszFile ) )
                    $szServerDump = $aszFile;
                else 
                {
                    $szServerDump = "";
                    foreach(  $aszFile as $line )
                        $szServerDump .= $line;
                }
                $_SESSION['gErrorManager']->setError( ERR_WARNING, "GDAS server 'GetData' request failed.  ".
                                "The server returned the following: ".$szServerDump );
            }
            
            // read the document
            foreach( $aszFile as $line )
            {
                // process line
                $data .= $line;            
            }
                      
            // include the necessary php_utils XML parsing info
            include_once( realpath( COMMON."/xml_utils/XMLObject.php" ) );

            // construct a new XML object
            $oXMLObj = new XMLObject( $data );
            
            // get the array of tags to loop through
            $axFrameset = $oXMLObj->children[0]->children[1]->children[2]->children;

            // intialize counters
            $i=0;
            
            // initialize the javascript string var
            $szJavascript = "";
            
            // loop through each frameset and process
            foreach( $axFrameset as $oFraProperties )
            {
                // intialize dataset counter
                $j = 0;
                
                // initialize javascript array
                $szJavascript .= "gaDropDownValues[".$i.
                                                      "] = new Array();\n";
                // loop through the frameset children's properties and find
                // all the datasets
                foreach ( $oFraProperties->children as $oFrameProperty )
                {
                    // check if name
                    if ( $oFrameProperty->name == "name" )
                    {
                        // add name to the array
                        $szJavascript .= "gaDropDownValues[".$i."][0] = '".
                                                $oFrameProperty->value."';\n";
                    }
                    
                    // look for the dataset
                    if  ( strtoupper( $oFrameProperty->name ) == "DATASET" )
                    {
                        // initialize the attribute counter
                        $k = 0;
                        
                        // initialize javascript array
                        $szJavascript .= "gaDropDownValues[".$i.
                                                       "][1] = new Array();\n";
                        $szJavascript .= "gaDropDownValues[".$i."][1][".$j.
                                                         "] = new Array();\n";
                        $szJavascript .= "gaDropDownValues[".$i.
                                           "][1][".$j."][1] = new Array();\n";
                            
                       
                        // loop through the dataset properties and find attributes
                        foreach( $oFrameProperty->children as $oDSProperty )
                        {                            
                            // check if name
                            if ( $oDSProperty->name == "name" )
                            {
                                // add name to the array
                                $szJavascript .= "gaDropDownValues[".$i.
                                    "][1][".$j."][0] = '".
                                    $oDSProperty->value."';\n";
                            }
                            
                            // check if is attribute
                            if ( strtoupper( $oDSProperty->name ) == 
                                                                "ATTRIBUTE" )
                            { 
                                // loop through the attribute properties 
                                // and get name
                                foreach( $oDSProperty->children as 
                                                                $oATTProperty )
                                {      
                                    // find name
                                    if ( $oATTProperty->name == "name" )
                                    {
                                        $szJavascript .= "gaDropDownValues[".$i.
                                            "][1][".$j."][1][".$k."] = '".
                                            $oATTProperty->value."';\n";
                                    }
                                }
                                
                                // increment the attribute counter
                                $k++;
                            }
                        }
                        
                        // increment the dataset counter
                        $j++;
                    }
                }
                
                // increment the frameset counter
                $i++;
            }
            
            // set the session variable
            $_SESSION["szJavascript"] = $szJavascript;
        }
        else 
        {
            // copy from session
            if ( isset( $this->mszContent ) )
                $szJavascript = $_SESSION["szJavascript"];
        }

/* ============================================================================
 * Define the function to populate the javascript global arrays
 * ========================================================================= */
        $szJsFunctionName = "globalTagValueArrays";
        $szFunction = <<<EOT
/**
 * {$szJsFunctionName}
 * function to populate the global arrays
 */
function {$szJsFunctionName}(obj)
{
    {$szJavascript}
    
    // clear the previous list
    {$this->mszHTMLForm}.GEOLINK_FRAME.length = 0;
        
    // populate the framework select box and select appropriate one
    for(i=0;i<gaDropDownValues.length;i++)
    {
        opt = new Option( gaDropDownValues[i][0], gaDropDownValues[i][0], true, true );
        eval("{$this->mszHTMLForm}.GEOLINK_FRAME.options["+ i +"]=opt;");
    }    
    
    // loop and select the correct item
    bSelectedAtt = 0;
    for ( i=0;i<{$this->mszHTMLForm}.GEOLINK_FRAME.length;i++ )
    {
        // check for a match
        if ( {$this->mszHTMLForm}.GEOLINK_FRAME.options[i].value == {$this->mszHTMLForm}.SEL_GEOLINK_FRAME.value )
        {    
            {$this->mszHTMLForm}.GEOLINK_FRAME.options[i].selected = true;
            changeFramework( {$this->mszHTMLForm}.GEOLINK_FRAME );
            bSelectedAtt = 1;
            break;
        }
    }
    
    // select the first item if nothing selected
    if ( bSelectedAtt == 0 && {$this->mszHTMLForm}.GEOLINK_FRAME.length > 0 )
    {
        {$this->mszHTMLForm}.GEOLINK_FRAME.options[0].selected = true;
        changeFramework( {$this->mszHTMLForm}.GEOLINK_FRAME );
    }
    
    return;
}
EOT;
        $aReturn[$szJsFunctionName] = $szFunction;               

        // return       
        return $aReturn;        
        
    }

    /**
     * get the HTML variables for this widget
     */
    function GetHTMLHiddenVariables()
    {
        parent::GetHTMLHiddenVariables();

        // set the name
        $szVariable = "SEL_GEOLINK_FRAME";
        
        // determine if value has been passed through URL
        if ( $this->isVarSet($szVariable) &&
             $this->getVar($szVariable) != "")
            $szVal = $this->getVar($szVariable);
        else 
            $szVal = "";
        $szValue = " <INPUT TYPE=hidden NAME=$szVariable VALUE=\"".$szVal."\">\n";
        $aReturn[$szVariable] = $szValue;

        // set next variable name
        $szVariable = "SEL_GEOLINK_DS";
        
        // determine if value has been passed through URL
        if ( $this->isVarSet($szVariable) &&
             $this->getVar($szVariable) != "")
            $szVal = $this->getVar($szVariable);
        else 
            $szVal = "";
        $szValue = " <INPUT TYPE=hidden NAME=$szVariable VALUE=\"".$szVal."\">\n";
        $aReturn[$szVariable] = $szValue;
        
        // set next variable name
        $szVariable = "SEL_GEOLINK_ATT";
        
        // determine if value has been passed through URL
        if ( $this->isVarSet($szVariable) &&
             $this->getVar($szVariable) != "")
            $szVal = $this->getVar($szVariable);
        else 
            $szVal = "";
        $szValue = " <INPUT TYPE=hidden NAME=$szVariable VALUE=\"".$szVal."\">\n";
        $aReturn[$szVariable] = $szValue;        
        
        return $aReturn;
    }

    /**
     * process the url looking for ecostrat requests
     */
    function  ParseURL()
    {
        parent::ParseURL();

        // process the eco query only if requested
        if ( $this->isVarSet("SUBMIT_GEO") &&
             strlen( trim($this->getVar("SUBMIT_GEO") ) ) > 0 )
        {
            // check to see that the framework, dataset, and attribute info
            // has been passed through the url
            if  ( $this->isVarSet("SEL_GEOLINK_FRAME")  && 
                  strlen( $this->getVar("SEL_GEOLINK_FRAME") ) > 0 &&
                  $this->isVarSet("SEL_GEOLINK_DS")  && 
                  strlen( $this->getVar("SEL_GEOLINK_DS") ) > 0 &&
                  $this->isVarSet("SEL_GEOLINK_ATT")  && 
                  strlen( $this->getVar("SEL_GEOLINK_ATT") ) > 0 )
            {
                // issue geolink command
                $szGeolink = $this->maParams["GEOLINKINGSERVICE"]."?request=geolink&version=0.8.1&framework=".$this->getVar("SEL_GEOLINK_FRAME")."&gdas=".$this->maParams["GDAS"]."&dataset=".$this->getVar("SEL_GEOLINK_DS")."&attribute=".$this->getVar("SEL_GEOLINK_ATT")."&cache=1";
                $data = "";
                $aszFile = file( $szGeolink );
                
                // check the first few chars to see if a doc was returned
                if ( !is_array( $aszFile ) || 
                     strtoupper( substr( $aszFile[0], 0, 11 ) ) != "<GS_GEOLINK" )
                {
                    // give error
                    if ( !is_array( $aszFile ) )
                        $szServerDump = $aszFile;
                    else 
                    {
                        $szServerDump = "";
                        foreach(  $aszFile as $line )
                            $szServerDump .= $line;
                    }
                    $_SESSION['gErrorManager']->setError( ERR_WARNING, "Geolinking Service 'Geolink' request failed.  ".
                                    "The server returned the following: ".$szServerDump );
                }
                
                // combine the array into 1 file
                foreach(  $aszFile as $line )
                    $data .= $line;
                    
                // extract the datastore url
                $szTmp = strtoupper( $data );
    
                // find opening tag
                $nDSStart = strpos( $szTmp, "<HOST>" );
                
                // check if start has been found
                if ( $nDSStart === false ) return false;
                
                // find closing tag
                $nDSEnd = strpos( $szTmp, "</HOST>" );
                
                // check if found
                if ( $nDSEnd === false ) return false;
                
                // get dataset component
                $szDataStore = substr( $data, $nDSStart + 6, $nDSEnd - $nDSStart - 6 );

                // get WMS server capabilites and parse them
                $nServerID = $this->parseServerCap( $szDataStore );
                
                // get the layer id to process
                $nLayerID = $this->getLayerId( $nServerID );
                if ( $nLayerID === false )
                    $nLayerID = 1;

                // add the WMS layer
                if ( !$this->addWMSLayer( $nLayerID, $this->moMapObject->oMap, $_SESSION['gErrorManager'] ) )
                    $_SESSION['gErrorManager']->setError( ERR_WARNING, "Error adding geolink layer." );
                    
            }
            else 
            {
                // give error
                $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                                            "Geolink widget values not set." );
            }
        }
    }
    
    /**
     * Get layer ID based on server ID
     **/
    function getLayerId( $nServerId )
    {
        // init return value
        $xReturn = false;
        
        // ensure dbase is loaded
        if (PHP_OS == "WINNT" || PHP_OS == "WIN32")
        {
            if (!extension_loaded("dbase")) dl("php_dbase.dll");
        }
        else
        {
            if (!extension_loaded("dbase")) 
            {
                $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                    "dBase support for PHP was not found.  You must include ".
                    "dbase support when compiling PHP." );
            }                        
        }
        
        // get the cap dbf name
        $szCap = str_replace( "\\", "/", 
                 realpath( $_SESSION["gszServerDataPath"] ) )."/".DB_CAPABILITIES;

        // open the cap database
        $dbCap = @dbase_open( $szCap, 0 );
    
        // check if opened
        if ( !( $dbCap === false ) )
        {
            // get the layer ID
            $nCount = dbase_numrecords( $dbCap );
            for ( $i=1; $i<=$nCount; $i++ )
            {
                // get next record
                $aRow = dbase_get_record_with_names( $dbCap, $i );
                
                // check if server id matches
                if ( $aRow["server_id"] == $nServerId && 
                     trim( $aRow["depth"] ) == ".." )
                {
                    // set layer id
                    $xReturn = $aRow["layer_id"];
                    
                    // exit loop
                    break;
                }
            }
            
            // close the database
            dbase_close( $dbCap );
        }
        
        // return
        return $xReturn;
    }
    

    /**
     * draw this widget
     */
    function DrawPublish()
    {
        // don't draw widget if hidden
        if (!$this->mbVisible)
            return "<!-- Geolink is hidden -->";            
   
        // format the table
        $szResult = "<table border=0 cellpadding=2 cellspacing=0>\n".
                    "<tr>\n<td>";
            
        // create the ecozone drop-down for attributes
        $szResult .= "<select name=\"GEOLINK_FRAME\" onchange=\"".
                    "changeFramework(this)\">\n".
                     "<option selected>----------</option>\n".
                    "</select>";

        // create the ecozone drop-down for source
        $szResult .= "<select name=\"DATASET_DD\" onchange=\"".
                    "changeDataset(this)\">\n".
                    "<option selected>----------</option>\n".
                    "</select>";

        // create the ecozone drop-down for fields
        $szResult .= "<select name=\"GEOLINK_ATT\" onchange=\"".
                    "changeAttribute(this)\">\n".
                    "<option selected>----------</option>\n".
                    "</select>";

        // add the go button
        $szResult .= "<td align=\"right\">".
                     "<input type=\"submit\" name=\"SUBMIT_GEO\" value=\"go\">".
                     "</td>\n</tr>\n</table>";
        
        // publish the results
        $szResult = $this->moLabel->DrawPublish( $szResult );
        return $szResult;
    }

    /**
     * buildGeolinkerXML()
     *
     * Postcondition: This function takes the given dbase file and converts
     *                it to an XML file suitable for use with the geolinker.
     *
     * @param $szdBaseFile String - Path and filename of dbase file.
     * @param $szXMLFile String - Path and filename of the XML file to create.
     * @param $szIdField String - ID field name.
     * @param $szValueField String - Value field name.
     * @param $szDataSourceFile String - Datasource file XML parameter.
     * @param $szDataSourceTable String - Datasource table XML parameter.
     * @param $szDataSourceField String - Datasource field XML parameter.
     * @return Boolean - True if successful,false if not.
     * @desc Converts dbase file to geolinker XML data file.
     */    
    function buildGeolinkerXML( $szdBaseFile, $szXMLFile, 
        $szIdField, $szValueField, $szDataSourceFile, 
        $szDataSourceTable, $szDataSourceField )
    {
        // check if the dbase file exists
        if ( !file_exists( $szdBaseFile ) )
            return false;
            
/* ============================================================================
 * Ensure that the dbase module is loaded
 * ========================================================================= */
        if (PHP_OS == "WINNT" || PHP_OS == "WIN32")
        {
            if (!extension_loaded("dbase")) dl("php_dbase.dll");
        }
        else
        {
            if (!extension_loaded("dbase")) 
                $_SESSION['gErrorManager']->setError( ERR_WARNING, "dBase support for PHP was not ".
                   "found.  You must include dbase support when compiling PHP." );
        }            

/* ============================================================================
 * Process dbf file
 * ========================================================================= */
        // open the dbase file
        if ( !$fp = @dbase_open( $szdBaseFile, 0 ) )
            return false;
        
        // delete existing xml file if necessary
        if ( file_exists( $szXMLFile ) )
            unlink( $szXMLFile );
            
        // open a file pointer to the xml doc
        $fXML = fopen( $szXMLFile, 'w');
        
        // write opening tags to the buffer
        $szBuffer = "<?xml version=\"1.0\" standalone=\"no\"?>\n".
                    "<geolinker>\n".
                    "  <datasource>\n".
                    "    <file>".$szDataSourceFile."</file>\n".
                    "    <table>".$szDataSourceTable."</table>\n".
                    "    <field>".$szDataSourceField."</field>\n".
                    "  </datasource>\n".
                    "  <rowset>\n";
          
        // loop and add rows to buffer
        $nCount = dbase_numrecords( $fp );
        for ( $i=1; $i<=$nCount; $i++ )
        {
            // get the next row
            $axRow = dbase_get_record_with_names( $fp, $i );
            
            // double check that the requested field names are valid
            if ( !isset( $axRow[$szIdField] ) )
               return false;
            if ( !isset( $axRow[$szValueField] ) )
                return false;
                
            // watch for deleted rows
            if ( $axRow["deleted"] != 1 )
            {
                $szBuffer .= "    <row>\n".
                             "      <i>".$axRow[$szIdField]."</i>\n".
                             "      <v>".$axRow[$szValueField]."</v>\n".
                             "    </row>\n";
            }
        }
        
        // finish the buffer
        $szBuffer .= "  </rowset>\n".
                     "</geolinker>\n";
        
        // write and close the file
        fwrite( $fXML, $szBuffer );
        
        // close the dbase file
        fclose( $fXML );
                       
        // return success
        return true;
        
    // end buildGeolinkerXML() function
    }   
    
    /**
     * parseServerCap()
     *
     * Postcondition: This function gets the server capabilites document and
     *                parses it.
     *
     * @param $szServerURL String - URL to the WMS server.
     * @return boolean - True if successful, false if not.
     * @desc Retrieves and parses server capabilities.
     */    
    function parseServerCap( $szServerURL )
    {
        // ensure dbase is loaded
        if (PHP_OS == "WINNT" || PHP_OS == "WIN32")
        {
            if (!extension_loaded("dbase")) dl("php_dbase.dll");
        }
        else
        {
            if (!extension_loaded("dbase")) 
            {
                $_SESSION['gErrorManager']->setError( ERR_WARNING, 
                    "dBase support for PHP was not found.  You must include ".
                    "dbase support when compiling PHP." );
            }                        
        }
        
        // include the supporting php code
        include_once( COMMON."/phpwms/manage_servers.php" );

        // initialize var
        $nMaxID = 0;
        
        // create a new server manager object
        $oServerManager = new ServerManager( $_SESSION["gszServerDataPath"], 
                                                 $_SESSION["gszWMSParseFile"]);

        // get the server dbf name
        $szServer = str_replace( "\\", "/", 
                    realpath( $_SESSION["gszServerDataPath"] ) )."/".DB_SERVER;

        // open the server database
        $dbServ = @dbase_open( $szServer, 0 );
    
        // check if opened
        if ( !( $dbServ === false ) )
        {
            // get the maximum server ID
            $nCount = dbase_numrecords( $dbServ );
            for ( $i=1; $i<=$nCount; $i++ )
            {
                // get next record
                $aRow = dbase_get_record_with_names( $dbServ, $i );
                
                // check if id is larger
                $nMaxID = max( $aRow["server_id"], $nMaxID );
            }
            
            // close the database
            dbase_close( $dbServ );
        }
        
        // add a new server & parse
        $oServerManager->add( $szServerURL, "Geolinker", $nMaxID, 
                                    "This server was created automatically.");
        // return server ID
        return $nMaxID + 1;
        
    // end parseServerCap()
    }
    
    /**
     * This function fetches the layer data for the given layer id and adds it
     * as a layer to the given map object.
     *
     * @param $nLayerID integer - The ID of the layer to fetch.
     * @param $oMap object - The mapscript map object to add the layer to.
     * @return boolean - True if successful, false if not.
     **/
    function addWMSLayer( $nLayerID, $oMap, $oErrorManager )
    {
        $oWMSDatabase = new WMSDatabase($_SESSION["gszServerDataPath"]);
        $oWMSDatabase->oErrorManager = $_SESSION['gErrorManager'];
    
/* ============================================================================
 * Step 1 - get capabilities information
 * ========================================================================= */
        // open the capabilities file
        $dbCap = $oWMSDatabase->getDB( DB_CAPABILITIES );
    
        // check for failure on opening
        if ( $dbCap === false )
        {
            // failed so give message and return
            $oErrorManager->setError( ERR_WARNING, "Failed to open capabilities file"." [".DB_CAPABILITIES."].");
            return false;
        }
    
        // locate the record
        $nRecNo = $oWMSDatabase->find_record( $dbCap, "layer_id", $nLayerID );
    
        // check if found
        if ( $nRecNo === false )
        {
            // not found so give error and return
            $oErrorManager->setError( ERR_WARNING, "LayerID was not found in capabilities file"." [".DB_CAPABILITIES."]." );
            dbase_close( $dbCap );
            return false;
        }
    
        // put the record into an associative array
        $axCapRec = dbase_get_record_with_names( $dbCap, $nRecNo );
    
        // close the database
        dbase_close( $dbCap );
    
/* ============================================================================
 * Step 2 - get server information
 * ========================================================================= */
        // open the server database
        $dbServ = $oWMSDatabase->getDB( DB_SERVER );
    
        // check for failure on opening
        if ( $dbServ === false )
        {
            // failed so give message and return
            $oErrorManager->setError( ERR_WARNING, "Failed to open server file"." [".DB_SERVER."].");
            return false;
        }

        // locate the record
        $nRecNo = $oWMSDatabase->find_record( $dbServ, "server_id", $axCapRec["server_id"] );
    
        // check if found
        if ( $nRecNo === false )
        {
            // not found so give error and return
            $oErrorManager->setError( ERR_WARNING, "ServerID [".$axCapRec["server_id"].
                            "] was not found in server file [".DB_SERVER."]." );
            dbase_close( $dbServ );
            return false;
        }
    
        // put the record into an associative array
        $axServRec = dbase_get_record_with_names( $dbServ, $nRecNo );
    
        // close the database
        dbase_close( $dbServ );
    
/* ============================================================================
 * Step 3 - get bbox information
 * ========================================================================= */
        // open the bbox database and locate the bbox for the layer
        // there can be 0 or more BBOX
        $szBBOX = "";
        $bbox_id = $axCapRec['bbox_id'];
    
        // open the bbox database
        $dbBBOX = $oWMSDatabase->getDB( DB_BBOX );
    
        // check for failure on opening
        if ( $dbBBOX === false )
        {
            // failed so give message and return
            $oErrorManager->setError( ERR_WARNING, "Failed to open bbox file [".
                                                            $dbBBOX."].");
            return false;
        }
    
        // loop and process ALL bbox ids
        while ($bbox_id >= 0)
        {
            // locate the record
            $nRecNo = $oWMSDatabase->find_record( $dbBBOX, "bbox_id", $bbox_id );
    
            // check if found
            if ( $nRecNo === false )
            {
                // not found so give error and return
                $oErrorManager->setError( ERR_WARNING, "BBOX ID [".$bbox_id.
                                "] was not found in bbox file [".DB_BBOX."]." );
                dbase_close( $dbBBOX );
                return false;
            }
    
            // put the record into an associative array
            $axBBOXRec = dbase_get_record_with_names( $dbBBOX, $nRecNo );
    
            // add to the BBOX string
            $szBBOX .= trim($axBBOXRec['SRS'])." ".
                        $axBBOXRec['minx']." ".$axBBOXRec['miny']." ".
                        $axBBOXRec['maxx']." ".$axBBOXRec['maxy']." ";
    
            // update the ID
            $bbox_id = $axBBOXRec['next_id'];
        }
    
        // close the database
        dbase_close( $dbBBOX );

/* ============================================================================
 * Step 3 and a half - get abstract information
 * ========================================================================= */
        if (defined("LAYER_ABSTRACT") && LAYER_ABSTRACT == true &&
            trim($axCapRec["abstractid"]) != -1)
        {
            if (file_exists($_SESSION['gszServerDataPath']."/a".trim(strval($axCapRec["server_id"])).
                            "abstract.txt"))
            {
    
                // open the abstract text file for the correct server and access the data
                $aFile = file( $_SESSION['gszServerDataPath']."/a".trim(strval($axCapRec["server_id"])).
                                                                "abstract.txt");
    
                // initialize value
                $szAbstract = "";
    
                // scan lines until we get to the one that we need
                // 'abstract_id'
                $abstract_id = $axCapRec["abstractid"];
            
                $nAbstract = count($aFile);
    
                for ($i=0; $i<$nAbstract; $i++)
                {
                    if ( $i == $abstract_id )
                    {
                        // capture srs information if correct line
                        $szAbstract .= trim( $aFile[$i] );
                    }
                }
            }
            else
                $oErrorManager->setError(ERR_WARNING, "Error opening (".$_SESSION['gszServerDataPath']."/s".trim(strval($axCapRec["server_id"]))."abstract.txt)");
        }
    
/* ============================================================================
 * Step 4 - get srs information
 * ========================================================================= */
        if (trim($axCapRec["srs_ids"]) != "")
        {
            if (file_exists( $_SESSION['gszServerDataPath']."/s".trim(strval($axCapRec["server_id"])).
                             "srs.txt"))
            {
                // open the srs text file for the correct server and access the data
                $fp = fopen( $_SESSION['gszServerDataPath']."/s".trim(strval($axCapRec["server_id"])).
                                                                "srs.txt", "r" );
                // initialize value
                $szSRS = "";
    
                // scan lines until we get to the one that we need
                // 'srs_ids' contains a comma-delimited list of SRS record ids
                $srs_ids = split(",", $axCapRec["srs_ids"]);
                $i = 0;
    
                while (!feof ($fp))
                {
                    if ( in_array($i, $srs_ids) )
                    {
                        // capture srs information if correct line
                        $szSRS .= trim( fgets($fp, 4096) ) ." ";
                    }
                    else
                    {
                        // Just skip this one
                        fgets($fp, 4096);
                    }
                    // increment the line counter
                    $i++;
                }
    
                // close the file pointer
                fclose( $fp );
            }
            else
                $oErrorManager->setError(ERR_WARNING, "Error opening (".$_SESSION['gszServerDataPath']."/s".trim(strval($axCapRec["server_id"]))."srs.txt)");
        }
    
/* ============================================================================
 * Step 4 and a half - get style information
 * ========================================================================= */
        if (defined("LAYER_STYLE") && LAYER_STYLE == true)
        {
            // open the style database
            $dbStyle = $oWMSDatabase->getDB( DB_STYLE );
    
            // check for failure on opening
            if ( $dbStyle === false )
            {
                // failed so give message and return
                $oErrorManager->setError( ERR_WARNING, "Failed to open style file [".
                                                            DB_STYLE."].");
                return false;
            }
    
            $nStyleID = $axCapRec['style_id'];
            $aStyle = array();
            $bFirst = true;
            $szStyleList = "";
            while ($nStyleID != -1)
            {
                $nStyleRec = $oWMSDatabase->find_record($dbStyle, "style_id", $nStyleID);
                $styleRec = dbase_get_record_with_names($dbStyle, $nStyleRec);
    
                // validate name and title mapserver don't allow
                // quotes in metadata
                if (strchr($styleRec['name'], "\"") !== false ||
                    strchr($styleRec['title'], "\"") !== false)
                {
                    $oErrorManager->setError( ERR_FATAL, "NAME or TITLE can't have quote in it.");
                    return false;            
                }
    
                $szStyleList .= ",".trim($styleRec['name']);
                
                if ($bFirst)
                {
                    $szStyle = trim($styleRec['name']);
                    $bFirst = false;
                }
    
                $aStyle[trim($styleRec['name'])] = array(trim($styleRec['title']), 
                                                   trim($styleRec['leg_width'])." ".
                                                   trim($styleRec['leg_height'])." ".
                                                   trim($styleRec['leg_format'])." ".
                                                   trim($styleRec['legendurl']));
    
    
                $nStyleID = $styleRec['next_id'];
            }
    
            $szStyleList = substr($szStyleList, 1);
    
            // close the database
            dbase_close( $dbStyle );
        }
    
    
/* ============================================================================
 * Step 5 - begin assembling layer information
 * ========================================================================= */
        // set the WMS layer name (replace spaces with "_")
        $szWMSName = trim( $axServRec["name"] )."-".trim( $axCapRec["name"] );
        $szWMSName = ereg_replace(" ", "_", $szWMSName);
    
        // set type
        $nWMSType = MS_LAYER_RASTER;
    
        // metadata item - title
        $szWMSMetaData = "wms_title=".trim( $axCapRec["title"] );
    
        // name
        $szWMSMetaData .= "|wms_name=".trim($axCapRec["name"]);
    
        // metadata item - server version
        $szWMSMetaData .= "|wms_server_version=".trim($axServRec['version']);
    
        // map url
        $szWMSMetaData .= "|wms_onlineresource=".urlencode(trim($axServRec['map_url']));
    
        // queriable
        if ( $axCapRec["queryable"] == 1 )
            $szWMSMetaData .= "|wms_queriable=1";
    
        // check for the wms_srs metadata
        if ( $szSRS != "" )
            $szWMSMetaData .= "|wms_srs=".$szSRS;
    
        // check for the wms_abstract metadata
        if ( isset($szAbstract) )
            $szWMSMetaData .= "|wms_abstract=".str_replace("\"", "'", $szAbstract);
    
        
        // check for the wms_stylelist metadata
        if ( isset($szStyleList) )
            $szWMSMetaData .= "|wms_stylelist=".$szStyleList;
    
        if ( isset($szStyle))
            $szWMSMetaData .= "|wms_style=".$szStyle;
    
        if ( isset($aStyle) && is_array($aStyle))
            foreach($aStyle as $szKey => $aStyleAttr)
            {
                $szWMSMetaData .= "|wms_style_".$szKey."_title=".$aStyleAttr[0];
                $szWMSMetaData .= "|wms_style_".$szKey."_legendurl=".urlencode($aStyleAttr[1]);
            }
    
        // assemble the LLBBOX
        $szLLBBOX = $axCapRec['ll_minx']." ".
                    $axCapRec['ll_miny']." ".
                    $axCapRec['ll_maxx']." ".
                    $axCapRec['ll_maxy'];
    
        // check for the wms_extents metadata
        if ( $szLLBBOX != "" )
            $szWMSMetaData .= "|wms_latlonboundingbox=".$szLLBBOX;
    
        // check for BBOX
        if ( $szBBOX != "" )
            $szWMSMetaData .= "|wms_boundingbox=".$szBBOX;
    
        if (isset($axCapRec["extractabl"]) && $axCapRec["extractabl"] == 1)
            $szWMSMetaData .= "|wms_extractable=1";
    
        // get the mapscript and WMS versions
        $szVersion = ms_GetVersion();
        $szWMSVersion = trim( $axServRec["version"] );
    
        // Find an image format recognized by both this client and the remote svr
        $aszFormats = split( ",", trim( $axServRec['formats'] ) );
        $szFormat = "";
    
        // Try to match formats in this order: GIF, PNG, JPEG
        foreach( array("GIF", "PNG", "JPEG") as $szFmt)
        {
            // Does MapScript support this format?
            if (strpos($szVersion, "OUTPUT=".$szFmt) > 0)
            {
                // Yes... so does the remote server support it as well?
                foreach( $aszFormats as $thisFormat )
                {
                    if ( strtoupper(trim($thisFormat)) == $szFmt ||
                         strtoupper(trim($thisFormat)) == "IMAGE/".$szFmt )
                    {
                        // Found a match!
                        $szFormat = $thisFormat;
                        break 2;  // Get out of 2 loops
                    }
                }
            }
        }
    
        // metadata item - server format list
        $szWMSMetaData .= "|wms_formatlist=".trim($axServRec['formats']);
    
        // metadata item - server format
        $szWMSMetaData .= "|wms_format=".$szFormat;
    
        // check if the layer is queriable
        if ( $axCapRec['queryable'] = "1" )
        {
            // update the query string
            $szQueryLayers = "&QUERY_LAYERS=".urlencode( trim($axCapRec['name']) );
        }
        else
        {
            // update the query string
            $szQueryLayers = "";
        }
    
        // set connection string
        //printf("usrl = %s <br>", $aszLayer["URL"]);
        $szURL = trim( $axServRec['map_url'] );
        $nLength = strlen( $szURL );
        if (strstr($szURL, '?') !== false)
        {
            if (substr($szURL, -1) != "?" && substr($szURL, -1) != "&")
                $szQuestionMark = "&";
            else
                $szQuestionMark = "";
        }
        else
            $szQuestionMark ="?";
    
        // build connection string
        $szConnection = $szURL.$szQuestionMark;
                        //."SERVICE=WMS&VERSION=".$szWMSVersion
                        //."&LAYERS=".$axCapRec['name']
                        //.$szQueryLayers
                        //."&FORMAT=".$szFormat."&TRANSPARENT=TRUE";
        
    
        // only add the layer if format is set
        if ($szFormat != "")
        {
            // build an array of layerparameters
            $axParams["metadata"] =  $szWMSMetaData;
            $axParams["connectiontype"] = MS_WMS;
            $axParams["connection"] = $szConnection;
    
            // create the layer
            $this->addLayer( $oMap, $szWMSName, $nWMSType, $axParams, $oErrorManager );
    
            // get the current map projection
            //$szTempMapProj = $oMap->getprojection();
    
            // only set the projection if it's set in the map
            //if ( strlen( trim( $szTempMapProj ) ) > 0 )
            //{
                // set layer projection
            //    $oLayer = $oMap->getlayerbyname( $szWMSName );
            //    $oLayer->setprojection( $szTempMapProj );
            //}
        }
        else
        {
            $oErrorManager->setError( ERR_WARNING, "Acceptable image format not found" );
        }
    
        // return success
        return true;
    
    // end addWMSLayer function
    }
    
    /**
     * This function adds a new layer to the given mapscript map object.  The
     * layer will NOT be added if the name is already in use.
     *
     * @param $oMap object - Mapscript map object.
     * @param $szName string - New layer name.
     * @param $nType integer - The layer type.
     * @param $axParams array - Mixed associative array of optional layer
     *              properties:
     *              $axParams["status"] - layer status (default = MS_ON)
     *              $axParams["metadata"] - comma de-limited list of metadata
     *                                      items separated by "metadivider".
     *              $axParams["metadivider"] - metadata divider that separates
     *                                      metadata name from value (default = "=")
     *              $axParams["connectiontype"] - connection type
     *              $axParams["connection"] - the connection string
     *              $axParams["group"] - the group name for the layer
     *              $axParams["data"] - the data string
     *              $axParams["scalemin"] - the minimum scale for the layer
     *              $axParams["scalemax"] - the maximum scale for the layer
     *
     * @param $oLog object - The logger object to log events.
     * @return boolean - True if successful, false if not.
     **/
    function addLayer( $oMap, $szName, $nType, $axParams, $oErrorManager )
    {
        // get an array of all layer names already present in the map
        $aszLayers = $oMap->getAllLayerNames();
      
        // Check if anything was returned
        if ( is_array( $aszLayers ) )
        {
            // init vars
            $szOrgName = $szName;
            $i = 1;
            do
            {
                // initialize the variable
                $bLayerFound = false;
                
                // loop through all the layers and determine if name is found
                foreach ( $aszLayers as $szLayer )
                {
                    // check for a match
                    if ( strtoupper($szLayer) == strtoupper($szName) )
                    {
                        // set flag
                        $bLayerFound = true;
        
                        // exit loop
                        break;
                    }
                    
                // end foreach
                }
                
                // update name if necessary
                if ( $bLayerFound ) $szName = $szOrgName."_".$i;
                
                // increment counter
                $i++;                
            
            }while( $bLayerFound );
        }
    
        // only add the layer if the layer is not already present
        if (!$bLayerFound)
        {
            // create the layer object
            $oNewLayer = ms_newLayerObj($oMap);
    
            // set name
            $oNewLayer->set("name",$szName);
    
            // set type
            $oNewLayer->set("type",$nType);
    
            // set status
            if ( isset( $axParams["status"] ) )
                $oNewLayer->set( "status", $axParams["status"] );
            else
                $oNewLayer->set( "status", MS_ON );
    
            // set group name
            if ( isset( $axParams["group"] ) )
                $oNewLayer->set( "group", $axParams["group"] );
    
            // set the data tag
            if ( isset( $axParams["data"] ) )
                $oNewLayer->set( "data", $axParams["data"] );
    
            // set the min scale
            if ( isset( $axParams["scalemin"] ) &&
                 is_numeric( $axParams["scalemin"] ) &&
                 $axParams["scalemin"] > 0 )
                $oNewLayer->set( "minscale", $axParams["scalemin"] );
    
            // set the max scale
            if ( isset( $axParams["scalemax"] ) &&
                 is_numeric( $axParams["scalemax"] ) &&
                 $axParams["scalemax"] >= $axParams["scalemin"] )
                $oNewLayer->set( "maxscale", $axParams["scalemax"] );
    
            // set metadata
            if ( isset( $axParams["metadata"] ) )
            {
                // determine the metadata divider
                if ( isset( $axParams["metadivider"] ) )
                    $sz_meta_div_str = $axParams["metadivider"];
                else
                    $sz_meta_div_str = "=";
    
                // create an array of metadata
                $aszMetaData1 = explode( "|", $axParams["metadata"] );
    
                // loop and add metadata
                foreach ($aszMetaData1 as $MetaData)
                {
                    // separate the value from the label
                    $aszMetaData2 = explode( $sz_meta_div_str, $MetaData );
    
                    // add
                    $oNewLayer->setMetaData(trim($aszMetaData2[0]),
                                            urldecode(trim($aszMetaData2[1])));
                }
            }
    
            // set connecttion
            if ( isset( $axParams["connectiontype"] ) )
                $oNewLayer->set( "connectiontype",$axParams["connectiontype"] );
    
            // set connection string
            if ( isset( $axParams["connection"] ) )
                $oNewLayer->set( "connection", $axParams["connection"] );
    
            // return success
            $bReturn = true;
        }
        // otherwise the layer exists
        else
        {
            // return failure
            $bReturn = false;
        }
    
        // return status
        return $bReturn;
    
    // end addLayer function
    }  
}

//register widget
if (function_exists( "RegisterWidget" ))
    RegisterWidget( "Geolink", "GeolinkWidget" );
?>
