<?php
/**
 * MapEditor Module
 * 
 * @project     PHP Wrapper Class
 * @revision    $Id: map_editor.php,v 1.2 2002/03/05 15:02:14 bronsema Exp $
 * @purpose     This file contains classes related to managing map editing. 
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2001, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*****************************************************************************
 * $Log: map_editor.php,v $
 * Revision 1.2  2002/03/05 15:02:14  bronsema
 * Completed basic functionality
 *
 * Revision 1.1  2002/03/04 18:46:16  bronsema
 * Initial Creation
 *
 *****************************************************************************/
 
/**
 * This class supports modifying the contents of the map file and would 
 * typically be used to add/remove layers and make other physical changes to 
 * the map file.
 * It is assummed that the phpmapscript module is loaded prior to 
 * instantiaiting this class.
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class MapEditor extends Logger
{       
    /**
     * The current session object (private).
     */
    var $oSession;
  
    /**
     * Construct a new MapNavigator instance and initialize it.
     *
     * @param oSession object - MapSession class object.
     */
    function MapEditor( &$oSession )
    {
        // initialize variables
        $this->oSession = &$oSession;
         
        // set the scope for the logging functions
        $this->setScope( "MapEditor" );
    
    // end constructor
    }
    
    /**
     * This function takes the supplied parameters and adds a new layer to the
     * current map object.
     *
     * @param szName string - The name of the layer.
     * @param nType integer - The layer type as defined in mapscript.
     * @param aProperties array - Optional array of property values.  The array
     *                            can contain any number of proerties to be 
     *                            set.
     * The list of property values that can be set in the array are as follows:
     *      $aProperties["name"] - string
     *      $aProperties["type"] - integer
     *      $aProperties["status"] - integer
     *      $aProperties["classitem"] - string
     *      $aProperties["group"] - string
     *      $aProperties["data"] - string
     *      $aProperties["tolerance"] - double
     *      $aProperties["toleranceunits"] - integer
     *      $aProperties["symbolscale"] - double
     *      $aProperties["minscale"] - double
     *      $aProperties["maxscale"] - double
     *      $aProperties["labelminscale"] - double
     *      $aProperties["labelmaxscale"] - double
     *      $aProperties["maxfeatures"] - integer
     *      $aProperties["offsite"] - integer
     *      $aProperties["annotate"] - integer
     *      $aProperties["transform"] - integer
     *      $aProperties["labelcache"] - integer
     *      $aProperties["postlabelcache"] - integer
     *      $aProperties["labelitem"] - string
     *      $aProperties["labelsizeitem"] - string
     *      $aProperties["labelangleitem"] - string
     *      $aProperties["tileitem"] - string
     *      $aProperties["tileindex"] - string
     *      $aProperties["header"] - string
     *      $aProperties["footer"] - string
     *      $aProperties["connection"] - string
     *      $aProperties["connectiontype"] - integer
     *      $aProperties["filteritem"] - string
     *      $aProperties["projection"] - string
     *      $aProperties["metadata"] - array of metadata items:
     *                                      $aMetadata["metadata1"] = "value1
     *                                      $aMetadata["metadata2"] = "value2
     *                                      $aMetadata["metadata3"] = "value3
     *
     * @return object - Returns a pointer to the new layer object.
     */
    function addLayer( $aProperties = array() )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "addLayer() starting" );

        // get the map object
        $oMap = $this->oSession->getMapObj();
        
        // check that the map object is valid
        if ( $oMap == "" || $oMap == null )
        {
            // generate error and exit
            $this->error( ERR_CRITICAL, "No valid map object is set." );
            return "";
        }

        // create the layer object
        $oNewLayer = ms_newLayerObj($oMap);

        // loop through each item in the array and set the property
        foreach ($aProperties as $key => $value)
        {
            // treat metadata and projection separately
            switch ($key) 
            {
                case "metadata":

                    // set the metadata items
                    foreach ($value as $mKey => $mValue)
                    {
                        $oNewLayer->setMetaData($mKey, $mValue);
                        $this->log( LOG_QUIET, "addLayer - set metadata item ".
                                    $mKey." to ".$mValue.".");
                    }                    
                    break;
                
                case "projection":
                    
                    // set projection
                    $oNewLayer->setprojection($value);
                    $this->log( LOG_QUIET, "addLayer - set projection to: ".
                                                                $value.".");
                    
                    break;
               
                default:
                
                    // not metadata or projection so just set
                    $oNewLayer->set($key,$value);

                    // log
                    $this->log( LOG_QUIET, "addLayer - set ".$key." to ".
                                                                $value."." );
            }
            
        // end properties array loop
        }
        
        // log exit
        $this->logFuncEnd( LOG_ALL, "addLayer() done" );
      
        // return a pointer to the layer object
        return $oNewLayer;
            
        
    // end addLayer function
    }  

    /**
     * This function sets the given layer's status to MS_DELETE
     *
     * @param nLayer integer - The index of the layer to delete.
     *
     * @return boolean - True if successful, false if not.
     */
    function deleteLayer( $nLayer )
    {
    // log entry
    $this->logFuncStart( LOG_VERBOSE, "deleteLayer(".$nLayer.
                                                        ") starting" );
    // get the map object
    $oMap = $this->oSession->getMapObj();        

    // check that the map object is valid
    if ( $oMap == "" || $oMap == null )
    {
        // generate error and exit
        $this->error( ERR_CRITICAL, "No valid map object is set." );
        return "";
    }

    // set the status of the layer to MS_DELETE
    $oLayer = $oMap->getlayer($nLayer);
    $oLayer->set("status", MS_DELETE);

    // log
    $this->log( LOG_QUIET, "Layer[".$nLayer.
                                        "] was successfully deleted." );               
    // log exit
    $this->logFuncEnd( LOG_ALL, "deleteLayer(".$nLayer.") done" );

    // return success
    return true;

    // end deleteLayer function
    }
 
    /**
     * This function promotes the given layer in the current map file such that
     * the layer will be drawn later than before.
     *
     * @param nLayer integer - The index of the layer to promote.
     * @param nLevels integer - Optional number of levels to traverse.
     */
    function promoteLayer( $nLayer, $nLevels = 1 )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "promoteLayer(".$nLayer.
                                                            ") starting" );
        // get the map object
        $oMap = $this->oSession->getMapObj();        

        // check that the map object is valid
        if ( $oMap == "" || $oMap == null )
        {
            // generate error and exit
            $this->error( ERR_CRITICAL, "No valid map object is set." );
            return "";
        }

        // promote the layer
        for ( $i=0; $i<$nLevels; $i++ )
        {
            $oMap->moveLayerDown( $nLayer );
        }

        // log
        $this->log( LOG_QUIET, "Layer[".$nLayer.
                                            "] was successfully promoted." );               
        // log exit
        $this->logFuncEnd( LOG_ALL, "promoteLayer(".$nLayer.") done" );
    
    // end promoteLayer function
    }
    
    /**
     * This function demotes the given layer in the current map file such that
     * the layer will be drawn earlier than before.
     *
     * @param nLayer integer - The index of the layer to demote.
     * @param nLevels integer - Optional number of levels to traverse.
     */
    function demoteLayer( $nLayer, $nLevels = 1 )
    {
        // log entry
        $this->logFuncStart( LOG_VERBOSE, "demoteLayer(".$nLayer.
                                                            ") starting" );
        // get the map object
        $oMap = $this->oSession->getMapObj();        

        // check that the map object is valid
        if ( $oMap == "" || $oMap == null )
        {
            // generate error and exit
            $this->error( ERR_CRITICAL, "No valid map object is set." );
            return "";
        }

        // demote the layer
        for ( $i=0; $i<$nLevels; $i++ )
        {
            $oMap->moveLayerUp( $nLayer );
        }

        // log
        $this->log( LOG_QUIET, "Layer[".$nLayer.
                                            "] was successfully demoted." );               
        // log exit
        $this->logFuncEnd( LOG_ALL, "demoteLayer(".$nLayer.") done" );
    
    // end demoteLayer function
    }
    
// end MapEditor class    
}

?>