/**********************************************************************
 * $Id: wmsparse.c,v 1.35 2003/05/12 15:04:24 sacha Exp $
 *
 * Name:     wmsparse.c
 * Project:  MapLab/MapBrowser
 * Purpose:  Parse a OGC WMS capabilities file into a set of dbase files
 * Author:   Paul Spencer, spencer@dmsolutions.ca
 *
 **********************************************************************
 * Copyright (c) 2001,2002, DM Solutions Group Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 **********************************************************************
 * $Log: wmsparse.c,v $
 * Revision 1.35  2003/05/12 15:04:24  sacha
 * remove the space added beetwee formats
 *
 * Revision 1.34  2002/11/26 14:01:53  julien
 * Valid the 3 optional legendURL field
 *
 * Revision 1.33  2002/11/21 22:47:53  daniel
 * OOpps... fixed a typo in usage output (WMSPARSE instead of WSMPARSE)
 *
 * Revision 1.32  2002/11/21 22:06:24  daniel
 * Contert strings from UTF-8 (libxml2 default) to ISO-8859-1 (our default)
 *
 * Revision 1.31  2002/11/20 04:46:24  daniel
 * Added wmsTrim() to trim spaces in layer titles from some servers
 *
 * Revision 1.30  2002/11/19 22:35:23  julien
 * change the stylesheeturl field in style.dbf to stylesheet.
 *
 * Revision 1.29  2002/11/19 22:31:31  daniel
 * Fixed more memory leaks
 *
 * Revision 1.28  2002/11/19 21:25:40  daniel
 * Added -v (version) command-line switch and added version/date info to
 * usage message.
 *
 * Revision 1.27  2002/11/19 21:09:50  julien
 * Add legendurl height, width and format for the mapcontext in the style.dbf
 *
 * Revision 1.26  2002/11/19 21:04:05  daniel
 * Added copyright header.
 * Run into Purify and fixed some memory issues.
 *
 */

//standard headers
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//libxml headers
#include <libxml/parser.h>
#include <libxml/xpath.h>

//dbase support taken from shapelib
#include "shapefil.h"

//global variables
DBFHandle hServerDB; //handle to the server DBF file
DBFHandle hCapabDB; //handle to the capability DBF file
DBFHandle hBBoxDB; //handle to the BoundingBox DBF file
DBFHandle hStyleDB; //handle to the style DBF file
FILE * hSRSfile; //handle to the SRS text file
int nSRSLine = 0; //current line number in the SRS text file
FILE * hAbstractFile; //handle to the abstract text file
int nAbstractLine = 0; //current line number in the abstract text file

// Extract global variables
xmlChar ** gpapszExtractName = NULL;
xmlChar ** gpapszExtractType = NULL;
xmlChar ** gpapszExtractFormat = NULL;
xmlChar *  gpszExtractURL;

// Variable to pre parse the optional tags.
// There is 3 module to check: 1-Abstract 2-Style 3-Extract 
//                             4-onlineresource of GetCapab
int ganEnableModule[4] = { 0, 0, 0, 0 };


/**
 * return the max value from an id field.  Assumes that this value is in
 * the last record (may not be valid).
 *
 * @param hDB handle a handle to an open db
 * @param szField string the name of the field to query
 * @result integer the max value.
 */
int getMaxID( DBFHandle hDB, char * szField )
{
    int nIDFld, nRec, nCurrent, result = 0;

    nIDFld = DBFGetFieldIndex( hDB, szField );

    nRec = DBFGetRecordCount( hDB );

    if ( nRec > 0 )
    {
        result = DBFReadIntegerAttribute( hDB, nRec -1, nIDFld );
    }

    return result;
}

/**
 * hack xmlGetNodePath for this application because it finds the root
 * node in the document type (they have the same name). This will replace
 * the index of the root node with 1 instead of 2.
 */
xmlChar * getXPathNodePath( xmlNodePtr pNode )
{
  xmlChar * result;

  result = xmlGetNodePath( pNode );

  if (result && xmlStrncmp(result, "/WMT_MS_Capabilities[2]", 23) == 0)
    result[21] = '1';

  //printf( "result is %s\n", result );

  return result;
}

/**
 * return an Attribute from the node at the given path
 *
 * @param pContext xmlXPathContextPtr the context to evaluate in
 * @param pszPath xmlChar* the expression to evaluate
 * @param pszAttribute xmlChar* the attribute to retrieve
 *
 * @return xmlChar* the attribute or an empty string. This must be free'd by
 *         the caller.  The output string is converted from UTF8 to ISO-8859-1
 *         since UTF8 is the default in libxml2 but we want isolatin.
 */
xmlChar * getXPathNodeAttribute( xmlXPathContextPtr pContext,
                                 xmlChar * pszPath, xmlChar * pszAttribute )
{
    xmlXPathObjectPtr result;
    xmlChar * pszValue = NULL;

    //evaluate the expression
    result = xmlXPathEval( pszPath, pContext );

    //result can be NULL indicating failure to evaluate the context
    if ( result != NULL )
    {
        //nodevalset can be NULL or 0 indicating no match
        if ( result->nodesetval != NULL && result->nodesetval->nodeNr > 0 )
            pszValue = xmlGetProp(result->nodesetval->nodeTab[0],pszAttribute);

        xmlXPathFreeObject( result );
    }

    if (pszValue)
    {
        // Convert pszValue from UTF8 to ISO-8859-1
        xmlChar *pszTmp;
        int nIn, nOut;

        nIn = nOut = xmlStrlen(pszValue);
        // Alloc buffer for isolatin string output which will be <= to original
        pszTmp = xmlStrdup(pszValue);
        UTF8Toisolat1(pszValue, &nOut, pszTmp, &nIn);
        pszValue[nOut] = '\0';  // Yes, we have to terminate it ourselves!
        xmlFree(pszTmp);
    }
    else
    {
        //massage return value into something that has to be xmlFree'd
        pszValue = (xmlChar *) xmlMalloc( sizeof( xmlChar ) );
        pszValue[0] = '\0';
    }

    return pszValue;
}

/*
 * return the content of a node
 *
 * @param pContext xmlXPathContextPtr the context to evaluate in
 * @param pszPath xmlChar* the expression to evaluate
 *
 * @return the content of the node or an empty string.  Must be free'd by
 *         the caller
 */
xmlChar * getXPathNodeContent( xmlXPathContextPtr pContext, xmlChar * pszPath)
{
    xmlXPathObjectPtr result;
    xmlChar * pszContent = NULL;

    //evaluate the expression
    result = xmlXPathEval( pszPath, pContext );

    //the result can be NULL
    if (result != NULL)
    {
        //the nodesetval can be NULL or 0
        if ( result->nodesetval != NULL && result->nodesetval->nodeNr > 0 )
            pszContent = xmlNodeGetContent( result->nodesetval->nodeTab[0]);

        //free the result object
        xmlXPathFreeObject( result );
    }

    if (pszContent)
    {
        // Convert pszContent from UTF8 to ISO-8859-1
        xmlChar *pszTmp;
        int nIn, nOut;

        nIn = nOut = xmlStrlen(pszContent);
        // Alloc buffer for isolatin string output which will be <= to original
        pszTmp = xmlStrdup(pszContent);
        UTF8Toisolat1(pszContent, &nOut, pszTmp, &nIn);
        pszContent[nOut] = '\0';  // Yes, we have to terminate it ourselves!
        xmlFree(pszTmp);
    }
    else
    {
        //massage return value into something that has to be xmlFree'd
        pszContent = (xmlChar*) xmlMalloc( sizeof( xmlChar ) );
        pszContent[0] = '\0';
    }
    return pszContent;
}

/*
 * concatenate two strings, result is malloc'd so free it when you're done.
 */
xmlChar * xmlConcat( xmlChar * first , xmlChar * second )
{
    xmlChar * result;
    result = (xmlChar*) xmlMalloc( sizeof(xmlChar) * ( 1 + \
                                   xmlStrlen( first ) + \
                                   xmlStrlen( second ) ) );
    result[0] = '\0';
    strcat( result, first );
    strcat( result, second );

    return result;
}

/*
 * Get rid of spaces at beginning and end of a string.
 * Returns a realloc'd string (or the same pointer if not modified)
 */
xmlChar * wmsTrim( xmlChar *pszString )
{
    xmlChar *pszPtr, *pszRetString=NULL;

    if (pszString == NULL || pszString[0] == '\0')
        return pszString;

    for(pszPtr = pszString+strlen(pszString)-1; pszPtr >= pszString; pszPtr--)
    {
        if (!isspace(*pszPtr))
        {
            *(pszPtr+1) = '\0';
            break;
        }
    }

    for(pszPtr = pszString; pszPtr != '\0'; pszPtr++)
    {
        if (!isspace(*pszPtr))
        {
            break;
        }
    }

    if (pszPtr == pszString)
    {
        pszRetString = pszString;
    }
    else
    {
        // Realloc a new string pointing to new start
        pszRetString = xmlStrdup(pszPtr);
        xmlFree(pszString);
    }

    return pszRetString;
}


/**
 * Encode \n, \r, and \ into \ n , \ r, and \ \ 
 */
xmlChar * wmsEncode( xmlChar * pszStrContent)
{
    int buflen, i;
    xmlChar *pszNewString;
    const xmlChar *c;

    // Start with 30 extra chars for replacements... 
    // should be good enough for most cases
    buflen = xmlStrlen(pszStrContent) + 30;
    pszNewString = (xmlChar*)xmlMalloc(buflen+1*sizeof(xmlChar*));
    if (pszNewString == NULL)
    {
        printf("Memory Allocation error!");
        return NULL;
    }

    for(i=0, c=pszStrContent; *c != '\0'; c++)
    {
        // Need to realloc buffer?
        if (i+2 > buflen)
        {
            // Realloc another 50 extra bytes
            buflen += 50;
            pszNewString = (xmlChar*)xmlRealloc(pszNewString, 
                                             buflen+1*sizeof(xmlChar*));
            if (pszNewString == NULL)
            {
                printf("Memory Allocation error!");
                return NULL;
            }
        }

        switch(*c)
        {
          case '\n':
            strcpy(pszNewString+i, "\\n");
            i += 2;
            break;
          case '\r':
            strcpy(pszNewString+i, "\\r");
            i += 2;
            break;
          case '\\':
            strcpy(pszNewString+i, "\\\\");
            i += 2;
            break;
          default:
            pszNewString[i++] = *c;
        }
    }

    pszNewString[i++] = '\0';

    return pszNewString;
}

/**
 *
 */
void addCapability( int nLayerID, int nServerID, xmlChar* pszName, 
                    xmlChar* pszTitle, xmlChar *pszSRS, float ll_minx, 
                    float ll_miny, float ll_maxx, float ll_maxy, int bbox_id, 
                    int style_id, char* pszDepth, int nQueryable, 
                    int nAbstractID, int nExtractID )
{
    int nRec;

    nRec = DBFGetRecordCount( hCapabDB );
    DBFWriteIntegerAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "layer_id" ), \
                              nLayerID );
    DBFWriteIntegerAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "server_id" ), \
                              nServerID );
    DBFWriteStringAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "name" ), \
                              pszName );
    DBFWriteStringAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "title" ), \
                              pszTitle );
    DBFWriteStringAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "srs_ids" ), \
                              pszSRS );
    if( ganEnableModule[0] == 1 )
    {
        DBFWriteIntegerAttribute( hCapabDB, nRec, 
                                 DBFGetFieldIndex( hCapabDB, "abstractid" ), 
                                 nAbstractID );
    }
    DBFWriteDoubleAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "ll_minx" ), \
                              ll_minx );
    DBFWriteDoubleAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "ll_miny" ), \
                              ll_miny );
    DBFWriteDoubleAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "ll_maxx" ), \
                              ll_maxx );
    DBFWriteDoubleAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "ll_maxy" ), \
                              ll_maxy );
    DBFWriteIntegerAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "bbox_id" ), \
                              bbox_id );
    if( ganEnableModule[1] == 1 )
    {
        DBFWriteIntegerAttribute( hCapabDB, nRec, \
                                  DBFGetFieldIndex( hCapabDB, "style_id" ), \
                                  style_id );
    }
    DBFWriteStringAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "depth" ), \
                              pszDepth );
    DBFWriteIntegerAttribute( hCapabDB, nRec, \
                              DBFGetFieldIndex( hCapabDB, "queryable" ), \
                              nQueryable );

    if( ganEnableModule[2] == 1 )
    {
        if(nExtractID != -1 && gpapszExtractName[nExtractID] != NULL)
        {
            DBFWriteIntegerAttribute( hCapabDB, nRec,
                                      DBFGetFieldIndex(hCapabDB,"extractabl"),
                                      1 );
            DBFWriteStringAttribute( hCapabDB, nRec,
                                     DBFGetFieldIndex(hCapabDB,"datatype"),
                                     gpapszExtractType[nExtractID] );
            DBFWriteStringAttribute( hCapabDB, nRec,
                                     DBFGetFieldIndex(hCapabDB,"formats"),
                                     gpapszExtractFormat[nExtractID] );
            DBFWriteStringAttribute( hCapabDB, nRec,
                                     DBFGetFieldIndex(hCapabDB,"extracturl"),
                                     gpszExtractURL );
        }
        else
        {
            DBFWriteIntegerAttribute( hCapabDB, nRec,
                                      DBFGetFieldIndex(hCapabDB,"extractabl"),
                                      0 );
            DBFWriteStringAttribute( hCapabDB, nRec,
                                     DBFGetFieldIndex(hCapabDB,"datatype"),
                                     " " );
            DBFWriteStringAttribute( hCapabDB, nRec,
                                     DBFGetFieldIndex(hCapabDB,"formats"),
                                     " " );
            DBFWriteStringAttribute( hCapabDB, nRec,
                                     DBFGetFieldIndex(hCapabDB,"extracturl"),
                                     gpszExtractURL );
        }
    }
    //printf( "added layer as record %d\n", nRec );
}

int addBBox( xmlChar * pszSRS, float nMinX, float nMinY, float nMaxX, \
              float nMaxY, int bNextID )
{
    int nRec, nBBoxID, nNextID;


    nRec = DBFGetRecordCount( hBBoxDB );
    nBBoxID = getMaxID( hBBoxDB, "bbox_id" ) + 1;
    if (bNextID == 1)
        nNextID = nBBoxID + 1;
    else
        nNextID = -1;

    DBFWriteIntegerAttribute( hBBoxDB, nRec, \
                              DBFGetFieldIndex( hBBoxDB, "bbox_id" ), \
                              nBBoxID );
    DBFWriteStringAttribute( hBBoxDB, nRec, \
                              DBFGetFieldIndex( hBBoxDB, "SRS" ), \
                              pszSRS );
    DBFWriteDoubleAttribute( hBBoxDB, nRec, \
                              DBFGetFieldIndex( hBBoxDB, "minx" ), \
                              nMinX );
    DBFWriteDoubleAttribute( hBBoxDB, nRec, \
                              DBFGetFieldIndex( hBBoxDB, "miny" ), \
                              nMinY );
    DBFWriteDoubleAttribute( hBBoxDB, nRec, \
                              DBFGetFieldIndex( hBBoxDB, "maxx" ), \
                              nMaxX );
    DBFWriteDoubleAttribute( hBBoxDB, nRec, \
                              DBFGetFieldIndex( hBBoxDB, "maxy" ), \
                              nMaxY );
    DBFWriteIntegerAttribute( hBBoxDB, nRec, \
                              DBFGetFieldIndex( hBBoxDB, "next_id" ), \
                              nNextID );
    //printf( "added bbox as record %d\n", nRec );
    return nBBoxID;
}

int addStyle(xmlChar* pszName, xmlChar* pszTitle, xmlChar* pszLegendURL,
             int nLegendHeight, int nLegendWidth, xmlChar* pszLegendFormat,
             xmlChar* pszStyleSheetURL, xmlChar* pszStyleURL, int bNextID )
{
    int nRec, nStyleID, nNextID, nField;


    nRec = DBFGetRecordCount( hStyleDB );
    nStyleID = getMaxID( hStyleDB, "style_id" ) + 1;
    if (bNextID == 1)
        nNextID = nStyleID + 1;
    else
        nNextID = -1;

    DBFWriteIntegerAttribute( hStyleDB, nRec, 
                              DBFGetFieldIndex( hStyleDB, "style_id" ), 
                              nStyleID );
    DBFWriteStringAttribute( hStyleDB, nRec, \
                              DBFGetFieldIndex( hStyleDB, "name" ), \
                              pszName );
    DBFWriteStringAttribute( hStyleDB, nRec, \
                              DBFGetFieldIndex( hStyleDB, "title" ), \
                              pszTitle );
    DBFWriteStringAttribute( hStyleDB, nRec, \
                              DBFGetFieldIndex( hStyleDB, "legendurl" ), \
                              pszLegendURL );
    if((nField = DBFGetFieldIndex( hStyleDB, "leg_height" )) != -1)
        DBFWriteIntegerAttribute( hStyleDB, nRec, \
                                  nField, \
                                  nLegendHeight );
    if((nField = DBFGetFieldIndex( hStyleDB, "leg_width" )) != -1)
        DBFWriteIntegerAttribute( hStyleDB, nRec, \
                                  nField, \
                                  nLegendWidth );
    if((nField = DBFGetFieldIndex( hStyleDB, "leg_format" )) != -1)
        DBFWriteStringAttribute( hStyleDB, nRec, \
                                 nField, \
                                 pszLegendFormat );
    DBFWriteStringAttribute( hStyleDB, nRec, \
                              DBFGetFieldIndex( hStyleDB, "stylesheet" ), \
                              pszStyleSheetURL );
    DBFWriteStringAttribute( hStyleDB, nRec, \
                              DBFGetFieldIndex( hStyleDB, "styleurl" ), \
                              pszStyleURL );
    DBFWriteIntegerAttribute( hStyleDB, nRec, \
                              DBFGetFieldIndex( hStyleDB, "next_id" ), \
                              nNextID );
    //printf( "added style as record %d\n", nRec );
    return nStyleID;
}

/**
 * process all layers extract entries and put them in 3 global variables
 *
 * @param pContext xmlXPathContextPtr the context for XPath
 */
void processExtractLayer( xmlXPathContextPtr pContext )
{
    xmlChar * expression, * pszNodePath;
    xmlXPathObjectPtr result, result2;
    int i, j, numExtract, nFormatLen;
    xmlChar * pszName, * pszFormats=NULL, * pszType, * pszTempFormat;

    if( ganEnableModule[2] == 0 )
        return;

    //look for style entries
    gpszExtractURL = getXPathNodeAttribute( pContext, "//VendorSpecificCapabilities/OASIS_montaj/Extract/DCPType/HTTP/Get", "onlineResource" );

    result = xmlXPathEvalExpression( "//VendorSpecificCapabilities/OASIS_montaj/Extract/ExtractableLayers/ExtractableLayer", pContext );

    if ( result != NULL)
    {
        if ( result->nodesetval == NULL )
            xmlXPathFreeObject( result );
        else
        {
            // set global extract variables
            if (gpapszExtractName)
            {
                while(*gpapszExtractName != NULL)
                {
                    xmlFree(*gpapszExtractName);
                    gpapszExtractName++;
                }
                xmlFree(gpapszExtractName);
            }
            if (gpapszExtractType)
            {
                while(*gpapszExtractType != NULL)
                {
                    xmlFree(*gpapszExtractType);
                    gpapszExtractType++;
                }
                xmlFree(gpapszExtractType);
            }
            if (gpapszExtractFormat)
            {
                while(*gpapszExtractFormat != NULL)
                {
                    xmlFree(*gpapszExtractFormat);
                    gpapszExtractFormat++;
                }
                xmlFree(gpapszExtractFormat);
            }
            gpapszExtractName = (xmlChar**)xmlRealloc(gpapszExtractName,
                              (result->nodesetval->nodeNr+2)*sizeof(xmlChar*));
            gpapszExtractType = (xmlChar**)xmlRealloc(gpapszExtractType,
                              (result->nodesetval->nodeNr+2)*sizeof(xmlChar*));
            gpapszExtractFormat = (xmlChar**)xmlRealloc(gpapszExtractFormat,
                              (result->nodesetval->nodeNr+2)*sizeof(xmlChar*));
            
            //there were actually some elements
            for ( i=0; i<result->nodesetval->nodeNr; i++ )
            {
                pszNodePath = getXPathNodePath(result->nodesetval->nodeTab[i]);
                // Name
                pszName = getXPathNodeAttribute(pContext, pszNodePath, "name");
                gpapszExtractName[i] = xmlStrdup( pszName );
                gpapszExtractName[i+1] = NULL;
                // Type
                pszType = getXPathNodeAttribute(pContext, pszNodePath, "type");
                gpapszExtractType[i] = xmlStrdup( pszType );
                gpapszExtractType[i+1] = NULL;
                // Format
                expression = xmlConcat(pszNodePath, "/ExtractFormat");
                result2 = xmlXPathEvalExpression( expression, pContext );
                xmlFree(expression);
                nFormatLen = 10;
                pszFormats = (xmlChar*)xmlMalloc(sizeof(xmlChar)*nFormatLen);
                pszFormats[0] = '\0';
                if ( result2->nodesetval->nodeNr >= 1 )
                {
                    for ( j=0; j<result2->nodesetval->nodeNr; j++ )
                    {
                        //determine the format string
                        expression = getXPathNodePath( 
                            result2->nodesetval->nodeTab[j] );
                        pszTempFormat = getXPathNodeAttribute( pContext, 
                                                               expression, 
                                                               "name" );
                        xmlFree( expression );
                        //record it.
                        pszFormats = (xmlChar*) xmlRealloc(pszFormats, 
                                      sizeof(xmlChar)*(strlen(pszTempFormat)+2+nFormatLen));
                        strcat( pszFormats, pszTempFormat);
                        if ( j < result2->nodesetval->nodeNr - 1 )
                        {
                            strcat( pszFormats, ";" );
                        }
                        xmlFree( pszTempFormat );
                        nFormatLen += strlen(pszTempFormat)+2;
                    }   
                }
                gpapszExtractFormat[i] = xmlStrdup( pszFormats );
                gpapszExtractFormat[i+1] = NULL;

                xmlFree( pszName );
                xmlFree( pszType );
                xmlFree( pszFormats );
                xmlFree( pszNodePath );
                xmlXPathFreeObject( result2 );
            }
            xmlXPathFreeObject( result );
        }
    }
}

/**
 * get the ID of the extract layer corresponding to the name
 *
 * @param pszName xmlChar* the name of the layer we want the ID
 *
 * @return the ID of the layer with the name pszName
 */
int getExtractLayerID( xmlChar * pszName )
{
    int i=0, nExtractID=-1;

    if(gpapszExtractName == NULL)
        return -1;

    while(gpapszExtractName[i] != NULL)
    {
        if(strcasecmp( gpapszExtractName[i], pszName ) == 0 )
        {
            nExtractID = i;
            break;
        }
        i++;
    }

    if(nExtractID == -1)
        nExtractID = i;

    return nExtractID;
}

/**
 * process a layer's style entries
 *
 * @param pContext xmlXPathContextPtr the context for XPath
 * @param pszPath xmlChar* the path in the xml file to the layer element
 *
 * @return the id of the first style or -1 if none added
 */
int processLayerStyle( xmlXPathContextPtr pContext, xmlChar * pszPath , 
                       xmlChar * pszVersion)
{
    xmlChar * expression;
    xmlXPathObjectPtr result;
    int i, nFirstStyleID, nNextID, nLegendHeight, nLegendWidth;
    xmlChar *pszStyle=NULL, *pszName=NULL, *pszTitle=NULL, *pszLegendURL=NULL;
    xmlChar *pszStyleSheetURL=NULL, *pszStyleURL=NULL, *pszNodePath=NULL;
    xmlChar *pszLegendFormat=NULL;

    nFirstStyleID = -1;

    //look for style entries
    expression = xmlConcat( pszPath, "/Style" );
    result = xmlXPathEvalExpression( expression, pContext );
    xmlFree( expression );

    if ( result != NULL)
    {
        if ( result->nodesetval == NULL )
            xmlXPathFreeObject( result );
        else
        {
            //there were actually some elements
            for ( i=0; i<result->nodesetval->nodeNr; i++ )
            {
                pszNodePath = getXPathNodePath(result->nodesetval->nodeTab[i]);
                expression = xmlConcat( pszNodePath, "/Name" );
                pszName = getXPathNodeContent(pContext,expression);
                xmlFree( expression );
                expression = xmlConcat( pszNodePath, "/Title" );
                pszTitle = wmsTrim(getXPathNodeContent(pContext,expression));
                xmlFree( expression );
                // WMS 1.0.7 and earlier LegendURL
                if (strcasecmp(pszVersion, "1.0.7") <= 0) 
                {
                    expression = xmlConcat( pszNodePath, "/LegendURL" );
                    pszLegendURL = getXPathNodeContent(pContext,expression);
                    nLegendHeight = atoi(
                        getXPathNodeAttribute(pContext, expression,"height"));
                    nLegendWidth = atoi(
                        getXPathNodeAttribute(pContext, expression,"width"));
                    pszLegendFormat = 
                        getXPathNodeAttribute(pContext, expression,"format");
                    xmlFree( expression );
                }
                else //WMS 1.1 and later LegendURL
                {
                    xmlChar *pszVal;

                    expression =xmlConcat(pszNodePath,
                                          "/LegendURL/OnlineResource");
                    pszLegendURL = getXPathNodeAttribute( pContext, 
                                                          expression, "href" );
                    xmlFree( expression );

                    expression =xmlConcat(pszNodePath, "/LegendURL");
                    pszVal = getXPathNodeAttribute(pContext, expression, 
                                                   "height");
                    nLegendHeight = atoi(pszVal);
                    xmlFree(pszVal);

                    pszVal = getXPathNodeAttribute(pContext, expression, 
                                                   "width");
                    nLegendWidth = atoi(pszVal);
                    xmlFree(pszVal);

                    xmlFree( expression );
                    expression =xmlConcat(pszNodePath,
                                          "/LegendURL/Format");
                    pszLegendFormat = getXPathNodeContent(pContext,expression);
                    xmlFree( expression );
                }
                // WMS 1.0.7 and earlier StyleSheetURL
                if (strcasecmp(pszVersion, "1.0.7") <= 0) 
                {
                    expression = xmlConcat( pszNodePath, "/StyleSheetURL" );
                    pszStyleSheetURL =getXPathNodeContent(pContext,expression);
                }
                else //WMS 1.1 and later StyleSheetURL
                {
                    expression = xmlConcat(pszNodePath,
                                          "/StyleSheetURL/OnlineResource" );
                    pszStyleSheetURL = getXPathNodeAttribute( pContext, 
                                                          expression, "href" );
                }
                xmlFree( expression );
                // WMS 1.0.7 and earlier StyleURL
                if (strcasecmp(pszVersion, "1.0.7") <= 0) 
                {
                    expression = xmlConcat( pszNodePath, "/StyleURL" );
                    pszStyleURL = getXPathNodeContent(pContext,expression);
                }
                else //WMS 1.1 and later StyleURL
                {
                    expression =xmlConcat(pszNodePath,
                                          "/StyleURL/OnlineResource");
                    pszStyleURL = getXPathNodeAttribute( pContext, 
                                                          expression, "href" );
                }
                xmlFree( expression );

                if (i < result->nodesetval->nodeNr - 1)
                    nNextID = 1;
                else
                    nNextID = 0;
                if ( i==0 )
                    nFirstStyleID = addStyle( pszName, pszTitle, pszLegendURL, 
                                              nLegendHeight, nLegendWidth,
                                              pszLegendFormat,pszStyleSheetURL,
                                              pszStyleURL, nNextID );
                else
                    addStyle( pszName, pszTitle, pszLegendURL, 
                              nLegendHeight, nLegendWidth, pszLegendFormat, 
                              pszStyleSheetURL, pszStyleURL, 
                              nNextID );

                xmlFree( pszNodePath );
                xmlFree( pszName );
                xmlFree( pszTitle );
                xmlFree( pszStyle );
                xmlFree( pszLegendURL );
                xmlFree( pszStyleSheetURL );
                xmlFree( pszStyleURL );
                xmlFree( pszLegendFormat );
            }
            xmlXPathFreeObject( result );
        }
    }
    //all done, return the first id (or -1);
    return nFirstStyleID;
}

/**
 * process a layer's bounding box entries
 *
 * @param pContext xmlXPathContextPtr the context for XPath
 * @param pszPath xmlChar* the path in the xml file to the layer
 *
 * @return the id of the first bounding box or -1 if none added
 */
int processLayerBBox( xmlXPathContextPtr pContext, xmlChar * pszPath )
{
    xmlChar * expression;
    xmlXPathObjectPtr result;

    float minx, miny, maxx, maxy;
    int i, nFirstBBoxID, nNextID;
    xmlChar * pszBBox, * pszSRS, *pszNodePath;

    nFirstBBoxID = -1;

    //look for bounding box entries
    expression = xmlConcat( pszPath, "/BoundingBox" );
    result = xmlXPathEvalExpression( expression, pContext );
    xmlFree( expression );

    if ( result != NULL)
    {
        if ( result->nodesetval == NULL )
            xmlXPathFreeObject( result );
        else
        {
            //there were actually some bboxes
            for ( i=0; i<result->nodesetval->nodeNr; i++ )
            {
                pszNodePath = getXPathNodePath(result->nodesetval->nodeTab[i]);
                pszSRS = getXPathNodeAttribute(pContext,pszNodePath, "SRS");
                pszBBox = getXPathNodeAttribute(pContext,pszNodePath,"minx");
                if ( !xmlStrEqual(pszBBox, "") ) minx = atof( pszBBox );
                xmlFree( pszBBox );
                pszBBox = getXPathNodeAttribute(pContext,pszNodePath,"miny");
                if ( !xmlStrEqual(pszBBox, "") ) miny = atof( pszBBox );
                xmlFree( pszBBox );
                pszBBox = getXPathNodeAttribute(pContext,pszNodePath,"maxx");
                if ( !xmlStrEqual( pszBBox, "") ) maxx = atof( pszBBox );
                xmlFree( pszBBox );
                pszBBox = getXPathNodeAttribute(pContext,pszNodePath,"maxy");
                if ( !xmlStrEqual(pszBBox, "") ) maxy = atof( pszBBox );
                xmlFree( pszBBox );

                if (i < result->nodesetval->nodeNr - 1)
                    nNextID = 1;
                else
                    nNextID = 0;
                if ( i==0 )
                    nFirstBBoxID = addBBox( pszSRS, minx, miny, maxx, \
                                            maxy, nNextID );
                else
                    addBBox( pszSRS, minx, miny, maxx, maxy, nNextID );
                xmlFree( pszSRS );
                xmlFree( pszNodePath );
            }
            xmlXPathFreeObject( result );
        }
    }
    //all done, return the first id (or -1);
    return nFirstBBoxID;
}

/**
 * add a layer to the dbase file then recursively
 * scan the xml tree of capabilities to add sub-layers
 * to the database structure
 *
 * @param nLayerID int the ID of the layer to add
 * @param nServerID int the ID of the server in the server.dbf file
 * @param pContext xmlXPathContextPtr the context for XPath
 * @param pszPath xmlChar* the path to the layer in the xml file
 * @param pszSRS char* a comma separated list of SRS line numbers
 * @param ll_minx int lat/long minimum x
 * @param ll_miny int lat/long minimum y
 * @param ll_maxx int lat/long maximum x
 * @param ll_maxy int lat/long maximum y
 * @param bb_id int bounding box id
 * @param char * pszDepth
 *
 * @return integer the id of the last layer returned.
 */
int add_layer(int nLayerID, int nServerID, xmlXPathContextPtr pContext, 
              xmlChar * pszVersion, xmlChar* pszPath, 
              xmlChar* pszSRS, float ll_minx, float ll_miny, float ll_maxx, 
              float ll_maxy, int bb_id, char * pszDepth)
{
    //working variables
    xmlXPathObjectPtr result;
    char * pszNewDepth, *pszQueryable;
    int i, nSRSLen;
    //all the info about a layer
    xmlChar *pszName, *pszTitle, *pszNewSRS, *pszThisSRS, *pszAbstractContent;
    xmlChar *expression, *expression2, *pszBBox, *pszTempSRS;
    xmlChar *pszTempAbstract, *pszStrAbstract, szAbstract[15];
    int nQueryable=0;
    float fll_minx, fll_miny, fll_maxx, fll_maxy;
    int nBBoxID, nStyleID, nExtractID;
    int nAbstractID;

    fll_minx = ll_minx;
    fll_miny = ll_miny;
    fll_maxx = ll_maxx;
    fll_maxy = ll_maxy;

    //get the layer name
    expression = xmlConcat( pszPath, "/Name[1]" );
    pszName = getXPathNodeContent( pContext, expression );
    xmlFree( expression );

    //get the layer title
    expression = xmlConcat( pszPath, "/Title[1]" );
    pszTitle = wmsTrim( getXPathNodeContent( pContext, expression ) );
    xmlFree( expression );

    //check for a queryable attribute
    pszQueryable = getXPathNodeAttribute( pContext, pszPath, "queryable" );
    if (pszQueryable != "")
        nQueryable = atoi(pszQueryable);
    xmlFree( pszQueryable );

    //check for SRS entries
    expression = xmlConcat( pszPath, "/SRS" );
    result = xmlXPathEvalExpression( expression, pContext );

    //if the result is NULL, don't do anything (no SRS)
    if ( result == NULL )
    {
        xmlFree( expression );
    }
    else
    {
        //if the nodesetval is NULL, clean up  and stop processing (no SRS)
        if ( result->nodesetval == NULL )
        {
            xmlXPathFreeObject( result );
            xmlFree( expression );
        }
        else //we have an SRS tag (at least one)
        {
            //iterate through the nodeset results and process each SRS
            for ( i=0; i<result->nodesetval->nodeNr; i++ )
            {
                expression2 = getXPathNodePath(result->nodesetval->nodeTab[i]);
                
                pszThisSRS = getXPathNodeContent( pContext, expression2 );
                xmlFree( expression2 ); //??
                //remove and \n and \ chars in the SRS
                pszTempSRS = wmsEncode(pszThisSRS);
                //write the SRS to the current SRS file
                fprintf( hSRSfile, "%s ", pszTempSRS );
                //all done with pszThisSRS for this loop
                xmlFree(pszThisSRS);
                xmlFree(pszTempSRS);
            }
            //done with the expression
            xmlFree( expression );
            //write a line feed to finish the SRS line
            if( result->nodesetval->nodeNr > 0 )
            {
                fprintf( hSRSfile, "\n" );

                //keep track of the SRS line numbers
                nSRSLen = strlen(pszSRS);
                if (nSRSLen == 0)
                    sprintf( pszSRS + nSRSLen, "%d", nSRSLine );
                else
                    sprintf( pszSRS + nSRSLen, ",%d", nSRSLine );
                nSRSLine ++;
            }
            //if the srs gets bigger than 255 we're in trouble because that's
            //all that the dbf format can handle without memos and PHP doesn't
            //support memos
            nSRSLen = strlen(pszSRS);
            if (nSRSLen > 255)
            {
                //clean up and fail.
                return 0;
            }
        }
    }
    //all done, free the result
    xmlXPathFreeObject( result );   

  if(ganEnableModule[0] == 1)
  {
    //check for Abstract entries
    expression = xmlConcat( pszPath, "/Abstract" );
    pszAbstractContent =  getXPathNodeContent( pContext, expression );
    szAbstract[0] = '\0';

    //if the result is NULL, don't do anything (no Abstract)
    if ( !pszAbstractContent || strcasecmp(pszAbstractContent, "") == 0)
    {
        xmlFree(pszAbstractContent);
        xmlFree( expression );
        nAbstractID = -1;
    }
    else
    {
        xmlChar *pszEncodedAbstract;

        pszEncodedAbstract = wmsEncode(pszAbstractContent);
        //write the Abstract to the current Abstract file
        fprintf( hAbstractFile, "%s ", pszEncodedAbstract );
        //write a line feed to finish the Abstract line and change layer
        fprintf( hAbstractFile, "\n" );
        //all done
        xmlFree(pszEncodedAbstract);
        xmlFree(pszAbstractContent);
        xmlFree( expression );

        nAbstractID = nAbstractLine;
        nAbstractLine ++;
        
    }
  }
  else
  {
      nAbstractID = -1;
  }
    
    //check for a latlon bounding box
    expression = xmlConcat( pszPath, "/LatLonBoundingBox[1]" );
    result = xmlXPathEval( expression, pContext );
    if (result != NULL && result->nodesetval != NULL \
        && result->nodesetval->nodeNr > 0)
    {
      pszBBox = getXPathNodeAttribute( pContext, expression, "minx" );
      if ( !xmlStrEqual( pszBBox, "" ) ) fll_minx = atof( pszBBox );
      xmlFree(pszBBox);
      pszBBox = getXPathNodeAttribute( pContext, expression, "miny" );
      if ( !xmlStrEqual( pszBBox, "" ) ) fll_miny = atof( pszBBox );
      xmlFree(pszBBox);
      pszBBox = getXPathNodeAttribute( pContext, expression, "maxx" );
      if ( !xmlStrEqual( pszBBox, "" ) ) fll_maxx = atof( pszBBox );
      xmlFree(pszBBox);
      pszBBox = getXPathNodeAttribute( pContext, expression, "maxy" );
      if ( !xmlStrEqual( pszBBox, "") ) fll_maxy = atof( pszBBox );
      xmlFree(pszBBox);
    }
    xmlFree( expression );
    expression = NULL;
    if (result != NULL)
    {
        xmlXPathFreeObject( result ); 
        result = NULL;
    }

    //now check for extract entries
    if( ganEnableModule[2] == 1 )
    {
        nExtractID = getExtractLayerID( pszName );
    }

    //now check for bounding box entries
    nBBoxID = processLayerBBox( pContext, pszPath );

    //now check for style entries
    if( ganEnableModule[1] == 1 )
    {
        nStyleID = processLayerStyle( pContext, pszPath, pszVersion );
    }

    addCapability( nLayerID, nServerID, pszName, pszTitle, pszSRS, 
                   fll_minx, fll_miny, fll_maxx, fll_maxy, 
                   nBBoxID, nStyleID, pszDepth, nQueryable, nAbstractID, 
                   nExtractID);

    //don't need these anymore
    xmlFree( pszName );
    xmlFree( pszTitle );

    //okay, now parse for children
    expression = xmlConcat( pszPath, "/Layer" );
    result = xmlXPathEvalExpression( expression, pContext );

    //if the result is NULL, just return
    if ( result == NULL )
    {
        xmlFree( expression );
        return 0;
    }

    //if the nodesetval is NULL, free it and return
    if ( result->nodesetval == NULL )
    {
        xmlXPathFreeObject( result );
        xmlFree( expression );
        return 0;
    }

    //iterate through the nodeset results and add each layer
    for ( i=0; i<result->nodesetval->nodeNr; i++ )
    {
        expression2 = getXPathNodePath( result->nodesetval->nodeTab[i] );
        pszNewDepth = xmlConcat( pszDepth, "." );
        //TODO check return value
        nLayerID = add_layer( nLayerID + 1, nServerID, pContext, pszVersion, 
                              expression2, pszSRS, fll_minx, fll_miny, 
                              fll_maxx, fll_maxy, bb_id, pszNewDepth );
        //if return value is 0 then something went wrong in add_layer somewhere
        if ( nLayerID == 0 )
        {
            return 0;
        }
        pszSRS[nSRSLen] = '\0';
        xmlFree( pszNewDepth );
        xmlFree( expression2 ); //??
    }
    xmlFree( expression );

    //all done, free the result
    xmlXPathFreeObject( result );

    //return the last layer added.
    return nLayerID;
}


/**
 * Function to enable optional tags.
 * There is 3 module to check: 1-Abstract 2-Style 3-Extract 4-capab_url
 *
 */
int enableOptions()
{

  // enable module

  if( DBFGetFieldIndex( hCapabDB, "abstractid" ) >= 0 )
      ganEnableModule[0] = 1;
  else
      ganEnableModule[0] = 0;

  if( DBFGetFieldIndex( hCapabDB, "style_id" ) >= 0 )
      ganEnableModule[1] = 1;
  else
      ganEnableModule[1] = 0;

  if( DBFGetFieldIndex( hCapabDB, "extractabl" ) >= 0 )
      ganEnableModule[2] = 1;
  else
      ganEnableModule[2] = 0;

  if( DBFGetFieldIndex( hServerDB, "capab_url" ) >= 0 )
      ganEnableModule[3] = 1;
  else
      ganEnableModule[3] = 0;

  return 0;
}


/**
 * application mainline
 *
 * expects several arguments - xml file name, server db name, capabilities
 * db name, bbox db name, style db name, srs text file name, and server id
 *
 * xml file name - the path and name of an WMS Capabilities XML file
 *
 * server db name - the path and name of a dbase file for server info
 *
 * capabilities db name - the path and name of a dbase for for layer info
 *
 * bbox db name - the path and name of a dbase file for bounding box info
 *
 * srs file name - the path and name of a text file for srs info
 *
 * abstract file name - the path and name of a text file for abstract info
 *
 * server id - the id of the server we are parsing for (in the server db)
 */
int main( int argc, char * argv[] )
{
  //declare all vars first, dummy
  xmlDocPtr pDoc;
  xmlXPathContextPtr pContext;
  xmlChar * expression, *expression2;
  xmlXPathObjectPtr result;
  xmlXPathObjectPtr result2;
  xmlNodePtr pCurrentNode;
  int nNodeSetLength;
  int i, ret;
  char szSRS[300] = ""; //SRS values can only get to 255 chars

  int nServerIDFld, nCapabURLFld,nMapURLFld,nVersionFld, nFormatsFld, nNameFld;
  int nServerID, nLayerID, nDirLen;
  int nServerRec = -1;

  xmlChar * pszVersion, * pszFormat, * pszDirectory, * pszFile;
  xmlChar * pszOnlineResource, * pszMapResource;
  xmlChar * pszServiceTitle;
  char   szFormats[255]; //max size of a dbf field anyway
  char *pszSRSFileName, *pszAbstractFileName;
  int nSRSSize, nAbstractSize;

  //printf( "in my program !!!!\n" );

  // Return version information with '-v' command-line switch
  if ( argc == 2 && strcasecmp(argv[1], "-v") == 0 )
  {
      printf( "WMSPARSE - ($Revision: 1.35 $, $Date: 2003/05/12 15:04:24 $)\n");
      return 0;
  }
  if ( argc < 8 && argc != 3 && argc != 4 )
  {
      printf( "WMSPARSE - ($Revision: 1.35 $, $Date: 2003/05/12 15:04:24 $)\n\n");
      printf( "ERROR: invalid number of arguments!\n\n" );
      printf( "Usage: \n");
      printf( " %s -v\n", argv[0] );
      printf( "or\n" );
      printf( " %s xmlfile directory [serverid]\n", argv[0] );
      printf( "or\n" );
      printf( " %s xmlfile serverdb layerdb ", argv[0] );
      printf( "bboxdb styledb\n\tsrsfile abstractfile [serverid]\n\n" );
      return -1;
  }

  if(argc == 3 || argc == 4)
  {
      pszDirectory = xmlStrdup(argv[2]);
      nDirLen = strlen(pszDirectory);
      if(pszDirectory[nDirLen - 1] != '/' && pszDirectory[nDirLen - 1] != '\\')
          pszDirectory = xmlConcat( pszDirectory, "/" );

      //look for an xml document in argv
      pDoc = xmlParseFile( argv[1] );

      if ( pDoc == NULL )
      {
          printf( "error opening or parsing %s\n", argv[1] );
          return -2;
      }

      //open the server dbf file
      expression = xmlConcat( pszDirectory, "server.dbf" );
      hServerDB = DBFOpen( expression, "rb+" );
      if ( hServerDB == NULL )
      {
          printf( "invalid Server DBF file %s\n", expression );
          return -3;
      }

      //open the capability dbf file
      expression = xmlConcat( pszDirectory, "layer.dbf" );
      hCapabDB = DBFOpen( expression, "rb+" );
      if ( hCapabDB == NULL )
      {
          printf( "invalid Capability DBF file %s\n", expression );
          return -4;
      }

      //open the bbox dbf file
      expression = xmlConcat( pszDirectory, "bbox.dbf" );
      hBBoxDB = DBFOpen( expression, "rb+" );
      if ( hBBoxDB == NULL )
      {
          printf( "invalid BBox DBF file %s\n", expression );
          return -5;
      }

      // pre parse to enable optional tags
      ret = enableOptions();
      if( ret < 0 )
      {
          return ret;
      }

      //open the style dbf file
      if( ganEnableModule[1] == 1 )
      {
          expression = xmlConcat( pszDirectory, "style.dbf" );
          hStyleDB = DBFOpen( expression, "rb+" );
          if ( hStyleDB == NULL )
          {
              printf( "invalid Style DBF file %s\n", expression );
              return -6;
          }
      }

      //get server ID
      if ( argc == 4 )
          nServerID = atoi(argv[3]);
      else
          nServerID = -1;

      //open the srs txt file
      pszFile = (xmlChar*)xmlMalloc(  sizeof(xmlChar) * 15 );
      sprintf(pszFile, "s%dsrs.txt", nServerID );
      expression = xmlConcat( pszDirectory, pszFile );
      pszSRSFileName = xmlStrdup( expression );
      hSRSfile = fopen( pszSRSFileName, "w" );
      if ( hSRSfile == NULL )
      {
          printf( "invalid SRS file file %s\n", expression );
          return -7;
      }
      xmlFree(pszFile);

      //open the abstract txt file
      if( ganEnableModule[0] == 1 )
      {
          pszFile = (xmlChar*)xmlMalloc(  sizeof(xmlChar) * 20 );
          sprintf(pszFile, "a%dabstract.txt", nServerID );
          expression = xmlConcat( pszDirectory, pszFile );
          pszAbstractFileName = xmlStrdup( expression );
          hAbstractFile = fopen( pszAbstractFileName, "w" );
          if ( hAbstractFile == NULL )
          {
              printf( "invalid abstract file file %s\n", expression );
              return -8;
          }
          xmlFree(pszFile);
      }
  }
  else
  {
      //look for an xml document in argv
      pDoc = xmlParseFile( argv[1] );

      if ( pDoc == NULL )
      {
          printf( "error opening or parsing %s\n", argv[1] );
          return -2;
      }

      //open the server dbf file
      hServerDB = DBFOpen( argv[2], "rb+" );
      if ( hServerDB == NULL )
      {
          printf( "invalid Server DBF file %s\n", argv[2] );
          return -3;
      }

      //open the capability dbf file
      hCapabDB = DBFOpen( argv[3], "rb+" );
      if ( hCapabDB == NULL )
      {
          printf( "invalid Capability DBF file %s\n", argv[3] );
          return -4;
      }

      //open the bbox dbf file
      hBBoxDB = DBFOpen( argv[4], "rb+" );
      if ( hBBoxDB == NULL )
      {
          printf( "invalid BBox DBF file %s\n", argv[4] );
          return -5;
      }

      // pre parse to enable optional tags
      ret = enableOptions();
      if( ret < 0 )
      {
          return ret;
      }

      //open the style dbf file
      if( ganEnableModule[1] == 1 )
      {
          hStyleDB = DBFOpen( argv[5], "rb+" );
          if ( hStyleDB == NULL )
          {
              printf( "invalid Style DBF file %s\n", argv[5] );
              return -6;
          }
      }

      //open the SRS txt file
      hSRSfile = fopen( argv[6], "w" );
      pszSRSFileName = xmlStrdup( argv[6] );
      if ( hSRSfile == NULL )
      {
          printf( "invalid SRS file file %s\n", argv[6] );
          return -7;
      }

      //open the abstract txt file
      if( ganEnableModule[0] == 1 )
      {
          hAbstractFile = fopen( argv[7], "w" );
          pszAbstractFileName = xmlStrdup( argv[7] );
          if ( hAbstractFile == NULL )
          {
              printf( "invalid abstract file file %s\n", argv[7] );
              return -8;
          }
      }
      if ( argc == 9 )
          nServerID = atoi(argv[8]);
      else
          nServerID = -1;
  }

  //initialize XPath
  xmlXPathInit();

  //get an xPath context
  pContext = xmlXPathNewContext( pDoc );

  pszVersion = getXPathNodeAttribute( pContext, \
                                     "/WMT_MS_Capabilities[@version]", \
                             "version" );

  //printf( "WMT MS Version is %s\n", pszVersion );
  
  //check for WMS Version Differences
  if( ganEnableModule[3] == 1)
  {
      if (strcasecmp(pszVersion, "1.0.7") <= 0) // WMS 1.0.7 and earlier
          pszOnlineResource = getXPathNodeAttribute( pContext, \
                                           "//Capabilities/DCPType/HTTP/Get", \
                                              "onlineResource" );
      else //WMS 1.1 and later
          pszOnlineResource = getXPathNodeAttribute( pContext, \
                           "//GetCapabilities/DCPType/HTTP/Get/OnlineResource",
                           "href" );

      //printf( "online resource is %s\n", pszOnlineResource );
  }
  if (strcasecmp(pszVersion, "1.0.7") <= 0) // WMS 1.0.7 and earlier
      pszMapResource = getXPathNodeAttribute( pContext, \
                                             "//Map/DCPType/HTTP/Get", \
                                             "onlineResource" );
  else //WMS 1.1 and later
      pszMapResource = getXPathNodeAttribute( pContext, \
                                    "//GetMap/DCPType/HTTP/Get/OnlineResource",
                                    "href" );

  //find the GetMap formats
  if (strcasecmp(pszVersion, "1.0.7") <= 0) // WMS 1.0.7 and earlier
      expression = "//Capability/Request/Map/Format/descendant::*";
  else //WMS 1.1 and later
      expression = "//Capability/Request/GetMap/Format";
  result = xmlXPathEvalExpression( expression, pContext );
  if ( result->nodesetval->nodeNr < 1 )
  {
    printf( "no map formats?\n" );
  }
  else
  {
      szFormats[0] = '\0';
      for ( i=0; i<result->nodesetval->nodeNr; i++ )
      {
          //determine the format string
          if (strcasecmp(pszVersion, "1.0.7") <= 0) // WMS 1.0.7 and earlier
          {
              pszFormat = xmlStrdup( result->nodesetval->nodeTab[i]->name );
          }
          else
          {
              expression2 = getXPathNodePath( result->nodesetval->nodeTab[i] );
              pszFormat = getXPathNodeContent( pContext, expression2 );
              xmlFree( expression2 );
          }
          //record it.
          strcat( szFormats, pszFormat);
          if ( i < result->nodesetval->nodeNr - 1 )
          {
              strcat( szFormats, "," );
          }
          xmlFree( pszFormat );
      }   
  }
  //printf( "formats: %s\n", szFormats );
  xmlXPathFreeObject( result );

  pszServiceTitle = wmsTrim(getXPathNodeContent( pContext, "//Service/Title"));
  //printf( "service title is %s\n", pszServiceTitle );

  //that worked, put the info into the server dbf
  //first find the server record
  nServerIDFld = DBFGetFieldIndex( hServerDB, "server_id" );
  nCapabURLFld = DBFGetFieldIndex( hServerDB, "capab_url" );
  nMapURLFld = DBFGetFieldIndex( hServerDB, "map_url" );
  nFormatsFld = DBFGetFieldIndex( hServerDB, "formats" );
  nNameFld = DBFGetFieldIndex( hServerDB, "svc_title" );
  nVersionFld = DBFGetFieldIndex( hServerDB, "version" );

  if ( nServerID == -1 )
  {
      //add a server record (mostly for testing)
      nServerRec = DBFGetRecordCount( hServerDB );
      nServerID = DBFReadIntegerAttribute( hServerDB, \
                                           nServerRec-1, nServerIDFld) + 1;
  }
  else
  {
      for ( i=0; i<DBFGetRecordCount( hServerDB ); i++ )
      {
          if (nServerID==DBFReadIntegerAttribute(hServerDB,i,nServerIDFld))
          {
              nServerRec = i;
              break;
          }
      }
  }

  if( ganEnableModule[3] == 1 )
  {
      if ( nCapabURLFld > - 1 )
          DBFWriteStringAttribute( hServerDB, nServerRec, nCapabURLFld, pszOnlineResource );
  }
  if ( nMapURLFld > - 1 )
      DBFWriteStringAttribute( hServerDB, nServerRec, nMapURLFld, pszMapResource );
  if ( nVersionFld > -1 )
      DBFWriteStringAttribute( hServerDB, nServerRec, nVersionFld, pszVersion );

  if ( nFormatsFld > -1 )
      DBFWriteStringAttribute( hServerDB, nServerRec, nFormatsFld, szFormats );

  if ( nNameFld > -1 )
      DBFWriteStringAttribute( hServerDB, nServerRec, nNameFld, pszServiceTitle );

  //parse the layers
  expression = "/WMT_MS_Capabilities[1]/Capability[1]/Layer[1]";
  result = xmlXPathEvalExpression( expression, pContext );
  if ( result == NULL )
  {
          printf( "root layer invalid!" );
          return -9;
  }
  if ( result->nodesetval == NULL || result->nodesetval->nodeNr < 1 )
  {
    printf( "root layer was valid but had no sub-layers\n" );
    return -10;
  }
  else
  {
      //get the maximum layer id to start adding at
      nLayerID = getMaxID( hCapabDB, "layer_id" ) + 1;

      //check for extract entries
      if( ganEnableModule[2] == 1 )
      {
          processExtractLayer( pContext );
      }

      //recursively add all the layers.
      if (add_layer( nLayerID, nServerID, pContext, pszVersion, \
                 "/WMT_MS_Capabilities[@version]/Capability[1]/Layer[1]", \
                 szSRS, 0.0, 0.0, 0.0, 0.0, -1, "." )  == 0)
      {
          printf( "add_layer failed?\n" );
          return -11;
      }
  }
  xmlXPathFreeObject( result );

  //free the context and doc. tree
  xmlXPathFreeContext( pContext );
  xmlFreeDoc(pDoc);

  xmlFree( pszVersion );
  if( ganEnableModule[3] == 1 )
      xmlFree( pszOnlineResource );
  xmlFree( pszMapResource );
  xmlFree( pszServiceTitle );

  //close file handles
  DBFClose( hServerDB );
  DBFClose( hCapabDB );
  DBFClose( hBBoxDB );
  if( ganEnableModule[1] == 1 )
  {
      DBFClose( hStyleDB );
  }
  fclose( hSRSfile );
  if( ganEnableModule[0] == 1 )
  {
      fclose( hAbstractFile );
  }

  // Check if SRS and abstract file are empty. if empty erase it
  hSRSfile = fopen(pszSRSFileName, "rb");
  fseek(hSRSfile, 0, SEEK_END);
  nSRSSize = ftell(hSRSfile);
  fclose(hSRSfile);
  if(nSRSSize == 0)
      remove(pszSRSFileName);
  free(pszSRSFileName);

  if( ganEnableModule[0] == 1 )
  {
      hAbstractFile = fopen(pszAbstractFileName, "rb");
      fseek(hAbstractFile, 0, SEEK_END);
      nAbstractSize = ftell(hAbstractFile);
      fclose(hAbstractFile);
      if(nAbstractSize == 0)
          remove(pszAbstractFileName);
      free(pszAbstractFileName);
  }

  //all done.
  return 0;
}

