<?php
/**
 * SearchAttribute  Class definition of search attribute
 *
 * @project     PHPGeoCoder
 * @revision    $Id:
 * @purpose     Class definition of search attribute.
 * @author      Sacha Fournier (sfournier@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/**
 * Check for a defined COMMON variable containing the absolute path.  If not 
 * found then check ../ then ./ else failure.
 */
if ( defined( "COMMON" ) && is_dir( COMMON ) )
{
    // check for closing "\" or "/"
    if ( substr( COMMON, strlen( COMMON )- 1, 1 ) == "\\" ||
         substr( COMMON, strlen( COMMON ) - 1, 1 ) == "/" )
        include_once( COMMON."logger/logger.php");
    else
        include_once( COMMON."/logger/logger.php");
}
elseif (file_exists("../logger/logger.php"))
{
    include_once("../logger/logger.php");
}
else
{
    include_once("./logger/logger.php");
} 

/**
 *  Array of Canadian provinces with
 *  equivalent code.
 */
$aszCanadianProvince = array();
$aszCanadianProvince["NF"] = 10;
$aszCanadianProvince["PE"] = 11;
$aszCanadianProvince["NS"] = 12;
$aszCanadianProvince["NB"] = 13;
$aszCanadianProvince["QC"] = 24;
$aszCanadianProvince["ON"] = 35;
$aszCanadianProvince["MB"] = 46;
$aszCanadianProvince["SK"] = 47;
$aszCanadianProvince["AB"] = 48;
$aszCanadianProvince["BC"] = 59;
$aszCanadianProvince["YT"] = 60;
$aszCanadianProvince["NT"] = 61;

class SearchAttribute extends Logger
{
    var $szName;
    var $szValue;
    var $aszPossibleValue;

    // This value is used when looking for possible values
    // from an array for a specific value. Values are compared
    // and can match from 0 to 100 (0 => not match at all, and
    // 100 perfect match).
    var $nMinimumMatchScore = 80;

    function SearchAttribute($szAttributeName="", $szAttributeValue="")
    {
        parent::Logger("SearchAttribute");

        $this->log(LOG_VERBOSE, "SearchAttribute($szAttributeName, $szAttributeValue)");

        $this->szName = $szAttributeName;
        $this->szValue = $szAttributeValue;

        $this->aszPossibleValue = array();
    }


    /**
     *  This function check for a equivalent in array for
     *  szValue. Array must be structured that way:
     */
    function checkMatch($szValue)
    {
        $this->log(LOG_VERBOSE, "checkMatch($szValue)");

        $nScore = 0;
        $szFinalValue = "";

        foreach(array_keys($this->aszPossibleValue) as $szKey1)
        {
            foreach(array_keys($this->aszPossibleValue[$szKey1]) as $szKey2)
            {
                // Look for similirarities
                similar_text(strtoupper($szValue), $this->aszPossibleValue[$szKey1][$szKey2], $nTmpScore);

                // Check if score is at least the minimum required
                // and check if greater than last greater value.
                if ($nTmpScore >= $this->nMinimumMatchScore && 
                    $nTmpScore > $nScore)
                {
                    $nScore = $nTmpScore;
                    $szFinalValue = $this->aszPossibleValue[$szKey1][0];
                }
            }
        }

        return $szFinalValue;
    }

    function getName()
    {
        return $this->szName;
    }

    /**
     *  Return generic value (not the original) from 
     *  aszPossibleValues array. If not set return
     *  value entered by user.
     */
    function getValue()
    {
        if (count($this->aszPossibleValue) > 0)
        {
            $szValue = $this->checkMatch($this->szValue);

            if ($szValue == "")
                $szValue = $this->szValue;
        }
        else
            $szValue = $this->szValue;

        return $szValue;
    }
}

class PostalCode extends SearchAttribute
{
    function PostalCode($szAttributeValue)
    {
        parent::SearchAttribute("POSTAL_CODE", $szAttributeValue);

    }
}

class PlaceName extends SearchAttribute
{
    function PlaceName($szAttributeValue)
    {
        parent::SearchAttribute("PLACE_NAME", $szAttributeValue);
    }
}

class WatershedCode extends SearchAttribute
{
    function WatershedCode($szAttributeValue)
    {
        parent::SearchAttribute("CODE", $szAttributeValue);
    }
}

class MapSheet extends SearchAttribute
{
    function MapSheet($szAttributeValue)
    {
        parent::SearchAttribute("MAP_SHEET", $szAttributeValue);
    }
}

class Scale extends SearchAttribute
{
    function Scale($nScaleValue)
    {
        parent::SearchAttribute("SCALE", $nScaleValue);
    }
}


class StateProvince extends SearchAttribute
{
    function StateProvince($szAttributeValue)
    {
        parent::SearchAttribute("STATE_PROVINCE", $szAttributeValue);

        array_push($this->aszPossibleValue, array("NF", "10", "NEWFOUNDLAND", "TERRE NEUVE"));
        array_push($this->aszPossibleValue, array("PE", "11", "PRINCE EDWARD ISLAND", "ILE DU PRINCE EDOUARD"));
        array_push($this->aszPossibleValue, array("NS", "12", "NOVA SCOTIA", "NOUVELLE ECOSSE"));
        array_push($this->aszPossibleValue, array("NB", "13", "NEW BRUNSWICK", "NOUVEAU BRUNSWICK"));
        array_push($this->aszPossibleValue, array("QC", "24", "QUEBEC", "PQ"));
        array_push($this->aszPossibleValue, array("ON", "35", "ONTARIO"));
        array_push($this->aszPossibleValue, array("MB", "46", "MANITOBA"));
        array_push($this->aszPossibleValue, array("SK", "47", "SASKATCHEWAN"));
        array_push($this->aszPossibleValue, array("AB", "48", "ALBERTA"));
        array_push($this->aszPossibleValue, array("BC", "59", "BRITISH COLUMBIA", "COLOMBIE BRITANIQUE"));
        array_push($this->aszPossibleValue, array("YT", "60", "YUKON", "YUKON TERRITORY", "TERRITOIRE DU YUKON"));
        array_push($this->aszPossibleValue, array("NT", "61", "NORTHWEST", "NORTHWEST TERRITORIES", "TERRITOIRES DU NORD OUEST"));
    }
}

class Country extends SearchAttribute
{
    function Country($szAttributeValue)
    {
        parent::SearchAttribute("COUNTRY", $szAttributeValue);

        array_push($this->aszPossibleValue, array("CA", "CANADA"));
        array_push($this->aszPossibleValue, array("US", "UNITED STATES", "ETATS UNIES"));
    }
}

class LatLong extends SearchAttribute
{
    function LatLong($nLat=0, $nLong=0)
    {
        parent::SearchAttribute("LAT_LONG", $nLat.",".$nLong);
    }

    function getLat()
    {
        $aszLatLong = explode(",", $this->szValue);

        return $aszLatLong[0];
    }

    function getLong()
    {
        $aszLatLong = explode(",", $this->szValue);

        return $aszLatLong[1];
    }
}

class Polygon extends SearchAttribute
{
    function Polygon($aPoints=array())
    {
        $szValue = "";
        foreach($aPoints as $aPoint)
        {
            $szValue .= $aPoint[0].",".$aPoint[1];
        }
          
        parent::SearchAttribute("POLYGON", $szValue);
    }

    function getPoint($nPoint)
    {
        $aszPolygon = explode(",", $this->szValue);

        if (isset($aszPolygon[$nPoint*2]) && isset($aszPolygon[$nPoint*2+1]))
        {
            return array($aszMinx[$nPoint*2], $aszMinx[$nPoint*2+1]);
        }

        return false;
    }

    function addPoint($nX, $nY)
    {
        if ($this->szValue == "")
            $this->szValue = $nX.",".$nY;
        else
            $this->szValue .= ",".$nX.",".$nY;
    }
}

class BBox extends SearchAttribute
{
    function BBox($nMinx=0, $nMiny=0, $nMaxx, $nMaxy)
    {
        parent::SearchAttribute("BBOX", $nMinx.",".$nMiny.",".$nMaxx.",".$nMaxy);
    }

    function getMinx()
    {
        $aszBBox = explode(",", $this->szValue);

        return $aszMinx[0];
    }

    function getMiny()
    {
        $aszBBox = explode(",", $this->szValue);

        return $aszMinx[1];
    }

    function getMaxx()
    {
        $aszBBox = explode(",", $this->szValue);

        return $aszMinx[2];
    }

    function getMaxy()
    {
        $aszBBox = explode(",", $this->szValue);

        return $aszMinx[3];
    }
}

class Entity extends SearchAttribute
{
    function Entity($szAttributeValue)
    {
        parent::SearchAttribute("ENTITY", $szAttributeValue);
        $this->aszEntity = array();
        
        /*
         * Some day, should replace english name by some call
         * to MLT to support multilingual.
         */
        array_push($this->aszPossibleValue, array("City", "CITY/VIL1"));
        array_push($this->aszPossibleValue, array("Town", "OWN/VIL2"));
        array_push($this->aszPossibleValue, array("Village", "VILG/VILG"));
        array_push($this->aszPossibleValue, array("Hamlet", "HAM/HAM"));
        array_push($this->aszPossibleValue, array("Other municipal/district area - maj. aggl.", "MUN1/AZM1"));
        array_push($this->aszPossibleValue, array("Other municipal/district area - misc.", "MUN2/AZM2"));
        array_push($this->aszPossibleValue, array("Unincorporated area", "UNP/LNO"));
        array_push($this->aszPossibleValue, array("Indian Reserve", "IR/RI"));
        array_push($this->aszPossibleValue, array("Geographical area", "EOG/GOG"));
        array_push($this->aszPossibleValue, array("Conservation area", "PARK/PARC"));
        array_push($this->aszPossibleValue, array("Military area", "MIL/MIL"));
        array_push($this->aszPossibleValue, array("River", "IV/CDE"));
        array_push($this->aszPossibleValue, array("River feature", "IVF/EFLV"));
        array_push($this->aszPossibleValue, array("Falls", "FALL/CHUT"));
        array_push($this->aszPossibleValue, array("Lake", "AKE/LAC"));;
        array_push($this->aszPossibleValue, array("Sea", "SEA/MER"));;
        array_push($this->aszPossibleValue, array("Undersea feature", "SEAU/SMER"));
        array_push($this->aszPossibleValue, array("Channel", "HAN/CHEN"));
        array_push($this->aszPossibleValue, array("Rapids", "AP/RAP"));
        array_push($this->aszPossibleValue, array("Bay", "BAY/BAEI"));
        array_push($this->aszPossibleValue, array("Cape", "APE/CAP"));
        array_push($this->aszPossibleValue, array("Beach", "BCH/PLAG"));
        array_push($this->aszPossibleValue, array("Shoal", "SHL/H-FD"));
        array_push($this->aszPossibleValue, array("Island", "ISL/LE"));
        array_push($this->aszPossibleValue, array("Cliff", "LF/ESC"));
        array_push($this->aszPossibleValue, array("Mountain", "MTN/MNT"));
        array_push($this->aszPossibleValue, array("Valley", "VALL/VALL"));
        array_push($this->aszPossibleValue, array("Plain", "PLN/PLNE"));
        array_push($this->aszPossibleValue, array("Cave", "AVE/CAV"));
        array_push($this->aszPossibleValue, array("Crater", "RAT/CRAT"));
        array_push($this->aszPossibleValue, array("Glacier", "LAC/GLAC"));
        array_push($this->aszPossibleValue, array("Low vegetation", "VEGL/VGB"));
        array_push($this->aszPossibleValue, array("Miscellaenous", "MISC/DIV"));
        array_push($this->aszPossibleValue, array("Province", "PROV/PROV"));
        array_push($this->aszPossibleValue, array("Territory", "ERR/TERR"));
    }
}
?>
