<?php
/**
 * LogFile Module
 * 
 * @project     PHP Wrapper Class
 * @revision    $Id: logfile.php,v 1.8 2002/06/13 14:56:16 sacha Exp $
 * @purpose     this file contains classes related to managing log files 
 * @author      William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*****************************************************************************
 * $Log: logfile.php,v $
 * Revision 1.8  2002/06/13 14:56:16  sacha
 * Var is not supported in finction.
 *
 * Revision 1.7  2002/06/13 14:53:07  sacha
 * Change copyright info and declare some vars.
 *
 * Revision 1.6  2002/02/28 00:58:45  bronsema
 * Tweaked some comments and functions
 *
 * Revision 1.5  2002/02/23 19:41:45  pspencer
 * added indent/dedent and optional truncation on creation.
 *
 * Revision 1.4  2002/02/22 16:31:49  bronsema
 * Updated various functions.  Modified log and error functions in logger to recieve logfile and error manager objects by reference.
 *
 * Revision 1.3  2002/02/22 03:53:33  bronsema
 * Updated various functions
 *
 * Revision 1.2  2002/02/22 02:30:54  bronsema
 * Completed Construction
 *
 * Revision 1.1  2002/02/21 18:18:49  bronsema
 * Under Construction
 *
 *****************************************************************************/
 
// define the log types

// LOG_TO_SYSTEM - message is sent to PHP's system logger, using the Operating 
//                 System's system logging mechanism.
// LOG_TO_EMAIL  - message is sent by email to the address in the destination 
//                 parameter.
// LOG_TO_DEBUG  - message is sent through the PHP debugging connection. 
//                 This option is only available if remote debugging has been 
//                 enabled.
// LOG_TO_FILE   - message is appended to the file destination.

define("LOG_TO_SYSTEM",0);
define("LOG_TO_EMAIL",1);
define("LOG_TO_DEBUG",2);
define("LOG_TO_FILE",3);

define("INDENT_STEP", 2);

/**
 * The purpose of this class is to encapsulate a file to which messages can 
 * be logged.  
 *
 * @author William A. Bronsema, C.E.T. (bronsema@dmsolutions.ca)
 *
 */
class LogFile
{       
    /**
     * The destination either path & filename or mail recipient (private).
     */
    var $szLogDestination;    

    /**
     * The type of logging to be done (private).
     */
    var $nLogType;

    /**
     * The level at which to log errors (private).
     */
    var $nMaxLogLevel;    
    
    var $nIndentLevel;
    
    /**
     *  Construct a new LogFile instance and initialize it.
     *
     *  @param szLogDestination string - The optional destination of the log 
     *                                   message. Either path & filename or mail 
     *                                   recipient.
     *  @param nLogType integer - The optional type of logging to be done.
     *
     *  @param bTruncate - optionally truncate an existing log file (append by
     *         default).
     */
    function LogFile($szLogDestination = "logfile.log", 
                     $nLogType = LOG_TO_FILE, $bTruncate = false )
    {
        $fp = 0;
        
        // initialize variables
        $this->szLogDestination = $szLogDestination;     
        $this->nLogType = $nLogType;
        $this->nMaxLogLevel = 0;
        $this->nIndentLevel = 1;
        
        //truncate the file if necessary
        if ( $bTruncate )
        {
            if ( file_exists( $this->szLogDestination ) )
            {
                $fp = @fopen( $this->szLogDestination, "w+" );
                if ( $fp )
                    fclose( $fp );
                
            }
        }
    
    // end constructor
    }
    
    /**
     * This function sets the maximum allowable log level.
     *
     * @param nMaxLogLevel integer - The level to set.
     */
    function setMaxLogLevel($nMaxLogLevel)
    {
        $this->nMaxLogLevel = $nMaxLogLevel;
    }    

    /**
     * This function gets the maximum allowable log level.
     *
     * @return integer - Returns the max log level.
     */
    function getMaxLogLevel()
    {
        if ( is_numeric($this->nMaxLogLevel) )
            return $this->nMaxLogLevel;
        else
            return 0;
    }    
    
    /**
     * This function logs the message according to the set parameters.
     *
     * @param nLogLevel integer - The log level for this message.
     * @param szMessage string - The message to log.
     */
    function logMessage($nLogLevel, $szMessage)
    {
        // log the message if below max level
        if ( $nLogLevel <= $this->nMaxLogLevel )
        {
            // add the time stamp
            $szMessage = date("[D.M.d.Y.G:i:s]").
                         str_repeat(" ", $this->nIndentLevel).
                         $szMessage."\r\n";
            
            // log message
            error_log ( $szMessage, $this->nLogType, $this->szLogDestination); 
        }
       
    // end logMessage function
    }
    
    function indent()
    {
        $this->nIndentLevel += INDENT_STEP;
    }
    
    function dedent()
    {
        $this->nIndentLevel -= INDENT_STEP;
        if ($this->nIndentLevel < 1)
            $this->nIndentLevel = 1;
    }

// end LogFile class    
}

?>
