<?php
/**
 * Buttonizer Utility functions
 *
 * @project     CWC2
 * @revision    $Id: buttonizer.php,v 1.29 2004/06/03 15:11:37 wbronsema Exp $
 * @purpose     Helper functions to create buttons from images and labels
 * @author      DM Solutions Group (spencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.

buttonizer.php provides a basic button building service.  A button is a single
graphic image that is composed of several components:

 - a border
 - a background
 - a button graphic
 - a label

The two parts of the button are the border and the center.  The border is
composed of eight (8) separate graphic images that can be individually
specified to allow for the creation of complex buttons.

The center portion is the rectangular area outlined by the border and its size
is determined by the overall size specified less the border width/height.  The
center is filled with a background color.  An optional background graphic is
aligned centrally within the center area if it doesn't fill it, or it is
cropped if it is too big.

A button is made by calling buttonize() with an array of parameters as follows:

graphic=<image file> : [optional] the image to put in the button, default is no
graphic

label=<string> : [optional] the label to draw in the button, default is no label

labelcolor=<hex color rrggbb> : [optional] the color to draw the label text in,
default is black

labelfont=<font name> : [optional] the font to draw the label in, default is
verdana or arial?

labelalign=<string> : [optional] the horizontal alignment of the label, one of
left, center or right.

labelsize=<point size> : [optional] the point size of font to use, default is 10

backgroundcolor=<hex color rrggbb> or transparent : [optional] the color to fill
the image with, default is transparent

backgroundimage=<imagefile> : [optional] the image to fill the background with,
default is transparent.  The background image is drawn on top of the background
fill color and is centered on the central portion of the button

width=<pixels wide> : [optional] minimum width of resulting image, may be wider
if the graphic or the text end up being wider than this.

height=<pixels high> : [optional] minimum height of resulting image, may be
higher if the graphic or the text end up being higher that this.

border_xx_image=<image file> : [optional] the image to put in the xx border of
the button, where xx is:

 t - top (border_t_image)
 r - right (border_r_image)
 b - bottom (border_b_image)
 l - left (border_l_image)
 tl - top left (border_tl_image)
 tr - top right (border_tr_image)
 bl - bottom left (border_bl_image)
 br - bottom right (border_br_image)

The image may be stretched vertically and/or horizontally to fit the dimensions
of the output button.

debug=<true|false> : [optional] default is false, if true then output debugging
information (as an html page) instead of just an image.

Image resizing logic:

The general preference will be to try to keep images at their natural
dimensions with adjustments being made as necessary to fill the space.  The
three top images (tl, t, tr) will all be forced to the same height, which is
the maximum of the three heights.  Similarly, the three bottom images will be
forced to the maximum of the heights of bl, b, and br.  The left and right
edges will be forced to the maximum width of tl, l, bl, and tr, r, br.

Once the height of the top and bottom border is determined, and the width of
the left and right border is determined, the width t and b borders will be
adjusted to (width - l_border_width - r_border_width) and the height of the l
and r borders will be adjusted to (height - t_border_height - b_border_height).

All border images are then scaled to fit these dimensions as required.  Any
dimensions smaller than the calculated values will be stretched, larger values
will be cropped.

The dimensions of the center of the button are calculated to be the same as the
width of the top border and the height of the left border.

The graphic image will be cropped to fit the height of the center portion.

The label will be positioned with the requested alignment in the remaining
horizontal space and will be cropped both horizontally and vertically within
this space.
*/

// border position definitions
define( "TOP_LEFT", 0 );
define( "TOP", 1 );
define( "TOP_RIGHT", 2 );
define( "RIGHT", 3 );
define( "BOTTOM_RIGHT", 4 );
define( "BOTTOM", 5 );
define( "BOTTOM_LEFT", 6 );
define( "LEFT", 7 );

if (!defined( "PHP_SHLIB_SUFFIX" ))
{
    if ( PHP_OS == "WINNT" || PHP_OS == "WIN32" )
        define( "PHP_SHLIB_SUFFIX", "dll" );
    else
        define ("PHP_SHLIB_SUFFIX", "so" );
}

function html_ent_decode($szToDecode)
{
    if (trim($szToDecode) == "" )
        return "";

    // If php version is equal or greater of 4.3.0 use the system function
    if (version_compare(phpversion(), "4.3.0", ">="))
        return html_entity_decode($szToDecode);
    else
    {
        // Use the home made function
        $aszEntities = array(
                             "&Acirc;" => "",
                             "&Agrave;" => "",
                             "&Ccedil;" => "",
                             "&Eacute;" => "",
                             "&Ecirc;" => "",
                             "&Egrave;" => "",
                             "&Euml;" => "",
                             "&Icirc;" => "",
                             "&Ocirc;" => "",
                             "&Ucirc;" => "",
                             "&acirc;" => "",
                             "&agrave;" => "",
                             "&apos;" => "'",
                             "&ccedil;" => "",
                             "&eacute;" => "",
                             "&ecirc;" => "",
                             "&egrave;" => "",
                             "&euml;" => "",
                             "&iuml;" => "",
                             "&nbsp;" => " ",
                             "&ocirc;" => "",
                             "&quot;" => "\"",
                             "&ucirc;" => "",
                             "&amp;" => "&");

        foreach ($aszEntities as $szEntity => $szEquiv)
            if (strpos($szToDecode, "&") !== false)
                $szToDecode = str_replace($szEntity, $szEquiv, $szToDecode);
            else
                break;

        return $szToDecode;
    }
}

function buttonize( $szOutputName, $aParams )
{
    $GLOBALS["bDebug"] = getParameter( $aParams, "debug", false );

    // set the GD version 
    $szGDModule = "php_gd2.".PHP_SHLIB_SUFFIX;
    if (!extension_loaded("gd"))
        dl($szGDModule);

    // set the free type version
    $szFreeType = "FreeType";
    
    //initialize variables
    $aoBorders = array( 0 => "", 1 => "", 2 => "", 3 => "",
                         4 => "", 5 => "", 6 => "", 7 => "" );

    $anBorderWidths = array( 0, 0, 0, 0, 0, 0, 0, 0 );
    $anBorderHeights = array( 0, 0, 0, 0, 0, 0, 0, 0 );

    //get values from the URL

    //width/height first - fail if one or the other is zero.
    $nWidth = getParameter( $aParams, "width", 0 );
    $nHeight = getParameter( $aParams, "height", 0 );

    if ($nWidth == 0 || $nHeight == 0)
        exit;

    //now build the output image so we can use it to allocate colors
    if (getParameter( $aParams, "truecolor", false ))
    {
      $oFinal = imagecreatetruecolor( $nWidth, $nHeight );
    }
    else
    {
      $oFinal = imagecreate( $nWidth, $nHeight );
    }

    //create a transparent color to be used
    $szTransparentColor = "abcdef";
    $nTransparentColor = allocateImageFromHexValue( $oFinal, $szTransparentColor );
    imagecolortransparent( $oFinal, $nTransparentColor );
    imagefill( $oFinal, 0, 0, $nTransparentColor );

    //process the border images
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_tl_image", TOP_LEFT );
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_t_image", TOP );
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_tr_image", TOP_RIGHT );
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_r_image", RIGHT );
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_br_image", BOTTOM_RIGHT );
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_b_image", BOTTOM );
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_bl_image", BOTTOM_LEFT );
    processBorderImage( $aParams, $aoBorders, $anBorderWidths, $anBorderHeights, "border_l_image", LEFT );

    // calculate the border sizes
    $nTopHeight = max( $anBorderHeights[TOP_LEFT], $anBorderHeights[TOP], $anBorderHeights[TOP_RIGHT] );
    $anBorderHeights[TOP_LEFT] = $anBorderHeights[TOP] = $anBorderHeights[TOP_RIGHT] = $nTopHeight;

    $nBottomHeight = max( $anBorderHeights[BOTTOM_LEFT], $anBorderHeights[BOTTOM], $anBorderHeights[BOTTOM_RIGHT] );
    $anBorderHeights[BOTTOM_LEFT] = $anBorderHeights[BOTTOM] = $anBorderHeights[BOTTOM_RIGHT] = $nBottomHeight;

    $nLeftWidth = max( $anBorderWidths[TOP_LEFT], $anBorderWidths[LEFT], $anBorderWidths[BOTTOM_LEFT] );
    $anBorderWidths[TOP_LEFT] = $anBorderWidths[LEFT] = $anBorderWidths[BOTTOM_LEFT] = $nLeftWidth;

    $nRightWidth = max( $anBorderWidths[TOP_RIGHT], $anBorderWidths[RIGHT], $anBorderWidths[BOTTOM_RIGHT] );
    $anBorderWidths[TOP_RIGHT] = $anBorderWidths[RIGHT] = $anBorderWidths[BOTTOM_RIGHT] = $nRightWidth;

    //calculate width of TOP/BOTTOM and height of LEFT/RIGHT
    $anBorderWidths[TOP] = $anBorderWidths[BOTTOM] = ($nWidth - $nLeftWidth - $nRightWidth);
    $anBorderHeights[LEFT] = $anBorderHeights[RIGHT] = ($nHeight - $nTopHeight - $nBottomHeight);

    //for ($i=0; $i<8; $i++)
    //    debug( " [$i] border width = ".$anBorderWidths[$i].", border height = ".$anBorderHeights[$i] );


    //top border images
    if ( $aoBorders[TOP_LEFT] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[TOP_LEFT],
                            0, 0, 0, 0,
                            $anBorderWidths[TOP_LEFT], $anBorderHeights[TOP_LEFT],
                            imagesx($aoBorders[TOP_LEFT]), imagesy($aoBorders[TOP_LEFT]) );
    }
    if ( $aoBorders[TOP] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[TOP],
                            $anBorderWidths[TOP_LEFT], 0, 0, 0,
                            $anBorderWidths[TOP], $anBorderHeights[TOP],
                            imagesx($aoBorders[TOP]), imagesy($aoBorders[TOP]) );
    }
    if ( $aoBorders[TOP_RIGHT] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[TOP_RIGHT],
                            $anBorderWidths[TOP_LEFT] + $anBorderWidths[TOP], 0, 0, 0,
                            $anBorderWidths[TOP_RIGHT], $anBorderHeights[TOP_RIGHT],
                            imagesx($aoBorders[TOP_RIGHT]), imagesy($aoBorders[TOP_RIGHT]) );
    }

    //left and right border images
    if ( $aoBorders[RIGHT] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[RIGHT],
                            $anBorderWidths[TOP_LEFT] + $anBorderWidths[TOP], $anBorderHeights[TOP_RIGHT], 0, 0,
                            $anBorderWidths[RIGHT], $anBorderHeights[RIGHT],
                            imagesx($aoBorders[RIGHT]), imagesy($aoBorders[RIGHT]) );
    }
    if ( $aoBorders[LEFT] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[LEFT],
                            0, $anBorderHeights[TOP_LEFT], 0, 0,
                            $anBorderWidths[LEFT], $anBorderHeights[LEFT],
                            imagesx($aoBorders[LEFT]), imagesy($aoBorders[LEFT]) );
    }

    //bottom border images
    if ( $aoBorders[BOTTOM_LEFT] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[BOTTOM_LEFT],
                            0, $anBorderHeights[TOP_LEFT] + $anBorderHeights[LEFT], 0, 0,
                            $anBorderWidths[BOTTOM_LEFT], $anBorderHeights[BOTTOM_LEFT],
                            imagesx($aoBorders[BOTTOM_LEFT]), imagesy($aoBorders[BOTTOM_LEFT]) );
    }
    if ( $aoBorders[BOTTOM] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[BOTTOM], $anBorderWidths[BOTTOM_LEFT],
                            $anBorderHeights[TOP_LEFT] + $anBorderHeights[LEFT], 0, 0,
                            $anBorderWidths[BOTTOM], $anBorderHeights[BOTTOM],
                            imagesx($aoBorders[BOTTOM]), imagesy($aoBorders[BOTTOM]) );
    }
    if ( $aoBorders[BOTTOM_RIGHT] != "" )
    {
        imagecopyresized( $oFinal, $aoBorders[BOTTOM_RIGHT],
                            $anBorderWidths[BOTTOM_LEFT] + $anBorderWidths[BOTTOM],
                            $anBorderHeights[TOP_LEFT] + $anBorderHeights[LEFT], 0, 0,
                            $anBorderWidths[BOTTOM_RIGHT], $anBorderHeights[BOTTOM_RIGHT],
                            imagesx($aoBorders[BOTTOM_RIGHT]), imagesy($aoBorders[BOTTOM_RIGHT]) );
    }

    //done with the border images
    for ($i=0; $i<8; $i++)
    {
        if ($aoBorders[$i] != "")
        {
            debugImage( $aoBorders[$i], "border$i.png" );
            imagedestroy( $aoBorders[$i] );
        }
    }
    //debug( "final image with borders:" );
    debugImage( $oFinal, "final2.png" );

    //calculate center size
    $nCenterWidth = $anBorderWidths[TOP];
    $nCenterHeight = $anBorderHeights[LEFT];
    $oCenter = imagecreate( $nCenterWidth, $nCenterHeight );

    //background values
    $szBackgroundColor = getParameter( $aParams, "backgroundcolor", "-1-1-1" );
    $nBackgroundColor = allocateImageFromHexValue( $oCenter, $szBackgroundColor );
    //debug( "Background color: $szBackgroundColor ($nBackgroundColor)" );
    //fill the image with the background color
    if ($nBackgroundColor != -1)
    {
        imagefill( $oCenter, 0, 0, $nBackgroundColor );
        //debug( "final image with background color fill: " );
        debugImage( $oCenter, "center0a.png" );
    }
    else
    {
        debug( "center transparent" );
        $nCenterTransparent = allocateImageFromHexValue( $oCenter, $szTransparentColor );
        imagecolortransparent( $oCenter, $nCenterTransparent );
        imagefill( $oCenter, 0, 0, $nCenterTransparent );
        debugImage( $oCenter, "center0b.png" );
    }

    //check to see if there is a background graphic
    $szBackgroundGraphic = getParameter( $aParams, "backgroundgraphic", "" );
    if ($szBackgroundGraphic != "")
    {
        if (file_exists($szBackgroundGraphic))
        {
            
            $oBackgroundGraphic = imagecreatefrompng( $szBackgroundGraphic );
            $nBackgroundWidth = min( imagesx($oBackgroundGraphic), $nCenterWidth );
            $nBackgroundHeight = min( imagesx($oBackgroundGraphic), $nCenterHeight );
            imagecopy( $oCenter, $oBackgroundGraphic, 0, 0, 0, 0, $nBackgroundWidth, $nBackgroundHeight );

            //debug( "background image ($szBackgroundGraphic):" );
            debugImage( $oBackgroundGraphic, "background.png" );
            debugImage( $oCenter, "center1.png" );

            imagedestroy( $oBackgroundGraphic );
        }
        else
        {
            sessionError( 'Error: missing backgroundgraphic image '.$szBackgroundGraphic );

        }
    }

    //options for the label graphic
    $nGraphicWidth = 0;
    $nGraphicHeight = 0;
    $szGraphic = getParameter( $aParams, "graphic", "" );
    if ($szGraphic != "")
    {
        $bOk = false;
        if (file_exists($szGraphic))
        {
            $bOk = true;
        }
        else if (($h = @fopen($szGraphic)) !== false)
        {
            fclose( $h );
            $bOk = true;
        }
        
        if ($bOk)
        {
            $oGraphic = imagecreatefrompng( $szGraphic );
            imagecolortransparent( $oGraphic, 0 );
            $nGraphicWidth = imagesx( $oGraphic );
            $nGraphicHeight = imagesy($oGraphic);

            //paste the graphic into the center image.
            if ($nGraphicHeight > $nCenterHeight)
            {
                //paste cropped
                $nDestX     = 0;
                $nDestY     = 0;
                $nSrcX      = 0;
                $nSrcY      = ($nGraphicHeight - $nCenterHeight)/2;
                $nSrcWidth  = $nGraphicWidth;
                $nSrcHeight = $nCenterHeight;
            }
            else
            {
                //paste centered vertically
                $nDestX     = 0;
                $nDestY     = ($nCenterHeight - $nGraphicHeight)/2;
                $nSrcX      = 0;
                $nSrcY      = 0;
                $nSrcWidth  = $nGraphicWidth;
                $nSrcHeight = $nGraphicHeight;
            }
            //debug( "Graphic image: $szGraphic" );
            //debug( "Graphic size: $nGraphicWidth x $nGraphicHeight" );
            //debug( "Graphic copy: $nDestX, $nDestY, $nSrcX, $nSrcY, $nSrcWidth, $nSrcHeight" );

            imagecopy( $oCenter, $oGraphic, $nDestX, $nDestY, $nSrcX, $nSrcY, $nSrcWidth, $nSrcHeight );

            debugImage( $oGraphic, "graphic.png" );
            debugImage( $oCenter, "center2.png" );

            imagedestroy( $oGraphic );
        }
        else
        {
            sessionError( 'Error: missing graphic '.$szGraphic );
        }
    }

    //label options
    $szLabel = html_ent_decode(getParameter( $aParams, "label", "" ));
    if ($szLabel != "" )
    {
        $nLabelSize = getParameter( $aParams, "labelsize", "10" );
        $szLabelAlign = strtolower(getParameter( $aParams, "labelalign", "left" ));

        if (strcmp( $szLabelAlign, "left" ) != 0 &&
            strcmp( $szLabelAlign, "center" ) != 0 &&
            strcmp( $szLabelAlign, "right" ) != 0)
        {
            $szLabelAlign = "left";
        }

        $szLabelFont = getParameter( $aParams, "labelfont", "Vera.ttf" );
        if (!is_file( $szLabelFont ))
        {
            sessionError( 'Error: font file ('.$szLabelFont.' not found, trying default (Vera.ttf)' );
            $szLabelFont = "Vera.ttf";
        }
        $szLabelFont = realpath( $szLabelFont );
        
        if (file_exists($szLabelFont))
        {
            if (strcasecmp($szFreeType,"FreeType") == 0)
            {
                $szFontPath = "GDFONTPATH=".dirname( $szLabelFont );
                putenv($szFontPath);
                $szLabelFont = basename($szLabelFont, ".ttf");
            }
            if ( !(PHP_OS == "WINNT" || PHP_OS == "WIN32") && $szFreeType != "TTF" )
            {
                //$nLabelSize = $nLabelSize * 72 / 96;
            }            
        
            $bAntiAlias = false;
            //if we are using FreeType 2, then we need this to actually find the
            //font reliably on Windows because the font doesn't start with /

            $nPadding = getParameter( $aParams, "padding", 0 );
            $nNudge = getParameter( $aParams, "nudge", 0 );

            $nLabelWidth = $nCenterWidth - $nPadding - $nGraphicWidth;
            $nLabelHeight = $nCenterHeight;

            //debug( "Label size: $nLabelWidth x $nLabelHeight" );
            //debug( "Label text: $szLabel" );
            //debug( "Label font: $szLabelFont" );
            //debug( "Label align: $szLabelAlign" );
            if ($nLabelWidth == 0)
                $nLabelWidth = 1;
            if ($nLabelHeight == 0)
                $nLabelHeight = 1;
            $oLabel = imagecreate( $nLabelWidth, $nLabelHeight );
            if ($nBackgroundColor == -1)
                $nLabelTransparent = allocateImageFromHexValue( $oLabel, $szTransparentColor );
            else
                $nLabelTransparent = allocateImageFromHexValue( $oLabel, $szBackgroundColor );

            debug ( "label background index is $nLabelTransparent" );
            imagecolortransparent( $oLabel, $nLabelTransparent );
            imagefill( $oLabel, 0, 0, $nLabelTransparent );
            //$nLabelBackground = allocateImageFromHexValue( $oLabel, $szBackgroundColor );
            //imagefill( $oLabel, 0, 0, $nLabelBackground );

            $szLabelColor = getParameter( $aParams, "labelcolor", "000000" );
            $nLabelColor = allocateImageFromHexValue( $oLabel, $szLabelColor );

            $bAntiAlias = getParameter( $aParams, "antialias", $bAntiAlias );
            if (!$bAntiAlias)
            {
                $nLabelColor = -1 * $nLabelColor;
            }
            //debug( "Label color: $szLabelColor ($nLabelColor)" );

            $anLabelSize = imagettfbbox( $nLabelSize, 0, $szLabelFont, $szLabel );
            $nTextWidth = abs($anLabelSize[6] - $anLabelSize[2]);

            //take a consistent letter to calculate the height value
            $anASize = imagettfbbox( $nLabelSize, 0, $szLabelFont, "I" );
            $nTextHeight = abs($anASize[5]);


            //debug( "Text size: $nTextWidth x $nTextHeight" );
            //debug( "Text box: " );
            //for ($i=0; $i<8; $i++)
            //    debug( "  [$i] = ".$anLabelSize[$i] );
            //check vertical size
            if ($nTextHeight > $nLabelHeight)
            {
                //text is higher than we have available to draw it so we draw it
                //aligned at the bottom and will lose some text off the top
                $nTextY = $nLabelHeight;
            }
            else
            {
                //text to be centered vertically in the available space
                $nTextY = (($nLabelHeight - $nTextHeight) / 2) + $nTextHeight + $nNudge;
            }

            //check horizontal size
            if ($nTextWidth > $nLabelWidth)
            {
                //text is wider than we have available to draw it, left align it
                $nTextX = 0;
            }
            else
            {
                //text fits, align as per label align
                if (strcmp( $szLabelAlign, "left" ) == 0)
                    $nTextX = 0;
                else if (strcmp( $szLabelAlign, "center" ) == 0)
                {
                    //echo "center";
                    $nTextX = ($nLabelWidth - $nTextWidth) / 2;
                }
                else
                    $nTextX = ($nLabelWidth - $nTextWidth);
            }
            //debug( "Text X,Y: $nTextX, $nTextY" );

            //adjust horizontal alignment from text metrics
            $nTextX = $nTextX - $anLabelSize[0];

            //now check for multiline text
            $aszLabels = explode( '\\'.'n', $szLabel );

            $nLabels = count($aszLabels);

            if ($nLabels > 1)
            {
                //draw each line of text
                //TODO handle vertical alignment
                $nTextY = $nTextY - ($nLabels/2 * $nTextHeight) + $nTextHeight / 2;
                $nTempY = $nTextY;
                for ($i=0; $i < $nLabels; $i++)
                {
                    imagettftext ( $oLabel, $nLabelSize, 0, $nTextX, $nTempY, $nLabelColor, $szLabelFont, $aszLabels[$i] );
                    $nTempY = $nTempY + $nTextHeight;
                }
            }
            else
            {

                //draw text on label image
                imagettftext ( $oLabel, $nLabelSize, 0, $nTextX, $nTextY, $nLabelColor, $szLabelFont, $szLabel );
            }

            debugImage( $oLabel, "label.png" );

            imagecopy( $oCenter, $oLabel, $nGraphicWidth + $nPadding, 0, 0, 0, $nLabelWidth, $nLabelHeight );
            debugImage( $oCenter, "center3.png" );

            //done with the label image
            imagedestroy( $oLabel );
        }
        else
        {
            sessionError( 'Error: missing font '.$szLabelFont );
        }
    }

    //copy the center image into the final image
    //imagecopy($oFinal, $oCenter, $anBorderWidths[LEFT], $anBorderHeights[TOP], 0, 0, $nCenterWidth, $nCenterHeight );
    imagecopyresized($oFinal, $oCenter, $anBorderWidths[LEFT], $anBorderHeights[TOP], 0, 0, $nCenterWidth, $nCenterHeight, $nCenterWidth, $nCenterHeight );
    //done with the center image
    imagedestroy( $oCenter );

    //imagetruecolortopalette( $oFinal, false, 256 );

    //output final image
    if (!$GLOBALS["bDebug"])
    {
        if ( $szOutputName == "" )
        {
            header("content-type: image/png");
            imagepng($oFinal);
        }
        else
            imagepng( $oFinal, $szOutputName );
    }
    else
    {
        debugImage( $oFinal, "final.png" );
    }
    imagedestroy($oFinal);
}

function debug( $str )
{
    if ($GLOBALS["bDebug"])
        echo $str."<BR>";
}

function debugImage( $oImage, $szName )
{
    if ($GLOBALS["bDebug"])
    {
        imagepng( $oImage, $szName );
        echo "$szName: <IMG SRC=\"$szName\" WIDTH=".imagesx($oImage)." HEIGHT=".imagesy($oImage)." BORDER=0><BR>";
    }

}

function getParameter( $aParams, $szValue, $mDefault )
{
    if (isset($aParams[$szValue]))
        return $aParams[$szValue];
    else
        return $mDefault;
}

function processBorderImage( $aParams, &$aoBorders, &$anBorderWidths, &$anBorderHeights, $szImageName, $nImagePosition )
{
    $image = getParameter( $aParams, $szImageName, "" );
    if ($image != "")
    {
        if (file_exists( $image ))
        {
            $aoBorders[$nImagePosition] = imagecreatefrompng( $image );
            $anBorderWidths[$nImagePosition] = imagesx( $aoBorders[$nImagePosition] );
            $anBorderHeights[$nImagePosition] = imagesy( $aoBorders[$nImagePosition] );
            //debug( "border image $nImagePosition: $szImageName - ".
            //       $anBorderWidths[$nImagePosition].
            //       "x".$anBorderHeights[$nImagePosition] );
        }
        else
        {
            sessionError( 'Error: missing border image '.$image );
        }
    }
}

function sessionError( $szText )
{
    if (isset($_SESSION['gErrorManager']))
    {
        $_SESSION['gErrorManager']->setError(ERR_WARNING, "Buttonizer Error: ".$szText );
    }
}

function allocateImageFromHexValue( $im, $szHexColor )
{
    $nRed = intval(substr($szHexColor,0,2),16);
    $nGreen = intval(substr($szHexColor,2,2),16);
    $nBlue = intval(substr($szHexColor,4,2),16);

    if ($nRed == -1 && $nGreen == -1 && $nBlue == -1)
    {
        debug( "setting transparent from $szHexColor" );
        $nColor = -1; //assumes first color allocated is the transparent color?
        //$nColor = imagecolortransparent( $im );
    }
    else
        $nColor = imagecolorallocate( $im, $nRed, $nGreen, $nBlue );
    return $nColor;
}
?>
