<?php
/**
 * Macro Template Processor
 *
 * @project     CWC2
 * @revision    $Id: TemplateProcessor.php,v 1.6 2004/04/16 18:04:19 pspencer Exp $
 * @purpose     Process macros in a template file
 * @author      DM Solutions Group (spencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

include_once( "TemplateParser.php" );

/**
 * The general purpose of this class is to process an HTML template that contains
 * special template macros that act like code.  The macros refer to values that
 * are provided in a structured array that create virtual "objects" from the keys
 * of the primary array.  Under each primary key is an array of objects of that
 * "type".  Each object has an array of attribute_name = value pairs.  The general
 * structure is:
 *
 * array[ object_type ][ object_index ][ "attribute_name" ] = "value";
 *
 * The macro language appears as commands inside square brackets.  Some macros
 * operate on blocks and must have an associated [end] macro to define the end
 * of the macro.
 *
 * The macro language supports three basic operations:
 *
 * [repeat <object_type>]...[end] - this outputs up to the associated [end] for
 * each object of <object_type>.  This defines all attributes of the object
 * within the current processing scope, making the values accessible to any
 * [if] macros or [<attribute>] macros within the block.  Attribute values will
 * overwrite existing attribute values from other objects regardless of scope,
 * so be careful with  * naming attributes between objects (they should be
 * different).  Once an [end] block is reached for a [repeat] macro, the associated
 * attributes are taken out of the current scope
 *
 * [if <attribute> <operator> <value>]...[end] - this outputs up to the associated
 * [end] if the expression is true.  The <attribute> is some valid attribute for
 * one of the current objects.  The <operator> is = or !=.  The <value> is either
 * a quoted literal value or, if not quoted, an attribute of a current object.
 *
 * [<attribute>] - outputs the value of the attribute.  The attribute must be defined
 * for an object that is in scope (i.e. in a repeat block).
 */
 class TemplateProcessor
 {
    var $moParser;
    var $mszPattern = "/(\[(.*)\])/iUs";

    function TemplateProcessor()
    {
        $this->moParser = new TemplateParser($this, $this->mszPattern);
    }
    
    /**
     * required by TemplateParser interface, return a template to be parsed
     * @param szTemplateName the template to load
     * @return a string containing the contents of the template
     */
    function GetTemplate( $szTemplateName )
    {
        if (function_exists('file_get_contents'))
            return file_get_contents($szTemplateName);
        else
            return implode("\n", file($szTemplateName));
    }

    /**
     * required by TemplateParser interface, always return false
     */
    function isVarSet( $szVar )
    {
        return false;
    }
    
    /**
     * required by TemplateParser interface, always return false
     */
    function getVar( $szVar )
    {
        return false;
    }
    
    function ProcessTemplate( $szTemplate, $aszValues )
    {
        $aszTemplate = file($szTemplate);
        $this->moParser->SetTemplate(implode("",$aszTemplate));
        if (!$this->moParser->Parse())
            echo "error parsing template $szTemplate<BR>\n";

        $nUnmatchedBraces = 0;
        $nObjectIndex = 0;
        $nLoopIndex = 0;

        $aszDescope = array();
        $bInIfStatement = false;
        $nElements = $this->moParser->NumElements();
        for ($i=0; $i<$nElements; $i++)
        {
            $szTag = $this->moParser->GetElement( $i );

            $szContents = substr( $szTag, 1, strlen($szTag) - 2 );

            if (strcasecmp(substr($szContents, 0, 6), "repeat") == 0)
            {
                $nObjectIndex ++;
                $szObject = substr( $szContents, 7, strlen($szContents) - 7 );

                $szCode = "<?php for( \$i".$nObjectIndex."=0; \$i".$nObjectIndex." < count(\$aObjects[\"".$szObject."\"]); \$i".$nObjectIndex."++)\n";
                $szCode .= "{ \$elm".$nObjectIndex." = \$aObjects[\"".$szObject."\"][\$i".$nObjectIndex."];\n";

                /* bring variables into scope */

                $aszDescope[$nUnmatchedBraces] = "foreach(array_keys(\$elm$nObjectIndex) as \$key) \n";
                $aszDescope[$nUnmatchedBraces] .= "  unset(\$aScope[\$key]);\n";


                $szCode .= "foreach(array_keys(\$elm$nObjectIndex) as \$key) \n";
                $szCode .= "  \$aScope[\$key] = \$elm".$nObjectIndex."[\$key];\n";
                $szCode .= " ?>";

                ++ $nUnmatchedBraces;

                $nDescope = $nUnmatchedBraces;
                $this->moParser->SetElement( $i, $szCode );
            }
            elseif (strcasecmp( substr($szContents, 0, 2), "if") == 0)
            {
                ++ $nUnmatchedBraces;
                $szStatement = trim(substr( $szContents, 3, strlen($szContents) - 2 ));

                $aszElms = explode( " ", $szStatement );
                if (count($aszElms) != 3)
                {
                    echo "invalid if statement format";
                }

                if ($aszElms[1] == "=")
                    $aszElms[1] = "==";

                if (substr(trim($aszElms[2]), 0, 1) !="\"")
                    $aszElms[2] = "\$aScope[\"".trim($aszElms[2])."\"]";

                $szCode = "<?php if (\$aScope[\"".$aszElms[0]."\"] ".$aszElms[1]." ".$aszElms[2].") { ?>";
                $this->moParser->SetElement( $i, $szCode );
                $bInIfStatement = true;
            }
            elseif (strcasecmp( $szContents, "else") == 0)
            {
                if ($bInIfStatement)
                {
                    $this->moParser->SetElement( $i, "<?php } else { ?>" );
                }
                else
                {
                    echo "ELSE is invalid outside of an IF ... END block";
                }
            }
            elseif (strcasecmp($szContents, "end") == 0)
            {

                if ($nUnmatchedBraces > 0)
                {
                    $szCode = "<?php ";
                    if ($nUnmatchedBraces == $nObjectIndex)
                        -- $nObjectIndex;

                    -- $nUnmatchedBraces;

                    if (isset($aszDescope[$nUnmatchedBraces]))
                    {
                        /* bring current variables out of scope before ending element */
                        $szCode .= $aszDescope[$nUnmatchedBraces];
                        unset( $aszDescope[$nUnmatchedBraces] );
                    }
                }
                else
                {
                    echo "error parsing template, [END] without block";
                }
                $szCode .= " } ?>";
                $this->moParser->SetElement( $i, $szCode );
                if ($bInIfStatement)
                    $bInIfStatement = false;
            }
            else
            {
                $szCode = "<?php if ( isset(\$aScope[\"".trim($szContents)."\"]) ) echo stripslashes(\$aScope[\"".trim($szContents)."\"]); else echo \"".$szTag."\"; ?>";
                $this->moParser->SetElement( $i, $szCode);
            }

        }

        if ($nUnmatchedBraces != 0)
        {
            echo "error processing template, unmatched block element somewhere!";
        }
        else
        {
            $this->moParser->ProcessElements();
            $szOutput = "?><?php\n";

            $szOutput .= $this->ArrayToCode( "aObjects", $aszValues );

            $szOutput .= "\$aScope = array();\n";
            $szOutput .= "?>\n";
            $szOutput .= $this->moParser->mszTemplate;

            /*
            echo "<PRE>";
            print_r( htmlentities($szOutput) );
            echo "</PRE>";
            */
            ob_start();

            eval( $szOutput );

            $szResult = ob_get_contents();
            ob_end_clean();

            return $szResult;

        }
    }

    function ArrayToCode( $szArrayName, $aszArray )
    {
        $szCode = "\$".$szArrayName." = array();\n";

        $aszKeys = array_keys($aszArray);

        for ($i=0; $i<count($aszKeys); $i++)
        {
            $szKey = $aszKeys[$i];
            $elm = $aszArray[$szKey];

            if (is_array( $elm ))
            {
                $szCode .= $this->ArrayToCode( "_".$szArrayName, $elm );
                $szValue = "\$_".$szArrayName;
            }
            else
            {
                $szValue = "\"".addslashes($aszArray[$szKey])."\"";
            }
            $szCode .= "\$".$szArrayName."[\"".$szKey."\"] = ".$szValue.";\n";
        }
        return $szCode;
    }
 }
?>
