<?php
/**
 * @project     CWC2
 * @revision    $Id: cwc2.php,v 1.7 2004/07/09 19:01:42 pspencer Exp $
 * @purpose     CWC2 Service Instance
 * @author      DM Solutions Group (pspencer@dmsolutions.ca)
 * @copyright
 * <b>Copyright (c) 2002, DM Solutions Group Inc.</b>
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * this script implements the service instance model from
 * the original CWC2 but now as a Chameleon application.
 *
 * The service is invoked by providing a URI to cwc2.php and passing a number
 * of optional and mandatory parameters as part of the query string.
 *
 * The parameters accepted by the service instance are:
 *
 * SERVICE   - VCG (Viewer Client Generator), mandatory
 * REQUEST   - GetCapabilities or GetApplication
 * VERSION   - any version up to and including the latest version supported by
 *             a particular service instance.  Initial release version was 
 *             0.1.0
 * LANGUAGES - list of languages supported
 * TEMPLATES - list of templates associated with each language
 * CONTEXTS  - list of contexts to load
 * LANGUAGE  - the language to use to generate the initial view of the
 *             application
 * SRS       - The projection (EPSG code) to display the context in
 * BBOX      - The initial extents to use when displaying the context 
 */
 
/*
 * variables used in this script:
 *
 * $oApp                  - the Chameleon application instance
 * $anServiceVersion      - an array of three ints containing the version 
                            numbers
 * $szServiceInstancePath - the URL to cwc2.php (this file)
 * $szServerDataPath      - the path to wmsparse data files
 * $bAllowUploadContext   - control access to remote contexts
 * $bAllowUploadTemplate  - control access to remote templates
 * $bCacheTemplate        - control local caching of remote templates
 * $szDefaultContext      - the context to use if no others are provided
 * $szDefaultTemplate     - the template to use if no others are provided
 * $szContextRoot         - all local contexts must be relative to this
 * $szTemplateRoot        - all local templates must be relative to this
 * $szMaxContextSize      - the maximum size of uploaded contexts
 * $szFontListFile        - a MapServer font file
 * $szSymbolFile          - a MapServer symbol file
 * $bValidVersion         - true if the VERSION parameter is valid
 * $anUserVersion         - array of ints passed in VERSION parameter
 * $nLanguages            - the number of languages in the LANGUAGES parameter
 * $aszLanguages          - array of language codes from LANGUAGES parameter
 * $nTemplates            - the number of templates in the TEMPLATES parameter
 * $aszTemplates          - array of template names from TEMPLATES parameter
 * $nContexts             - the number of contexts in the CONTEXTS parameter
 * $aszContexts           - array of context names from the CONTEXTS parameter
 */
 /**
  * getting set up, this is a chameleon application so it has to do the
  * normal 'application' things plus interpret a request
  */
include( "../../htdocs/chameleon.php" );

//our customized chameleon application just uses the
//session RW mode.
class ServiceInstanceApp extends Chameleon
{
  function ServiceInstanceApp()
  {
    parent::Chameleon();
    $this->moMapSession = new MapSession_RW;
    $this->moMapSession->setTempDir( getSessionSavePath());
  }
}


$oApp =  new ServiceInstanceApp();

/*
 * the current version of the CWC2 protocol that this service instance
 * understands ...
 */
$anServiceVersion = array( 0, 1, 0 );

/*
 * Configuration Parameters are retrieved from the AppContext
 *
 * $szServiceInstancePath - the URL to cwc2.php (this file)
 * $szServerDataPath      - the path to wmsparse data files
 * $bAllowUploadContext   - control access to remote contexts
 * $bAllowUploadTemplate  - control access to remote templates
 * $bCacheTemplate        - control local caching of remote templates
 * $szDefaultLanguage     - the language to use if no others are provided
 * $szDefaultContext      - the context to use if no others are provided
 * $szDefaultTemplate     - the template to use if no others are provided
 * $szContextRoot         - all local contexts must be relative to this
 * $szTemplateRoot        - all local templates must be relative to this
 * $szMaxContextSize      - the maximum size of uploaded contexts
 * $szFontListFile        - a MapServer font file
 * $szSymbolFile          - a MapServer symbol file
 */

// define a new AppContext object and initialize some variables from it
$oCWC2Context = new AppContext( CHAMELEON_CONFIG_PATH."cwc2.xml" );

$szServiceInstancePath = $oCWC2Context->getContextValue( "service_instance_path" );
$szServerDataPath = $oCWC2Context->getContextValue( "server_data_path" );
$bAllowUploadContext = $oCWC2Context->getContextValue( "allow_upload_context" );
$bAllowUploadTemplate = $oCWC2Context->getContextValue( "allow_upload_template" );
$bCacheTemplate = $oCWC2Context->getContextValue( "cache_template" );
$szDefaultLanguage = $oCWC2Context->getContextValue( "default_language" );
$szContextRoot = $oCWC2Context->getContextValue( "context_root" );
$szDefaultContext = $oCWC2Context->getContextValue( "default_context" );
if (strlen(trim($szDefaultContext)) != 0)
{
    $szDefaultContext = $oApp->resolvePath2( $szDefaultContext, $szContextRoot);
}
$szTemplateRoot = $oCWC2Context->getContextValue( "template_root" );
$szDefaultTemplate = $oCWC2Context->getContextValue( "default_template" );
if (strlen(trim($szDefaultTemplate)) != 0)
{
    $szDefaultTemplate = $oApp->resolvePath2( $szDefaultTemplate, $szTemplateRoot);
}
$szMaxContextSize = $oCWC2Context->getContextValue( "context_maxsize" );
$szFontListFile = $oCWC2Context->getContextValue( "fontlist_file" );
$szSymbolFile = $oCWC2Context->getContextValue( "symbol_file" );

/*
 * test cases
 *
 * All tests to be performed in upper, lower and mixed case versions
 *
 * Test:
 *   EMPTY QUERY STRING
 * Expected Result:
 *   Service Exception
 * 
 * Test:
 *   invalid REQUEST
 * Expected Result
 *   Service Exception
 * 
 * Test:
 *   valid REQUEST=GetCapabilities
 *   invalid version (incomplete, badly formatted ...)
 * Expected Result:
 *   Service Exception
 *
 * Test:
 *   valid REQUEST=GetCapabilities
 *   valid VERSION=0.1.0
 * Expected Result:
 *   capabilities document
 *
 * with templateroot set
 *
 *
 * without templateroot set
 *
 *
 * with contextroot set
 *
 *
 * without contextroot set
 *
 *
 *
 */
 
 ///////////////////////////////////////////////////////////////////
 //
 // TODO: SESSION CACHING OF ALL SERVICE INSTANCE PARAMETERS
 //
 ///////////////////////////////////////////////////////////////////
 
/*
 * pre-process the GET variables to make uppercase versions of the keys
 * available
 */
$_GET = array_merge( $_GET, array_change_key_case( $_GET, CASE_UPPER) );

/** 
 * SERVICE handling
 * 
 * MANDATORY - generate a service exception if missing or not equal to
 *             VCG
 */
if (isset($_GET['SERVICE']))
{
    if (strcasecmp($_GET['SERVICE'], "VCG" ) != 0)
    {
        generateServiceException( "Invalid SERVICE (".$_GET['SERVICE'].
                                  ") requested, must be set to VCG." );
    }
}
else
{
    generateServiceException( "Missing mandatory SERVICE parameter" );
}


/**
 * REQUEST handling
 *
 * MANDATORY - generate service exception if missing ...
 */
if (isset($_GET['REQUEST']))
{
    $_GET['REQUEST'] = strtoupper($_GET['REQUEST']);
    //the statement inside the switch traps a possible error
    //if the value of $_GET['REQUEST'] is 0, all strings
    //automatically match 0 and so the first case would be
    //called rather than the default case.
    switch ($_GET['REQUEST'] === 0 ? '' : $_GET['REQUEST'])
    {
        case "GETCAPABILITIES":
            echo "GetCapabilities<BR>";
            break;
        case "GETAPPLICATION":
            //echo "GetApplication<BR>";
            break;
        default:
            $szMessage  = "REQUEST parameter invalid: ";
            $szMessage .= $_GET['REQUEST']." specified, should be one of ";
            $szMessage .= " GETCAPABILITIES or GETAPPLICATION.";
            generateServiceException( $szMessage );
            break;
    }       
}
else
{
    generateServiceException( "Missing mandatory REQUEST parameter" );
}

//if we get to here, the request is valid
 
/**
 * VERSION handling
 *
 * MANDATORY - generate service exception if missing
 */
if (isset($_GET['VERSION']))
{
    $bValidVersion = false;
    
    //version validation
    $szVersionRegEx = "/\d+.\d+.\d+/i";
    if (preg_match( $szVersionRegEx, $_GET['VERSION'] ))
    {
        $anUserVersion = explode( ".", $_GET['VERSION'] );
        
        //compare to current version
        if ($anUserVersion[0] > $anServiceVersion[0])
        {
            $szMessage = "Requested VERSION ".$_GET['VERSION'];
            $szMessage .=" major version number (";
            $szMessage .= $anUserVersion[0].") exceeds Service Instance ";
            $szMessage .= "major version number ".$anServiceVersion[0];
            $szMessage .= " .Service Instance VERSION is ";
            $szMessage .= implode( ".", $anServiceVersion );
            generateServiceException( $szMessage );
        }
        else if ($anUserVersion[0] == $anServiceVersion[0])
        {
            if ($anUserVersion[1] > $anServiceVersion[1])
            {
                $szMessage = "Requested VERSION ".$_GET['VERSION'];
                $szMessage .=" minor version number (";
                $szMessage .= $anUserVersion[1].") exceeds Service Instance ";
                $szMessage .= "minor version number ".$anServiceVersion[1];
                $szMessage .= " .Service Instance VERSION is ";
                $szMessage .= implode( ".", $anServiceVersion );
                generateServiceException( $szMessage );
            }
            else if ($anUserVersion[1] == $anServiceVersion[1])
            {
                if ($anUserVersion[2] > $anServiceVersion[2])
                {
                    $szMessage = "Requested VERSION ".$_GET['VERSION'];
                    $szMessage .=" sub version number (";
                    $szMessage .= $anUserVersion[2].") exceeds Service ";
                    $szMessage .= "Instance sub-version number ";
                    $szMessage .= $anServiceVersion[2];
                    $szMessage .= " .Service Instance VERSION is ";
                    $szMessage .= implode( ".", $anServiceVersion );
                    generateServiceException( $szMessage );
                }
            }
        }
    }
    else
    {
        $szMessage = $_GET['VERSION']." is an invalid format for the ";
        $szMessage .= "VERSION parameter,must be in n.n.n format";
        generateServiceException( $szMessage );
    }
}
else
{
    generateServiceException( "Missing mandatory VERSION parameter" );
}

//if we get to here, the version is valid

/**
 * LANGUAGES and TEMPLATES test for existance
 */
if (isset($_GET['LANGUAGES']) && 
    $_GET['LANGUAGES'] != "" &&
    !isset($_GET['TEMPLATES']))
{
    $szMessage = "Invalid request. LANGUAGES specified without ";
    $szMessage .= "TEMPLATES. There must be an equal number of LANGUAGES ";
    $szMessage .= "and TEMPLATES in the request.";
    generateServiceException( $szMessage );
}

if (isset($_GET['TEMPLATES']) && 
    $_GET['TEMPLATES'] != "" &&
    !isset($_GET['LANGUAGES']))
{
    $szMessage = "Invalid request. TEMPLATES specified without ";
    $szMessage .= "LANGUAGES. There must be an equal number of LANGUAGES ";
    $szMessage .= "and TEMPLATES in the request.";
    generateServiceException( $szMessage );
}
//if we get here then we know that a valid combination of LANGUAGES and
//TEMPLATES were specified ...

/**
 * LANGUAGES handling
 *
 * OPTIONAL, if set, then it is mandatory to specify TEMPLATES and there
 * must be the same number of templates as languages
 */
$nLanguages = 0;
if (isset($_GET['LANGUAGES']))
{
    $aszLanguages = explode( ",", $_GET['LANGUAGES'] );
    $nLanguages = count( $aszLanguages );
    
    $szLanguageRegEx = "/[a-za-z]-[A-ZA-Z]/i";
    //validate that the languages match the xx-XX pattern
    for($i=0; $i<$nLanguages; $i++)
    {
        if (!preg_match( $szLanguageRegEx, $aszLanguages[$i] ))
        {
            $szMessage = $aszLanguages[$i]." is an invalid language ";
            $szMessage .= "specifier, it must be in the format xx-XX";
            generateServiceException( $szMessage );
        }
    }

}

//if we get here, the languages are valid

/**
 * TEMPLATES handling
 *
 * OPTIONAL, if set, then it is mandatory to specify LANGUAGES and there
 * must be the same number of templates as languages
 */
$nTemplates = 0;
if (isset($_GET['TEMPLATES']))
{
    $aszTemplates = explode( ",", $_GET['TEMPLATES'] );
    $nTemplates = count( $aszTemplates );
    
    //validate the templates
    for($i=0; $i<$nTemplates; $i++)
    {
        $aszTemplates[$i] = urldecode($aszTemplates[$i]);
        //test to see if it is a URL or a relative path
        if (strtoupper(substr($aszTemplates[$i], 0, 7)) != "HTTP://")
        {
            //not a URL, test file system path ...
            if (!$oApp->isAbsolutePath($aszTemplates[$i]))
            {    
                $aszTemplates[$i] = $oApp->resolvePath2($aszTemplates[$i],
                                                        $szTemplateRoot );
            }
            if (strcasecmp($szTemplateRoot,
                substr($aszTemplates[$i], 0, strlen($szTemplateRoot))) != 0)
            {
                $szMessage = "Access denied to template $szTemplateRoot";
                generateServiceException( $szMessage );
            }
        }
        else if (!$bAllowUploadTemplate)
        {
            $szMessage = "Invalid template specified.  This service instance ";
            $szMessage .= "does not allow use of remote templates.";
            generateServiceException( $szMessage );
        }
        else if ($bCacheTemplate)
        {
            //download template and cache it in the user session
            $aszTemplate = @file($aszTemplates[$i]);
            if ($aszTemplate === false)
            {
                $szMessage = "Invalid template specified (".$aszTemplates[$i].
                             "), it could not be opened.  Perhaps the URL is".
                             " invalid?";
                generateServiceException( $szMessage );
            }
            $szCachedTemplate = tempnam(getSessionSavePath(), uniqid('')).".html";

            $fh = fopen($szCachedTemplate, "w");

            $nbLine = count($aszTemplate);

            for($i=0; $i<$nbLine; $i++)
            {
                fwrite($fh, $aszTemplate[$i]);
            }
            fclose($fh);
            $aszTemplates[$i] = $szCachedTemplate;
        }
    }
}

//if we get here, the templates are valid

/**
 * validate that there are as many languages as templates
 */

if ($nLanguages != $nTemplates)
{
    $szMessage = "Invalid request.  There must be the same number of ";
    $szMessage = "LANGUAGES as TEMPLATES.  $nLanguages were found and ";
    $szMessage = "$nTemplates TEMPLATES were found.";
    generateServiceException( $szMessage );
}

/**
 * CONTEXTS handling
 * OPTIONAL, if set then it contains a comma separated list of contexts to
 * load.  Validate that the contexts are either URLs or are relative to
 * the context_root ...
 * If not set then use the default context ...
 * And if that is not set, then generate a service exception
 */
$nContexts = 0;
if (isset($_GET['CONTEXTS']))
{
    $aszContexts = explode( ",", $_GET['CONTEXTS'] );
    $nContexts = count($aszContexts);
    for ($i=0; $i<$nContexts; $i++)
    {
        $szContext = urldecode($aszContexts[$i]);
        if (strtoupper(substr($szContext, 0, 7)) == "HTTP://")
        {
            if ($bAllowUploadContext)
            {
                $aszFile = @file($szContext);
                if ($aszFile === FALSE)
                {
                    $szMessage = "Invalid context specified.  ".
                                 $szContext." could not be opened. ".
                                 "Perhaps the URL is invalid?";
                    generateServiceException( $szMessage );
                }
                $szCachedContext = tempnam(getSessionSavePath(), uniqid('')).
                                                                       ".cml";
                $fh = fopen($szCachedContext, "w");
                $nbLine = count($aszFile);

                for($j=0; $j<$nbLine; $j++)
                  fwrite($fh, $aszFile[$j]);

                fclose($fh);

                $szContext = $szCachedContext;
            }
            else
            {
                $szMessage = "Invalid context specified. This service ";
                $szMessage .= "instance does not allow use of remote ";
                $szMessage .= "contexts.";
                generateServiceException( $szMessage );
            }
        }
        else
        {
            //test for empty contexts=
            if (strlen(trim($szContext)) == 0)
            {
                $szMessage = "Invalid CONTEXTS parameter, no contexts provided.";
                generateServiceException( $szMessage );
            }
            
            //local access to contexts have to be validated to context_root
            if (!$oApp->isAbsolutePath($szContext))
            {    
                $szContext = $oApp->resolvePath2($szContext,
                                                  $szContextRoot );
            }
            if (strcasecmp($szContextRoot,
                substr($szContext, 0, strlen($szContextRoot))) != 0)
            {
                $szMessage = "Access denied to context $szContextRoot";
                generateServiceException( $szMessage );
            }
        }
        $aszContexts[$i] = $szContext;
    }
}
else
{
    //try loading the default context
    if ($szDefaultContext != "")
    {
        $nContexts = 1;
        $aszContexts = array( $szDefaultContext );
    }
    else
    {
        $szMessage = "Invalid request.  No CONTEXTS parameter was provided ";
        $szMessage .= "and the Service Instance has no default context";
        generateServiceException( $szMessage );
    }
}

/**
 * handle PROJECTION request
 *
 * OPTIONAL, if supplied, use this PROJECTION for all contexts
 */
$szProjection = '';
if (isset($_GET['SRS']))
{
    $szEPSGRegEx = "/epsg:\d+/i";
    if (preg_match( $szEPSGRegEx, $_GET['SRS'] ))
    {    
        //mapserver expects EPSG to be in lower case
        $szProjection = strtolower(urldecode($_GET['SRS']));
    }
    else
    {
        $szMessage = "requested SRS (".
                     $_GET['SRS'].
                     ") is invalid.  It must be in ".
                     "the form EPSG:XXXXX.";
        generateServiceException( $szMessage );
    }
}

/**
 * handle BBOX
 *
 * optional, if provided then it is four values separated by commas and
 * is assumed to be in the units of the defined projection (either the
 * supplied SRS parameter or the SRS of the last context loaded)
 */
$szExtents = "";
if (isset($_GET['BBOX']))
{
    $szExtents = urldecode( $_GET['BBOX'] );
    $anExtents = split(",", $szExtents);
    if (count($anExtents) != 4)
    {
        $szMessage = "Invalid BBOX ($szExtents), it contains ".
                     count($anExtents)." values and should contain ".
                     "4 comma separated numbers that are in the same ".
                     "units as the SRS for the loaded contexts.";
        generateServiceException( $szMessage );
    }
}

/**
 * test for a LANGUAGE in the URL, if not set then set one based on either the
 * default language or the first of the requested languages
 */
$szCurrentLanguage = "";
if (isset($_POST['LANGUAGE']))
{
    $szCurrentLanguage = $_POST['LANGUAGE'];
}
else if (isset($_GET['LANGUAGE']))
{
    $szCurrentLanguage = $_GET['LANGUAGE'];
}

if ($szCurrentLanguage == "")
{
    if ($nLanguages > 0)
        $szCurrentLanguage = $aszLanguages[0];
    else
        $szCurrentLanguage = $szDefaultLanguage;
}
$oApp->mszCurrentLanguage = $szCurrentLanguage;

//add regional templates from the supplied languages and templates
if ($nLanguages > 0)
{
    for ($i=0; $i<$nLanguages; $i++)
    {
        $oApp->CWCAddRegionalTemplate( $aszLanguages[$i], $aszTemplates[$i]);
    }
}
else
{
    $oApp->CWCAddRegionalTemplate( $szDefaultLanguage, $szDefaultTemplate); 
}

$oApp->CWCInitialize( $szDefaultTemplate ); //, $szMapFile  );

//this will happen every time unless we trap it somehow :(
    
if ($nContexts > 0 && !$oApp->isVarSet( "NAV_CMD" ))
{
    $oApp->moMapSession->oMap->web->set( "imageurl", $_SESSION["gszTmpWebPath"]);
    $oApp->moMapSession->oMap->web->set( "imagepath", $_SESSION["gszTmpImgPath"]);
    $oApp->moMapSession->oMap->selectOutputFormat( $_SESSION['gszImgType'] );
    $oApp->moMapSession->oMap->setFontSet( $szFontListFile );
    $oApp->moMapSession->oMap->setSymbolSet( $szSymbolFile );
    for ($i=0;$i<$nContexts;$i++)
    {
        $oApp->moMapSession->oMap->loadMapContext( $aszContexts[$i] );
    }
    if ($szProjection != "")
    {
        $oApp->moMapSession->oMap->setProjection( "init=".$szProjection, MS_TRUE );
    }
    if ($szExtents != "")
    {
        $oApp->moMapSession->oMap->setExtent( $anExtents[0], $anExtents[1],
                                              $anExtents[2], $anExtents[3] );
    }
}

$oApp->CWCExecute();

function generateServiceException( $szMessage )
{
    $uri = "http";
    if (isset($_SERVER['HTTPS']) && strcasecmp($_SERVER['HTTPS'], "on") == 0)
        $uri .= "s";
    $uri .= "://";
    $uri .= $_SERVER['SERVER_NAME'];
    if (isset($_SERVER['ORIG_PATH_INFO']))
    {
        $uri .= $_SERVER['ORIG_PATH_INFO'];
    }
    else
    {
      $uri .= $_SERVER['PATH_INFO'];
    }
    echo "request URI was $uri<BR>";
    echo "query string was ".$_SERVER['QUERY_STRING']."<BR>";
    echo $szMessage;
    exit;
}
?>